/*
 * ***** BEGIN LICENSE BLOCK *****
 * Zimbra Collaboration Suite Web Client
 * Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc.
 *
 * The contents of this file are subject to the Common Public Attribution License Version 1.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at: https://www.zimbra.com/license
 * The License is based on the Mozilla Public License Version 1.1 but Sections 14 and 15
 * have been added to cover use of software over a computer network and provide for limited attribution
 * for the Original Developer. In addition, Exhibit A has been modified to be consistent with Exhibit B.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.
 * See the License for the specific language governing rights and limitations under the License.
 * The Original Code is Zimbra Open Source Web Client.
 * The Initial Developer of the Original Code is Zimbra, Inc.  All rights to the Original Code were
 * transferred by Zimbra, Inc. to Synacor, Inc. on September 14, 2015.
 *
 * All portions of the code are Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2014, 2015, 2016 Synacor, Inc. All Rights Reserved.
 * ***** END LICENSE BLOCK *****
 */


/**
 * Utility function (NOT object!) that loads external files in a specified
 * order (thus it'll work fine if scripts have dependencies, you just need to
 * pass them all in the includes array).  If a callback is given, it will be
 * called when all the scripts in the includes array (and possibly dependencies
 * generated by them using document.write) are loaded.
 *
 * Note that scripts are loaded asynchronously -- this function returns
 * immediately, but there's no guarantee that scripts have finished loading by
 * that time.  DO pass a callback if you want to execute code that rely on
 * those scripts.
 *
 * @param {array}	includes 	An array of strings or object; each is the URL to
 *                              some script OR an object with a required src property
 *                              and an optional id property. The id, if specified,
 *                              will be attached to the SCRIPT tag so that the caller
 *                              can access the element later.
 * @param {string}	baseurl 	the URL that will be prepended to scripts with a
 *                            relative address (not starting with http://, ftp:// or /)
 * @param {AjxCallback}	callback the callback will be called when all scripts were processed
 *
 * @author Mihai Bazon
 * 
 * @private
 */
AjxInclude = function(includes, baseurl, callback, proxy) {
	var head = document.getElementsByTagName("head")[0];

	function loadNextScript(script) {
		if (AjxEnv.isIE && script && !/loaded|complete/.test(script.readyState))
			return;
		if (script) {
			// Clear the event handler so IE won't leak.  (Did you know
			// that Microsoft knew about the mem. leak bug in 1998
			// and by the end of 2005 it's still not fixed? :-p
			// http://www.bazon.net/mishoo/home.epl?NEWS_ID=1281 )
			script[AjxInclude.eventName] = null;
		}
		window.status = "";
		if (includes.length > 0) {
            var object = includes.shift();
			var fullurl = typeof object == "string" ? object : object.src;
			var orig = fullurl;
			if (!/^((https?|ftps?):\x2f\x2f|\x2f)/.test(fullurl)) {
				if (baseurl)
					fullurl = baseurl + fullurl;
				if (cacheKillerVersion)
					fullurl += "?v=" + cacheKillerVersion;
			} else if (proxy && fullurl.indexOf('/') != 0) {
 				// fully qualified URL-s will go through our proxy
 				fullurl = proxy + AjxStringUtil.urlEncode(fullurl);
			}
			var script = document.createElement("script");
			var handler = AjxCallback.simpleClosure(loadNextScript, null, script);
			if (script.attachEvent) {
				script.attachEvent("onreadystatechange", handler);
				script.attachEvent("onerror", handler);
			}
            else if (script.addEventListener) {
				script.addEventListener("load", handler, true);
				script.addEventListener("error", handler, true);
			}
			script.type = "text/javascript";
			script.src = fullurl;
			window.status = "Loading script: " + orig;

            var oscript = null;
            if (object.id) {
                script.id = object.id;
                oscript = document.getElementById(object.id);
            }
            if (oscript) {
                oscript.parentNode.replaceChild(script, oscript);
            }
            else {
                head.appendChild(script);
            }
		} else if (includes.length == 0) {
			script = null;
			head = null;
			if (callback)
				callback.run();
		}
	};

	loadNextScript(null);
};
