#   Copyright (c) 2020 Veritas Technologies LLC. All Rights Reserved
#
#   VERITAS,  the  VERITAS Logo and all other  VERITAS  product names and
#   slogans are trademarks  or registered trademarks of  VERITAS Software
#   Corporation.  VERITAS and the VERITAS Logo Reg.  U.S. Pat. & Tm. Off.
#   Other product names and/or slogans mentioned herein may be trademarks
#   or registered trademarks of their respective companies.
#
#   The Licensed Software and Documentation are  deemed to be  commercial
#   computer software  and  commercial computer software documentation as
#   defined in FAR Sections 12.212 and DFARS Section 227.7202.
#
#   $Id: RALUSCommon.pm,v 1.66 2013/12/31 05:07:04  $
#   $Source: R:/Source/cvsvault/unixinst/RALUSCommon.pm,v $
#
###########################################################################

package RALUSCommon;

require VxIF::Utils;

################################################
# Product specific Defines                     #
################################################
$CONF{RALUSCFG}="/etc/VRTSralus/ralus.cfg";
$CONF{RALUSINSTALLDIR}="/opt/VRTSralus";
$CONF{RALUSCFGFILENAME}="ralus.cfg";
$CONF{RALUSADVMEDSRVRKEY}="Software\\Veritas\\Backup Exec For Windows\\Backup Exec\\Engine\\Agents\\Agent Directory List";

$CFG{PREVRALUSMDSRVRS}=[];
$CFG{CERTIFICATEINFO}=[];
$CFG{ORACLELINUXINFO}=[];
$CFG{RALUSINFO}=[];

################################################
# Product Info Interfaces                      #
################################################

#
# Get the localized product title for this product. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#
# Return: the complete localized title for this product.
#
sub get_product_title (\%) {
 my ($self,$ctx) = @_;
 
 $$ctx{LOGGER}->entering("RALUSCommon::get_product_title");
 my $title = Utils::_tr("Backup Exec Agent for Linux and Unix Servers");

 return ($title);
}

# Get the product version number (i.e. 4.1.3.10). For some products, the product version
# number may be different from the main package version number. The main package version 
# number is optional and will be absent for the current product version. If present, 
# the main package version will be different the current main package version. 
# The reason is that for product upgrade and product downgrade, VxIF needs a list of 
# packages associated with a particular product version so they can properly removed.
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the main package version; this parameter is optional;
#
# Return: product version number
#
sub get_product_version (\%;$) {
 my ($self, $ctx, $pkg) = @_;
 my $trg_upi = $ctx->get_target_upi();
 
 $$ctx{LOGGER}->entering("RALUSCommon::get_product_version");
 # Need to have placeholder "_RALUS_PKG_VERSION_" without the underscores
 # always in place of the actual VRTSralus return value so that the builder 
 # will update during Install Media Creation Process.
 if ($trg_upi eq "VRTSralus") { 
 	return "22.0.1193-1605";
 } # elsif ($trg_upi eq "VRTSvxmsa") {
 # 	return "VXMSPKGVERSION"
 #}
}

#
# Get the localized product description for this product. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: localized product description.
#
sub get_product_description (\%) {
 my ($self, $ctx) = @_;
 
 $$ctx{LOGGER}->entering("RALUSCommon::get_product_description");
 return Utils::_tr("Backup Exec Agent for Linux and Unix Servers."); 
}

#
# Get the localized product copyright information. 
#
# This interface is optional. If product copyright information is not provided, 
# VxIF will use the generic VERITAS copyright information. If product copyright 
# information is given, VxIF will use it instead of the generic VERITAS 
# copyright information.
#
# Input: 1) the reference to the installation context;
#
# Return: the localized product copyright information.
#
#sub get_product_copyright (\%) {
#}

#
# Determine whether an older or a newer version of the product can be configured using
# the current version of the product script. This interface will be invoked if an older 
# or a newer version of this product exist on the destination host.
#
# This interface is option. If not implemented, VxIF assumes any existing version of this
# product is not configurable using the current version of the product script.
#
# Input: 1) the reference to the installation context;
#        2) the version of the product currently installed on the destination host;
#
# Return: 1 if this version can be configured using the current version of the product script. 
#         Otherwise, return 0;
#
sub is_product_configurable (\%$) {

 my ($self, $ctx) = @_;
 
 $$ctx{LOGGER}->entering("RALUSCommon::is_product_configurable");
 
 return 1;
}

#
# Determine whether an older version of the product can be upgraded to the current version. 
# This interface will be invoked if an older version of this product exist on the
# destination host. If this product is upgradable, VxIF will try to perform a product
# upgrade rather than new product install. VxIF will invoke product upgrade interfaces,
# such as pre_product_upgrade(), post_product_upgrade(), etc instead of the new installation
# interfaces.
#
# This interface is option. If not implemented, VxIF assumes any existing version of this
# product is not upgradable.
#
# Product upgrade may not be supported on all VxIF platforms. If product upgrade is not
# supported by a VxIF platform, VxIF will first attempt remove the existing product and
# perform a new install with the current version. Product group should implement 
# pre_product_upgrade() to properly backup the data, and implement post_product_upgrade() 
# to properly migrate the data over after the packages are installed.
#
# Input: 1) the reference to the installation context;
#        2) the version of the product currently installed on the destination host;
#
# Return: 1 if this version can be upgraded to the current version. Otherwise, return
#         0;
#
sub is_product_upgradable (\%$) {
 my ($self, $ctx) = @_;
 
 $$ctx{LOGGER}->entering("RALUSCommon::is_product_upgradable");
 
 return 1;
}

#
# Determine whether a newer version of the product can be downgraded to the current version. 
# This interface will be invoked if an newer version of this product exist on the
# destination host. If this product is downgradable, VxIF will try to perform a product
# downgrade rather than new product install. VxIF will invoke product downgrade interfaces,
# such as pre_product_downgrade(), post_product_downgrade(), etc instead of the new installation
# interfaces.
#
# This interface is option. If not implemented, VxIF assumes any existing version of this
# product is not downgradable.
#
# Product downgrade may not be supported on all VxIF platforms. If product downgrade is not
# supported by a VxIF platform, VxIF will first attempt remove the existing product and
# perform a new install with the current version. Product group should implement 
# pre_product_downgrade() to properly backup the data, and implement post_product_downgrade() 
# to properly migrate the data over after the packages are installed.
#
# Input: 1) the reference to the installation context;
#        2) the version of the product currently installed on the destination host;
#
# Return: 1 if this version can be downgraded to the current version. Otherwise, return
#         0;
#
#sub is_product_downgradable (\%$) {
#}

#
# Get the product train directory. This is similar to getting the {TRAINDIR} in IFRTA and IFRTB.
# This interface, if implemented, will be used to determine the location of the product packages.
#
# This interface is optional. If not implement, VxIF to determine the packages location from
# the get_package_location() interface. This interface must be implemented in the absence of 
# get_package_location() interface.
#
# Input: 1) the reference to the installation context;
#        
# Return: the product directory.
#
#sub get_product_train_dir (\%) {
#}

#
# Return the different installation modes for this product. For example, "Complete", 
# "Typical", etc. 
#
# This interface is optional and must be omitted if there is no installation modes.
#
# This interface shall return the reference of a hash with mode names as the keys
# with the following properties.
#
# {DESCRIPTION} - the localized mode description; This property is mandatory;
# {NAME}    	- the localized mode alternative name; This property is mandatory;
#
#
# For example,
#
# $modes{Typical}{NAME} = "Typical";
# $modes{Typical}{DESCRIPTION} = "Install the Foo service only.";
# $modes{Complete}{NAME} = "Complete";
# $modes{Complete}{DESCRIPTION} = "Install Foo service and client.";
# $modes{Optional}{NAME} = "Optional";
# $modes{Optional}{DESCRIPTION} = "Install optional debug files for Foo.";
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %CFG hash;
#
# Return: the reference of a hash with the installation modes as the keys and the localized
#         description as the values.
#
#sub get_product_installation_modes (\%\%) {
#}


#
# Get the order in which the product installation modes will be displayed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %CFG hash;
#
# Return: the list of product installation modes in the order in which to be displayed.
#
#sub get_product_installation_modes_ordering (\%\%) {
#}


################################################
# System Validation Interfaces                 #
################################################

#
# Determine whether the destination host OS is supported.
#
# Input: 1) the reference to the installation context;
#        2) host name;
#        3) host OS;
#        4) host OS version number;
#        5) host OS architecture;
#
# Return: 1 if the given OS is supported. Otherwise, return 0;
#
#sub is_os_supported (\%$$$$) {
#}

#
# Check to make sure the system names are valid.
#
# This interface is optional. By default, VxIF do some generic checking, such as making
# sure the name contains the valid characters. Apart from that it is up the each individual
# product to impose additional contraints.
#
# Input: 1) the reference to the installation context;
#        2) the array of systems that will be installing this product;
#
# Return: 1 if the system names are valid; 0 otherwise.
#
#sub check_system_names (\%@) {
#}


################################################
# Common Package and Patch Interfaces          #
################################################

#
# Some of the interfaces will likely be implemented by the train coordination team. For example, 
# the train coordination team can determine whether a product package can be compressed 
# to reduce space on a train CD.
#

#
# Check for any version of this product installed on the target host and return its version number.
# Some OS such as OSF1 (TRU64) does not have a consistent and reliable way of checking for existing
# product status. It entirely depended on customized subset control programs (SCP) to check for 
# product installation status. With this limitation, it is up the the product itself to determine
# the installation status and feed this information back to VxIF.
#
# This interface is optional. If this interface is implemented, VxIF will use it instead of 
# get_instpkgvers().
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: the installed version number or null if the product does not exist on target host.
#
#sub get_installed_package_version (\%$) {
#}

#
# Check for installed dependents of the given package and return the hash containing the 
# dependents' information. If this interface is implemented, VxIF will use it instead 
# of get_pkgdep().
#
# This interface shall return the reference of a hash with package names as the keys
# with the following properties.
#
# {VERSION}     - the dependent version installed on target host; This is a mandatory attribute;
# {DESCRIPTION} - the dependent package description; This is an optional attribute;
#
# For example,
#
# $deps{VRTSfoo}{VERSION} = "1.2.3.4";
# $deps{VRTSfoo}{DESCRIPTION} = "This package provides foo";
# $deps{VRTSbar}{VERSION} = "2.3.4.5";
# $deps{VRTSbar}{DESCRIPTION} = "This package provides bar";
#
# This interface is optional. If it is implemented, VxIF will use it instead of get_pkgdep().
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: the reference to the dependents information hash.
#
#sub check_for_package_dependents (\%$) {
# my ($self, $ctx) = @_;
# 
# $$ctx{LOGGER}->entering("RALUSCommon::check_for_package_dependents);
#
# $deps{VRTSfoo}{VERSION} = "1.2.3.4";
# $deps{VRTSfoo}{DESCRIPTION} = "This package provides foo"; 
#}

#
# Get the order in which the product packages will be installed. This interface
# is used for backward compatibility support only. Some old product scripts
# may not set the {DEPS} correctly. Basically, $PROD{<RALUS>}{PKGS} instructs the
# installer the order in which to install the product packages.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the list of product packages in the order in which to be installed.
#
#sub get_product_packages_ordering (\%$$$;$) {
#}

#
# Get the name of the main package. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the selected installation mode if one is available;
#
# Return: the name of the main package.
# 
#sub get_product_main_package (\%$) {
#}

#
# For a given package, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: Return the directory path where the given package is located.
#
#sub get_package_location (\%$) {
#}

#
# For a given patch name, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>/patches.
#
# Input: 1) the reference to the installation context;
#        2) patch name;
#
# Return: Return the directory path where the given patch is located.
#
#sub get_patch_location (\%$) {
#}

#
# Indicate whether a given package is in a compressed file. 
#
# This interface is optional. By default, VxIF assumes a given package is not compressed.
#
# Input: 1) the reference to the installation context;
#        2) package name; 
#
# Return: 0 if the given package is not compressed;
#         1 if the given package is compressed using gunzip (.tar.gz);
#         2 if the given package is compressed in other format; If the package
#           is compressed in other format, product must implement get_package_decompressor() 
#           method to provide a custom decompressor;
#
#sub get_package_compression_type (\%$) {
#}

#
# Indicate whether a given patch is in a compressed file. 
#
# This interface is optional. By default, VxIF assumes a given patch is not compressed.
#
# Input: 1) the reference to the installation context;
#        2) patch name; 
#
# Return: 0 if the given patch is not compressed;
#         1 if the given patch is compressed using gunzip (.tar.gz);
#         2 if the given patch is compressed in other format; If the patch
#           is compressed in other format, product must implement get_patch_decompressor() 
#           method to provide a custom decompressor;
#
#sub get_patch_compression_type (\%$) {
#}

#
# Return the compressed filename, without the path, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed filename, without the path, for the given package.
#
#sub get_package_compressed_filename (\%$) {
#}

#
# Return the compressed filename, without the path, for the given patch.
#
# This interface is mandatory if any product patches are compressed. This interface 
# must be absent all product patches are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) patch name;
#
# Return: the compressed filename, without the path, for the given patch.
#
#sub get_patch_compressed_filename (\%$) {
#}

#
# Return the compressed file size, in kilobytes, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed file size, in kilobytes, for the given package.
#
#sub get_package_compressed_filesize (\%$) {
#}

#
# Return the compressed file size, in kilobytes, for the given patch.
#
# This interface is mandatory if any product patches are compressed. This interface 
# must be absent all product patches are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed file size, in kilobytes, for the given patches.
#
#sub get_patch_compressed_filesize (\%$) {
#}

#
# This is the package decompression callback. Given a package name, return the 
# reference of the interface to handle the decompression of the compressed package file.
#
# This interface is mandatory if the package compression type is 2.
#
# VxIF will pass the following parameters into the callback in the order as specified.
# 
# 1) the reference to the installation context;
# 2) the package name;
# 3) the compressed file name associated with this package;
# 4) the directory where the compressed file is located;
# 5) the directory to uncompress the files into;
#
# This callback shall return 1 if the given package has been successfully uncomressed;
# otherwise, 0 shall be returned.
#
# Here's an example on how to customize the decompression operation.
#
# sub get_package_decompressor (\%$$$$$) {
#   my ($self,$ctx,$pkg,$cfile,$cdir, $ucdir) = @_;
#   my $my_decompressor;
#   if ($pkg eq "package1") {
#      $my_decompressor = sub (\%$$$$) {
#        my ($ctx, $pkg, $cfile, $cdir, $ucdir) = @_;
#        ... customize the decompression for package1 here
#      };
#   } elsif ($pkg eq "package2") {
#      $my_decompressor = sub (\%$$$) {
#        my ($ctx, $pkg, $cfile, $cdur, $ucdir) = @_;
#        ... customize the decompression for package2 here
#      };
#   }
#
#   return \&$my_decompressor;
# }
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the compressed file name associated with this pacakge;
#        4) the directory where the compressed file is localted;
#        5) the directory to uncompress the files into;
#
# Return: the reference of the customized package decompressor.
#
#sub get_package_decompressor (\%$$$$) {
#}

#
# This is the patch decompression callback. Given a patch name, return the 
# reference of the interface to handle the decompression of the compressed patch file.
#
# This interface is mandatory if the patch compression type is 2.
#
# VxIF will pass the following parameters into the callback in the order as specified.
# 
# 1) the reference to the installation context;
# 2) the patch name;
# 3) the compressed file name associated with this patch;
# 4) the directory where the compressed file is located;
# 5) the directory to uncompress the files into;
#
# This callback shall return 1 if the given patch has been successfully uncomressed;
# otherwise, 0 shall be returned.
#
# Here's an example on how to customize the decompression operation.
#
# sub get_patch_decompressor (\%$$$$) {
#   my ($self,$patch,$cfile,$cdir, $ucdir) = @_;
#   my $my_decompressor;
#   if ($patch eq "patch1") {
#      $my_decompressor = sub (%$$$$) {
#        my (%ctx, $pkg, $cfile, $ucdir) = @_;
#        ... customize the decompression for package1 here
#      };
#   } elsif ($patch eq "patch2") {
#      $my_decompressor = sub (%$$$$) {
#        my (%ctx, $pkg, $cfile, $ucdir) = @_;
#        ... customize the decompression for patch2 here
#      };
#   }
#
#   return \&$my_decompressor;
# }
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the compressed file name associated with this patch;
#        4) the directory where the compressed file is localted;
#        5) the directory to uncompress the files into;
#
# Return: the reference of the customized package decompressor.
#
#sub get_patch_decompressor (\%$$$$) {
#}


################################################
# Common Product Installation Interfaces       #
################################################
#
# This method checks to see if anyone is currently using the Port 10000
# which is currently the default port used by Backup Exec and RALUS.
#
# Input: 1) the reference to the installation context;
#
# Return: 1) Port 10000 in use 0) Port 10000 NOT in use. 
#
sub _ralus_is_port_10000_inuse {
        my ($ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_is_port_10000_in_use");

	# Locate netstat and use full path (location differs on different platforms)
        my $netstat = 'netstat';
	if (Utils::vxif_fc("/bin/netstat", $ctx)) {
		$netstat = '/bin/netstat';
	} elsif (Utils::vxif_fc("/usr/bin/netstat", $ctx)) {
		$netstat = '/usr/bin/netstat';
	}

	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

        #  The Default Port notation for 'netstat' is .[port number] 
        my $be_port = ".10000";
	if ($trg_os_name eq "Linux") {
                $be_port = ":10000";
        }
        # Default Column for Local Address is typically 4th
        my $LAddr_Idx = 4;
        if ($trg_os_name eq "SunOS") {
                $LAddr_Idx = 1;
        }

        # Display message to user that we are now checking for Port 10000
        my $msg = Utils::_tr("Checking for port 10000", 250, 1002);
	Utils::vxif_pbl($msg);
        # Format the correct command on the target machine to check for 
        # Port 10000 and filter with grep and Execute and retrieve Results
        my $command;
        if ($trg_os_name eq "SunOS")
        {
           $command = $netstat . " -anv | grep " . $be_port;
        }
        else
        {
           $command = $netstat . " -atn | grep " . $be_port;
        }
	$$ctx{LOGGER}->info("Check for Port 10000 on Target Server using: $command");
	my ($Results) = Utils::vxif_dor("${command}", "", $ctx);
	my (@lines) = split(/\n/,$Results);
        foreach my $line (@lines) 
	{
		my ($nCount) = 0;
                if (length($line)) 
		{
 			$$ctx{LOGGER}->info("netstat result = $line");

			#my ($proto, $recvQ, $sendQ, $LAddr) = split(/ /,$line);
			my (@items) = split(/ /,$line);
                        foreach my $item (@items)
                        {
                                if ($item ne "")
                                {
                                        # Check if we are at the Local Address Column yet
                                        if ($nCount == ($LAddr_Idx-1))
                                        {
			                        # Local Address using Port 10000 ????
			                        if ($item =~ m/$be_port$/)
			                        {
                                                        $$ctx{LOGGER}->warning("WARNING (V-225-248): Port 10000 in Use!!");
			                                return 1;
                	                        } else {
		        	                        last;
			                        }
                                        } else {
                                                $$ctx{LOGGER}->info("item: $item");
                                                $nCount = $nCount + 1;
                                        }
                                }
                        }
		}
        }
        # Notify user that we have finished checking for Port 10000
	Utils::vxif_pbr($msg, Utils::_tr("Done", 250, 1003));
        
        # Nothing is using Port 10000
        return 0;
}


#
# This method checks for an existing BE GRFS Agent and then attempts
# to read the Media Servers list and then renames the config file and Agent
# so that the Media Servers can be migrated to the new RALUS Agent and the
# GRFS Agent will not startup in future system startups.
#
# This interface is optional for upgrade support for previous generation GRFS agents.
#
# Input: 1) the reference to the installation context;
#
# Return: 1) for success or 0) for failure.
#
sub _ralus_migrate_GRFS_servers {
        my ($ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_migrate_GRFS_servers");

	# Migrate the Media Server entries in the agent.cfg
        # file and rename the file so we don't do it again

	my ($mark_for_del) = 0;
        my $sf = "/etc/bkupexec/agent.cfg";
        my $df = "/etc/bkupexec/agent.cfg.migrated";
	Utils::vxif_dor("cp $sf $df", "", $ctx);
        if ( Utils::vxif_fc("$df", $ctx) ) 
	{
		$$ctx{LOGGER}->info("Backed up the original GRFS agent config file $sf to $df.");
		$mark_for_del = 1;
	}

        if ( Utils::vxif_fc("$sf", $ctx) )
        {
               	my $tmp_sf = "$$ctx{TMPPATH}/agent.cfg";

		$$ctx{LOGGER}->info("Copy GRFS agent config file $sf to $tmp_sf.");
        	Utils::vxif_copy($sf,$tmp_sf,$ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
                if ( -f "$tmp_sf" )
                {
			$$ctx{LOGGER}->info("Read local GRFS agent config file $tmp_sf.");
		        my ($buffer) = Utils::vxif_readf($tmp_sf);
		        if ($buffer ne "") 
			{
				my (@servers);
		        	my @lines = split(/\n/,$buffer);
			        foreach my $n(0..$#lines) 
				{
			        	my $lookfor = "tell ";
				        if ($lines[$n] =~ m/^($lookfor)/) 
					{
				       		my ($server) = substr($lines[$n],5);
					        chomp($server);
	                                        push(@servers, $server);
						$$ctx{LOGGER}->info("Adding $server to Migrated Server List.");
	                                }
	                        }

				# Add Media Servers to Hash CFG Item
	                        if ($#servers>=0) 
				{
	                        	$CFG{MIGRATEMDSRVRS}=[];
	                                @{$CFG{MIGRATEMDSRVRS}}=@servers;
	                        }
				else
				{
					$$ctx{LOGGER}->info("No Media Servers found in GRFS agent config file.");
				}
	                }
			
			# Remove the file from the TMP directory
			$$ctx{LOGGER}->info("Deleting local copy of the GRFS agent config file: $tmp_sf.");
			Utils::vxif_dol("rm -f $tmp_sf");
	        }
		else
		{
			$$ctx{LOGGER}->error("ERROR (V-225-249): GRFS agent config file failed to be copy to local TMP: $tmp_sf.");
		}	
        }

	if ($mark_for_del == 1)
	{
		# Remove the source file from the target machine since we already backed it up
		$$ctx{LOGGER}->info("Remove the GRFS agent config file from Target: $sf.");
		Utils::vxif_dor("rm -f $sf", "", $ctx);
	}

	return 1;
}

sub _ralus_disable_GRFS_Agent {
        my($self, $ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_disable_GRFS_Agent");

	my($sf,$df);

        # Stop the agent.be process if it is running
        #_comm_stop_processes("agent.be");

	# Stop all threads/processes for "agent.be"
	# we need to filter out the PID which is the 2nd Column
	# Value in the return of the get_pid function call
	my $trg_host = $ctx->get_target_hostname();
	my $host_os_installer = $ctx->get_host_os_installer($trg_host);
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");
        
	my $kill_msg = Utils::_tr("Killing GRFS threads and processes", 250, 1004);
	Utils::vxif_pbl($kill_msg);

	my $command = "ps -ef | grep agent.be | grep -v grep";
	$$ctx{LOGGER}->info("Getting GRFS pids via: $command");
	my ($results) = Utils::vxif_dor($command, "", $ctx );

        # Set Column to retrieve PID for given O/S
        my $PID_col = 5; # set as default
        if ( $trg_os_name eq "AIX" )
        {
                $PID_col = 6;
        }

	my ($success) = 1;
	my (@lines) = split(/\n/,$results);
        foreach my $line (@lines) 
	{
		my ($nCount) = 0;
                if (length($line)) 
		{
 			$$ctx{LOGGER}->info("PS result = $line");

			my (@items) = split(/ /,$line);
			foreach my $item (@items)
			{
				if ($nCount == $PID_col) # get the PID
				{
		 			$$ctx{LOGGER}->info("Pid for GRFS = $item");
					$$ctx{LOGGER}->entering("kill_proc for GRFS agent");
					my $rc = $host_os_installer->kill_proc($ctx, "agent.be", $item);
					if (!$rc)
					{
						$success = 0;
					}
					last;
                		}
				else
				{
					$nCount = $nCount + 1;
				}
			}
		}
        }

	# Display Kill Process completed
	Utils::vxif_pbr($kill_msg, Utils::_tr("Done", 250, 1003));

	if ( $success == 0 )
	{
		$$ctx{LOGGER}->warning("WARNING: Killing GRFS threads and processes failed or timedout.");
	} else {
		$$ctx{LOGGER}->info("Successfully killed GRFS threads and processes.");
	}

        # Rename the agent.be process so it wont load again
        $sf = "/etc/bkupexec/agent.be";
        $df = "/etc/bkupexec/agent.be.disabled";
	$$ctx{LOGGER}->info("Copy $sf to $df to disable GRFS agent.");
        #Utils::vxif_copy($sf, $df, $ctx, $trg_host, $trg_host,"");
        Utils::vxif_dor("cp $sf $df", "", $ctx);
        if ( Utils::vxif_fc("$df", $ctx) )
        {
                # We successful made a backup so Force the removal of the original
                # /etc/bkupexec/agent.be GRFS Agent
		$$ctx{LOGGER}->info("Remove the agent $sf.");
        	Utils::vxif_dor("rm -f $sf", "", $ctx);
        }

        return (_ralus_migrate_GRFS_servers($ctx));
}

sub _ralus_migrate_MediaServers {
	my ($self, $ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_migrate_MediaServers");

	# First check to see if we have the BE GRFS Unix Agent on this machine
        my $sf = $CONF{RALUSCFG};
	if ( !Utils::vxif_fc($sf, $ctx) )
	{
		return;
	}
        
        my $tmp_sf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";

	$$ctx{LOGGER}->info("Copy RALUS config file $sf to $tmp_sf.");
        Utils::vxif_copy($sf,$tmp_sf,$ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
        if ( -f "$tmp_sf" )
        {
	        $$ctx{LOGGER}->info("Read local RALUS config file $tmp_sf.");
	        my ($buffer) = Utils::vxif_readf($tmp_sf);
		if ($buffer ne "") 
		{
			my (@servers);
		       	my @lines = split(/\n/,$buffer);
		        foreach my $n(0..$#lines) 
			{
		       	        my $lookfor = "Agent Directory List";
				if ($lines[$n] =~ m/($lookfor)/) 
				{
                                        my @entry = split(/=/,$lines[$n]); 
                                        if ( $entry[1] ne "" )
                                        {
				                my ($server) = $entry[1];
				                chomp($server);
	                                        push(@servers, $server);
					        $$ctx{LOGGER}->info("Adding $server to Migrated Server List.");
                                        }
	                        }
                        }

			# Add Media Servers to Hash CFG Item
	                if ($#servers>=0) 
			{
	                        @{$CFG{PREVRALUSMDSRVRS}}=@servers;
				$$ctx{LOGGER}->info("Added ($#servers+1) media servers.");
	                }
			else
			{
				$$ctx{LOGGER}->info("No Media Servers found in ralus.cfg file.");
			}
	        }
			
	        # Remove the file from the TMP directory
	        $$ctx{LOGGER}->info("Deleting local copy of the ralus config file: $tmp_sf.");
	        Utils::vxif_dol("rm -f $tmp_sf");
        }
}
sub _ralus_migrate_Certificate_Info {
	my ($self, $ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_migrate_Certificate_Info");

	# First check to see if we have the BE GRFS Unix Agent on this machine
        my $sf = $CONF{RALUSCFG};
	if ( !Utils::vxif_fc($sf, $ctx) )
	{
		return;
	}
        
        my $tmp_sf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";

	$$ctx{LOGGER}->info("Copy RALUS config file $sf to $tmp_sf.");
        Utils::vxif_copy($sf,$tmp_sf,$ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
        if ( -f "$tmp_sf" )
        {
	        $$ctx{LOGGER}->info("Read local RALUS config file $tmp_sf.");
	        my ($buffer) = Utils::vxif_readf($tmp_sf);
		if ($buffer ne "") 
		{
			my (@certinfo);
		       	my @lines = split(/\n/,$buffer);
		        foreach my $n(0..$#lines) 
			{
		       	        my $certinfo_lookfor = "Certificates";
				if ($lines[$n] =~ m/($certinfo_lookfor)/) 
				{
			                my ($oneline) = $lines[$n];
			                chomp($oneline);
                                        push(@certinfo, $oneline);
				        $$ctx{LOGGER}->finer("Adding $oneline to Certificate Info  List.");
	                        }
                        }
			# Add Certificate Info to Hash CFG Item
	                if ($#certinfo >=0) 
			{
	                        @{$CFG{CERTIFICATEINFO}}=@certinfo;
	                        $tmpcount = $#certinfo+1;
				$$ctx{LOGGER}->info("Added $tmpcount rows for certificate information.");
	                }
			else
			{
				$$ctx{LOGGER}->info("No certificate information found in ralus.cfg file.");
			}
	        }
			
	        # Remove the file from the TMP directory
	        $$ctx{LOGGER}->info("Deleting local copy of the ralus config file: $tmp_sf.");
	        Utils::vxif_dol("rm -f $tmp_sf");
        }
}
sub _ralus_migrate_Oracle_Linux_Info {
	my ($self, $ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_migrate_Oracle_Linux_Info");

	# First check to see if we have the BE GRFS Unix Agent on this machine
        my $sf = $CONF{RALUSCFG};
	if ( !Utils::vxif_fc($sf, $ctx) )
	{
		return;
	}
        
        my $tmp_sf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";

	$$ctx{LOGGER}->info("Copy RALUS config file $sf to $tmp_sf.");
        Utils::vxif_copy($sf,$tmp_sf,$ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
        if ( -f "$tmp_sf" )
        {
	        $$ctx{LOGGER}->info("Read local RALUS config file $tmp_sf.");
	        my ($buffer) = Utils::vxif_readf($tmp_sf);
		if ($buffer ne "") 
		{
			my (@oracleinfo);
		       	my @lines = split(/\n/,$buffer);
		        foreach my $n(0..$#lines) 
			{
		       	        my $oracleinfo_lookfor_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Agents\\\\XBSA" ;
                                my $oracleinfo_lookfor_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Agents\\\\XBSA" ;
				if ($lines[$n] =~ m/($oracleinfo_lookfor_old)/) 
				{
			                my ($oneline) = $lines[$n];
                                        $oneline =~ s/Symantec/Veritas/ig;
			                chomp($oneline);
                                        push(@oracleinfo, $oneline);
				        $$ctx{LOGGER}->finer("Adding $oneline to Oracle Info  List.");
	                        }
                                elsif ($lines[$n] =~ m/($oracleinfo_lookfor_new)/)
                                {
                                        my ($newoneline) = $lines[$n];
			                chomp($newoneline);
                                        push(@oracleinfo, $newoneline);
				        $$ctx{LOGGER}->finer("Adding $newoneline to Oracle Info  List.");
                                }

            }
			# Add Oracle Info to Hash CFG Item
            if ($#oracleinfo >=0) 
			{
	                        @{$CFG{ORACLELINUXINFO}}=@oracleinfo;
	                        $tmpcount = $#oracleinfo+1;
				$$ctx{LOGGER}->info("Added $tmpcount rows for Oracle Linux information.");
	        }
			else
			{
				$$ctx{LOGGER}->info("No Oracle Linux information found in ralus.cfg file.");
			}
		}
			
	        # Remove the file from the TMP directory
	        $$ctx{LOGGER}->info("Deleting local copy of the ralus config file: $tmp_sf.");
	        Utils::vxif_dol("rm -f $tmp_sf");
        }
}

sub _ralus_migrate_ralus_cfg_Info {
	my ($self, $ctx) = @_;
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_migrate_ralus_cfg_Info");

	# First check to see if we have the BE GRFS Unix Agent on this machine
        my $sf = $CONF{RALUSCFG};
	if ( !Utils::vxif_fc($sf, $ctx) )
	{
		return;
	}
        
        my $tmp_sf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";

	$$ctx{LOGGER}->info("Copy RALUS config file $sf to $tmp_sf.");
        Utils::vxif_copy($sf,$tmp_sf,$ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
        if ( -f "$tmp_sf" )
        {
	        $$ctx{LOGGER}->info("Read local RALUS config file $tmp_sf.");
	        my ($buffer) = Utils::vxif_readf($tmp_sf);
		if ($buffer ne "") 
		{
	      		my (@ralusinfo);
		       	my @lines = split(/\n/,$buffer);
		        foreach my $n(0..$#lines) 
			{
		       	my $ralusinfo_tsafs_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\ShowTSAFS" ;
                        my $ralusinfo_tsafs_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\ShowTSAFS" ;

				my $ralusinfo_disableofo_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\DisableOFO" ;
				my $ralusinfo_disableofo_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\DisableOFO" ;

				my $ralusinfo_cachefile_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\AOFOL\\\\CacheFileMountPoint" ;
				my $ralusinfo_cachefile_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\AOFOL\\\\CacheFileMountPoint" ;

				my $ralusinfo_cachefilesize_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\AOFOL\\\\CacheFileSize" ;
				my $ralusinfo_cachefilesize_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RALUS\\\\AOFOL\\\\CacheFileSize" ;
				
				my $ralusinfo_disablermal_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RMAL\\\\DisableRMAL" ;
				my $ralusinfo_disablermal_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\RMAL\\\\DisableRMAL" ;
				
				my $ralusinfo_vxbsalevel_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Debug\\\\VXBSAlevel" ;
				my $ralusinfo_vxbsalevel_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Debug\\\\VXBSAlevel" ;

				my $ralusinfo_agentconfig_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Debug\\\\AgentConfig" ;
				my $ralusinfo_agentconfig_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Debug\\\\AgentConfig" ;
				
				my $ralusinfo_ndmp_debug_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Logging\\\\RANT NDMP Debug Level" ;
				my $ralusinfo_ndmp_debug_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Backup Exec\\\\Engine\\\\Logging\\\\RANT NDMP Debug Level" ;
				
				my $ralusinfo_adv_port_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Agent Browser\\\\TcpIp\\\\AdvertisementPort" ;
				my $ralusinfo_adv_port_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Agent Browser\\\\TcpIp\\\\AdvertisementPort" ;

                my $ralusinfo_cert_info_old = "Software\\\\Symantec\\\\Backup Exec For Windows\\\\Common\\\\Backup Exec\\\\Engine\\\\Agents\\\\Security\\\\Certificates" ;
				my $ralusinfo_cert_info_new = "Software\\\\Veritas\\\\Backup Exec For Windows\\\\Common\\\\Backup Exec\\\\Engine\\\\Agents\\\\Security\\\\Certificates" ;
				
				
				if ($lines[$n] =~ m/($ralusinfo_tsafs_old)/ || $lines[$n] =~ m/($ralusinfo_disableofo_old)/ || $lines[$n] =~ m/($ralusinfo_cachefile_old)/ || $lines[$n] =~ m/($ralusinfo_cachefilesize_old)/
				|| $lines[$n] =~ m/($ralusinfo_disablermal_old)/ || $lines[$n] =~ m/($ralusinfo_vxbsalevel_old)/ || $lines[$n] =~ m/($ralusinfo_agentconfig_old)/ || $lines[$n]=~ m/($ralusinfo_adv_port_old)/
				|| $lines[$n] =~ m/($ralusinfo_ndmp_debug_old)/ || $lines[$n] =~ m/($ralusinfo_cert_info_old)/ )
				{
			                my ($oneline) = $lines[$n];
                            $oneline =~ s/Symantec/Veritas/ig;
			                chomp($oneline);
                            push(@ralusinfo, $oneline);
				            $$ctx{LOGGER}->finer("Adding $oneline to Ralus Info  List.");
                }
				if ($lines[$n] =~ m/($ralusinfo_tsafs_new)/ || $lines[$n] =~ m/($ralusinfo_disableofo_new)/ || $lines[$n] =~ m/($ralusinfo_cachefile_new)/ || $lines[$n] =~ m/($ralusinfo_cachefilesize_new)/
				|| $lines[$n] =~ m/($ralusinfo_disablermal_new)/ || $lines[$n] =~ m/($ralusinfo_vxbsalevel_new)/ || $lines[$n] =~ m/($ralusinfo_agentconfig_new)/ || $lines[$n] =~ m/($ralusinfo_adv_port_new)/ 
				|| $lines[$n] =~ m/($ralusinfo_ndmp_debug_new)/ || $lines[$n] =~ m/($ralusinfo_cert_info_new)/ ) 
                {
                           my ($newoneline) = $lines[$n];
			               chomp($newoneline);
                           push(@ralusinfo, $newoneline);
				           $$ctx{LOGGER}->finer("Adding $newoneline to Ralus Info  List.");
                }
            }
			# Add ralus Info to Hash CFG Item
            if ($#ralusinfo >=0) 
			{
	                        @{$CFG{RALUSINFO}}=@ralusinfo;
	                        $tmpcount = $#ralusinfo+1;
				$$ctx{LOGGER}->info("Added $tmpcount rows for RALUS cfg information.");
	        }
			else
			{
				$$ctx{LOGGER}->info("No information found in ralus.cfg file for user-defined parameters.");
			}
		}
			
	        # Remove the file from the TMP directory
	        $$ctx{LOGGER}->info("Deleting local copy of the ralus config file: $tmp_sf.");
	        Utils::vxif_dol("rm -f $tmp_sf");
        }
}

sub _ralus_check_BEGRFS_UnixAgent {
	my ($self, $ctx) = @_;
	
	# First check to see if we have the BE GRFS Unix Agent on this machine
	if ( !Utils::vxif_fc("/etc/bkupexec/agent.be", $ctx) )
	{
		return;
	}

	# We need to notify the user that they will need to change the name
        # of the BE GRFS Unix Agent so it wont Collide with the new RALUS
        # advertising protocol.
	my $msg = Utils::_tr("A previous generation of the Backup Exec Unix agent has been found on this computer. Because the Backup Exec Agent for Linux and Unix Servers and the Backup Exec Unix Agent cannot run at the same time, the Backup Exec Unix Agent will be disabled, but not uninstalled. For more information, see your Backup Exec Agent for Linux and Unix servers documentation.\n\n", 250, 1005);
	Utils::vxif_pbl($msg);
	my $ask_q = Utils::_tr("Do you want to continue installing RALUS?", 250, 1006);
	my $ans = Utils::vxif_ayny($ask_q);
	if ($ans eq "N")
	{
		#_die(Utils::_tr("Exiting the $PROD{RALUS}{ABBR} installer."));
		return;
	}

	# We need to kill all agent.be processes and
        # disable the GRFS Agent on the local computer.
	_ralus_disable_GRFS_Agent($self, $ctx);
	return 1;
}

sub _ralus_add_MediaServers {
	my ($ctx) = @_;
        
	my($n);
        
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_add_MediaServers");
        
	Utils::vxif_title(Utils::_tr("Backup Exec Agent Media Server Migration", 250, 1007));
	Utils::vxif_pbl(Utils::_tr("The following Backup Exec servers are being migrated during this install of the agent:\n\n", 250, 1008));
	foreach $n(0..$#{$CFG{PREVRALUSMDSRVRS}}) {
		Utils::vxif_pl("\t\t${$CFG{PREVRALUSMDSRVRS}}[$n]");
	}

	Utils::vxif_prtc($ctx, 0);
}

sub _ralus_add_GRFS_MediaServers {
	my($n,$ayn);

	Utils::vxif_title(Utils::_tr("Backup Exec Agent Media Server Migration", 250, 1007));
	Utils::vxif_pbl(Utils::_tr("Do you want to migrate the following Backup Exec servers from the previous Backup Exec Unix agent for use with the new agent:\n\n", 250, 1064));
	foreach $n(0..$#{$CFG{MIGRATEMDSRVRS}}) {
		Utils::vxif_pl("${$CFG{MIGRATEMDSRVRS}}[$n]");
	}

	$ayn=Utils::vxif_ayny(Utils::_tr("\nAdd all to my current configuration?", 250, 1009));

	# If they don't want to migrate the list of old Media Servers
	# over to the new configuration then we need to just erase them. 
	if ($ayn eq "N") {
		$CFG{MIGRATEMDSRVRS} = [];
	}
}

sub _ralus_ask_address {
	my($address,$ask);
	($address)=(@_);
	while (1) {
		$COMM{DEFANSWER}=$address;
		$MSG{HELP}=Utils::_tr("A directory host can be any of the following: IP Address, Host Name or LDAP Address.", 250, 1010);

		$ask = Utils::vxif_ask(Utils::_tr("Enter a directory host:", 250, 1011));
		return if ($FLAG{BACK});
                return "$ask";
	}
}

sub _ralus_add_addresses {
	my ($ctx) = @_;
	my(@addresses,$address,$done,$ayn);
        while (!$done) {
        	if ( $#{$CFG{MIGRATEMDSRVRS}} >= 0 ) {
	        	_ralus_add_GRFS_MediaServers($ctx);
		}
                
                if ( $#{$CFG{PREVRALUSMDSRVRS}} >= 0 ) {
                        _ralus_add_MediaServers($ctx);
                }
                
	        my $conf_msg = Utils::_tr("Agent and Backup Exec server Configuration", 250, 1012);
	        Utils::vxif_title($conf_msg);
	        undef(@addresses);

		my $msg = Utils::_tr("Enter the names or IP addresses of the Backup Exec servers that you want the agent on '$$ctx{TARGET}{HOST}' to communicate with.\n", 250, 1013, "$$ctx{TARGET}{HOST}");
		Utils::vxif_pl($msg);

		Utils::vxif_pl(Utils::_tr("\tAn IP Address: XXX.XXX.XXX.XXX", 250, 1014));
		Utils::vxif_pl(Utils::_tr("\tA Host Name: COMPUTERNAME\n", 250, 1015));
	
                my $ayn = "Y";
		while ( $ayn eq "Y" ) {
			$address=_ralus_ask_address();
                        if ( $address ne "" ) {
			        push(@addresses, $address);
                                $ayn=Utils::vxif_aynn(Utils::_tr("Do you want to add another name or address for this agent?", 250, 1016));
                        } else {
                                $ayn = "N";
                        }
		}

                # Verify entered Addresses from the user before going to the next
                # configuration phase.
		if ( $#addresses >= 0 ) {
		        # Verify entries
	                Utils::vxif_title($conf_msg);
		        Utils::vxif_bpl(Utils::_tr("Hostnames and/or IP addresses verification for '$$ctx{TARGET}{HOST}':\n", 250, 1017,"$$ctx{TARGET}{HOST}"));
		        my $msg = "";
		        foreach $n(0..$#addresses) {
			        $msg.= Utils::_tr("\tAddress: $addresses[$n]\n", 250, 1018, "$addresses[$n]");
		        }
		        Utils::vxif_pl("$msg");
		        $ayn = Utils::vxif_ayny(Utils::_tr("Is this information correct?", 250, 1019));
		        $done = 1 if ($ayn eq "Y");
                } else {
	                Utils::vxif_prtc($ctx, 0);
                        $done = 1;
                }
	}

	$CFG{ADDRESSES}=[];
        @{$CFG{ADDRESSES}}=@addresses;
        
	return 1;
}

#
# This function will check Port 10000 to see if it's in use by another product. If it is
# it will alert the user with a message and allow them to EXIT out of the install process
# or continue with the understanding that the Remote Agent will not startup without redirecting
# to another unused PORT.
#
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the user wants to continue with the Install otherwise, return 0;
#
sub _ralus_alert_user_port10000_inuse {
        my ($ctx) = @_;
        my ($result) = 0;

        $$ctx{LOGGER}->entering("RALUSCommon::_ralus_alert_user_port10000_inuse");
        
	Utils::vxif_title(Utils::_tr("Agent Configuration Warning!!!", 250, 1020));
	my $msg = Utils::_tr("The Agent Installer has determined that port 10000 is currently in use by another application. Do you want to continue installing the agent on '$$ctx{TARGET}{HOST}' anyways?", 250, 1021,"$$ctx{TARGET}{HOST}");
        my $ayn=Utils::vxif_ayny( $msg );
        if ( $ayn eq "Y" ) {
                $result = 1;
        }

        return $result;            
}

#
# This method will check to see if the installer is running as ROOT user
# and will return 1 for TRUE or 0 for FALSE.
#
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the user is running as ROOT otherwise, return 0;
#
sub _ralus_is_root_user {
        my ($ctx) = @_;
        my ($result) = 0;

        $$ctx{LOGGER}->entering("RALUSCommon::_ralus_is_root_user");
	
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

	# Locate id and use full path (location differs on different platforms)
        my $id_cmd = 'id';
	my $root_switch = ' -un';
        my $use_revised_id = 0;
        
	# SunOS uses a different binary which supports the -un switch
        if ($trg_os_name eq "SunOS") {
        	if (Utils::vxif_fc("/usr/xpg4/bin/id", $ctx)) {
			$id_cmd = '/usr/xpg4/bin/id';
		} else {
                        $use_revised_id = 1;
			$root_switch = " | awk '{print \$1}'";
		}
        } else {
		if (Utils::vxif_fc("/usr/bin/id", $ctx)) {
			$id_cmd = '/usr/bin/id';
		} elsif (Utils::vxif_fc("/bin/id", $ctx)) {
			$id_cmd = '/bin/id';
		}
	}

	my $command = $id_cmd . $root_switch;
        $$ctx{LOGGER}->info("Checking for 'root' user during install via: $command.");
        my ($Results) = Utils::vxif_dor("${command}", "", $ctx);
        $$ctx{LOGGER}->fine("Results: $Results.");
        if ($use_revised_id == 1) {
		if ($Results eq "uid=0(root)") {
			$result = 1;
		}
	} else {
        	if ($Results eq "root") {
                	$result = 1;
		}
        }
        
        return $result;
}

#
# This method will check to see if the Target system has Backup Exec
# for Linux Servers installed and will return 1 for TRUE or 0 for FALSE.
#
# Input: 1) the reference to the installation context
#
# Return: 1 if Backup Exec for Linux Servers is installed otherwise, return 0;
#
sub _ralus_is_BELS_installed {
        my ($ctx) = @_;
        my ($result) = 0;

        $$ctx{LOGGER}->entering("RALUSCommon::_ralus_is_BELS_installed");
	
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

	# We only need to check on Linux Targets if BELS is installed
        if ($trg_os_name eq "Linux") {
		# Check for the ndmpd binary which is an installed component
		# for the BELS fileset.
        	if (Utils::vxif_fc("/opt/bkupexec/bin/ndmpd", $ctx)) {
			$$ctx{LOGGER}->info("RALUS Installer has found BELS on Target Host.");
			$result = 1;
		}
	}

        return $result;
}

#
# Product pre check uninstallation script. This interface will be called before the product
# is being uninstalled.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the pre check uninstallation has an error condition and must abort the uninstall
# otherwise, return 0 to cont
#
sub pre_check_product_uninstall (\%;$) {
 my $self = shift;
 my ($ctx) = @_;
 
 $$ctx{LOGGER}->entering("RALUSCommon::pre_check_product_uninstall");
 
        # Check to see if we have the Remote Agent installed or not
        if ( !Utils::vxif_fc("/opt/VRTSralus/bin/beremote", $ctx) ) {

                # If not installed we need to log entry in logfile and notify user via message and exit
                $$ctx{LOGGER}->info("WARNING: Agent wasn't found on target machine '$$ctx{TARGET}{HOST}'. Uninstallation now exiting.");
                
                Utils::vxif_title(Utils::_tr("Agent UnInstaller Warning!!!", 250, 1024));
                my $msg = "The uninstallation program has detected that the agent is not installed on host '$$ctx{TARGET}{HOST}'. The uninstallation program will now exit.", 250, 1063, "$$ctx{TARGET}{HOST}";
                Utils::vxif_pl($msg);
                Utils::vxif_prtc($ctx, 0);
                return 1; # returning 1 will cause VxIF to Exit gracefully with logfile
        }
 
        return 0;
}

#
# Product pre-installation script. This interface will be called before the product
# is being installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-installation is successful. Otherwise, return 0;
#
sub pre_product_install (\%\%;$) {
 my $self = shift;
 my ($ctx, $mode) = @_;
 my $msg;
 
 $$ctx{LOGGER}->entering("RALUSCommon::pre_product_install");
 
 # if we are NOT running as root then we need to notify the user if
 # we are not running in silent mode and EXIT
 if ( _ralus_is_root_user($ctx) == 0 )
 {
        if ( !$$ctx{RESPONSEFILE} )
        {
	        Utils::vxif_title(Utils::_tr("Agent Installer Warning!!!", 250, 1022));
		$msg = Utils::_tr("The Agent Installer has determined that the current user is not running as root on '$$ctx{TARGET}{HOST}'. Please retry installing the agent as 'root' user.\n", 250, 1023,"$$ctx{TARGET}{HOST}");
	        Utils::vxif_pl($msg);
	        Utils::vxif_prtc($ctx, 0);
        }
        
        $$ctx{LOGGER}->error("ERROR (V-225-247): Exiting RALUS Installer due to Non 'root' user.");
        return 0;       
 }

 # If we are installing on a Target machine running BELS we must notify
 # the user and abort the install of RALUS
 if ( _ralus_is_BELS_installed($ctx) == 1 )
 {
	if ( !$$ctx{RESPONSEFILE} )
	{
		# Display the BELS installed message to user and abort install to target message
		Utils::vxif_title(Utils::_tr("Agent Installer Warning!!!", 250, 1022));
		$msg = Utils::_tr("The installation program has detected that Backup Exec for Linux and Unix Servers is already installed on host '$$ctx{TARGET}{HOST}'. The Remote Agent for Linux and Unix Servers cannot be installed on the same host as Backup Exec for Linux and Unix Servers.", 250, 1058, "$$ctx{TARGET}{HOST}" );
		Utils::vxif_pl($msg);

	        Utils::vxif_pl("\n\n");

		$msg = Utils::_tr("Installation will continue on the next target host. If there are no more targets, the installation program will exit.", 250, 1059);
		Utils::vxif_pl($msg);
		Utils::vxif_prtc($ctx, 0);
	}

        $$ctx{LOGGER}->info("Exiting RALUS Installer due to BELS on Target server.");
	return 0;
 }

 my $Port10000_IsInUse = _ralus_is_port_10000_inuse($ctx);
 
 if (!$$ctx{RESPONSEFILE})
 {
        # If Port 10000 is in use and the user doesn't want to continue
        # with the install then we simply exit out
         if (( $Port10000_IsInUse ) && (_ralus_alert_user_port10000_inuse($ctx) == 0))
         { 
                return 0;
         }
         
	 # We need to check for a current install of
	 # the Backup Exec GRFS Unix Agent before Installing
	 _ralus_check_BEGRFS_UnixAgent($self, $ctx);

         # Now we need to check if we are upgrading from a previous version of RALUS
         # and if so notify the user what media servers are currently in the list
         # of Advertising to Media Servers
         _ralus_migrate_MediaServers($self, $ctx);
         _ralus_migrate_Certificate_Info($self, $ctx);
		 _ralus_migrate_Oracle_Linux_Info($self, $ctx);
		 _ralus_migrate_ralus_cfg_Info($self, $ctx);
         my $trg_host = $ctx->get_target_hostname();
         my $trg_os_name = $ctx->get_host_os_name($trg_host);
         $$ctx{LOGGER}->info("Target OS name: $trg_os_name");
         if ($trg_os_name ne "Darwin")
         {
	 	_ralus_create_beoper_group($ctx);
	 }

 } else {
         # Now we need to check if we are upgrading from a previous version of RALUS
         # and if so notify the user what media servers are currently in the list
         # of Advertising to Media Servers
         _ralus_migrate_MediaServers($self, $ctx);
         _ralus_migrate_Certificate_Info($self, $ctx);
		 _ralus_migrate_Oracle_Linux_Info($self, $ctx);
		 _ralus_migrate_ralus_cfg_Info($self, $ctx);
	 _ralus_create_beoper_group($ctx);

 	$$ctx{LOGGER}->finer("Skipping Questions in Silent Mode.");
 }
 
 return 1;
}

# Product post-installation script. This interface will be called after all the packages
# has been installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
#sub post_product_install (\%\%;$) {
#}

#
# Product configuration. This interface will be called after all the packages
# has been installed and post_product_install().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
#sub per_host_product_postinstall_questions (\%\%;$) {
#}

#
# Product configuration. This interface will be called after all the packages
# has been installed and post_product_install().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
#sub per_os_family_product_postinstall_questions (\%\%;$) {
#}

#
# Product configuration. This interface will be called after all the packages
# has been installed and post_product_install().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
#sub product_postinstall_questions (\%\%;$) {
#}

#
# Product configuration. This interface will be called after post-installation.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
sub create_product_install_config (\%\%;$) {
        my $self = shift;
	my ($ctx,$cfg,$mode) = @_;
	my ($cfile) = "/etc/VRTSralus/ralus.cfg";

	$$ctx{LOGGER}->entering("RALUSCommon::create_product_install_config");

        if (Utils::vxif_fc($cfile, $ctx)) {
		return $self->create_configfiles($ctx, $cfg, $mode, $$cfile);
	} else {
		$$ctx{LOGGER}->error("ERROR (V-225-246): $cfile not found on $$ctx{TARGET}{HOST}");
	}

	return 0;
}
							
#
# Product completion messages. This interface will be called after the product installation
# operations completed successfully.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the localized product completion message.
#
#sub get_product_completion_message (\%\%;$) {
#}

#
# Product pre-uninstallation script. This interface will be called before the product
# is being uninstalled.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-uninstallation is successful. Otherwise, return 0;
#
sub pre_product_uninstall (\%\%;$) {
 my $self = shift;
 my ($ctx, $mode) = @_;

        $$ctx{LOGGER}->entering("RALUSCommon::pre_product_uninstall");
        
        # if we are NOT running as root then we need to notify the user if
        # we are not running in silent mode and EXIT
        if ( _ralus_is_root_user($ctx) == 0 )
        {
                if ( !$$ctx{RESPONSEFILE} )
                {
	                Utils::vxif_title(Utils::_tr("Agent UnInstaller Warning!!!", 250, 1024));
	                my $msg = Utils::_tr("The Agent UnInstaller has determined that the current user is not running as root on '$$ctx{TARGET}{HOST}'. Please retry uninstalling the agent as 'root' user.\n", 250, 1025,"$$ctx{TARGET}{HOST}");
	                Utils::vxif_pl($msg);
	                Utils::vxif_prtc($ctx, 0);
                }
        
                $$ctx{LOGGER}->error("ERROR (V-225-247): Exiting RALUS UnInstaller due to Non 'root' user.");
                return 0;       
        }
	
	# return all is good
	return 1;
}

#
# Product post-uninstallation script. This interface will be called after all the packages
# has been uninstalled.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-uninstallation is successful. Otherwise, return 0;
#
#sub post_product_uninstall (\%\%;$) {
#}

#
# Product pre-patch script. This interface will be called before the patches
# are being applied. This interface will only be invoked for patch-only install.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-patch is successful. Otherwise, return 0;
#
#sub pre_product_patch (\%\%;$) {
#}

#
# Product post-patch script. This interface will be called after all the product patches
# have been installed. This interface will only be invoked for patch-only install.
#
# This interface is optional.
#
# Input: 1) license key if the product is a licensable product;
#        2) %cfg array reference;
#        3) the reference to the installation context;
#
# Return: 1 if the post-patch is successful. Otherwise, return 0;
#
#sub post_product_patch (\%\%;$) {
#}

#
# Product configuration. This interface will be called after all the packages
# has been patched and post_product_patch().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the patch configuration is successful. Otherwise, return 0;
#
#sub patch_product_configuration (\%\%;$) {
#}

#
# Patch configuration. This interface will be called after post-patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_patch_install_config (\%\%;$) {
#}

#
# Product pre-upgrade script. This interface will be called before the product
# are being upgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-upgrade is successful. Otherwise, return 0;
#
#sub pre_product_upgrade (\%\%;$) {
#}

#
# Product post-upgrade script. This interface will be called after the product
# have been upgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-upgrade is successful. Otherwise, return 0;
#
#sub post_product_upgrade (\%\%;$) {
#}

#
# Product configuration. This interface will be called after all the packages
# has been upgraded and post_product_upgrade().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the upgrade configuration is successful. Otherwise, return 0;
#
#sub upgrade_product_configuration (\%\%;$) {
#}

#
# Product upgrade configuration. This interface will be called after post-upgrade.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_product_upgrade_config (\%\%;$) {
#}

#
# Product pre-downgrade script. This interface will be called before the product
# are being downgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-downgrade is successful. Otherwise, return 0;
#
#sub pre_product_downgrade (\%\%;$) {
#}

#
# Product post-downgrade script. This interface will be called after the product
# have been downgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the installation mode if applicable;
#        3) %cfg array reference;
#
# Return: 1 if the post-downgrade is successful. Otherwise, return 0.
#
#sub post_product_downgrade (\%\%;$) {
#}

#
# Product downgrade configuration. This interface will be called after post-downgrade.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_product_downgrade_config (\%\%;$) {
#}

#
# Start the product processees after product installation operations.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version; this parameter is optional;
#        3) the installation mode if applicable;
#
# Return: 1 if successful; 0 otherwise.
#
#sub start_product_processees (\%;$$) {
#}

#
# Stop the product processees, for a given version of the product, before product
# installation operations.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version; this parameter is optional;
#        3) the installation mode if applicable; this parameter is optional;
#
# Return: 1 if successful; 0 otherwise.
#
#sub stop_product_processees (\%;$$) {
#}

#
# Serialize %cfg array. The serialized %cfg will be written directly to the
# response file. The handler must return a buffer which contains the serialzied %cfg array.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the installation type;
#           0 - product install;
#           1 - product patch;
#           2 - product upgrade;
#           3 - product downgrade;
#        3) %cfg array reference;
#        4) the installation mode if applicable;
#
# Return: the reference to the %cfg serialization hander or null ("").
#
#sub serialize_cfg (\%$\%;$) {
#}

#
# Verify the %cfg array to make sure the response file contents are properly loaded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the installation type;
#           0 - product install;
#           1 - product patch;
#           2 - product upgrade;
#           3 - product downgrade;
#        3) %cfg array reference;
#        4) the installation mode if applicable;
#
# Return: 1 if everything is successful; 0 otherwise.
#
#sub verify_responsefile (\%$\%;$) {
#}


################################################
# Common Package Installation Interfaces       #
################################################

#
# Stop the processes associated with a given package. This interface, if implemented, will be
# invoked before a package, or a patch for this package, is installed/upgraded/downgraded.
#
# This interface is optional. This interface need not be implemented if the stop_product_processes()
# has already shutdown all the processes.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name of the patch to be applied;
#
# Return: 1 if successful; 0 otherwise.
#
#sub stop_package_processes (\%$$) {
#}

#
# Start the processes associated with a given package. This interface, if implemented, will be
# invoked after a package, or a patch for this package, is installed/upgraded/downgraded.
#
# This interface is optional. This interface need not be implemented if the start_product_processes()
# will start all the processes.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name of the patch applied;
#
# Return: 1 if successful; 0 otherwise.
#
#sub start_package_processes (\%$$) {
#}

# 
# Pre system reboot script. This interface will be invoked before system reboot take place.
# This interface will be invoked if {REQREBOOT} is set to 1 for a given package.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#
# Return: 1 if pre-system reboot is successful. Otherwise, return 0.
#
#sub pre_package_system_reboot (\%$) {
#}

# 
# Pre system reboot script. This interface will be invoked before system reboot take place.
# This interface will be invoked if {REQREBOOT} is set to 1 for a given patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#
# Return: 1 if pre-system reboot is successful. Otherwise, return 0.
#
#sub pre_patch_system_reboot (\%$) {
#}

#
# This is the system reboot callback. Given a package name, return the 
# reference of the interface to handle the system reboot. 
# Instead of using default reboot mechanism (i.e. reboot command),
# VxIF will using this handler to handle system reboot.
#
# This interface is optional.
#
# Here's an example on how to customize the reboot operation.
#
# sub get_package_system_reboot_handler (%$) {
#   my($self,%context,$package_name) = @_;
#   my $my_reboot = sub {
#     # handle the system reboot here.
#   }
#
#   return \&$my_reboot;
# }
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#
# Return: the reference of the system reboot handler.
#
#sub get_package_system_reboot_handler (\%$) {
#}

#
# This is the system reboot callback. Given a patch name, return the 
# reference of the interface to handle the system reboot. 
# Instead of using default reboot mechanism (i.e. reboot command),
# VxIF will using this handler to handle system reboot.
#
# This interface is optional.
#
# Here's an example on how to customize the reboot operation.
#
# sub get_patch_system_reboot_handler (%$) {
#   my($self,%context,$patch_name) = @_;
#   my $my_reboot = sub {
#     # handle the system reboot here.
#   }
#
#   return \&$my_reboot;
# }
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#
# Return: the reference of the system reboot handler.
#
#sub get_patch_system_reboot_handler (\%$) {
#}

#
# What to do if a given package already exist on the target host and its version is different 
# from the one this product is trying to install? 
#
# This interface is option. By default, VxIF assumes the existing package is not compatible
# with this product and stop the installation.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the actual existing package version on the target host;
#        4) the package version to be installed;
#
# Return: 1 if the existing package on the target host is compatible with this product
#         and use it instead of installing the new version; 2 if the existing package on
#         target host is not compatible with this product and abort the installation;
#         0 to remove the existing package on the target host and install the current
#         version.
# 
#sub is_package_version_supported (\%$$$) {
#}
sub is_package_version_supported {
   my $self = shift;
   my ($ctx, $pkg, $version) = @_;
   my $rtn = 1;
   my ($major, $minor, $build, $patch) = split(/\./,$version);
   my $low_vers_major = 10;
   my $low_vers_minor = 0;
   my $low_vers_build = 0;

   $$ctx{LOGGER}->entering("RALUSCommon::is_package_version_supported");

   if (($low_ver_major <= $major) && (($low_vers_minor < $minor) ||($low_vers_build < $build))) {
      $rtn = 0;
   } elsif (($low_ver_major == $major) && ($low_vers_minor == $minor) && ($low_vers_build == build)) {
      $rtn = 1;
   } else {
      $$ctx{LOGGER}->warning("WARNING (V-225-250): Version ${version} of Package ${pkg} is not supported.  A manual uninstall will be required to proceed");
      $rtn = 2;
   }

   $$ctx{LOGGER}->exiting("RALUSCommon::is_package_version_supported");
   return $rtn;
}

#
# Pre package installation operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_package_install (\%$$$$) {
#}

#
# Post package installation operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_package_install (\%$$$$) {
#}

#
# Pre package removal operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_package_uninstall (\%$$$$) {
#}

#
# Post package removal operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_package_uninstall (\%$$$$) {
#}

#
# Pre package upgrade operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be upgraded;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_package_upgrade (\%$$$$) {
#}

#
# Post package upgrade operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be upgraded;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_package_upgrade (\%$$$$) {
#}

#
# Pre package downgrade operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be downgraded;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_package_downgrade (\%$$$$) {
#}

#
# Post package downgrade operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be downgraded;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_package_downgrade (\%$$$$) {
#}

#
# Pre patch installation operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the name of the host for which this patch will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_patch_install (\%$$$$) {
#}

#
# Post patch installation operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the name of the host for which this patch will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_patch_install (\%$$$$) {
#}

#
# Pre patch removal operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the name of the host for which this patch will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_patch_uninstall (\%$$$$) {
#}

#
# Post patch removal operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the name of the host for which this patch will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_patch_uninstall (\%$$$$) {
#}

sub pre_upgrade {
   my $self = shift;
   my ($ctx, $pkg, $version) = @_;
   $$ctx{LOGGER}->entering("RALUSCommon::pre_upgrade");

   #copy configu files, etc... here.
   $self->ralus_migrate_cfg($ctx);

   $$ctx{LOGGER}->exiting("RALUSCommon::pre_upgrade");
   return 1;
}

sub post_upgrade {
   my $self = shift;
   my ($ctx, $pkg, $version) = @_;
   $$ctx{LOGGER}->entering("RALUSCommon::post_upgrade");

   #copy configu files, etc... here.

   $$ctx{LOGGER}->exiting("RALUSCommon::post_upgrade");
   return 1;
}

sub _ralus_add_GRFS_MediaServers {
        my($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_add_GRFS_MediaServers");

	Utils::vxif_title("");
	Utils::vxif_pbl(Utils::_tr("Do you want to migrate the following Backup Exec servers from the previous Backup Exec Unix agent for use with the new agent on '$$ctx{TARGET}{HOST}':\n\n", 250, 1008, "$$ctx{TARGET}{HOST}"));
	foreach my $server(@{$CFG{MIGRATEMDSRVRS}}) 
	{
		$$ctx{LOGGER}->info("Migrate GRFS Server: $server ?");
		Utils::vxif_pl( Utils::_tr("Media Server: $server", 250, 1026, "$server"));
	}

	my $msg = Utils::_tr("\nAdd all to my current configuration?", 250, 1009);
	my ($ayn) = Utils::vxif_ayny($msg, "", "", "");
	# If they don't want to migrate the list of old Media Servers
	# over to the new configuration then we need to just erase them. 
	if ($ayn eq "N") 
	{
		$$ctx{LOGGER}->warning("WARNING: User NOT Migrating GRFS Servers to current install.");
		$CFG{MIGRATEMDSRVRS} = [];
	}
	else
	{
		$$ctx{LOGGER}->info("User Migrating GRFS Servers to current install.");
	}

	return 1;
}

sub _ralus_add_oracle_to_beopergrp {
	my($ctx, $grpname) = @_;
	my $Success = 0;

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_add_oracle_to_beopergrp");

	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");
        # Only check if we are running on Linux Platforms
        if ($trg_os_name ne "Linux") {
	        $$ctx{LOGGER}->info("Exiting due to non-Linux Platform for Oracle support.");
                return 0;
        }

	# SunOS uses a different binary which supports the "-G -n" switch
        my $id_cmd = 'id';
	if (Utils::vxif_fc("/usr/bin/id", $ctx)) {
	        $id_cmd = '/usr/bin/id';
	} elsif (Utils::vxif_fc("/bin/id", $ctx)) {
	        $id_cmd = '/bin/id';
	}
	
        # Is Oracle installed and if so what is the Oracle User name
        my ($oracle_installed, $oracle_user) = _ralus_get_oracle_user_name($ctx);
        if ( $oracle_installed == 0 ) { 
	        $$ctx{LOGGER}->info("Exiting due to Oracle not being installed.");
                return 0;
        } 
        
	# Get all of the supplimentary groups for Oracle to append
	# when we add root to the beoper group
	my $command = $id_cmd . " -G -n $oracle_user";
	$$ctx{LOGGER}->fine("Get supplimentary groups for '$oracle_user' via: $command.");
	my ($Results) = Utils::vxif_dor($command, "", $ctx);
	$$ctx{LOGGER}->fine("Supplimentary Groups = $Results");

	my ($Supp_Grps) = "";
       	my @groups = split(/ /,$Results);
        foreach my $n(0..$#groups) 
	{
		if ( length($Supp_Grps) != 0 ) 
		{
			$Supp_Grps .= "," . $groups[$n];
		} else {
			$Supp_Grps = $groups[$n];
		}
		$$ctx{LOGGER}->fine("Adding group: $groups[$n] to Supp_Grps: $Supp_Grps");
	}

	# Locate usermod and use full path (location differs on different platforms)
	my $usermod = 'usermod';
	if (-f '/usr/sbin/usermod') {
		$usermod = '/usr/sbin/usermod';
	}

        # we need to add the user to the "beoper" group
	$command = $usermod . " -G " . $Supp_Grps . "," . $grpname . " " . $oracle_user;
	$$ctx{LOGGER}->info("Adding Oracle User to '$grpname' group via: $command.");
	$Results = Utils::vxif_dor("${command}", "", $ctx);
	if ($Results == "") {
	        $$ctx{LOGGER}->info("Results: Successfully added Oracle User to beoper group.");
		$Success = 1;
	} else {
	        $$ctx{LOGGER}->fine("Results: $Results.");
        }

	return $Success;
}

sub _ralus_add_root_to_beopergrp {
	my($ctx, $grpname) = @_;
	my $Success = 0;

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_add_root_to_beopergrp");

	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

	# Locate id and use full path (location differs on different platforms)
	# SunOS uses a different binary which supports the "-G -n" switch
        my $id_cmd = 'id';
        if ($trg_os_name eq "SunOS") {
        	if (Utils::vxif_fc("/usr/xpg4/bin/id", $ctx)) {
			$id_cmd = '/usr/xpg4/bin/id';
		}
        } else {
		if (Utils::vxif_fc("/usr/bin/id", $ctx)) {
			$id_cmd = '/usr/bin/id';
		} elsif (Utils::vxif_fc("/bin/id", $ctx)) {
			$id_cmd = '/bin/id';
		}
	}
	
	# Get all of the supplimentary groups for Root to append
	# when we add root to the beoper group
	my $command = $id_cmd . " -G -n root";
	$$ctx{LOGGER}->info("Get supplimentary groups for 'root' via: $command.");
	my ($Results) = Utils::vxif_dor($command, "", $ctx);
	$$ctx{LOGGER}->fine("Supplimentary Groups = $Results");

	my ($Supp_Grps) = "";
       	my @groups = split(/ /,$Results);
        foreach my $n(0..$#groups) 
	{
		if ( length($Supp_Grps) != 0 ) 
		{
			$Supp_Grps .= "," . $groups[$n];
		} else {
			$Supp_Grps = $groups[$n];
		}
		$$ctx{LOGGER}->info("Adding group: $groups[$n] to Supp_Grps: $Supp_Grps");
	}

	# Locate usermod and use full path (location differs on different platforms)
	my $usermod = 'usermod';
	if (-f '/usr/sbin/usermod') {
		$usermod = '/usr/sbin/usermod';
	}

	$command = $usermod . " -G " . $Supp_Grps . "," . $grpname . " root";
	$$ctx{LOGGER}->info("Adding 'root' to '$grpname' user group via: $command.");
	$Results = Utils::vxif_dor("${command}", "", $ctx);
	$$ctx{LOGGER}->fine("Results: $Results.");

	if ($Results == "") {
		$Success = 1;
	}

	return $Success;
}

sub _ralus_create_beoper_group {
	my($ctx) = @_;
	my($gid, $done, $ayn, $msg, $command, $groupCreated, $rootAdded);

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_create_beoper_group");

	# Set default values
	my $groupname = "beoper";
	my $rootQuestion = Utils::_tr("Do you want to add the 'root' user to '$groupname' user group?", 250, 1027, "$groupname");

	# Locate groupadd and use full path (location differs on different platforms)
	my $groupadd = 'groupadd';
	if (-f '/usr/sbin/groupadd') {
		$groupadd = '/usr/sbin/groupadd';
	}

	# Locate usermod and use full path (location differs on different platforms)
	#my $usermod = 'usermod';
	#if (-f '/usr/sbin/usermod') {
	#	$usermod = '/usr/sbin/usermod';
	#}

	Utils::vxif_title(Utils::_tr("Agent User Group Configuration", 250, 1028));
	Utils::vxif_pl(Utils::_tr("To perform backups, you must have a '$groupname' user group configured with root account priviledges on the system. The '$groupname' user group can be created on all systems except NIS Servers.\n\nAfter pressing [Return], the system '$$ctx{TARGET}{HOST}' will be scanned to detect a '$groupname' user group with root account privileges. It will also scan for a NIS server.\n", 250, 1029, "$groupname", "$groupname", "$$ctx{TARGET}{HOST}", "$groupname"));
	Utils::vxif_prtc($ctx, 0);

	my $nis_msg = Utils::_tr("Checking for NIS server:", 250, 1030);
        my $yes_msg = Utils::_tr("Yes\n", 250, 1031);
        my $no_msg = Utils::_tr("No\n", 250, 1032);
	Utils::vxif_pbl($nis_msg);
	my ($NISFound, $NISServerName) = _ralus_check_for_NIS($ctx);
	if ( $NISFound == 1 ) {
		Utils::vxif_pbr($nis_msg, $yes_msg);
		$$ctx{LOGGER}->info("System uses NIS Server.");
	} else {
		Utils::vxif_pbr($nis_msg, $no_msg);
		$$ctx{LOGGER}->info("System does NOT use NIS Server.");
	}

    	if ($NISFound) {
		$$ctx{LOGGER}->info("System uses NIS Server so we are exiting the creation of the 'beoper' group.");
        	my $nis_notify_msg = Utils::_tr("The installer found that your system is using NIS server '$NISServerName'. The installer cannot create '$groupname' user group on this system '$$ctx{TARGET}{HOST}'. You must manually create '$groupname' user group and make root user member of that group before running backup jobs.", 250, 1033, "$NISServerName", "$groupname", "$$ctx{TARGET}{HOST}", "$groupname");
        	Utils::vxif_pl($nis_notify_msg);
		Utils::vxif_prtc($ctx,0);
        	return;
	}

	#Utils::vxif_pl(Utils::_tr("The installer will now check your system for the '$groupname' user group and root membership.\n", 250, 1034, "$groupname"));
	#Utils::vxif_prtc($ctx,0);

	my $group_msg = Utils::_tr("Checking for '$groupname' user group:", 250, 1035, "$groupname");
	my $found_msg =  Utils::_tr("Found", 250, 1036);
	my $notfound_msg = Utils::_tr("Not Found", 250, 1037);
	Utils::vxif_pbl($group_msg);
	my ($groupfound, $rootfound) = _ralus_check_for_beoper_group($ctx, $groupname);
	if ( $groupfound == 1 ) {
		Utils::vxif_pbr($group_msg, $found_msg);
	} else {
		Utils::vxif_pbr($group_msg, $notfound_msg);
	}

	my $chkroot_msg = Utils::_tr("Checking for 'root' user membership in '$groupname' user group:", 250, 1038, "$groupname");
	Utils::vxif_pbl($chkroot_msg);
	if ( $rootfound == 1 ) {
		Utils::vxif_pbr($chkroot_msg, $found_msg);
	} else {
		Utils::vxif_pbr($chkroot_msg, $notfound_msg);
	}

	if ($groupfound && $rootfound) {
                # Support systems with beoper already installed and configured 
		my $success = _ralus_add_oracle_to_beopergrp($ctx,$groupname);
		$$ctx{LOGGER}->info("_ralus_add_oracle_to_beopergrp result: $success");

		$$ctx{LOGGER}->info("System already contains required $groupname and membership of root.");
		Utils::vxif_pl(Utils::_tr("\nYour system '$$ctx{TARGET}{HOST}' already has a required user group and membership.", 250, 1039, "$$ctx{TARGET}{HOST}"));
        	Utils::vxif_prtc($ctx,0);
		Utils::vxif_pl("\n");
		return;
	}

	if (!$groupfound) {
                if ($$ctx{RESPONSEFILE}) {
                        $ayn = "Y";
                } else {
		        Utils::vxif_pl(Utils::_tr("\nYou can create '$groupname' user group manually or you can choose to have it created automatically.\n", 250, 1040, "$groupname"));
		        $ayn = Utils::vxif_ayny(Utils::_tr("Do you want installer to create '$groupname' user group on '$$ctx{TARGET}{HOST}'?", 250, 1041, "$groupname", "$$ctx{TARGET}{HOST}"));
                }
		if ($ayn eq "Y") {
                        # Get the Target OS flavor
	                my $trg_host = $ctx->get_target_hostname();
	                my $trg_os_name = $ctx->get_host_os_name($trg_host);
                        $$ctx{LOGGER}->info("Target OS name: $trg_os_name");

                        if ($trg_os_name ne "AIX")
                        {
			        if (!$$ctx{RESPONSEFILE}) {
			                $ayn=Utils::vxif_aynn(Utils::_tr("\nDo you want to use specific group ID when creating '$groupname' user group?", 250, 1042, "$groupname"));
                                } else {
                                        $ayn = "N";
                                }
                                $gid = 0;
			        if ($ayn eq "Y") {
			                Utils::vxif_pl("");
				        $gid = _ralus_ask_group_id($ctx, "", $groupname);
				        $msg = Utils::_tr(" with group ID '$gid'", 250, 1043, "$gid");
				        $command = $groupadd . " -g $gid $groupname";
				        $$ctx{LOGGER}->info("Setting '$groupname' with specific group ID of '$gid'.");
				        Utils::vxif_pl("\n");
			        } else {
				        $command = $groupadd . " $groupname";
				        Utils::vxif_pl("\n");
			        }
                        } else {
                                # Create AIX command to create "beoper" group
                                $command = "/usr/bin/mkgroup " . $groupname;

                        }

                        if ($trg_os_name eq "Darwin")
                        {
                                # Darwin command to add "beoper" group to the NetInfo DB
                                if ($gid == 0)
                                {
                                        $command = "echo 'beoper:*:80:root' | sudo niload -v group /";
                                } else {
                                        $command = "echo 'beoper:*:" . $gid . ":root' | sudo niload -v group /";
                                }
                        }

			# Now create the user group
			my $creategrp_msg = Utils::_tr("Creating group '$groupname'$msg:", 250, 1044, "$groupname", "$msg");
		        my $done_msg = Utils::_tr("Done", 250, 1003);
			my $failed_msg = Utils::_tr("Failed\n", 250, 1045);
			Utils::vxif_pbl($creategrp_msg);
			my ($Results) = Utils::vxif_dor("${command}", "", $ctx);
			#my ($result, $output) = @{$Results};

			$$ctx{LOGGER}->info("Setting '$groupname' with specific group ID of '$gid'.");
            
			# Verify group creation
			($groupCreated, $rootfound) = _ralus_check_for_beoper_group($ctx, $groupname);
			$$ctx{LOGGER}->info("Verifying: '$groupname' created: $groupCreated.");
			$$ctx{LOGGER}->info("Verifying: 'root' member added to '$groupname': $rootfound.");
			if ($groupCreated) {
				Utils::vxif_pbr($creategrp_msg, $done_msg);
			} else {
				Utils::vxif_pbr($creategrp_msg, $failed_msg);
				Utils::vxif_pl(Utils::_tr("The user group could not be created on '$$ctx{TARGET}{HOST}'. Please manually create the user group.", 250, 1046, "$$ctx{TARGET}{HOST}"));
				Utils::vxif_prtc($ctx,0);
			}

			Utils::vxif_pl("\n");
		} elsif ($ayn eq "N") {
			Utils::vxif_pl(Utils::_tr("\nMake sure you have '$groupname' user group on '$$ctx{TARGET}{HOST}' before running backup jobs.", 250, 1047, "$groupname", "$$ctx{TARGET}{HOST}"));
			Utils::vxif_prtc($ctx,0);
		}
	} else {
		$rootQuestion = Utils::_tr("\nYour system '$$ctx{TARGET}{HOST}' already has the required group.\nDo you want to add the 'root' user to the existing '$groupname' user group?", 250, 1048, "$groupname");
	}

	if (($groupfound && !$rootfound) || ($groupCreated && !$rootfound)) {
                if ($$ctx{RESPONSEFILE}) {
                        $ayn = "Y";
                } else {
		        $ayn = Utils::vxif_ayny($rootQuestion);
                }
		if ($ayn eq "Y") {
			Utils::vxif_pl("\n");

			# Add root to newly created group
			my $addroot_msg = Utils::_tr("Adding 'root' user to '$groupname' user group:", 250, 1049, "$groupname");
			my $done_msg = Utils::_tr("Done\n", 250, 1050);
			my $failed_msg = Utils::_tr("Failed\n", 250, 1045);
			Utils::vxif_pbl($addroot_msg);
			
			# Add root to the beoper group
			my $success = _ralus_add_root_to_beopergrp($ctx,$groupname);
			$$ctx{LOGGER}->info("_ralus_add_root_to_beopergrp result: $success");
                        # Add Oracle User to beoper if applicable 
        		$success = _ralus_add_oracle_to_beopergrp($ctx,$groupname);
	        	$$ctx{LOGGER}->info("_ralus_add_oracle_to_beopergrp result: $success");

			#$command = $usermod . " -G $groupname root";
			#$$ctx{LOGGER}->info("Adding 'root' to '$groupname' user group via: $command.");
			#my ($Results) = Utils::vxif_dor("${command}", "", $ctx);
			#my ($result, $output) = @{$Results};
			#$$ctx{LOGGER}->info("Results: $Results.");

			# Verify root addition
			($groupCreated, $rootAdded) = _ralus_check_for_beoper_group($ctx, $groupname);
			$$ctx{LOGGER}->info("Verifying: '$groupname' created: $groupCreated.");
			$$ctx{LOGGER}->info("Verifying: 'root' member added to '$groupname': $rootAdded.");
			if ($rootAdded) {
				Utils::vxif_pbr($addroot_msg, $done_msg);
			} else {
				Utils::vxif_pbr($addroot_msg, $failed_msg);
				Utils::vxif_pl(Utils::_tr("\nThe 'root' user could not be added to the '$groupname' user group. Please manually add the 'root' user to '$groupname' in the /etc/group file on '$$ctx{TARGET}{HOST}'.", 250, 1051, "$groupname", "$groupname", "$$ctx{TARGET}{HOST}"));
			}
		} else {
			Utils::vxif_pl(Utils::_tr("\nMake sure you add 'root' user to '$groupname' user group on '$$ctx{TARGET}{HOST}' before running backup jobs.", 250, 1052, "$groupname", "$$ctx{TARGET}{HOST}"));
		}

		# Done with the work!
		Utils::vxif_prtc($ctx,0);
	}
}

sub _ralus_ask_group_id {
	my($ask, $gid, $msg, $groupname);
	($ctx, $gid, $groupname) = (@_);

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_ask_group_id");

	while (1) {
		$COMM{DEFANSWER}=$gid;
		$MSG{HELP}=Utils::_tr("Group ID is a unique number given to the groups. Values between 0 and 99 are typically reserved for system groups.", 250, 1053);
		$msg = Utils::_tr("Enter group ID for '$groupname' user group:", 250, 1054, "$groupname");
		$ask=Utils::vxif_ask($msg);
		return if ($FLAG{BACK});
		return "$ask" if (_ralus_verify_group_id($ctx, $ask));
	}
}

sub _ralus_verify_group_id {
	my($gid, $ayn);
	($ctx, $gid) = (@_);

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_verify_group_id");
	
	$gid = sprintf("%d", $gid);

	# Check if user given gid is between 0 and 99, if so print warning
	if ($gid <= 0 || $gid > 999) {
		Utils::vxif_pl(Utils::_tr("\nInvalid group ID. Please enter a number greater than 0 and less than 999.\n", 250, 1055));
		return "0";
	} elsif ($gid > 0 && $gid <= 99) {
		$ayn = Utils::vxif_aynn(Utils::_tr("\nWarning: Group ID between 0 and 99 is typically reserved for system groups.\nDo you still want to use group ID '$gid'?", 250, 1056, "$gid"));
		Utils::vxif_pl("");
		if ($ayn eq "Y") {
			return "1";
		} else {
			return "0";
		}
	}

	return "1";
}

# Return the oracle user if installed on the system
sub _ralus_get_oracle_user_name {
        my ($ctx) = @_;
        my $nResult = 0;
        my $oracle_user = "";
        my $oratab_file = "/etc/oratab";

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_get_oracle_user_name");

	if (Utils::vxif_fc($oratab_file, $ctx))
        {
                #$oracle_user = Utils::vxif_dor("/bin/ls -l /etc/oratab | /bin/awk '{print $3}' ", "", $ctx);
                my $output = Utils::vxif_dor("/bin/ls -l /etc/oratab", "", $ctx);
                if ( $output =~ m/ls:/ )
                {
	                $$ctx{LOGGER}->info("Error in listing file for owner: $oratab_file.");
                } 
                else
                {
                        # Let's get the creater / oracle user for the file
	                $$ctx{LOGGER}->info("Listing for $oratab_file: $output");
	                my (@items) = split(/ /, $output);
                        my ($nUserIndx) = 2;
			foreach my $nCount (0..$#items) 
			{
                                my $item = @items[$nCount];
	                        $$ctx{LOGGER}->info("Parsing thru listing $nCount:$item");

                                # If we find an empty entry increase our index by 1 
                                if ( $item eq "" )
                                {
	                                $$ctx{LOGGER}->info("Empty entry found. Increase Index and skip to next entry.");
                                        $nUserIndx = $nUserIndx + 1;
                                        next;
                                }
                                
                                # Check to see if we are getting Owner of file 
				if ($nCount == $nUserIndx)
				{
                                        $oracle_user = @items[$nCount];
	                                $$ctx{LOGGER}->info("Oracle file found. Successfully got username: $oracle_user");
                                        $nResult = 1;
			                last;
                                }
                        }
                }
                
                # Log WARNING that Oracle resources will not be protected until this is manually added
                if ( $nResult == 0 )
                {
	                $$ctx{LOGGER}->info("WARNING: Unable to locate Oracle User thus unable to add to beoper group for Oracle resource backups.");
                }
        }
        else
        {
	        $$ctx{LOGGER}->info("Oracle file not found: $oratab_file. Oracle not installed.");
        }

        return ($nResult, $oracle_user);

}

# Oracle is only supported on Linux platforms so we will make that assumption
# that we will only be called on Linux platforms
sub _ralus_check_for_oracle {
	my ($ctx, $groupname) = @_;
	
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_check_for_oracle");

	my $oracleingroup = 0;
	my $oracleuserfound = 0;
	my $grep = 'grep';
        my $cat = 'cat';
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

        my ($command);

        # First let's see if we have an "oracle" user on the linux box
	# Locate cat and use full path (location differs on different platforms)
	if (Utils::vxif_fc("/bin/cat", $ctx)) {
		$cat = '/bin/cat';
	} elsif (Utils::vxif_fc("/usr/bin/cat", $ctx)) {
		$cat = '/usr/bin/cat';
	}
        
        # Get the Oracle User name if it exist
        my ($oracle_installed, $oracle_user) = _ralus_get_oracle_user_name($ctx);
        return (0,0) if ($oracle_installed == 0);

        # get list of users on the target linux box
        $command = $cat . " /etc/passwd | cut -d\":\" -f1";
	$$ctx{LOGGER}->info("Search for existance of 'beoper' group via: $command");
	my (@users) = Utils::vxif_dor("${command}", "", $ctx);
        for my $user (@users) {
                if ($user =~ m/$oracle_user/)
                {
			$$ctx{LOGGER}->info("'$user' for 'beoper' group was found.");
			$oracleuserfound = 1;
			last;
                }
        }
        # if oracle user not found we need to bail since no need to add to beoper group
        if ( $oracleuserfound == 0)
        {
		$$ctx{LOGGER}->info("Oracle user not found.");
                return (0,0);
        }
        
	# Locate grep and use full path (location differs on different platforms)
	if (Utils::vxif_fc("/bin/grep", $ctx)) {
		$grep = '/bin/grep';
	} elsif (Utils::vxif_fc("/usr/bin/grep", $ctx)) {
		$grep = '/usr/bin/grep';
	}
        
	# grep for group in /etc/group
	$command = $grep . " \"" . "${groupname}" . "\"" .  ' /etc/group';
	$$ctx{LOGGER}->info("Search for existance of 'beoper' group via: $command");
	my ($Results) = Utils::vxif_dor("${command}", "", $ctx);
	$$ctx{LOGGER}->fine("Results: $Results");

	my ($name, $temp, $gid, @members) = split(/:/, $Results);
	if ($name eq $groupname) {
		# Group found, now check for memeber
		$$ctx{LOGGER}->info("'$groupname' group was found.");
		$groupfound = 1;

		for my $member (@members) {
			if ($member =~ m/$oracle_user/) 
			{
				$$ctx{LOGGER}->info("'$member' for '$groupname' group was found.");
				$oracleingroup = 1;
				last;
			}
		}
	}

	return ($oracleingroup, $oracleuserfound);
}
        
sub _ralus_check_for_beoper_group {
	my ($ctx, $groupname) = @_;
	
	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_check_for_beoper_group");

	my $groupfound = 0;
	my $rootfound = 0;
	my $grep = 'grep';

	# Locate grep and use full path (location differs on different platforms)
	if (Utils::vxif_fc("/bin/grep", $ctx)) {
		$grep = '/bin/grep';
	} elsif (Utils::vxif_fc("/usr/bin/grep", $ctx)) {
		$grep = '/usr/bin/grep';
	}
        
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

        my ($command);

	if ($trg_os_name eq "Darwin")
	{
                # We need to check the NetInfo DB on Mac for 'beoper" group
                $command = "sudo nidump group . | " . $grep . " beoper:";

        } else {
	        # grep for group in /etc/group
	        $command = $grep . " \"" . "${groupname}" . "\"" .  ' /etc/group';
        }
	$$ctx{LOGGER}->info("Search for existance of 'beoper' group via: $command");
	my ($Results) = Utils::vxif_dor("${command}", "", $ctx);

	# Since we are using Null as the 2nd parameter we will not receive an Array in our Result
	#my $result = @{$Results}[0];
	#my $output = @{$Results}[1];
	$$ctx{LOGGER}->fine("Results: $Results");

	my ($name, $temp, $gid, @members) = split(/:/, $Results);
	if ($name eq $groupname) {
		# Group found, now check for memeber
		$$ctx{LOGGER}->info("'$groupname' group was found.");
		$groupfound = 1;

		for my $member (@members) {
			if ($member =~ m/root/) 
			{
				$$ctx{LOGGER}->info("'$member' for '$groupname' group was found.");
				$rootfound = 1;
				last;
			}
		}
	}

	return ($groupfound, $rootfound);
}

sub _ralus_check_for_NIS {
	my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSCommon::_ralus_check_for_NIS");

	# Locate ypwhich and use full path (location differs on different platforms)
	my $ypwhich = 'ypwhich';
	if (-f '/usr/bin/ypwhich') 
	{
		$ypwhich = '/usr/bin/ypwhich';
	}

	# Run ypwhich to see if the machine is part of NIS
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");

	my $command = $ypwhich . ' 2>&1';
	if ($trg_os_name eq "Darwin")
	{
		$command = $ypwhich . ' -m';
	}
	$$ctx{LOGGER}->info("Check for NIS on Target Server using: $command");
	my ($Results) = Utils::vxif_dor("${command}", "", $ctx);

	# Since we are using Null as the 2nd Parameter we will not get an Array back as a Result
	#my $result = @{$Results}[0];
	#my $output = @{$Results}[1];
	$$ctx{LOGGER}->fine("Results: $Results");

	my $NISServerName = "";
	if ($Results ne "")
	{
		$NISServerName = $Results;
	}

	# Check the output of ypwhich - it can be error or name of the NIS server
	if (length($NISServerName) > 0)
	{
		chop($NISServerName);
		if (($NISServerName =~ m/^ypwhich: can't get local yp domain/i) || 
		    ($NISServerName =~ m/^ypwhich:  the domainname hasn't been set on this machine/i) || 
		    ($NISServerName =~ m/^ypwhich: /i) || 
		    ($NISServerName =~ m/command not foun/i) || 
		    ($NISServerName =~ m/No such file or director/i) ||
		    ($NISServerName =~ m/^Can't get map list for domain/i)) 
		{
			$NISServerName = "";
		}
	} 
	else 
	{
		$NISServerName = "";
	}

	my $NISFound = (length($NISServerName) > 0) ? 1 : 0;
	return ($NISFound, $NISServerName);
}

# 
# Get the max index used for Agent Directory List from the existing config file (if it exists)
#
# There are 2 MAX indices...
# 
# 1. MaxIndex
# - tracks the MAX value used by Agent Directory List
# 
# 2. MaxIndexForMigratedServers
# - tracks the MAX value used by Agent Directory List for migrated servers
#
# This is done just so that we can easily distinguish migrated servers from user provided servers.
# Basically, when index is > BaseIndexForMigratedServers, it means that this is a migrated server.
# 
sub ralus_migrate_cfg {
	my $self = shift;
   	my ($ctx) = @_;
	my ($sf) = $CONF{RALUSCFG};

	$$ctx{LOGGER}->entering("RALUSCommon::ralus_migrate_cfg");

	if ( Utils::vxif_fc($sf,$ctx) ) 
	{
		# First make a temp copy of the existing file on local machine
		my $tf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";
		# Copy from Remote machine to Local Machine
		$$ctx{LOGGER}->finer("Copy ${sf} to ${tf}");
		Utils::vxif_copy($sf,$tf,$ctx,$$ctx{TARGET}{HOST},$$ctx{LOCAL}{HOST});
		$$ctx{LOGGER}->finer("Reading ${tf}");
		my $buffer = Utils::vxif_readf($tf);
		if ($buffer ne "") 
		{
			my (@servers, $n);
			my @lines = split(/\n/, $buffer);
			foreach $n (0..$#lines) 
			{
				my $lookfor = "Agent Directory List (.*)=";
				if ($lines[$n] =~ m/$lookfor/)
				{
					my ($key, $server) = split(/=/, $lines[$n]); 
					chomp($server);
	                                push(@servers, $server);
					$$ctx{LOGGER}->info("Adding $server to Migrated Server List.");
	                        }
			}

			# Add Media Servers to Hash CFG Item
	                if ($#servers>=0) 
		        {	
	                	$CFG{MIGRATEMDSRVRS}=[];
	                        @{$CFG{MIGRATEMDSRVRS}}=@servers;
				$$ctx{LOGGER}->info("Saving servers to MIGRATEDMDSRVRS.");
		        } else {
				$$ctx{LOGGER}->warning("WARNING (V-225-251): No Media Servers found in RALUS config file.");
	                }
		
			# Finally, remove the local temp copy
			Utils::vxif_dol("rm -f $tf");

		} else {
			$$ctx{LOGGER}->error("ERROR (V-225-246): Empty config file at ${tf}");
		}
	}
}

# Create RALUS Configfiles.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the CFG hash;
#        3) the installation directory;
#
# Return: 1 if successful; 0 otherwise.
#
sub create_configfiles {
	my ($self,$ctx,$cfg,$mode,$installdir) = @_;
        my @ADirListEntries;

	$$ctx{LOGGER}->entering("RALUSCommon::create_configfiles");

	# 
	# Base value for Migrated Servers. 
	# All migrated server entries will have index >= this vlaue.
	#
	my $BaseIndexForMigratedServers = 10000;
	my($msg, $n, $indx);
	
	my $createcfg_msg = Utils::_tr("Creating configuration files on $$ctx{TARGET}{HOST}", 250, 1057, "$$ctx{TARGET}{HOST}");
	Utils::vxif_pbl($createcfg_msg);
	if ( $CFG{USERNAME} ) {
		$msg = "Software\\Veritas\\Backup Exec\\Engine\\RALUS\\MachineUsername=$CFG{USERNAME}\n";
		$msg .= "Software\\Veritas\\Backup Exec\\Engine\\RALUS\\MachinePassword=$CFG{PASSWORD}";
		Utils::vxif_writef($msg, $CONF{RALUSCFG}, 1 ); # Set 1 for Appending for now
	}

	# 
	# Get the max index used for Agent Directory List from the existing config file (if it exists)
	#
	# There are 2 MAX indices...
	# 
	# 1. MaxIndex
	# - tracks the MAX value used by Agent Directory List
	# 
	# 2. MaxIndexForMigratedServers
	# - tracks the MAX value used by Agent Directory List for migrated servers
	#
	# This is done just so that we can easily distinguish migrated servers from user provided servers.
	# Basically, when index is > BaseIndexForMigratedServers, it means that this is a migrated server.
	# 
	my $MaxIndex = 0;
	my $MaxIndexForMigratedServers = 0;
	my ($sf) = $CONF{RALUSCFG};

	if ( Utils::vxif_fc($sf,$ctx) ) {
		# First make a temp copy of the existing file on local machine
		$tf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";
		# Copy from Remote machine to Local Machine by passing
		# 3rd Parameter with a 1.
		$$ctx{LOGGER}->finer("Copy ${sf} to ${tf}");
		Utils::vxif_copy($sf,$tf,$ctx,$$ctx{TARGET}{HOST},$$ctx{LOCAL}{HOST});
		$$ctx{LOGGER}->finer("Reading ${tf}");
		$buffer = Utils::vxif_readf($tf);
		if ($buffer ne "") {
			@lines = split(/\n/, $buffer);
			foreach $n (0..$#lines) {
				my $lookfor = "Agent Directory List (.*)=";
				if ($lines[$n] =~ m/$lookfor/) {
					if ($1 >= $BaseIndexForMigratedServers) {
						my $Temp = $1 - $BaseIndexForMigratedServers;
						if ($MaxIndexForMigratedServers < $Temp) {
							$MaxIndexForMigratedServers = $Temp;
						}
                    			} else {
						if ($MaxIndex < $1) {
							$MaxIndex = $1;
						}
					}

                                        # Add Agent Directory List to my local reference
                                        my $ValueIdx = rindex($lines[$n], "=");
                                        if ($ValueIdx > -1) {
                                                my $MediaServer = substr( $lines[$n], $ValueIdx+1 );

                                                push(@ADirListEntries, $MediaServer);
			                        $$ctx{LOGGER}->finer("Entry Found: $MediaServer");
                                        }
				}
			}
			$$ctx{LOGGER}->finer("MaxIndex=${MaxIndex}\nMaxIndexForMigratedServers=${MaxIndexForMigratedServers}");
		} else {
			$$ctx{LOGGER}->finer("Empty config file at ${tf}");
		}

		# Finally, remove the local temp copy
		Utils::vxif_dol("rm -f $tf");
	}

        my $cfg = $ctx->get_config();
        my $addresses = $cfg->{ADDRESSES};
	# Add Directory Host names and/or addresses to config file
	if ( $addresses || $CFG{ADDRESSES} || $CFG{PREVRALUSMDSRVRS} ) # $migratedsrvrs || $CFG{MIGRATEDSRVRS}
	{
                $$ctx{LOGGER}->finer("Add Directory Host names and/or ADDRESSES to config file");

                my @ServersToBeAdded;
		
                $msg="";
                my $indx = $MaxIndex;
                my @mediaservers = split(/ /, $addresses);
                foreach my $name (@mediaservers) {
                        my $Found = 0;
                        foreach my $item (@ServersToBeAdded) {
                                if ( $name eq $item ) {
                                        $Found = 1;
                                }
                        }
                        next if ($Found);
                        push(@ServersToBeAdded, $name);
		}

                foreach my $n (0..$#{$CFG{ADDRESSES}}) {
                        my $Found = 0;
                        foreach my $item (@ServersToBeAdded) {
                                if ( $item eq ${$CFG{ADDRESSES}}[$n] ) {
                                        $Found = 1;
                                }
                        }
                        next if ($Found);
                        push(@ServersToBeAdded, ${$CFG{ADDRESSES}}[$n]);
                }
                
                foreach my $n (0..$#{$CFG{PREVRALUSMDSRVRS}}) {
                        my $Found = 0;
                        foreach my $item (@ServersToBeAdded) {
                                if ( $item eq ${$CFG{PREVRALUSMDSRVRS}}[$n] ) {
                                        $Found = 1;
                                }
                        }
                        next if ($Found);
                        push(@ServersToBeAdded, ${$CFG{PREVRALUSMDSRVRS}}[$n]);
                }
                
                my $Count = 0;
                foreach my $server (@ServersToBeAdded)
                {
                        # Check to see if the New Server already exists in our list
                        my $Found = 0;
                        foreach my $ExistingSrvr (@ADirListEntries) {
                                if ( $ExistingSrvr eq $server ) {
                                        $Found = 1;
                                        $$ctx{LOGGER}->finer("create_configfiles: Duplicate Found : $server");
                                }
                        }

                        if (!$Found) {
                                if ($Count != 0) {
                                        $msg.= "\n";
                                } 
                                $Count = $Count + 1;
			        $indx = $indx + 1;
			        $msg.= $CONF{RALUSADVMEDSRVRKEY} . " $indx=$server";
                                $$ctx{LOGGER}->finer("create_configfiles : Adding: $server");
                        }
                }
        }	
		$$ctx{LOGGER}->info("Checking certificate information.");

		# Add Certificate Information to config file
		if ($#{$CFG{CERTIFICATEINFO}} >=0 )
		{
				 $$ctx{LOGGER}->info("Adding certificate information in cfg ");
				foreach my $n (0..$#{$CFG{CERTIFICATEINFO}}) {
								$msg.= "\n";
				$msg.= ${$CFG{CERTIFICATEINFO}}[$n];
						$$ctx{LOGGER}->finer("create_configfiles : Adding: $server");
				}
		}
		$$ctx{LOGGER}->info("Checking Oracle Linux information.");
		# Add Oracle Linux Information to config file
		if ($#{$CFG{ORACLELINUXINFO}} >=0 )
		{
				 $$ctx{LOGGER}->info("Adding Oracle Linux information in cfg ");
				foreach my $n (0..$#{$CFG{ORACLELINUXINFO}}) {
					$msg.= "\n";
					$msg.= ${$CFG{ORACLELINUXINFO}}[$n];
					$$ctx{LOGGER}->finer("create_configfiles : Adding: ${$CFG{ORACLELINUXINFO}}[$n]");
				}
		}
		
		$$ctx{LOGGER}->info("Checking Ralus cfg information.");
		# Add RALUS CFG Information to config file
		if ($#{$CFG{RALUSINFO}} >=0 )
		{
				 $$ctx{LOGGER}->info("Adding RALUS cfg information in cfg ");
				foreach my $n (0..$#{$CFG{RALUSINFO}}) {
					$msg.= "\n";
					$msg.= ${$CFG{RALUSINFO}}[$n];
					$$ctx{LOGGER}->finer("create_configfiles : Adding: ${$CFG{RALUSINFO}}[$n]");
				}
		}		
		
		# Write out to the RALUS.CFG for each system
		# that we have installed too...
		#foreach $SYS (@SYSI)
		#{
			#$sf = $CONF{RALUSCFG};
			if ((Utils::vxif_fc($sf,$ctx)) && ($msg ne ""))
			{
				# Log Message that we are about to write to config
				$$ctx{LOGGER}->finer("Prepare writing to ${tf}");
		
				$tf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";
				# Copy from Remote machine to Local Machine by passing
				# 3rd Parameter with a 1.
				$$ctx{LOGGER}->finer("Copy ${sf} to ${tf}");
				Utils::vxif_copy($sf,$tf,$ctx,$$ctx{TARGET}{HOST},$$ctx{LOCAL}{HOST});
				# Update local copy with Media Servers
				$$ctx{LOGGER}->finer("Writing to ${tf} following data:\n${msg}");
				Utils::vxif_writef($msg, $tf, 0); # Set 0 for Appending

				# Copy local copy to the Remote Machine
				$$ctx{LOGGER}->finer("Copy ${tf} to ${sf}");
				Utils::vxif_copy($tf,$sf,$ctx,$$ctx{LOCAL}{HOST},$$ctx{TARGET}{HOST});

				# Remove the local copy
				Utils::vxif_dol("rm -f $tf");
			} else {

				$$ctx{LOGGER}->info("Warning (V-225-246): Cannot find ralus.cfg on remote machine.");
			}
		#}
		$$ctx{LOGGER}->finer("Finished Writing out ralus.cfg file.");

        #Remove unwanted symantec keys from ralus.cfg
        $sf = $CONF{RALUSCFG};
        $tf = "$$ctx{TMPPATH}/$CONF{RALUSCFGFILENAME}";
        Utils::vxif_dor("grep -v 'Software\\\\Symantec' $sf > $tf", "", $ctx);
        Utils::vxif_dor("cp $tf $sf", "", $ctx);


	my $done_msg = Utils::_tr("Done", 250, 1003);
	Utils::vxif_pbr($createcfg_msg, $done_msg);
        
        # Startup the Agent if the User requested Automatic Startup after Install....
        # Ask user if they want us to automatically start the agent after install
	#Utils::vxif_title(Utils::_tr("Agent Startup", 250, 1061));
	Utils::vxif_pl("\n");
	my $msg = Utils::_tr("The agent installer can automatically start the agent after the installation completes. Do you want to automatically start the agent on '$$ctx{TARGET}{HOST}'?", 250, 1060, "$$ctx{TARGET}{HOST}");
        my $ayn=Utils::vxif_ayny( $msg );
        if ( $ayn eq "Y" ) {
                my $RetVal = launchAgent($ctx); 
        }

        # Display Message to user to configure trust relationship with media server
	my $msg = Utils::_tr("To establish a trust between the Backup Exec server and the agent, move to the Backup Exec server and add the agent computer to the list of servers.\n\nTo add the agent computer, click Add on the Backup and Restore tab.", 260, 1017);
        Utils::vxif_pl("\n"); 
        Utils::vxif_pl($msg); 
       
	return 1;
}

sub launchAgent {
        my ($ctx) = @_;
	my $msg;
        	
        Utils::vxif_pl("\n"); 
	my $startup_msg = Utils::_tr("Starting agent on $$ctx{TARGET}{HOST}", 250, 1062, "$$ctx{TARGET}{HOST}");
	Utils::vxif_pbl($startup_msg);
        
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	$$ctx{LOGGER}->info("Target OS name: $trg_os_name");
	if ($trg_os_name eq "Linux" || $trg_os_name eq "SunOS" || $trg_os_name eq "AIX") {
                $cmd = "/opt/VRTSralus/bin/beremote > /var/VRTSralus/beremote.service.log 2>/var/VRTSralus/beremote.service.log &";
        } else {
                $cmd = "SystemStarter start VRTSrams";
        }
	$$ctx{LOGGER}->info("vxif_dor($cmd)");
        my $result = Utils::vxif_dor($cmd, "", $ctx);
        if ($result =~ "FAILED") {
	        $msg = Utils::_tr("Failed", 250, 1045);
        } else {
	        $msg = Utils::_tr("Done", 250, 1003);
        } 
	$$ctx{LOGGER}->info("vxif_dor result: $result");
	Utils::vxif_pbr($startup_msg, $msg);
        
        return 1;
}

1;
