# Copyright (c) 2022 Veritas Technologies LLC. All Rights Reserved SY39-6648-5573-26-15-5
# RMALLinux interfaces
package RMALLinux;

use User::pwent;
use VxIF::ProductHelper;
use scripts::RALUSCommon;
@RMALLinux::ISA = qw(RALUSCommon ProductHelper);

sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $self = {@_,};

	bless($self, $class);

	$self->initialize();

	return $self;
}

sub initialize {
	my $self = shift;

	$self->ProductHelper::init();
}

################################################
# Product Dependency Interfaces                #
################################################

#
# Check to make sure the system names are valid.
#
# This interface is optional. By default, VxIF do some generic checking, such as making
# sure the name contains the valid characters. Apart from that it is up the each individual
# product to impose additional contraints.
#
# Input: 1) the reference to the installation context;
#        2) the array of systems that will be installing this product;
#
# Return: 1 if the system names are valid; 0 otherwise.
#
sub check_system_names (\%@) {
  my ($self,$ctx,@hosts) = @_;

  $$ctx{LOGGER}->entering("RMALLinux::check_system_names");

  return 1;
}

#
# Get the localized product title for this product.
#
# This interface is mandatory. 
#
# Input: 1) the reference to the installation context;
#
# Return: the complete localized title for this product.
#
sub get_product_title (\%) {
  my ($self, $ctx) = @_;

  $$ctx{LOGGER}->entering("RMALLinux::get_product_title");

  my $title = Utils::_tr("Backup Exec Remote Media Agent for Linux");

  return ($title);
}

#
# Get the external packages for which this product depended on.
# Any package that is not part of this product are consider external
# packages. VxIF will first check for the external dependent packages before 
# installing the product.
#
# This interface is optional.
#
# This interface shall return the reference of a hash with the dependent 
# package name as the keys with the following properties.
#
# {NAME}	- dependent package name; This is mandatory;
# {RPM}		- the rpm filename associated with this package;
# {VERSION}	- dependent package version; This is optional; If the dependent
#                 version is not specified; VxIF will assume this product depended
#                 on any version of this package;
# {DESC}	- localized dependent package description; This is optional;
#
# Input: 1) the reference to the installation context;
#        2) host OS version;
#        3) host OS architecture;
#        4) the selected installation mode if applicable;
#
# Return: the reference of a hash with the dependent package name as the key and the
#         dependent package version and the localized description as the values.
#
#sub get_external_dependent_packages (\%$$;$) {
#}

#
# What to do if a given external dependent package exist on the install host
# and its version is different from the one this product is expecting? 
#
# This interface is option. By default, VxIF assumes the existing external dependent
# package is not supported and stop the installation.
#
# Input: 1) the reference to the installation context;
#        2) the external dependent package name;
#        3) the actual dependent package version;
#        4) host OS version;
#        5) host OS architecture;
#        6) the expected dependent package version;
#
# Return: 1 if the existing external dependent package is compatible with this product
#         and continue with the installation; 0 if the existing external dependent 
#         package is not supported and stop the installation.
#
#sub is_external_dependent_version_supported (\%$$$$$) {
#}

#
# Get the list of requred patches, both external application and OS patches, for this product. 
# VxIF will first check for the required patches before installing the product.
#
# This interface is optional.
#
# This interface shall return the a list which contains the patch ID of the required
# patches for this product.
#
# Input: 1) the reference to the installation context;
#        2) host OS version;
#        3) host OS architecture;
#        4) the selected installation mode if applicable;
#
# Return: a list which contains the patch name of the required patches.
#
#sub get_required_external_patches (\%$$;$) {
#}


################################################
# Packages and Patches Definition Interfaces   #
################################################

#
# Return a list of patches for a given version of this product. The given 
# version may not be the current version. The reason is that for product upgrade 
# and product downgrade, VxIF needs a list of patches associated with a particular 
# version so they can properly removed.
#
# This interface is mandatory for patch-only install.
#
# This interface shall return the reference of a hash with patch IDs as the keys
# with the following properties.
#
# {NAME}    	- the patch name or title; This property is mandatory;
# {RPM}		- the RPM filename associate with this patch;
# {PACKAGE}	- the package this patch is intended for; This property is mandatory;
# {SPACE}	- a list which specified the spaces, in kilobytes, needed for this
#                 patch in /opt, /usr, /var, and /; this property is mandatory;
# {REQREBOOT}	- require system reboot after applying this patch? This property is
#                 optional;
# {OBSOLETES} 	- the obsoletes patches; This property is optional;
# {INCOMPAT}	- incompatible patches; This property is optional;
# {REQUIRES}	- a list patches, within this product, that this patch depended on;
#                 If a dependent patch is not part of this product, it must be
#                 included in get_required_external_patches(); This property is optional;
# {DESC}	- localized patch description; This property is optional;
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the reference of the hash which contains the packages to be installed.
#
#sub get_product_patches (\%$$$;$) {
#}


################################################
# Package and Patch Installation Interfaces    #
################################################

#
# Pre pkgadd operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_pkgadd (\%$$$$) {
#}

#
# Post pkgadd operations. 
#

# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_pkgadd (\%$$$$) {
#}

#
# Pre pkgrm operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_pkgrm (\%$$$$) {
#}

#
# Post pkgrm operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_pkgrm (\%$$$$) {
#}

#
# Pre patchadd operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch ID;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_patchadd (\%$$$$) {
#}

#
# Post patchadd operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch ID;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_patchadd (\%$$$$) {
#}

#
# Pre patchrm operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch ID;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub pre_patchrm (\%$$$$) {
#}

#
# Post patchrm operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch ID;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
#sub post_patchrm (\%$$$$) {
#}
#

# Product pre-uninstallation script. This interface will be called before the product
# is being uninstalled.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-uninstallation is successful. Otherwise, return 0;
#
sub pre_product_uninstall (\%\%;$) {
 my $self = shift;
 my ($ctx, $mode) = @_;

        $$ctx{LOGGER}->entering("RMALLinux::pre_product_uninstall");

	# Let's call the base class first for inital checks
        if ( $self->RALUSCommon::pre_product_uninstall($ctx, $mode) == 0 ) {
                return 0;
        }
	
	_ralus_linux_shutdown_ralus($ctx);

        return 1;
}

#
# Product pre-installation prerequisite checking script. This interface should be called during
# the pre_product_install process to insure it is part of the first pre-checking phase of the install
# process.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the pre-installation is successful. Otherwise, return 0;
#
sub _ralus_linux_check_prerequisites {
 my ($ctx) = @_;
 my $found = 0;   # Default to Not Found
 my $success = 1; # Set Default to TRUE for now since we will always allow the install
                  # to continue.

        $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_check_prerequisites");
               
        # Display the Prerequiste check message
        my $msg = Utils::_tr("Checking for prerequisites", 260, 1000);
        Utils::vxif_pbl($msg);
        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));

        my $reqfile = "/lib64/libnsl.so.1";
        my $linkfile = "/lib64/libnsl.so.2.0.0";
        my $redhatfile = "/etc/redhat-release";

        #if not a red hat distro just bail out as success do not go ahead and check as folder structures are different in other OS
        $found = Utils::vxif_fc($redhatfile, $ctx);
        if (! $found)
        {
           return $success;
        }

        $found = Utils::vxif_fc($reqfile, $ctx);
        my $foundlink = Utils::vxif_fc($linkfile, $ctx);

        if (! $found)
        {
           if (! $foundlink)
           {
               Utils::vxif_title(Utils::_tr("Backup Exec RALUS Agent Prerequisite Warning!!!", 260, 1014));
               $msg = Utils::_tr("The installer cannot find the libnsl package that is required to run the Linux agent version. The installation will continue without the package. Ensure that you install the package before you start the Linux agent.\n ", 260, 1015);
               Utils::vxif_pl($msg);
               Utils::vxif_pl("Refer to the following link for more information: https://www.veritas.com/support/en_US/article.100046985\n");
               $$ctx{LOGGER}->error("ERROR Prerequisite Warning - libnsl.so.1 missing.");
               Utils::vxif_prtc($ctx, 0);
           }
		   else
		   {
		       $$ctx{LOGGER}->info("INFO: We found package libnsl installed on machine but libnsl.so.1 does not exist so we will try to do a softlink for that file");
		   }
        }
		else
		{
		       $$ctx{LOGGER}->info("INFO: Backup Exec Agent pre-requisite check completed successfully");
		}

        return $success;
}

#
# Product pre-installation script. This interface will be called before the product
# is being installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-installation is successful. Otherwise, return 0;
#
sub pre_product_install (\%\%;$) {
 my $self = shift;
 my ($ctx, $mode) = @_;
 my $success = 0;

        $$ctx{LOGGER}->entering("RMALLinux::pre_product_install");

        # Shutdown RMAL if it's running
        _ralus_linux_shutdown_ralus($ctx);

        # Let's check to see if we have all Prerequistes installed for RMAL
        my $hdwr_plat = Utils::vxif_dor("uname -i", "", $ctx);
        if ( $hdwr_plat eq "x86_64" ) {
            $success = _ralus_linux_check_prerequisites($ctx);
        } 

        $success = $self->RALUSCommon::pre_product_install($ctx, $mode);

        return $success;
}

# Process to remove specific RMAL Product Directories.
#
# This process will remove all directories associated 
# with the RMAL package during the uninstall process.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_shutdown_ralus {
 my ($ctx) = @_;

        $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_shutdown_ralus");
        
	# setup my default initdir
	my $initdir = "/etc/init.d";
	if ( Utils::vxif_dc("/etc/rc.d/init.d", $ctx) )
	{
		$initdir = "/etc/rc.d/init.d";
	}
	$$ctx{LOGGER}->info("Default init script path set to: $initdir");
        
        # Define our runtime Init Script
        my $initfile = "$initdir/VRTSralus.init";

        # if the Init Script exists then we need to attempt to shutdown
        # the Agent if it is running before install/uninstalling
        if (Utils::vxif_fc($initfile, $ctx))
        {
        
                $msg = Utils::_tr("Stopping the Veritas agent service", 260, 1000);
                Utils::vxif_pbl($msg);
                $$ctx{LOGGER}->info("Stopping Veritas agent at: $initfile.");

                # Execute Shutdown of RMAL
        	$cmd="$initfile stop";
		$result = Utils::vxif_dor($cmd, "", $ctx);
                # Notify user that we have finished shutting down RMAL
                Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));

        }

        return 1; 
}

# Process to remove specific RMAL Product Directories.
#
# This process will remove all directories associated 
# with the RMAL package during the uninstall process.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_remove_pkgdirs {
  	my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_remove_pkgdirs");

        my($cmd,$result);
        # Manual UnInstall Steps to cleanup if necessary...
        if ( Utils::vxif_dc("/opt/VRTSralus", $ctx) )
        {
        	$cmd="rm -rf /opt/VRTSralus";
		$result = Utils::vxif_dor($cmd, "", $ctx);
                if (Utils::vxif_dc("/opt/VRTSralus", $ctx))
                {
			$$ctx{LOGGER}->error("ERROR (V-225-253): vxif_dor failed ${cmd}");
                        $$ctx{LOGGER}->finer("result=${result}");
		} else {
			$$ctx{LOGGER}->finer("Successful vxif_dor: ${cmd}");
		}
        }

        #if ((!Utils::vxif_fc("/opt/VRTSvxms/lib/libmap.so", $ctx))     || 
        #    (!Utils::vxif_fc("/opt/VRTSvxms/lib/libvfutil.so", $ctx)))
        #{
        #	$cmd="rm -rf /opt/VRTSvxms";
        #	$result = Utils::vxif_dor($cmd, "", $ctx);

                # Check to see if the VRTSvxms directory still exists on Target machine
        #        if (Utils::vxif_dc("/opt/VRTSvxms", $ctx))
        #        {
        #		$$ctx{LOGGER}->error("ERROR (V-225-253): vxif_dor failed ${cmd}, result=${result}");
        #	} else {
        #		$$ctx{LOGGER}->finer("Successful vxif_dor: ${cmd}");
        #        }
        #}

        # Check the /var/VRTSralus dir now ....
        if (Utils::vxif_dc("/var/VRTSralus", $ctx))
        {
        	$cmd="rm -rf /var/VRTSralus";
		$result = Utils::vxif_dor($cmd, "", $ctx);

                # Check to see if successful
		if (Utils::vxif_dc("/var/VRTSralus", $ctx))
                {
			$$ctx{LOGGER}->error("ERROR (V-225-253): vxif_dor failed ${cmd}");
                        $$ctx{LOGGER}->finer("result=${result}");
		} else {
			$$ctx{LOGGER}->finer("Successful vxif_dor: ${cmd}");
		}
        }

        # Check the /etc/VRTSralus dir now ....
        if (Utils::vxif_dc("/etc/VRTSralus", $ctx))
        {
        	$cmd="rm -rf /etc/VRTSralus";
		$result = Utils::vxif_dor($cmd, "", $ctx);

                # Check to see if successful
		if (Utils::vxif_dc("/etc/VRTSralus", $ctx))
                {
			$$ctx{LOGGER}->error("ERROR (V-225-253): vxif_dor failed ${cmd}");
                        $$ctx{LOGGER}->finer("result=${result}");
		} else {
			$$ctx{LOGGER}->finer("Successful vxif_dor: ${cmd}");
		}
	}

	return 1;
}

# Process to remove specific RMAL Product init scripts.
#
# This process will remove all linkednames and init scripts
# associated with the RMAL package during the uninstall process.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_remove_init_scripts {
 	my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_remove_pkgdirs");

	# setup my default startup directory
	my $rcdir = "/etc/rc.d";
	if ( Utils::vxif_dc("/etc/init.d/rc2.d", $ctx) )
	{
		$rcdir = "/etc/init.d";
	} elsif ( Utils::vxif_dc("/etc/rc2.d", $ctx) ) {
                # Add support for Debian/Ubuntu systems
                $rcdir = "/etc";
        }
	$$ctx{LOGGER}->info("Default init script path set to: $initdir");

	# Run chkconfig which will create the init runtime scripts for RALUS
        # and If we are on a SuSE or OES box chkconfig will call the insserv utility
	if ( Utils::vxif_fc("/sbin/chkconfig", $ctx) )
	{
		$cmd="/sbin/chkconfig --del VRTSralus.init";
	        Utils::vxif_dor($cmd, "", $ctx);
	        $$ctx{LOGGER}->info("Executed chkconfig cmd: $cmd");
        } else {
                $$ctx{LOGGER}->info("Unable to remove RMAL Init Runtime Scripts via chkconfig utility.");

	        # Remove linkedname in rc5.d if it exists
                $msg = Utils::_tr("Removing RMAL linkedname in rc5.d", 260, 1002);
                Utils::vxif_pbl($msg);
                $initfile="$rcdir/rc5.d/*VRTSralus.init";
                $$ctx{LOGGER}->info("Removing RMAL linkedname in rc5.d at: $initfile.");
                Utils::vxif_dor("rm $initfile","",$ctx);
                if ( Utils::vxif_fc("$initfile",$ctx) )
                {
                        Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
                        $$ctx{LOGGER}->error("ERROR (V-225-254): Removing RMAL linkedname in rc5.d Failed.");
                } else {
                        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
                        $$ctx{LOGGER}->info("Removing RMAL linkedname in rc5.d was successful.");
                }

	        # Remove linkedname in rc3.d if it exists
                $msg = Utils::_tr("Removing RMAL linkedname in rc3.d", 260, 1004);
                Utils::vxif_pbl($msg);
                $initfile="$rcdir/rc3.d/*VRTSralus.init";
                $$ctx{LOGGER}->info("Removing RMAL linkedname in rc3.d at: $initfile.");
                Utils::vxif_dor("rm $initfile","",$ctx);
                if ( Utils::vxif_fc("$initfile",$ctx) )
                {
                        Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
                        $$ctx{LOGGER}->error("ERROR (V-225-254): Removing RMAL linkedname in rc3.d Failed.");
                } else {
                        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
                        $$ctx{LOGGER}->info("Removing RMAL linkedname in rc3.d was successful.");
                }

	        # Remove linkedname in rc2.d if it exists
                $msg = Utils::_tr("Removing RMAL linkedname in rc2.d", 260, 1005);
                Utils::vxif_pbl($msg);
                $initfile="$rcdir/rc2.d/*VRTSralus.init";
                $$ctx{LOGGER}->info("Removing RMAL linkedname in rc2.d at: $initfile.");
                Utils::vxif_dor("rm $initfile","",$ctx);
                if ( Utils::vxif_fc("$initfile",$ctx) )
                {
                        Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
                        $$ctx{LOGGER}->error("ERROR (V-225-254): Removing RMAL linkedname in rc2.d Failed.");
                } else {
                        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
                        $$ctx{LOGGER}->info("Removing RMAL linkedname in rc2.d was successful.");
                }

        }
		# setup my default initdir
		my $initdir = "/etc/init.d";
		if ( Utils::vxif_dc("/etc/rc.d/init.d", $ctx) )
		{
				$initdir = "/etc/rc.d/init.d";
		}
		$$ctx{LOGGER}->info("Default init script path set to: $initdir");

		# Delete the VRTSralus.init file
			$msg = Utils::_tr("Removing RMAL init script", 260, 1006);
			Utils::vxif_pbl($msg);
			$initfile = "$initdir/VRTSralus.init";
			$$ctx{LOGGER}->info("Removing RMAL init script at: $initfile.");
			Utils::vxif_dor("rm $initfile","",$ctx);
			if ( Utils::vxif_fc("$initfile",$ctx) )
			{
					Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
					$$ctx{LOGGER}->error("ERROR (V-225-254): Removing RMAL init script Failed.");
			} else {
					Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
					$$ctx{LOGGER}->info("Removing RMAL init script was successful.");
			}

        return 1;
}

# Process to remove specific RALUS Product unit script.
#
# This process will remove unit script
# associated with the RALUS package during the uninstall process.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_remove_suse_unit_scripts {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSLinux::_ralus_linux_remove_suse_unit_scripts");

    # setup my default unitdir
	my $unitdir = "/etc/systemd/system";
	$$ctx{LOGGER}->info("Default unit script path set to: $unitdir");

    #First Disabling the unit file
    $cmd = "systemctl disable RALUSUnit.service";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->info("Disabling unit file from unit directory was successful.");

	# Delete the RALUSUnit.service file
		$msg = Utils::_tr("Removing RALUS unit script", 260, 1006);
		Utils::vxif_pbl($msg);
		$unitfile = "$unitdir/RALUSUnit.service";
		$$ctx{LOGGER}->info("Removing RALUS unit script at: $unitfile.");
		Utils::vxif_dor("rm $unitfile","",$ctx);
		if ( Utils::vxif_fc("$unitfile",$ctx) )
		{
				Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
				$$ctx{LOGGER}->error("ERROR (V-225-254): Removing RALUS unit script Failed.");
		} else {
				Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
				$$ctx{LOGGER}->info("Removing RALUS unit script was successful.");
		}

    return 1;
}

# Process to remove specific RALUS RHEL Product unit script.
#
# This process will remove rhel unit script
# associated with the RALUS package during the uninstall process.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_remove_rhel_unit_scripts {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSLinux::_ralus_linux_remove_rhel_unit_scripts");

    # setup my default unitdir
	my $unitdir = "/etc/systemd/system";
	$$ctx{LOGGER}->info("Default unit script path set to: $unitdir");

    #First Disabling the unit file
    $cmd = "systemctl disable RALUSRHELUnit.service";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->info("Disabling unit file from unit directory was successful.");

	# Delete the RALUSRHELUnit.service file
		$msg = Utils::_tr("Removing RALUS RHEL unit script", 260, 1006);
		Utils::vxif_pbl($msg);
		$unitfile = "$unitdir/RALUSRHELUnit.service";
		$$ctx{LOGGER}->info("Removing RALUS RHEL unit script at: $unitfile.");
		Utils::vxif_dor("rm $unitfile","",$ctx);
		if ( Utils::vxif_fc("$unitfile",$ctx) )
		{
				Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
				$$ctx{LOGGER}->error("ERROR (V-225-254): Removing RALUS RHEL unit script Failed.");
		} else {
				Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
				$$ctx{LOGGER}->info("Removing RALUS RHEL unit script was successful.");
		}

    return 1;
}

# Process to enable TEXTREL for specific product binaries.
#
# Product post-installation script. This interface will be called after all the
# packages have been installed. This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if the post-installation process is successful; otherwise 0.
#
sub _ralus_linux_textrel_binaries {
        my ($ctx) = @_;
        my $result = "";
        my $found = 1;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_textrel_binaries");

	# Let's make sure we have the Change Security Context utility available to us
	my $chconutil = "/usr/bin/chcon";
	if ( ! Utils::vxif_fc($chconutil, $ctx) ) 
        {
                # Let's see if we can locate chcon since it isn't in its 
                # default location
                $result = Utils::vxif_dor("which chcon","",$ctx);
                if ($result =~ m/no chcon in/)
                {
                        $$ctx{LOGGER}->warning("WARNING: Unable to locate chcon - Change Security Context utility on Target system missing.");
                        $found = 0;
                } else {
                        $chconutil = $result;
                        chomp($chconutil);
	                if ( ! Utils::vxif_fc($chconutil, $ctx) ) {
                                $$ctx{LOGGER}->warning("WARNING: Unable to locate chcon - Change Security Context utility on Target system missing.");
                                $found = 0;
                        }
                }
	}

        # if we have a valid path to chcon utility let's update the context
        if ( $found )
        {
        	$$ctx{LOGGER}->info("Change Security Context utility set to: $chconutil");

                # Enable dynamic relocations for libDeviceIo binary
                my $chcon_cmd = "$chconutil -t textrel_shlib_t /opt/VRTSralus/bin/libDeviceIo.so";
                $result = Utils::vxif_dor($chcon_cmd,"",$ctx);
                $$ctx{LOGGER}->finer("Executed CMD: $chcon_cmd with result: $result");
        }

        return 1;
}

# This interface will be called after the product has been uninstalled.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the uninstallation is successful. Otherwise, return 0;
#
sub _ralus_linux_uninstall_symsnap {
 my ($ctx) = @_;
 my $success = 0;

        $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_uninstall_symsnap");

	my $msg = Utils::_tr("Uninstalling SymSnap driver", 260, 1019);
	Utils::vxif_pbl($msg);

        if ( (Utils::vxif_fc("/etc/redhat-release", $ctx)) || (Utils::vxif_fc("/etc/SuSE-release", $ctx)) || (Utils::vxif_fc("/etc/centos-release", $ctx)) )
        {
              my $cmd = "rpm -e --nodeps symbdsnap";
              $success = Utils::vxif_dor($cmd, "", $ctx);
              $$ctx{LOGGER}->info("Uninstalling SymSnap Driver result: $success");
        }

        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));      
        return 1;
}

# Process Product Uninstall specific routines.
#
# Product post-uninstallation script. This interface will be called after all the
# packages has been uninstalled. This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable
#
# Returns: 
#	1 if the post-uninstallation is successful; otherwise 0.
#
sub post_product_uninstall {
  my ($self,$ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::post_product_uninstall");

        # Manual UnInstall Steps to cleanup if necessary...
        _ralus_linux_remove_pkgdirs($ctx);

        # Remove the init files
        _ralus_linux_remove_init_scripts($ctx);

        # Remove the unit files
        my $releasefile = "/proc/version";
        open(FH, $releasefile) or die("File $releasefile not found");
        while (my $osname = <FH>) {
	        if($osname =~ /SUSE/){
                _ralus_linux_remove_suse_unit_scripts($ctx);
            }
        }
        close(FH);

    my $resultrhel9 = Utils::vxif_dor("hostnamectl | grep \"Red Hat Enterprise Linux 9\" | wc -l", "", $ctx);
    if ($resultrhel9 eq 1)
    {
        $$ctx{LOGGER}->info("RALUSLinux:post_product_uninstall - We are uninstalling from a Red Hat 9.x system");
        _ralus_linux_remove_rhel_unit_scripts($ctx);
    }

    my $resultcentos9 = Utils::vxif_dor("hostnamectl | grep \"CentOS Stream 9\" | wc -l", "", $ctx);
    if ($resultcentos9 eq 1)
    {
        $$ctx{LOGGER}->info("RALUSLinux:post_product_install - We are un-installing on a CentOS Stream 9.x system");
        _ralus_linux_remove_rhel_unit_scripts($ctx);
    }

        # Remove the SymSnap Drivers during Uninstall
        _ralus_linux_uninstall_symsnap($ctx); 
}

#
# This interface will be called after the product has been installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#
# Return: 1 if the installation is successful. Otherwise, return 0;
#
sub _ralus_linux_install_symsnap {
 my ($ctx) = @_;
 my $success = 0;

        $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_install_symsnap");

	my $msg = Utils::_tr("Installing SymSnap driver", 260, 1018);
	Utils::vxif_pbl($msg);

        #uninstall old driver if present 
        my $uninstaofo = "rpm -e --nodeps symbdsnap";
        $result = Utils::vxif_dor($uninstaofo, "" , $ctx);
        my $sf="$ENV{VXIF_HOME}scripts/Veritas_symbdsnap.bin";
        my $tf="$$ctx{TMPPATH}/Veritas_symbdsnap.bin";
        $$ctx{LOGGER}->info("Copy ${sf} to ${tf}");
        Utils::vxif_copy($sf, $tf, $ctx, $$ctx{LOCAL}{HOST}, $$ctx{TARGET}{HOST});
        my $cmd = $tf;

	if ( Utils::vxif_fc($cmd, $ctx) )
        {
                $success = Utils::vxif_dor($cmd, "", $ctx);
	        $$ctx{LOGGER}->info("Installing SymSnap Driver result: $success");
        } else {
                $$ctx{LOGGER}->error("ERROR: SymSnap Driver not found at: $cmd");
        }

        if ($success)
        {
	 	if ($success =~ m/Succeeded/)
                {
		        Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
                        $$ctx{LOGGER}->info("Installing SymSnap Driver was successful.");
	        } else {
		        Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
                        $$ctx{LOGGER}->error("ERROR: Installing SymSnap Driver Failed.");
                }
	} else {
                Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
                $$ctx{LOGGER}->error("ERROR: Installing SymSnap Driver Failed.");
	}

        # Cleanup our AOFO binary on remote temp folder
        Utils::vxif_dor("rm -rf $tf", "", $ctx);
        
        return 1;
}

# Product post-installation script. This interface will be called after all the packages
# has been installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
sub post_product_install (\%\%;$) {
  my ($self,$ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::post_product_install");

	# Manual Install of Init Scripts for RMAL
	_ralus_linux_do_init_scripts($ctx);

    # Manual Install of Unit Scripts for RALUS
    my $releasefile = "/proc/version";
    open(FH, $releasefile) or die("File $releasefile not found");
    while (my $osname = <FH>) {
	    if($osname =~ /SUSE/){
            _ralus_linux_do_suse_unit_scripts($ctx);
        }
    }
    close(FH);

    my $resultrhel9 = Utils::vxif_dor("hostnamectl | grep \"Red Hat Enterprise Linux 9\" | wc -l", "", $ctx);
    if ($resultrhel9 eq 1)
    {
        $$ctx{LOGGER}->info("RALUSLinux:post_product_install - We are installing on a Red Hat 9.x system");
        _ralus_linux_do_rhel_unit_scripts($ctx);
    }

    my $resultcentos9 = Utils::vxif_dor("hostnamectl | grep \"CentOS Stream 9\" | wc -l", "", $ctx);
    if ($resultcentos9 eq 1)
    {
        $$ctx{LOGGER}->info("RALUSLinux:post_product_install - We are un-installing on a CentOS Stream 9.x system");
       _ralus_linux_remove_rhel_unit_scripts($ctx);
    } 

	# Add DB2 Support if applicable
	#_ralus_linux_add_db2_support($ctx);
        
        # Set the directory permissions for network security
        _ralus_linux_set_security_settings($ctx);

	# Reset all Files to beoper group ownership
	_ralus_linux_change_group_owner($ctx);

        # Enable TEXTREL Dynamic Relocations for libDeviceIo
        _ralus_linux_textrel_binaries($ctx);

        # Install the SymSnap Drivers for AOFO-Linux Support
        _ralus_linux_install_symsnap($ctx);

        # Install / Configure Linux client-side Dedupe support if 64-bit
        my $hdwr_plat = Utils::vxif_dor("uname -i", "", $ctx);
        if ( $hdwr_plat eq "x86_64" ) {
                _ralus_linux_install_dedupe($ctx);
        } 

       if ( $hdwr_plat eq "x86_64") {
        _ralus_linux_link_libnsl($ctx);
       }

	return 1;
}

sub _ralus_linux_link_libnsl
{
   my ($ctx) = @_;
   my $reqfile = "/lib64/libnsl.so.1";
   my $linkfile = "/lib64/libnsl.so.2.0.0";
   my $cmdlink = "ln -sf $linkfile $reqfile";

   $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_link_libnsl");

   if (! Utils::vxif_fc($reqfile, $ctx))
   {
         if (Utils::vxif_fc($linkfile, $ctx))
         {
                  Utils::vxif_dor($cmd, "", $ctx);
             if ( Utils::vxif_dc("$sfdir", $ctx) )
             {
                  $$ctx{LOGGER}->info("Creating symlink for $linkfile was successful.");
             } else {
                  $$ctx{LOGGER}->error("ERROR: Symlink for $linkfile failed.");
                  my $msg = Utils::_tr("\n\nThe installer was unable to do a soft link to create /lib64/libnsl.so.1. Create this soft link manually and then start the RALUS agent. Do not automatically start the agent post install step\n ", 260, 1015);
                  Utils::vxif_pbl($msg);
                  Utils::vxif_pl("Refer to the following link for more information on creating the soft link manually : https://www.veritas.com/support/en_US/article.100046985\n");
                  Utils::vxif_prtc($ctx, 0);

             } 
         }
   }
}

# Sets Permissions needed for Network Security.
#
# This routine will install the appropriate permissions for network security
# for the Linux platform for RALUS.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_set_security_settings {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_set_security_settings");

	my ($dir,$cmd,$result,$permission);
        $permission = "770";
        #$dir = "/opt/VRTSralus/bin";
        #$cmd = "chmod $permission $dir";
	#$$ctx{LOGGER}->info("Setting network security for: $dir via $cmd");
        #$result = Utils::vxif_dor($cmd, "", $ctx);
        #$$ctx{LOGGER}->info("vxif_dor result: $result");
        
        $dir = "/opt/VRTSralus/data";
        if (!Utils::vxif_dc($dir, $ctx) )
        {
                $cmd = "mkdir /opt/VRTSralus/data";
	        $$ctx{LOGGER}->info("Create Network Security directory: $dir via $cmd");
                $result = Utils::vxif_dor($cmd, "", $ctx);
                $$ctx{LOGGER}->finer("vxif_dor result: $result");
		if (!Utils::vxif_dc($dir,$ctx))
		{
		        $$ctx{LOGGER}->error("ERROR: Unable to create directory: $dir");
                        return 0;
		} else {
		        $$ctx{LOGGER}->info("Create directory '$dir' was successful.");
		}
        }

        $cmd = "chmod $permission $dir";
	$$ctx{LOGGER}->info("Setting network security for: $dir via $cmd");
        $result = Utils::vxif_dor($cmd, "", $ctx);
        $$ctx{LOGGER}->finer("vxif_dor result: $result");

        return 1;
} 

# Install RMAL PDDE Dedupe Support.
#
# This routine will install the appropriate init support  
# for PDDE Dedupe Support for RMAL for the Linux platform.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_install_dedupe {
    my ($ctx) = @_;

    $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_install_dedupe");

    my ($cmd, $libfile); 

    # We need to setup a Symlink to /opt/VRTSralus/lib for modules in PDDE
    # looking for the products lib folder for explicit searches
    my $sfdir = "/opt/VRTSralus/lib";
    $cmd = "ln -sf /opt/VRTSralus/bin $sfdir";
    Utils::vxif_dor($cmd, "", $ctx);
    if ( Utils::vxif_dc("$sfdir", $ctx) )
    {
         $$ctx{LOGGER}->info("Creating symlink for $sfdir was successful.");
    } else {
         $$ctx{LOGGER}->error("ERROR: Symlink for $sfdir failed.");
    }
    
    # Create Symbolic Link to libz.so.1
    #$libfile = "/opt/VRTSralus/lib/libz.so.1";
    #$cmd = "ln -sf /opt/VRTSralus/bin/libz.so.1.2.3 $libfile";
    #Utils::vxif_dor($cmd, "", $ctx);
    #if ( Utils::vxif_fc("$libfile", $ctx) )
    #{
    #     $$ctx{LOGGER}->info("Creating symlink for $libfile was successful.");
    #} else {
    #     $$ctx{LOGGER}->error("Symlink for $libfile failed.");
    #}

    #$libfile = "/opt/VRTSralus/lib/libcurl.so.4";
    #$cmd = "ln -sf /opt/VRTSralus/bin/libcurl.so.4.2.0 $libfile";
    #Utils::vxif_dor($cmd, "", $ctx);
    #if ( Utils::vxif_fc("$libfile", $ctx) )
    #{
    #     $$ctx{LOGGER}->info("Creating symlink for $libfile was successful.");
    #} else {
    #     $$ctx{LOGGER}->error("Symlink for $libfile failed.");
    #}

    #$libfile = "/opt/VRTSralus/bin/libssh2.so.1";
    #$cmd = "ln -sf /opt/VRTSralus/bin/libssh2.so.1.0.0 $libfile";
    #Utils::vxif_dor($cmd, "", $ctx);
    #if ( Utils::vxif_fc("$libfile", $ctx) )
    #{
    #     $$ctx{LOGGER}->info("Creating symlink for $libfile was successful.");
    #} else {
    #     $$ctx{LOGGER}->error("Symlink for $libfile failed.");
    #}

    #$libfile = "/opt/VRTSralus/bin/libxml2.so.2";
    #$cmd = "ln -sf /opt/VRTSralus/bin/libxml2.so.2.7.3 $libfile";
    #Utils::vxif_dor($cmd, "", $ctx);
    #if ( Utils::vxif_fc("$libfile", $ctx) )
    #{
    #     $$ctx{LOGGER}->info("Creating symlink for $libfile was successful.");
    #} else {
    #     $$ctx{LOGGER}->error("Symlink for $libfile failed.");
    #}

    $libfile = "/opt/VRTSralus/bin/libstspipdMT.so";
    $cmd = "ln -sf /opt/VRTSralus/bin/libstspipd.so $libfile";
    Utils::vxif_dor($cmd, "", $ctx);
    if ( Utils::vxif_fc("$libfile", $ctx) )
    {
         $$ctx{LOGGER}->info("Creating symlink for $libfile was successful.");
    } else {
         $$ctx{LOGGER}->error("ERROR: Symlink for $libfile failed.");
    }
    
    # Copy the pdregistry.cfg to the /etc until the PDDE team can
    # read the config file from the /etc/VRTSralus folder
    my $cfgfile = "pdregistry.cfg";
    $cmd = "cp /etc/VRTSralus/$cfgfile /etc/$cfgfile";
    $$ctx{LOGGER}->info("Copying $cfgfile via: $cmd");
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->finer("vxif_dor result: $result");
    if ( Utils::vxif_fc("/etc/$cfgfile", $ctx) )
    {
	$$ctx{LOGGER}->info("Copying $cfgfile to /etc was successful.");
    } else {
	$$ctx{LOGGER}->error("ERROR: Copying $cfgfile to /etc failed.");
    }

    #Run pddeupgrade.sh script to update configuration files for PDDE10
    $cmd = "/opt/VRTSralus/bin/pddeupgrade.sh agent";
    $$ctx{LOGGER}->info("Running pddeupgrade.sh script");
    $result = Utils::vxif_dor($cmd, "" , $ctx);
    $$ctx{LOGGER}->info("vxif_dor result: $result");

    #Etrack 4039599 - client side dedupe changes for DD15
    #Create pdconfigure/scripts/installers under /opt/VRTSralus to allow client side dedupe to run well with DD15
    $cmd = "mkdir /opt/VRTSralus/pdconfigure";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $cmd = "mkdir /opt/VRTSralus/pdconfigure/scripts";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $cmd = "mkdir /opt/VRTSralus/pdconfigure/scripts/installers";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->info("vxif_dor result: $result");
} 

# Install RMAL Init Scripts.
#
# This routine will install the appropriate init runtime scripts
# for the Linux platform for RMAL.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_do_init_scripts {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_do_init_scripts");

	my ($cmd,$result);
	# setup my default initdir
	my $initdir = "/etc/init.d";
	if ( Utils::vxif_dc("/etc/rc.d/init.d", $ctx) )
	{
		$initdir = "/etc/rc.d/init.d";
	}
	$$ctx{LOGGER}->info("Default init script path set to: $initdir");
	
	my $msg = Utils::_tr("Removing previous version of initialization scripts", 260, 1007);  
	my $initfile = "$initdir/VRTSralus.init";
	if ( Utils::vxif_fc($initfile, $ctx) )
	{
		# file exist and we need to remove it
		Utils::vxif_pbl($msg); 
		$cmd="rm -rf $initfile";
		$$ctx{LOGGER}->info("Remove previous version of RMAL init scripts via: $cmd");
		$result = Utils::vxif_dor($cmd, "", $ctx);
		$$ctx{LOGGER}->finer("vxif_dor result: $result");
		if (Utils::vxif_fc($initfile,$ctx))
		{
			$$ctx{LOGGER}->error("ERROR (V-225-254): Removing $initfile Failed.");
		} else {
			$$ctx{LOGGER}->info("Removing $initfile was successful.");
		}
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	}

	$msg = Utils::_tr("Copying new initialization scripts", 260, 1008);
	Utils::vxif_pbl($msg);
	my $sf = "/opt/VRTSralus/bin/VRTSralus.init";
        $cmd = "chmod 750 $sf";
	$$ctx{LOGGER}->info("chmod VRTSralus.init via: $cmd.");
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("chmod VRTSralus.init cmd result: $result");

	my $df = "$initdir/VRTSralus.init";
	$cmd = "cp $sf $df";
	$$ctx{LOGGER}->info("Copying $initfile via: $cmd.");
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("vxif_dor result: $result");
	if ( Utils::vxif_fc("$df", $ctx) )
	{
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
		$$ctx{LOGGER}->info("Copying new initialization scripts was successful.");
	} else {
		Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
		$$ctx{LOGGER}->error("ERROR (V-225-254): Copying new initialization scripts Failed.");
	}

	# setup my default startup directory
	my $rcdir = "/etc/rc.d";
	if ( Utils::vxif_dc("/etc/init.d/rc2.d", $ctx) )
	{
		$rcdir = "/etc/init.d";
	} elsif ( Utils::vxif_dc("/etc/rc2.d", $ctx) ) {
                # Add support for Debian/Ubuntu systems runtime
                $rcdir = "/etc";
        }
	$$ctx{LOGGER}->info("Default init script path set to: $initdir");

	# Check to see if linkedname exist if so delete and create
	# the new linkedname with the newly copied init file in RC2
	#$msg = Utils::_tr("Creating RMAL linkedname in rc2.d", 260, 1009);
	#Utils::vxif_pbl($msg);
	#$initfile = "$rcdir/rc2.d/S95VRTSralus.init";
	#if ( Utils::vxif_fc($initfile, $ctx) )
	#{
	#	Utils::vxif_dor("rm $initfile", "", $ctx);
	#	if (Utils::vxif_fc($initfile,$ctx))
	#	{
	#		$$ctx{LOGGER}->error("ERROR (V-225-254): Removing previous $initfile Failed.");
	#	} else {
	#		$$ctx{LOGGER}->info("Removing previous $initfile was successful.");
	#	}
	#}
	#$cmd="ln -s $initdir/VRTSralus.init $initfile";
	#Utils::vxif_dor($cmd, "", $ctx);
	#if ( Utils::vxif_fc($initfile, $ctx) )
	#{
	#	Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	#	$$ctx{LOGGER}->info($msg . " was successful.");
	#} else {
	#	Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
	#	$$ctx{LOGGER}->error("ERROR (V-225-254): $msg Failed.");
	#}
		
	# Check to see if linkedname exist if so delete and create
	# the new linkedname with the newly copied init file in RC3
	#$msg = Utils::_tr("Creating RMAL linkedname in rc3.d", 260, 1010);
	#Utils::vxif_pbl($msg);
	#$initfile="$rcdir/rc3.d/S95VRTSralus.init";
	#if ( Utils::vxif_fc($initfile, $ctx) )
	#{
	#	Utils::vxif_dor("rm $initfile", "", $ctx);
	#	if (Utils::vxif_fc($initfile,$ctx))
	#	{
	#		$$ctx{LOGGER}->error("ERROR (V-225-254): Removing previous $initfile Failed.");
	#	} else {
	#		$$ctx{LOGGER}->info("Removing previous $initfile was successful.");
	#	}
	#}
	#$cmd="ln -s $initdir/VRTSralus.init $initfile";
	#Utils::vxif_dor($cmd, "", $ctx);
	#if ( Utils::vxif_fc($initfile, $ctx) )
	#{
	#	Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	#	$$ctx{LOGGER}->info($msg . " was successful.");
	#} else {
	#	Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
	#	$$ctx{LOGGER}->error("ERROR (V-225-254): $msg Failed.");
	#}

	# Check to see if linkedname exist if so delete and create
	# the new linkedname with the newly copied init file in RC5
	#$msg = Utils::_tr("Creating RMAL linkedname in rc5.d", 260, 1011);
	#Utils::vxif_pbl($msg);
	#$initfile="$rcdir/rc5.d/S95VRTSralus.init";
	#if ( Utils::vxif_fc($initfile, $ctx) )
	#{
	#	Utils::vxif_dor("rm $initfile", "", $ctx);
	#	if (Utils::vxif_fc($initfile,$ctx))
	#	{
	#		$$ctx{LOGGER}->error("ERROR (V-225-254): Removing previous $initfile Failed.");
	#	} else {
	#		$$ctx{LOGGER}->info("Removing previous $initfile was successful.");
	#	}
	#}
	#$cmd="ln -s $initdir/VRTSralus.init $initfile";
	#Utils::vxif_dor($cmd, "", $ctx);
	#if ( Utils::vxif_fc("$initfile", $ctx) )
	#{
	#	Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	#	$$ctx{LOGGER}->info($msg . " was successful.");
	#} else {
	#	Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
	#	$$ctx{LOGGER}->error("ERROR (V-225-254): $msg Failed.");
	#}

	# Run chkconfig which will create the init runtime scripts for RMAL
        # and If we are on a SuSE or OES box chkconfig will call the insserv utility
	if ( Utils::vxif_fc("/sbin/chkconfig", $ctx) )
	{
		$cmd="/sbin/chkconfig --add VRTSralus.init";
	        Utils::vxif_dor($cmd, "", $ctx);
	        $$ctx{LOGGER}->info("Executed chkconfig cmd: $cmd");
        } else {
                $$ctx{LOGGER}->error("ERROR Unable to create RMAL Init Runtime Scripts via chkconfig utility."); 
        }

	return 1;
}

# Install RALUS Unit Script.
#
# This routine will install the appropriate Unit script
# for the SUSE Linux platform for RALUS.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_do_suse_unit_scripts {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSLinux::_ralus_linux_do_suse_unit_scripts");

	my ($cmd,$result);
	# setup my default unitdir
	my $unitdir = "/etc/systemd/system";
	
	$$ctx{LOGGER}->info("Default unit script path set to: $unitdir");

    my $msg = Utils::_tr("Removing previous version of unit file", 260, 1007);  
	my $unitfile = "$unitdir/RALUSUnit.service";
	if ( Utils::vxif_fc($unitfile, $ctx) )
	{
		# file exist and we need to remove it
		Utils::vxif_pbl($msg); 
		$cmd="rm -rf $unitfile";
		#$$ctx{LOGGER}->info($msg . " via: $cmd");
		$$ctx{LOGGER}->info("Remove previous version of RALUS unit scripts via: $cmd");
		$result = Utils::vxif_dor($cmd, "", $ctx);
		$$ctx{LOGGER}->fine("vxif_dor result: $result");
		if (Utils::vxif_fc($unitfile,$ctx))
		{
			$$ctx{LOGGER}->error("ERROR (V-225-254): Removing $unitfile Failed.");
		} else {
			$$ctx{LOGGER}->info("Removing $unitfile was successful.");
		}
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	}

    $msg = Utils::_tr("Copying new unit file", 260, 1008);
	Utils::vxif_pbl($msg);
	my $sf = "/opt/VRTSralus/bin/RALUSUnit.service";
        $cmd = "chmod 750 $sf";
	$$ctx{LOGGER}->info("chmod RALUSUnit.service via: $cmd.");
        $result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->fine("chmod RALUSUnit.service cmd result: $result");
	my $df = "$unitdir";
	$cmd = "cp $sf $df";
	$$ctx{LOGGER}->info("Copying $unitfile via: $cmd.");
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->fine("vxif_dor result: $result");
	if ( Utils::vxif_fc("$df", $ctx) )
	{
                $cmd = "chmod 750 $df";
                $result = Utils::vxif_dor($cmd, "", $ctx); 
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
		$$ctx{LOGGER}->info("Copying new unit file was successful.");
	} else {
		Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
		$$ctx{LOGGER}->error("ERROR (V-225-254): Copying new unit file Failed.");
	}

    $cmd = "systemctl enable RALUSUnit.service";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->info("Enabling new unit file was successful.");

    return 1;
}

# Install RALUS RHEL Unit Script.
#
# This routine will install the appropriate Unit script
# for the RHEL Linux platform for RALUS.
#
# Input: 1) the reference to the installation context;
#
# Returns: 
#	1 if successful; otherwise 0.
#
sub _ralus_linux_do_rhel_unit_scripts {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RALUSLinux::_ralus_linux_do_rhel_unit_scripts");

	my ($cmd,$result);
	# setup my default unitdir
	my $unitdir = "/etc/systemd/system";
	
	$$ctx{LOGGER}->info("Default unit script path set to: $unitdir");

    my $msg = Utils::_tr("Removing previous version of unit file", 260, 1007);  
	my $unitfile = "$unitdir/RALUSRHELUnit.service";
	if ( Utils::vxif_fc($unitfile, $ctx) )
	{
		# file exist and we need to remove it
		Utils::vxif_pbl($msg); 
		$cmd="rm -rf $unitfile";
		#$$ctx{LOGGER}->info($msg . " via: $cmd");
		$$ctx{LOGGER}->info("Remove previous version of RALUS RHEL unit scripts via: $cmd");
		$result = Utils::vxif_dor($cmd, "", $ctx);
		$$ctx{LOGGER}->fine("vxif_dor result: $result");
		if (Utils::vxif_fc($unitfile,$ctx))
		{
			$$ctx{LOGGER}->error("ERROR (V-225-254): Removing $unitfile Failed.");
		} else {
			$$ctx{LOGGER}->info("Removing $unitfile was successful.");
		}
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
	}

    $msg = Utils::_tr("Copying new unit file", 260, 1008);
	Utils::vxif_pbl($msg);
	my $sf = "/opt/VRTSralus/bin/RALUSRHELUnit.service";
        $cmd = "chmod 750 $sf";
	$$ctx{LOGGER}->info("chmod RALUSRHELUnit.service via: $cmd.");
        $result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->fine("chmod RALUSRHELUnit.service cmd result: $result");
	my $df = "$unitdir";
	$cmd = "cp $sf $df";
	$$ctx{LOGGER}->info("Copying $unitfile via: $cmd.");
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->fine("vxif_dor result: $result");
	if ( Utils::vxif_fc("$df", $ctx) )
	{
                $cmd = "chmod 750 $df";
                $result = Utils::vxif_dor($cmd, "", $ctx); 
		Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
		$$ctx{LOGGER}->info("Copying new unit file was successful.");
	} else {
		Utils::vxif_pbr($msg, Utils::_tr("Failed", 260, 1003));
		$$ctx{LOGGER}->error("ERROR (V-225-254): Copying new unit file Failed.");
	}

    $cmd = "systemctl enable RALUSRHELUnit.service";
    $result = Utils::vxif_dor($cmd, "", $ctx);
    $$ctx{LOGGER}->info("Enabling new unit file was successful.");

    return 1;
}

# Change Group ownership for all RMAL files from bin to beoper
#
# This routine will change the group ownership from bin to beoper
# for configure RMAL for DB2 and Oracle support 
# for the Linux platform for RMAL.
#
#	1 if successful; otherwise 0.
#
sub _ralus_linux_change_group_owner {
    my ($ctx) = @_;

	$$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_change_group_owner");

	# Display to user the configuring of the files
    	my $msg = Utils::_tr("Updating files for beoper group", 260, 1012);
    	Utils::vxif_pbl($msg);
    
	my $cmd = "chgrp -R beoper /opt/VRTSralus/bin/*";
        my $result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("Result for '$cmd' : $result");
    
        my $cmd = "chgrp -R beoper /opt/VRTSralus/data";
        my $result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("Result for '$cmd' : $result");

	$cmd = "chgrp -R beoper /etc/VRTSralus/ralus.cfg";
        $result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("Result for '$cmd' : $result");

	$cmd = "chgrp -R beoper /var/VRTSralus";
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("Result for '$cmd' : $result");

	# Change permissions for the /var/VRTSralus directory
	$cmd = "chmod 775 /var/VRTSralus";
	$result = Utils::vxif_dor($cmd, "", $ctx);
	$$ctx{LOGGER}->finer("Result for '$cmd' : $result");

	# Display to user the configuring of the files completed
	Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
    
	return 1;
}

# Add DB2 Support for RMAL.
#
# This routine will install and configure DB2 support 
# for the Linux platform for RMAL.
#
#	1 if successful; otherwise 0.
#
sub _ralus_linux_add_db2_support {
    my ($ctx) = @_;
    my ($Success) = 0;
    my ($n);
    
    $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_add_db2_support");

    # Let's check to see if db2 is installed on the target machine
    $cmd="ls -1 /var/db2/";
    my $Results = Utils::vxif_dor($cmd, 2, $ctx);
    my ($result, $output) = @{$Results};
    my (@entries) = split(/\n/, $output);
    foreach $n (0..$#entries)
    {
    	if (!($entries[$n] =~ "ls: "))
	{
	 	if ($entries[$n] =~ m/^v.*/i)
		{
			my $dir = "/var/db2/$entries[$n]";
			if ( Utils::vxif_dc($dir, $ctx) )
			{
                                $$ctx{LOGGER}->finer("Found DB2 dir: $dir");

				# Look to configure based on the profiles.reg
				# found in the given directory if found
				my $nCnt = _ralus_linux_configure_db2($ctx, $dir);
                                if ($nCnt > 0) 
                                {
                                   $Success = 1;
                                }
			}
	        } 
	} 
        else 
        { 
                $$ctx{LOGGER}->finer("DB2 not Installed or configured at /var/db2/"); 
        }
    }
    
    return $Success;
}

sub _ralus_linux_configure_db2 (\%\%) {
my ($ctx, $db2dir) = @_;
my ($n, $Entries) = 0;

   $$ctx{LOGGER}->entering("RMALLinux::_ralus_linux_configure_db2");
        
   # Let's open up the file and start parsing the file for db2 Instances
   my $regfile = $db2dir . "/profiles.reg";
   if ( Utils::vxif_fc( $regfile, $ctx ) )
   {
      # Display to user the configuring of the DB2 files
      my $msg = Utils::_tr("Configuring RMAL for DB2 support", 260, 1013);
      Utils::vxif_pbl($msg);
    
      $$ctx{LOGGER}->finer("Found DB2 Regfile: $regfile");
      
      # We need to copy the file locally for reading
      my $localfile = "$$ctx{TMPPATH}/profiles.reg";
      # Copy from Remote machine to Local Machine
      Utils::vxif_copy($regfile, $localfile, $ctx, $$ctx{TARGET}{HOST}, $$ctx{LOCAL}{HOST});
      my $buffer = Utils::vxif_readf($localfile);
      if ($buffer ne "") 
      {
         $$ctx{LOGGER}->finer("Successfully opened Regfile w/Buffer: $buffer");
	 
      	 my @lines = split(/\n/, $buffer);
	 foreach $n (0..$#lines) 
	 {
	    my $db2Inst = @lines[$n];
	    chomp($db2Inst);
	    
            $$ctx{LOGGER}->finer("Calling getpwnam w/ $db2Inst");
         
	    # Use Perl's User DB interface to retrieve Instance Information
	    my $pw = getpwnam($db2Inst);
            if ($pw)
            {
                $$ctx{LOGGER}->finer("Return from getpwnam dir: $pw->dir");
	    
                if ( Utils::vxif_dc($pw->dir, $ctx) )
	        {
	    	   # We need to now copy our DB2 file to this directory
		   my $srcdb2 = "/opt/VRTSralus/bin/db2uext2";
		   my $destdb2 = $pw->dir . "/sqllib/bin/";
		   Utils::vxif_copy($srcdb2, $destdb2, $ctx, $$ctx{TARGET}{HOST}, $$ctx{TARGET}{HOST});
		   $Entries = $Entries + 1;
                }
	    }
	 }
      }
      else 
      { 
         $$ctx{LOGGER}->finer("DB2 Regfile was empty.");
      }
	 
      # Notify user that configuring of the DB2 files is complete
      Utils::vxif_pbr($msg, Utils::_tr("Done", 260, 1001));
   }
   else 
   { 
      $$ctx{LOGGER}->finer("Regfile NOT Found at: $regfile.");
   }

   return $Entries;
}

1;
