package WindowsInstaller;

use VxIF::Context;

require VxIF::Utils;
require VxIF::Logger;
require VxIF::Response;
require VxIF::SunOS::CPISunOS;
require VxIF::Linux::CPILinux;
require VxIF::HPUX::CPIHPUX;
require VxIF::AIX::CPIAIX;
require VxIF::OSF1::CPIOSF1;
require VxIF::Darwin::CPIDarwin;
require VxIF::NativePerl::CPINativePerl;
require File::Spec;


#
# Determine the localhost OS so VxIF can launch the correct platform installer.
#
sub get_local_os {
	my $self = shift;

	if (Utils::running_on_windows()) {
		return "NativePerl";
	} else {
		my $searchpaths = $self->{CMDSEARCHPATHS};
		# search uname on the known paths
		for my $path (@$searchpaths) {
			if (-x "${path}/uname") {
				my $os = `${path}/uname`;
				chomp($os);
				$os =~ s/-//; # get rid of any - character. Mainly for HP-UX.
				if (Utils::vxif_list($os, @{$self->{SUPPORTEDOS}}) >= 0) {
					return $os;
				} else {
					die Utils::_tr("${os} is not a supported platform.", 38, 1000, "${os}");
				}
			}
		}
	}
	
	die Utils::_tr("Unable to find the uname command. Unable to determine the localhost OS type.", 38, 1001); 
}

#
# Create the CPI Logger.
#
# Input: 1) the reference of the installation context;
#
$create_logger = sub ($) {
  my ($self,$ctx) = @_;
  my ($logdir,$logfile);
  
  if ($self->{LOGGER}) {
    $$ctx{LOGGER} = $self->{LOGGER};
  } else {
    # If the environment variable "CPI_LOGLEVEL" is set, use it.
    ($ENV{CPI_LOGLEVEL}) && ($self->{LOGLEVEL} = $ENV{CPI_LOGLEVEL});
    (!$self->{LOGDIR}) && ($self->{LOGDIR} = "$self->{TMPPATH}/$self->{PROGRAM}$$ctx{TIME}");
    (!$self->{LOGFILE}) && ($self->{LOGFILE} = "$self->{LOGDIR}/$self->{PROGRAM}.log");
    
    # check local file system access by writing to $self->{LOGDIR}
    if (!Utils::vxif_mkdir($self->{LOGDIR}, $ctx)) {
      my ($logdir) = $self->{LOGDIR};
      die Utils::_tr("Cannot create ${logdir} directory on $$ctx{LOCAL}{HOST}.", 38, 1002, "${logdir}", "$$ctx{LOCAL}{HOST}");
    }
    Utils::vxif_dol("$$ctx{LOCAL}{CMD}{TOUCH} $self->{LOGFILE}");
    if (!Utils::vxif_fc($self->{LOGFILE}, $ctx)) {
      $logfile = $self->{LOGFILE};
      die Utils::_tr("Cannot write to ${logfile} on $$ctx{LOCAL}{HOST}", 38, 1003, "${logfile}", "$$ctx{LOCAL}{HOST}");
    }
    
    $$ctx{LOGGER} = Logger->new(LOGLEVEL 	=> $self->{LOGLEVEL},
  			        LOGORIGINATORID	=> 1,
  			        LOGFILE 	=> $self->{LOGFILE},
  	                        LOGAPPLICATION 	=> "VxIF");
  }

};


#
# Self initialization.
#
# Input: 1) the reference to self;
#        2) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
	my $self = shift;
	my ($ctx) = @_;
	my (@lt) = localtime();

	my $installers = $ctx->{INSTALLERS} = {};

	# get an instance of each supported OS installer
	for my $os (@{$self->{SUPPORTEDOS}}) {
		my $os_pkg = "CPI${os}";
		my $os_installer =
			$os_pkg->new(INSTALLER      => $self,
						 SUPPORTEDOS    => $self->{SUPPORTEDOS},
						 CMDSEARCHPATHS => $self->{CMDSEARCHPATHS},
						 PKGPATH        => $self->{PKGPATH});
		$os_installer->set_commands();
		$os_installer->initialize($ctx);

		$installers->{$os} = $os_installer;
	}

	# get the local host OS
	my $local_os = $self->get_local_os();
  
	$$ctx{LOCAL}{CMD} = $$ctx{INSTALLERS}{$local_os}->get_commands();
	$$ctx{TARGET}{CMD} = $$ctx{INSTALLERS}{$local_os}->get_commands();
	$$ctx{INSTALLERS}{$local_os}->get_local_host($ctx);
	$$ctx{TARGET}{HOST} = $$ctx{LOCAL}{HOST}; # initially target host is the same as localhost
     
	# setup the tmp directory
	$$ctx{TMPPATH} = $self->{TMPPATH};
  
	# setup time
	if (!$$ctx{TIME}) {
		$$ctx{TIME}=sprintf("%d%02d%02d%02d%02d", $lt[4]+1, $lt[3], $lt[2], $lt[1], $lt[0]);
	}

	# setup the temporary log directory path
	$$ctx{TLOG} = "$self->{TMPPATH}/$self->{PROGRAM}$$ctx{TIME}";

	# setup the log path
	my $logdir = "$$ctx{TLOG}";
	if (Utils::running_on_windows()) {
		#$logdir = "c:$logdir";
	}

	$$ctx{LOG} = $logdir;
	return 0 if (!Utils::vxif_mkdir($$ctx{LOG}, $ctx));

	# store the program name in the context
	(!defined($$ctx{PROGRAM})) && ($$ctx{PROGRAM} = $self->{PROGRAM});

	# create the logger
	&$create_logger($self, $ctx);

	return 1;
}


# Return: an instance of the Installer object.
#
sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $tmpdir = "/tmp";
	my $tmppath = "/var/tmp/vxif";

	my $self = {
		TMPDIR         => $tmpdir,
		TMPPATH        => $tmppath,
		PRODUCT        => '',
		LOGLEVEL       => VxIF::Logger::LEVEL_INFO,
		SUPPORTEDOS    => [ qw(SunOS Linux Darwin HPUX AIX OSF1) ],
		SUPPORTEDTASKS => [ qw(install uninstall) ],
		CMDSEARCHPATHS => [ qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin /local/bin) ],
		@_,
	};

	bless($self, $class);

	# make sure native perl platform is one of common installers
	my $native_perl_platform = "NativePerl";
	my $native_perl_specd = 0;
	my $supported_os = $self->{SUPPORTEDOS};
	for my $os (@$supported_os) {
		if ($os eq $native_perl_platform) {
			$native_perl_specd = 1;
			last;
		}
	}
	if (!$native_perl_specd) {
		push(@$supported_os, $native_perl_platform);
	}

	$self->process_args(@ARGV);

	return bless $self, $class;
}

#
# Start the installation of the product.
#
# Return: 1 if the installation is successful. Otherwise, 0 is returned.
#
sub install {
	my ($self) = @_;
	my $ctx = new Context();
	my $cfg = {};
	$response_file = $self->{RESPONSEFILE};

        if ($response_file) {
                # load the response file
                my $response_info = Response::read_response_file($response_file);

                # get data
                my $sections = $response_info->{sections};

                # populate ctx from the context section
                my $context = $sections->{context};
                if ($context) {
                        for my $key (keys(%$context)) {
                                my $val = $context->{$key};
                                if ($key eq "SYSTEMS") {
                                        $ctx->add_target_systems($context->{SYSTEMS});
                                        my @systems = split(/\s*,\s*/, $context->{SYSTEMS});
                                        $ctx->{SYSTEMS} = \@systems;
                                } elsif ($key =~ /^(UPI|HOST|MODE)$/) {
                                        if (!exists($ctx->{TARGET})) {
                                                $ctx->{TARGET} = {};
                                        }
                                        $ctx->{TARGET}{$key} = $val;
                                } elsif ($key eq "HOSTSPACE") {
                                        # command line, if specified (and,
                                        # therefore, already processed),
                                        # overrides response file
                                        if (!$self->{HOSTSPACE}) {
                                                $self->{HOSTSPACE} = $val;
                                        }
                                } else {
                                        $ctx->{$key} = $val;
                                }
                        }
                }

                # populate cfg from the config section
                my $config = $sections->{config};
                if ($config) {
                        for my $key (keys(%$config)) {
                                my $val = $config->{$key};
                                $cfg->{$key} = $val;
                        }
                }

                # post-process

                $ctx->{RESPONSEFILE} = $self->{RESPONSEFILE};
                $ctx->{ORIGTARGET}{UPI} = $ctx->{TARGET}{UPI};
                $ctx->{PASSWORD} = $self->{PASSWORD};
        } else {
                # no response file.  start with empty hashes.
        }

	$self->initialize($ctx);

        # There are places where ctx is, where we need cfg, but it's not passed.
        $ctx->set_config($cfg);

	my $rc = $self->install_products($ctx, $cfg);

	return 1;
}

#
# Return: 1 if the product is successfully added; Otherwise, 0 is returned.
#
sub add_product ($$) {
  my ($self,$upi,$product) = @_;

  if ($upi && $product) {
      $self->{PRODUCT}{$upi} = $product;
  } else {
    ($upi) || (Utils::vxif_pl(Utils::_tr("Unable to add product. UPI is null.", 38, 1004)));
    ($product) || (Utils::vxif_pl(Utils::_tr("Unable to add product. Product reference is null.", 38, 1005)));
    
    return 0;
  }
  
  return 1;
}

#
# Prompt user for the systems to be tasked.
#
# Input: 1) reference to self;
#        2) the reference to the installation context;
#
# Return: 1 if everything is successful; 0 otherwise.
#
$get_target_systems = sub (\%\%\%) {
  my ($self,$ctx, $cfg) = @_;
  my ($ayn,$tsl,$proxy, $os);
  my ($rc) = 1;

  @{$$ctx{SYSTEMS}} = @{$self->{SYSTEMS}} if ($#{$self->{SYSTEMS}} >= 0);
  $$ctx{PROXY} = $self->{PROXY} if ($self->{PROXY});

  # return if we already have a list of systems to be tasked.
  return 1 if ($$ctx{SYSTEMS} && $$ctx{PROXY});
    
  if (!defined($$ctx{TASK})) {
    $$ctx{LOGGER}->error(Utils::_tr("No task specified.", 38, 1006));
    return 0;
  }
  
  if (Utils::vxif_list($$ctx{TASK}, @{$self->{SUPPORTEDTASKS}}) < 0) {
    $$ctx{LOGGER}->error(Utils::_tr("$$ctx{TASK} is not one of the supported tasks.", 38, 1007, "$$ctx{TASK}"));
    return 0;
  }

  $proxy = Utils::vxif_ask(Utils::_tr("Enter the name of the proxy host:", 38, 1008));
  Utils::vxif_desp($proxy);
  $$ctx{PROXY} = $proxy;
    
  $tsl = Utils::vxif_ask(Utils::_tr("Enter the system names separated by spaces on which to $$ctx{TASK} $self->{UPI}:", 38, 1009, "$$ctx{TASK}", "$self"));
  Utils::vxif_desp($tsl);
  @{$$ctx{SYSTEMS}} = split(/\s+/, $tsl);

  # check to make sure the system names are valid
  #$rc = &$check_system_names($self, $ctx);
  #return $rc if ($rc == 0);
  
  Utils::vxif_pl();
  #return $rc;
  return 1;
};

#
# Run the installation logic.
#
# Input: 1) the reference of the installation context;
#        2) the reference to the configuration hash;
#
# Return: 1 if successfull; 0 otherwise.
#
sub install_products (\%\%) {
  my ($self,$ctx,$cfg) = @_;
  my $proxy;
  my @sys = @{$$ctx{SYSTEMS}};

  $$ctx{LOGGER}->entering("WindowsInstaller::install_products");

	if ($self->{TASK}) {
		$$ctx{TASK} = $self->{TASK};
	} else {
		$$ctx{LOGGER}->error(Utils::_tr("No task defined.", 38, 1010));
		$self->cleanup($ctx);	
	}

  # update the title
  if ($self->{PRODUCT}{$self->{UPI}}) {
	$$ctx{INSTALLER} = $self->{PRODUCT}{$self->{UPI}};
    $$ctx{INSTALLERTITLE} = $self->{PRODUCT}{$self->{UPI}}->get_product_title($ctx) . " " . $self->{PRODUCT}{$self->{UPI}}->get_product_version($ctx);
  }

  Utils::vxif_title($$ctx{INSTALLERTITLE});

  if ($$ctx{RESPONSEFILE}) {
	$$ctx{PROXY} = $$cfg{PROXY};
  }
  
    # get the list of systems to perform product installation task
  if (($#sys < 0)&&(!&$get_target_systems($self, $ctx, $cfg))) {
    $$ctx{LOGGER}->error(Utils::_tr("Unable to get the systems to perform product installation and configration operations.", 38, 1011));
    $self->cleanup($ctx);
  }
 
	$$cfg{PROXY} = $$ctx{PROXY};
 
    Utils::vxif_prtc($ctx) if (!$$ctx{RESPONSEFILE});
  
	Utils::vxif_title($$ctx{INSTALLERTITLE});

	#if (Utils::vxif_is_localhost($$ctx{PROXY}, $ctx)) {
	$hostinfo = $ctx->get_hostinfo();
	$verdict = $hostinfo->is_local_host($$ctx{PROXY});

	if ((!$verdict) && (!&$check_communication($self, $ctx, $cfg))) {
		Utils::vxif_bpl(Utils::_tr("Unable to communicate with proxy host!", 38, 1012));	

		Utils::vxif_prtc($ctx);
		$self->cleanup($ctx);
	}

	Utils::vxif_pl(Utils::_tr("VxIF is now ready to $$ctx{TASK} $self->{UPI} on target systems", 38, 1013, "$$ctx{TASK}", "$self"));

	Utils::vxif_prtc($ctx) if (!$$ctx{RESPONSEFILE});

	Utils::vxif_title($$ctx{INSTALLERTITLE});

  # now perform the selected task on the target hosts
  if ($$ctx{TASK} eq "install") {
	Utils::vxif_bpl(Utils::_tr("Now installing $self->{UPI}:", 38, 1014, "$self"));

    if (!&$install_product_on_target_hosts($self, $ctx, $cfg)) {
      $$ctx{LOGGER}->error(Utils::_tr("Unable to install $$ctx{TARGET}{UPI} to the target systems.", 38, 1015, "$$ctx{TARGET}{UPI}"));

	  Utils::vxif_prtc($ctx) if (!$$ctx{RESPONSEFILE});
      $self->cleanup($ctx);
    }
	Utils::vxif_bpl(Utils::_tr("Installation completed successfully", 38, 1016));
  } elsif ($$ctx{TASK} eq "uninstall") {
	Utils::vxif_bpl(Utils::_tr("Now uninstalling $self->{UPI}:", 38, 1017, "$self"));
    if (!&$uninstall_product_on_target_hosts($self, $ctx, $cfg)) {
      $$ctx{LOGGER}->error(Utils::_tr("Unable to uninstall $$ctx{TARGET}{UPI} to the target systems.", 38, 1018, "$$ctx{TARGET}{UPI}"));

	  Utils::vxif_prtc($ctx) if (!$$ctx{RESPONSEFILE});
      $self->cleanup($ctx);
    }
	Utils::vxif_bpl(Utils::_tr("Uninstall completed successfully", 38, 1019));
  }
  
  Utils::vxif_prtc($ctx) if (!$$ctx{RESPONSEFILE});

  # installer completion
  return &$installer_completion($self, $ctx, $cfg);	
}

#
# Check Communication with proxy
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
$check_communication = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;
	my $ret = 1;
	my $msgl, $msgr, $rshcmd;

	Utils::vxif_bpl(Utils::_tr("Checking communication with the proxy host:\n", 38, 1020));

	#check ping

	#check rsh
	$rshcmd .= $$ctx{PROXY};
	$rshcmd .= " ";
	$rshcmd .= "dir";

	$msgl = Utils::_tr("Checking rsh", 38, 1021);
	Utils::vxif_pbl($msgl);

	$ret = vxif_rsh($rshcmd);
	if (0 ne $ret) {
		$msgr = Utils::_tr("failed", 38, 1022);
		Utils::vxif_pbr($msgl,$msgr);
		$$ctx{LOGGER}->error(Utils::_tr("Cannot rsh to the proxy host $ret", 38, 1023, "$ret"));
		return 0;
	} else {
		$msgr = Utils::_tr("successful", 38, 1024);
		Utils::vxif_pbr($msgl,$msgr);
		$$ctx{LOGGER}->info(Utils::_tr("rsh successful", 38, 1025));
	}

	#check rcp?

	return 1;
};

#
# Perform completion tasks.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
$installer_completion = sub (\%\%\%) {
  my ($self,$ctx,$cfg) = @_;
  my $respfilename;
  
  # create response file
  if (!Response::write_response_file($ctx, $cfg)) {
	  $$ctx{LOGGER}->error(Utils::_tr("Unable to create the response file.", 38, 1026));
	  return 0;
  } else {
	$respfilename = Response::compute_response_file_name($ctx);

	Utils::vxif_pl(Utils::_tr("The installation responsefile is saved at:\n", 38, 1027));
    	Utils::vxif_pl("$self->{MSG}{INDENT}${respfilename}\n");
  }

  if (-f "$$ctx{LOGGER}->{LOGFILE}") {
    Utils::vxif_pl(Utils::_tr("The $$ctx{PROGRAM} log is saved at:\n", 38, 1028, "$$ctx{PROGRAM}"));
    Utils::vxif_pl("$self->{MSG}{INDENT}$$ctx{LOGGER}->{LOGFILE}\n");
  }

  return 1;
};

#
# Cleanup.
#
# Input: 1) the reference of the installation context;
#        2) display response file location?;
#        3) exit code;
#        3) display summary;
#
sub cleanup (\%;$$$) {
  my ($self,$ctx,$respfile,$exit,$sum)=@_;

  Utils::vxif_title($$ctx{INSTALLERTITLE});

  if (-f "$$ctx{LOGGER}->{LOGFILE}") {
    Utils::vxif_pl(Utils::_tr("The $$ctx{PROGRAM} log is saved at:\n", 38, 1028, "$$ctx{PROGRAM}"));
    Utils::vxif_pl("$self->{MSG}{INDENT}$$ctx{LOGGER}->{LOGFILE}\n");
  }

  exit(0);
}

$copy_files_to_proxy = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;
	my $rcpcmd, $msgl, msgr, $rc;
	my $location = $$ctx{INSTALLER}->get_package_location($ctx);
	my $source = $location . "/" . $$ctx{INSTALLER}->get_package_name($ctx);

	$$ctx{LOGGER}->entering("WindowsInstaller::copy_files_to_proxy");

	#if the package file doesn't exist, look for a compressed file
	if (!Utils::vxif_fc($source, $ctx)) {
		# make sure the gunzip tools exist on target
		if (!Utils::vxif_fc($$ctx{TARGET}{CMD}{GUNZIP}, $ctx)) {
			# search in CMDSEARCHPATHS
			foreach my $path (@{$self->{CMDSEARCHPATHS}}) {
				if (Utils::vxif_fc("$path/gunzip", $ctx)) {
					$$ctx{TARGET}{CMD}{GUNZIP} = "$path/gunzip";
					last;
				}
			}
		}

		if (!Utils::vxif_fc($$ctx{TARGET}{CMD}{GUNZIP}, $ctx)) {
			$msg = Utils::_tr("Unable to uncompress files. $$ctx{TARGET}{CMD}{GUNZIP} not found on $trg_host.", 38, 1029, "$$ctx{TARGET}{CMD}{GUNZIP}", "$trg_host");
			Utils::vxif_lpl($msg, "error", $ctx);
			return 0;
		}

		# uncompress the stuff
		# support only tar.gz format for now
		if ($$ctx{INSTALLER}->can(get_compressed_file_name)) {
			$cfilename = $$ctx{INSTALLER}->get_compressed_file_name($ctx);
		} else {
			# problem! file not found.
			$msg = Utils::_tr("Installer file not found", 38, 1030);
			Utils::vxif_lpl($msg, "error", $ctx);
			return 0;
		}

		$rc = Utils::vxif_dor("cd ${location}; $$ctx{TARGET}{CMD}{GUNZIP} $cfilename", 1, $ctx);
		if ($rc != 0) {
			$$ctx{LOGGER}->error(Utils::_tr("Unable to unzip $source. Return code = ${rc}", 38, 1031, "$source", "${rc}"));
			return 0;
		}

		# filename will change after it is unzipped
		$cfilename=~s/\.gz//;
		
		$rc = Utils::vxif_dor("cd ${location}; $$ctx{TARGET}{CMD}{TAR} -xvf ${cfilename}", 1, $ctx);
		if ($rc != 0) {
			$$ctx{LOGGER}->error(Utils::_tr("Unable to untar $source. Return code = $rc", 38, 1032, "$source", "$rc"));
			return 0;
		}
	}

	$rcpcmd = $source;
	$rcpcmd .= " ";
	$rcpcmd .= $$ctx{PROXY};
	$rcpcmd .= ':\\tmp';

	$msgl = Utils::_tr("Copying files to the proxy host", 38, 1033);
	Utils::vxif_pbl($msgl);

	if (0 eq vxif_rcp($rcpcmd)) {
		$msgr = Utils::_tr("done", 38, 1034);
		Utils::vxif_pbr($msgl, $msgr);
		return 1;
	} else {
		$msgr = Utils::_tr("failed", 38, 1022);
		Utils::vxif_pbr($msgl, $msgr);
		return 0;
	}
};

$trigger_install_on_proxy = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;
	my $rshcmd, $msgl, $msgr, $targethost, $ret=1, $rc,
		$pkgname = $$ctx{INSTALLER}->get_package_name($ctx);

	my $abspath, $hostinfo, $verdict;

	$$ctx{LOGGER}->entering("WindowsInstaller::trigger_install_on_proxy");

	foreach $targethost (@{$$ctx{SYSTEMS}}) {
		$hostinfo = $ctx->get_hostinfo();
		$verdict = $hostinfo->is_local_host($$ctx{PROXY});

		if ($verdict) {
			my $source = $$ctx{INSTALLER}->get_package_location($ctx) . "/" . 
				$pkgname . "/Setup.exe";
			
			$abspath = File::Spec->rel2abs($source);

			$rshcmd = $abspath;
			$rshcmd .= " \/s INSTALL_MODE=1 SOLUTION=3";
			$rshcmd .= " NODE=";
			$rshcmd .= $targethost;

			# if product has it's own install command, use that
			# else

			$msgl = Utils::_tr("Installing on target host $targethost", 38, 1035, "$targethost");
			Utils::vxif_pbl($msgl);

			$rc = Utils::vxif_dol($rshcmd);
			
			if ($rc =~ /The operation completed successfully/) {
				$msgr = Utils::_tr("done", 38, 1034);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 1;
			} else {
				$msgr = Utils::_tr("failed", 38, 1022);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 0;;
			}
		} else {
			$rshcmd = $$ctx{PROXY};
			$rshcmd .= " ";
			$rshcmd .= "\\tmp\\$pkgname\\Setup.exe";
			$rshcmd .= " \/s INSTALL_MODE=1 SOLUTION=3";
	# apparently there is a bug in VPI. If local host is specified using NODE= notation, 
	# VPI doesn't install the product.
			$rshcmd .= " NODE=";
			$rshcmd .= $targethost;

			

			# if product has it's own install command, use that
			# else

			$msgl = Utils::_tr("Installing on target host $targethost", 38, 1035, "$targethost");
			Utils::vxif_pbl($msgl);

			$rc = vxif_rsh($rshcmd);
			if (0 eq $rc) {
				$msgr = Utils::_tr("done", 38, 1034);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 1;
			} else {
				$msgr = Utils::_tr("failed", 38, 1022);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 0;;
			}
		}
	}

	return $ret;
};

$trigger_uninstall_on_proxy = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;
	my $rshcmd, $msgl, $msgr, $targethost, $ret=1, $rc,
	$pkgname = $$ctx{INSTALLER}->get_package_name($ctx);

$$ctx{LOGGER}->entering("WindowsInstaller::trigger_uninstall_on_proxy");
	my $verdict, $hostinfo;

	foreach $targethost (@{$$ctx{SYSTEMS}}) {
		$hostinfo = $ctx->get_hostinfo();
		$verdict = $hostinfo->is_local_host($$ctx{PROXY});

		if ($verdict) {
			my $source = $$ctx{INSTALLER}->get_package_location($ctx) . "/" . 
				$pkgname . "/Setup.exe";
			
			$abspath = File::Spec->rel2abs($source);

			$rshcmd = $abspath;
			$rshcmd .= " \/s INSTALL_MODE=5 SOLUTION=3";
			$rshcmd .= " NODE=";
			$rshcmd .= $targethost;

			# if product has it's own install command, use that
			# else

			$msgl = Utils::_tr("Uninstalling on target host $targethost", 38, 1036, "$targethost");
			Utils::vxif_pbl($msgl);

			$rc = Utils::vxif_dol($rshcmd);
			if ($rc =~ /The operation completed successfully/) {
				$msgr = Utils::_tr("done", 38, 1034);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 1;
			} else {
				$msgr = Utils::_tr("failed", 38, 1022);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 0;;
			}
		} else {
			$rshcmd .= $$ctx{PROXY};
			$rshcmd .= " ";
			$rshcmd .= "\\tmp\\$pkgname\\Setup.exe";
			$rshcmd .= " \/s INSTALL_MODE=5 SOLUTION=3";
	# apparently there is a bug in VPI. If local host is specified using NODE= notation, 
	# VPI doesn't install the product.
			$rshcmd .= " NODE=";
			$rshcmd .= $targethost;

			

			# if product has it's own uninstall command, use that
			# else

			$msgl = Utils::_tr("Uninstalling on target host $targethost", 38, 1036, "$targethost");
			Utils::vxif_pbl($msgl);

			$rc = vxif_rsh($rshcmd);
			if (0 eq $rc) {
				$msgr = Utils::_tr("done", 38, 1034);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 1;
			} else {
				$msgr = Utils::_tr("failed", 38, 1022);
				Utils::vxif_pbr($msgl, $msgr);
				$ret &= 0;;
			}
		}
	}

	return $ret;
};

# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
$install_product_on_target_hosts = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;

	$$ctx{LOGGER}->entering("WindowsInstaller::install_product_on_target_hosts");

	my $hostinfo = $ctx->get_hostinfo();
	my $verdict = $hostinfo->is_local_host($$ctx{PROXY});

	if (!$verdict) {
	# copy files to the proxy host
		if (!&$copy_files_to_proxy($self, $ctx, $cfg)) {
			$$ctx{LOGGER}->error(Utils::_tr("Unable to copy file to the proxy host.", 38, 1037));
			Utils::vxif_pl(Utils::_tr("Unable to copy files to the proxy host", 38, 1038));
			return 0;	
		}
	}

	#trigger installer on the proxy host
	if (!&$trigger_install_on_proxy($self, $ctx, $cfg)) {
		$$ctx{LOGGER}->error(Utils::_tr("Install failed.", 38, 1039));
		Utils::vxif_pl(Utils::_tr("Installation failed", 38, 1040));
		return 0;
	}

	return 1;
};

$cleanup_proxy_host = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;
	my $rshcmd, $msgl, $msgr, $targethost, $ret=1, $rc,
	$pkgname = $$ctx{INSTALLER}->get_package_name($ctx);

	$$ctx{LOGGER}->entering("WindowsInstaller::cleanup_proxy_host");
	my $verdict, $hostinfo;

	$hostinfo = $ctx->get_hostinfo();
	$verdict = $hostinfo->is_local_host($$ctx{PROXY});

	if ($verdict) {
		# no op. Shouldn't delete if running from a windows machine.
		# Since files were not copied to a temporary location.
	} else {
		$rshcmd .= $$ctx{PROXY};
		$rshcmd .= " ";
		$rshcmd .= "rmdir /s /q \\tmp\\$pkgname";
		
		$msgl = Utils::_tr("Cleaning up the proxy host", 38, 1041);
		Utils::vxif_pbl($msgl);

		$rc = vxif_rsh($rshcmd);
		if (0 eq $rc) {
			$msgr = Utils::_tr("done", 38, 1034);
			Utils::vxif_pbr($msgl, $msgr);
			$ret &= 1;
		} else {
			$msgr = Utils::_tr("failed", 38, 1022);
			Utils::vxif_pbr($msgl, $msgr);
			$ret &= 0;;
		}
	}

	return $ret;
};

# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
$uninstall_product_on_target_hosts = sub (\%\%\%) {
	my ($self,$ctx,$cfg) = @_;

	$$ctx{LOGGER}->entering("WindowsInstaller::uninstall_product_on_target_hosts");

	my $hostinfo = $ctx->get_hostinfo();
	my $verdict = $hostinfo->is_local_host($$ctx{PROXY});

	if (!$verdict) {
	# copy files to the proxy host
		if (!&$copy_files_to_proxy($self, $ctx, $cfg)) {
			$$ctx{LOGGER}->error(Utils::_tr("Unable to copy file to the proxy host.", 38, 1037));
			Utils::vxif_pl(Utils::_tr("Unable to copy files to the proxy host", 38, 1038));
			return 0;	
		}
	}

	#trigger installer on the proxy host
	if (!&$trigger_uninstall_on_proxy($self, $ctx, $cfg)) {
		$$ctx{LOGGER}->error(Utils::_tr("Uninstall failed.", 38, 1042));
		Utils::vxif_pl(Utils::_tr("uninstall failed", 38, 1043));
		return 0;
	}

	#cleanup the proxy host
	if (!&$cleanup_proxy_host($self, $ctx, $cfg)) {
		$$ctx{LOGGER}->error(Utils::_tr("Cleanup of directories on proxy failed.", 38, 1044));
		Utils::vxif_pl(Utils::_tr("Cleanup of directories on proxy failed", 38, 1045));
		return 0;
	}

	return 1;
};

# return 0 on success, 1 on failure
sub vxif_rcp {
	my $local_user = "root"; #local user is always root.
	my $remote_user = "root";
	my ($ip) = @_;
	my (@input) = split(/\s/, $ip);
	my $ret;

	use VxIF::NativePerl::Net::Rcp;
	$a=Net::Rcp->new();
  	$ret = $a->rcp($local_user,$remote_user,\@input);
	sleep(1);

	return $ret;
}

# argument parser.
# i/p = arguments to rsh sub.
# o/p : argv containing -
# argv[0] = target hostname 
# argv[1] = remote user (default root)
# argv[2] = command

sub parse {
	# set defaults;
	my @argv = qw(hostname root);

	my $UNAME=0,
	   $HOSTNAME=0;

	foreach my $token (split(/\s+/,join(" ",@_))) {
		if ($token eq "-l") {
			$UNAME = 1;
		}
		elsif (1 == $UNAME) {
			$argv[1] = $token;
		} elsif (1 == $HOSTNAME) {
			if ($argv[2]) {
				$argv[2] = join(" ",$argv[2],$token);
			} else {
				$argv[2] = $token;
			}

			next;
		} else {
			$argv[0] = $token;
			$HOSTNAME = 1;
		}
	}

	return @argv;						
}

# rsh subroutine. implements the rsh client function.
sub vxif_rsh {
	my (@argv) = parse(@_);

	my $host = $argv[0];
	my $local_user = "root"; #local user is always root.
	#my $local_user = "v021164"; #local user is always root except on win32.
	#my $local_user = getlogin(); #local user is always root.
	my $remote_user = $argv[1];
	my $cmd = $argv[2];

	use VxIF::NativePerl::Net::Rsh;
	$a=Net::Rsh->new();
  	my $out=$a->rsh($host,$local_user,$remote_user,$cmd);

	#return  @{$out};
	# for now we are only interested in the return code
	return @{$out}[0];
}
#
# Process the command line arguements. This routine should be called before calling install()
#
# Input: 1) the reference to self;
#        2) the command line arguements;
#
sub process_args (@) {
        my $self = shift;

        while (@_) {
                my $a = shift(@_);
                $a = Utils::vxif_desp($a);
                $a = lc($a);

                if ($a eq '-responsefile') {
                        $self->{RESPONSEFILE} = shift;
                } elsif ($a eq '-logdir') {
                        $self->{LOGDIR} = shift;
                } else {
                        push(@{$self->{SYSTEMS}},$a);
                }
	}
}
1;
