
package Utils;

# Adding as requested by the BE Folks
#use open qw(:utf8);
#use open ':locale';
#no warnings 'utf8';

use Encode qw(from_to);
##use Switch;

#
# Test to see if on windows.  Criteria may change,
# so run test through a single place (i.e. here).
#
sub running_on_windows {
	my $verdict = 0;

	if ($^O eq "MSWin32") {
		$verdict = 1;
	}

	return $verdict;
}

BEGIN {
  if (running_on_windows()) {
	  # just do this on windows.
	  require VxIF::NativePerl::PerlUnixSh;
  }
}

#
# Common utilities
#

###############################
# Setup Globals               #
###############################

# common commands
my %COMM;

$COMM{TERMX}=80; # default if no $CMD{TPUT}
$COMM{TERMY}=24; # default if no $CMD{TPUT}

# search for the tput command
if (-X "/usr/bin/tput") {
  $COMM{TPUT}{CMD} = "/usr/bin/tput";
} elsif (-X "/usr/local/bin/tput") {
  $COMM{TPUT}{CMD} = "/usr/local/bin/tput";
} elsif (-X "/bin/tput") {
  $COMM{TPUT}{CMD} = "/bin/tput";
} elsif (-X "/sbin/tput") {
  $COMM{TPUT}{CMD} = "/sbin/tput";
}

# set %COMM{TPUT}
if ($COMM{TPUT}{CMD}) {

	# xterm-color on solaris has problems with tput
	my $os = `uname -s`;
	chomp($os);

	if (("SunOS" eq $os) && ($ENV{TERM} eq "xterm-color")) {
			$ENV{TERM}="dtterm";
	}

  $COMM{TPUT}{BS}=`$COMM{TPUT}{CMD} bold`;	# Bold face Start
  $COMM{TPUT}{BE}=`$COMM{TPUT}{CMD} sgr 0`;	# Bold face End
  $COMM{TPUT}{CS}=`$COMM{TPUT}{CMD} clear`;	# Clear the Screen
  $COMM{TPUT}{SS}=`$COMM{TPUT}{CMD} smso`;	# Standout face Start
  $COMM{TPUT}{SE}=`$COMM{TPUT}{CMD} rmso`;	# Standout face End
  $COMM{TPUT}{US}=`$COMM{TPUT}{CMD} smul`;	# Underline face Start
  $COMM{TPUT}{UE}=`$COMM{TPUT}{CMD} rmul`;	# Underline face End
  $COMM{TPUT}{CL}=`$COMM{TPUT}{CMD} cub1`;	# cursor left
  $COMM{TPUT}{SC}=`$COMM{TPUT}{CMD} sc`;	# save cursor pos
  $COMM{TPUT}{RC}=`$COMM{TPUT}{CMD} rc`;	# restore cursor pos
  $COMM{TERMX}=`$COMM{TPUT}{CMD} cols`;		# number of columns
  $COMM{TERMY}=`$COMM{TPUT}{CMD} lines`;	# number of rows
  chomp($COMM{TERMX});
  chomp($COMM{TERMY});
}

# messages
my %MSG;

$MSG{INDENT}="    ";
$MSG{BACKKEY}{L}=_tr("b", 12, 1000);
$MSG{BACKKEY}{U}=_tr("B", 12, 1008);
$MSG{HELPKEY}=_tr("?", 12, 1001);
$MSG{QUITKEY}{L}=_tr("q", 12, 1002);
$MSG{QUITKEY}{U}=_tr("Q", 12, 1009);
$MSG{NULL} = "";
$MSG{NOKEY}{L}=_tr("n", 12, 1003);
$MSG{NOKEY}{U}=_tr("N", 12, 1010);
$MSG{YESKEY}{L}=_tr("y", 12, 1004);
$MSG{YESKEY}{U}=_tr("Y", 12, 1011);
$MSG{MENUBACK}=_tr("Back to previous menu", 12, 1005);
$MSG{PRTC}=_tr("Press [Return] to continue:", 12, 1006);

###############################
# Public Methods              #
###############################

#
# Print and log the localized message
#
# Input: 1) the message;
#        2) log level; "error", "warning", or "info";
#        3) the installation context;
#
sub vxif_lpl ($$$) {
  my($msg, $level, $ctx) = @_;
  
  vxif_bpl($msg);
  if (($ctx) && ($ctx->{LOGGER}) && ($ctx->{LOGGER}->can($level))) {
    $ctx->{LOGGER}->$level($msg);
  } else {
    $msg = "Invalid logger. \"$msg\" will not be logged.";
    vxif_bpl($msg);
  }
}

#
# Print a localized message to standard output.
#
# Input: 1) localized message;
#
sub vxif_p (;$) {
	my $lm = defined($_[0]) ? $_[0] : "";
	print STDOUT "$lm";
}

#
# Print a localized message to standard output. A new line character is added to the
# end of the line.
#
# Input: 1) localized message;
#
sub vxif_pl (;$) {
	my $lm = defined($_[0]) ? $_[0] : "";
	vxif_p($lm . "\n");
}

#
# Print a given localized message in bold.
#
# Input: 1) localized message to print in bold;
#
sub vxif_bp ($) {
	if (defined($COMM{TPUT}{BS})) {
		print "$COMM{TPUT}{BS}";
	}
	print "$_[0]"; 
	if (defined($COMM{TPUT}{BE})) {
		print "$COMM{TPUT}{BE}"; 
	}
}

#
# Print a given localized message in bold. A new line character is added to the end of
# the line.
#
# Input: 1) localized message to print in bold;
#
sub vxif_bpl ($) {
  my $lm = defined($_[0]) ? $_[0] : "";
  vxif_bp($lm . "\n");
}

#
# Ask a question. Users can optionally supply handlers to validate the answers,
# handle back key, and handle quit key.
#
# For example,
#
# require VxIF::Utils;
#
# $validator = sub ($) {
#   my($answer) = @_;
#   
#   if (($answer > 0)&&($answer < 6)) {
#     print "Cool\n";
#   } else {
#     print "No good.\n";
#     return 0;
#   }
#   
#   return 1;
# };
#
# sub back_key_handler {
#   print STDOUT "A back key has been pressed!\n";
# }
#
# sub quit_key_handler {
#   print STDOUT "Quiting!\n";
# }
# 
# $answer = ask("Hit me with a digit between 1 and 5?", 1, "I said between 1 and 5!", $validator, \&back_key_handler, \&quit_key_handler);
#
# Input: 1) localized question;
#        2) default answer;
#        3) localized help message;
#        4) validator;
#        5) back key handler;
#        6) quit key handler;
#        7) additional options;
#
# Return: the answer to the question or null (""). Null is returned if user pressed the 
#         back key or the quit key.
#
sub vxif_ask($;$$$$$$) {
  my($question,$default_answer,$help_msg,$validator,$back_key_handler,$quit_key_handler,$aopts) = @_;
  my($nl) = " ";
  my($ask);
  my($opt);
  
  (($default_answer) && ($default_answer = vxif_desp($default_answer)));
  ($aopts)&&($opt = $aopts . ",");
  $opt .= " $MSG{BACKKEY}{L}," if ($back_key_handler);
  $opt .= " $MSG{HELPKEY}," if ($help_msg);
  $opt .= " $MSG{QUITKEY}{L}," if ($quit_key_handler);
  $opt = vxif_desp($opt);
  chop($opt) if ($opt ne ""); # get rid of the last comma.
  $opt = "[${opt}]" if ($opt ne "");
  $question .= " ${opt}" if ($opt ne "");
  $question .= "$nl($default_answer)" if ($default_answer ne "");
  while (1) {
    vxif_bp($question);
    $ask = <STDIN>;
    $ask=vxif_desp($ask);
     
    if ((($ask eq $MSG{QUITKEY}{L}) || ($ask eq $MSG{QUITKEY}{U})) && ($quit_key_handler)) {
      &$quit_key_handler();
      return $MSG{NULL};
    } elsif ((($ask eq $MSG{BACKKEY}{L}) || ($ask eq $MSG{BACKKEY}{U})) && ($back_key_handler)) {
      &$back_key_handler();
      return $MSG{NULL};
    } elsif (($help_msg) && ($ask eq $MSG{HELPKEY})) {
      vxif_pl("\n${help_msg}\n");
    } else {
      $ask=$default_answer if (($default_answer ne "") && ($ask eq ""));
      if ($validator) {
        if (&$validator($ask)) {
          return "$ask";
        } else {
          ($help_msg && vxif_pl("\n${help_msg}\n"));
        }
      } else {
        return "$ask";
      }
    }
  }
}

#
# Asks a yes/no question. Users can optionally supply handlers to
# handle back key, and handle quit key.
#
# Input: 1) localized question;
#        2) default answer; Must not be "Y", "N", "y", or "n";
#        3) localized help message;
#        4) back key handler;
#        5) quit key handler;
#
# Return: "Y", "N", or null (""). Null is returned if user pressed the back key or the quit key.
#
sub vxif_ayn ($;$$$$) {
  my($question,$default_answer,$help_msg,$back_key_handler,$quit_key_handler) = @_;
  my ($opts) = "$MSG{YESKEY}{L}, $MSG{NOKEY}{L}";
  my ($ans);
  
  my $validator = sub {
    my($answer) = @_;
    
    return (($answer eq $MSG{YESKEY}{L}) || ($answer eq $MSG{YESKEY}{U}) || ($answer eq $MSG{NOKEY}{L}) || ($answer eq $MSG{NOKEY}{U}));
  };
  
  $ans = vxif_ask($question, $default_answer, $help_msg, $validator, $back_key_handler, $quit_key_handler, $opts);
  $ans = $MSG{YESKEY}{U} if ($ans eq $MSG{YESKEY}{L});
  $ans = $MSG{NOKEY}{U} if ($ans eq $MSG{NOKEY}{L});
  
  return $ans;
}

#
# Asks a yes/no question. The default answer is no. Users can optionally supply handlers to
# handle back key, and handle quit key.
#
# Input: 1) localized question;
#        2) localized help message;
#        3) back key handler;
#        4) quit key handler;
#
# Return: "Y", "N", or null (""). Null is returned if user pressed the back key or the quit key.
#
sub vxif_aynn ($;$$$) {
  my($question,$help_msg,$back_key_handler,$quit_key_handler) = @_;
  
  return vxif_ayn($question,$MSG{NOKEY}{L},$help_msg,$back_key_handler,$quit_key_handler);
}


#
# Asks a yes/no question. The default answer is yes. Users can optionally supply handlers to
# handle back key, and handle quit key.
#
# Input: 1) localized question;
#        2) localized help message;
#        3) back key handler;
#        4) quit key handler;
#
# Return: "Y", "N", or null (""). Null is returned if user pressed the back key or the quit key.
#
sub vxif_ayny ($;$$$) {
  my($question,$help_msg,$back_key_handler,$quit_key_handler) = @_;
  
  return vxif_ayn($question,$MSG{YESKEY}{L},$help_msg,$back_key_handler,$quit_key_handler);
}

#
# Compare the two given version numbers. Version numbers will be compared from 
# left to right. Each components of a version number must be separated by a 
# dot (.). Leading zeros in each component are insignificant.
#
# Input: 1) first version number;
#        2) second version number;
#
# Return: 0 if versions are the same; 1 if the first version number is higher; 
#         2 if the second version number is higher.
#
sub vxif_cv ($$) {
	my(@a1,@a2,$f,$n,$v1,$v2);	
	($v1,$v2)=(@_);	
	@a1=split(/[\.-]/,$v1);
	@a2=split(/[\.-]/,$v2);
	$n = ($#a1>$#a2) ? $#a1 : $#a2;
	foreach $f(0..$n) {
		$v1 = ($f<=$#a1) ? $a1[$f] : "0";
		$v2 = ($f<=$#a2) ? $a2[$f] : "0";
		return 1 if ($v1>$v2);
		return 2 if ($v2>$v1);
		# Assumes letters always trail numbers
		$v1=~s/[0-9]//g;
		$v2=~s/[0-9]//g;
		return 1 if (($v1 cmp $v2)>0);
		return 2 if (($v1 cmp $v2)<0);
	}
	return "0";
}

#
# Remove spaces from begginning and end of line. Spaces in between words 
# will be retained.
# 
# Input: 1) string
#
# Return: a string with the spaces at the beginning and end of line removed.
#
sub vxif_desp ($) {
  my($s)=(@_);
  $s=~s/^\s+//;
  $s=~s/\s+$//;
	

  return "$s";
}

#
# Duplicate an array to properly load the reference into a list.
#
# Input: 1) an array;
#
# Return: the reference of the duplicate.
#
sub vxif_dupa (@) { 
  my @a=@_; 
	

  return \@a;
}

#
# Duplicate a hash to properly load the reference into a list.
#
# Input: 1) a hash;
#
# Return: the reference of the duplicate.
#
sub vxif_duph ($) { 
  my $rh=shift;
  my %h=%$rh;
	

  return \%h;
}

#
# Check whether all the elements of a given one-dimentional hash 
# are the same. For example, this routine will return 1 for the 
# following array.
#
# $a{b} = "12345";
# $a{c} = "12345";
# $a{d} = "12345";
# $a{e} = "12345";
#
#
# vxif_hats(\%a);
#
# Input: 1) the reference of the one-dimentional hash array;
#
# Return: 1 if all elements of a hash array are the same; Otherwise, return 0.
#
sub vxif_hats (\%) {
  my($rh) = @_;
  my(%nh);
  
  foreach $key (keys(%$rh)) {
    $nh{$$rh{$key}} = 1;
  }
  
  return (scalar(keys(%nh)) == 1) ? 1 : 0;
}

#
# Determine if an IP address is v4 or v6.  Then refer it to
# to the correct check routine
#
# Input: 1) the IP address;
#
# Returns: 1 if the given IP address is valid; 0 otherwise, AND a 4 or a 6 indicating the 
# address is either IPv4 or IPv6
#
sub vxif_ipcheck ($) {
   my ($ip_in) = @_;
   my ($valid, $format);

   # jpk
   #print "IP Check -> $ip_in\n";
   if ($ip_in =~ /\:/) {
      $valid = Utils::vxif_ipv6check($ip_in);
      $format = 6;
   } elsif ($ip_in =~ /\./){
      $valid = Utils::vxif_ipv4check($ip_in);
      $format = 4;
   } else {
      # format is wrong
      return 0;
   }
   return ($valid,$format);
}

#
# Verify a IPv4 address string is in ##.##.##.## notation.
#
# Input: 1) the IP address;
#
# Return: 1 if the given IP address valid; 0 otherwise.
#
sub vxif_ipv4check ($) {
   my $self = shift;
   my(@f,$d,$ip,$n);
   ($ip)=(@_);
   @f=split(/\./,$ip);
   return 0 if ($#f!=3);
	

   foreach $n(0..3) {
      return 0 if ($f[$n] eq "");
      $d=$f[$n];
      $d =~ s/[0-9]//g;
      return 0 if (($d) || ($f[$n]>255));
   }
   return 1;
}

#
# Verify an IPv6 address string is in ####:####:####:####:####:#### format.
#
# Input: 1) an IPv6 address
#
# Return: 1 if the given IPv6 address is valid; 0 if not valid
sub vxif_ipv6check ($) {
   my (@f, @g, $temp, $ipv6, $a, $b);
   ($ipv6) = @_;
   my $tupple_count = 5; # This is a counter for the number of tupples in a IPv6 address

   # Remove any subnet mask that was included
   if ($ipv6 =~ /\//) {
      ($temp) = split(/\//,$ipv6);
      $ipv6 = $temp;
   }

   # get each of the byte spaces so they can be validated.
   @f = split(/\:/,$ipv6);
   if ($ipv6 =~ /1\/128/) {
      # This is the localhost address
      return 1;
   }
   if (($#f > 5) || ($#f < 3)) {
      # Hybrid format can be  ::ffff:10.255.1.10 or 2000::1:2
      # not long enough to be full IPv6 address.  
      return 0;
   } else {
      $tupple_count = $#f;
   }

   foreach $a (0..$tupple_count) {
      $b = length($f[$a]);
      if ($b > 4) {
         # entry is too long.  IPv6 byte is 4 characters.
         return 0;
      }
      until (length($f[$a]) == 4) {
         # pad short entries with zeros and fill in quad zeros represented by ::
         $f[$a] = "0".$f[$a];
      }
      if ($f[$a] !~ /^[0-9|a-f][0-9|a-f][0-9|a-f][0-9|a-f]$/) {
         # there is a unauthorized character somewhere.
         return 0;
      }
   } # end of foreach
   return 1;
}

#
# Check to see if the given string is an integer.
#
# Input: 1) a string;
#
# Return: 1 if the given string is an integer; 0 otherwise.
#
sub vxif_isint ($) {
  return ($_[0] =~ /^-?\d+$/);
}

#
# Check to see if the given string is a number.
#
# Input: 1) a string;
#
# Return: 1 if the given string is number; 0 otherwise.
#
sub vxif_isnum ($) {
  return ($_[0] =~ /^-?\d+\.?\d*$/);
}

#
# Check whether a string is a member of an array. There may be multiple matches. 
# This subroutine will only return the member number of the first match.
#
# Input: 1) a string;
#        2) the array;
#
# Return: the member number, or -1 if it is not.
#
sub vxif_list ($@) {
  my($match,@arr) = (@_);
  my($counter) = 0;
  
  foreach $element (@arr) {
    return $counter if ($element eq $match);
    ++$counter;
  }
	

  return "-1";
}

#
# Same as vxif_list, but supporting * as a wildcard in @arr.
#
# Input: 1) a string;
#        2) the array;
#
# Return the member number, or -1 if it is not.
#
sub vxif_wclist ($@) {
  my (@arr,$match,$n);
  
  ($match,@arr) = (@_);
  @arr=split(/\s+/,$arr[0]) if (!$#arr);
  foreach $n(0..$#arr) {
    if  ($arr[$n] =~ "\\*")  {
      return $n if $match =~ m/$arr[$n]/;
    } else {
      return "$n" if ($arr[$n] eq $match);
    }   
  }       
    
  return "-1";
}   

#
# Ask a question. Users can optionally supply handlers to validate the answers,
# handle back key, and handle quit key.
#
# For example,
#
# require VxIF::Utils;
#
# $validator = sub ($) {
#   my($answer) = @_;
#   
#   if (($answer > 0)&&($answer < 6)) {
#     print "Cool\n";
#   } else {
#     print "No good.\n";
#     return 0;
#   }
#   
#   return 1;
# };
#
# sub back_key_handler {
#   print STDOUT "A back key has been pressed!\n";
# }
#
# sub quit_key_handler {
#   print STDOUT "Quiting!\n";
# }
# 
# $answer = ask("Hit me with a digit between 1 and 5?", 1, "I said between 1 and 5!", $validator, \&back_key_handler, \&quit_key_handler);
#
# Input: 1) localized question;
#        2) the reference of the localized choices array;
#        3) default answer; Must not be null ("");
#        4) localized help message;
#        5) back key handler;
#        6) quit key handler;
#
# Return: the answer to the question or null (""). Null is returned if user pressed the 
#         back key or the quit key.
#
sub vxif_menu($$$$$$) {
  my($question,$choices,$default_answer,$help_msg,$back_key_handler,$quit_key_handler) = @_;
  my($nl) = " ";
  my($ask);
  my($opt);
  my($choice);
  my($index) = 0;
  my($cl) = scalar(@$choices) - 1;
  
  $default_answer = vxif_desp($default_answer);
  $opt .= ($cl > 0) ? ("[1-" . ($cl + 1) . ",") : "[1,";
  $opt .= " $MSG{BACKKEY}{L}," if ($back_key_handler);
  $opt .= " $MSG{HELPKEY}," if ($help_msg);
  $opt .= " $MSG{QUITKEY}{L}," if ($quit_key_handler);
  $opt = vxif_desp($opt);
  chop($opt) if ($opt ne ""); # get rid of the last comma.
  $opt .= "]";
  $question .= "${nl}${opt}";
  $question .= " ($default_answer)" if ($default_answer ne "");
  
  foreach $choice (@$choices) {
    $index=sprintf("%2d",$index+1);
    vxif_bp("$MSG{INDENT}${index}");
    vxif_pl(")$MSG{INDENT}${choice}");
  }
  
  if ($back_key_handler) {
    vxif_bp("$MSG{INDENT} $MSG{BACKKEY}{U}");
    vxif_p(")$MSG{INDENT}");
	vxif_pl(_tr("Back to previous menu", 12, 1005));
  }
  vxif_pl();
  
  while (1) {
    vxif_bp($question);
    $ask = <STDIN>;
    $ask = vxif_desp($ask);
    if ((($ask eq $MSG{QUITKEY}{L}) || ($ask eq $MSG{QUITKEY}{U})) && ($quit_key_handler)) {
      &$quit_key_handler();
      return $MSG{NULL};
    } elsif ((($ask eq $MSG{BACKKEY}{L}) || ($ask eq $MSG{BACKKEY}{U})) && ($back_key_handler)) {
      &$back_key_handler();
      return $MSG{NULL};
    } elsif (($help_msg) && ($ask eq $MSG{HELPKEY})) {
      vxif_p("\n${help_msg}\n");
    } else {
      next if ($ask =~ /\D/);
      $ask=$default_answer if (($default_answer ne "") && ($ask eq ""));
      
      if (($ask >=1 )&&($ask <= $index)) {
        return "$ask";
      } else {
        vxif_p("\n${help_msg}\n") if ($help_msg);
      }
    }
  }
}

#
# Verify a NIC string is in "lettersnumbers" notation.
# 
# Input: 1) NIC string;
#
# Return: 1 if the NIC string is not in "lettersnumbers" notation; Otherwise, 0 is returned.
#
sub vxif_niccheck ($) {
  my($l,$n,$nic);
  ($nic)=(@_);
  $l=$n=$nic;
  $l=~s/[0-9]//g;
  $n=~s/[A-Za-z]//g;
  return 1 if ((!$l) || ($n eq "") || ($nic ne "$l$n"));
  return 0;
}

#
# Progress bar, left side. Write the localized message to the left side of the screen.
#
# Input: 1) localized message;
#
sub vxif_pbl ($) { 
  vxif_p("$MSG{INDENT}$_[0] "); 
}

#
# Progress bar, right side. Write the localized message to the right side of the screen.
#
#
# Input: 1) localized message to appear at the left side of the screen;
#        2) localized message to appear at the right side of the screen;
#
sub vxif_pbr ($$) { 
  my($lm,$rm) = @_;
  my($p,$p1,$p2,$sp);
  $p1=length($lm);
  $p2=length($rm);
  # need a reliable length that works in ja locale
  if ($COMM{ENVLANG} && $COMM{ENVLANG}=~/^ja/) {
    $p1*=1.4;
    $p2*=1.4;
  }
  $sp=$COMM{TERMX}-length($MSG{INDENT})-10-$p1-$p2;
  #vxif_p("$MSG{INDENT}${lm} "); 
  vxif_p("...");
  foreach $p(1..$sp) { vxif_p("."); }
  vxif_pl(" ${rm}"); 
}

#
# Pause for a short period of time and continue.
#
# Input: 1) how many seconds to pause;
#
sub vxif_pause (;$) {
  sleep $COMM{SCREENPAUSE};
  return;
}

#
# Pause and display the "Press Return to Continue" message.
#
# Input: 1) the reference to the installation context;
#        2) the number of seconds to pause, this is optional;
#
sub vxif_prtc ($;$) {
	my ($ctx,$pausetime) = @_;

	if ($$ctx{RESPONSEFILE} || $pausetime) {
		if ($pausetime) {
			sleep $pausetime;
		} else {
			sleep 2;
		}
		return;
	}

	vxif_pl();
	vxif_bp(_tr("Press [Return] to continue:", 12, 1006));
	vxif_p(" ");
	my $prtc = <STDIN>;
	chomp($prtc);
	my $local_hostname = $ctx->get_local_hostname();
	my $os_inst = $ctx->get_host_os_installer($local_hostname);
	$os_inst->cleanup($ctx) if (($prtc eq $MSG{QUITKEY}{L}) || ($prtc eq $MSG{QUITKEY}{U}));
}

# 
# Read and return the content of a file.
#
# Input: 1) error handler;
#
# Return: the content of the file in a buffer.
#
sub vxif_readf ($) {
  my($file,$line,$rf,$error_handler);
  ($file, $error_handler)=(@_); 
  open(RF, $file) or ($error_handler && &$error_handler());
  while ($line=<RF>) { 
    $rf.=$line;
  }
  close(RF);
  return $rf;	
}

#
# Clean up zombie processes.
#
use POSIX ":sys_wait_h";
sub vxif_reaper {
  my($pid);
  while (($pid = waitpid(-1,WNOHANG)) > 0) {
    #_log("reaped $pid");
  }
}

#
# Cause the cursor to have a spinning char while waiting.
#
sub vxif_spin {
  if (!$FLAG{TPUT}) { sleep 1; return; }
  my (@ch,$n);
  @ch=("/","-","\\","|");
  print "$ch[0]$COMM{TPUT}{CL}";
  sleep 1;
  print "$ch[1]$COMM{TPUT}{CL}";
  sleep 1;
  print "$ch[2]$COMM{TPUT}{CL}";
  sleep 1;
  print "$ch[3]$COMM{TPUT}{CL}";
  sleep 1;
}

# 
# Display the title bar.
#
# Input: 1) localized title;
#
sub vxif_title ($) {
	my($title) = @_;
	my($es,$ns,$ss,$t);

	$ns=($COMM{TERMX}-length(${title}))/2;
	foreach $t(1..$ns) { 
		$ss .= " "; 
	}
	$ns=$COMM{TERMX}-$ns-length(${title});
	foreach $t(1..$ns) { 
		$es.=" "; 
	}
	# print newlines anyway so you can scroll back after a clear
	foreach $t(1..$COMM{TERMY}) {  
		vxif_pl();
	}
	if (defined($CMD{CLEAR})) {
		system("$CMD{CLEAR}") if ((!$FLAG{TPUT}) && (-x $CMD{CLEAR}));
	}
	if (defined($COMM{TPUT}{CS})) {
		#vxif_pl("$COMM{TPUT}{CS}$COMM{TPUT}{SS}$ss${title}$es$COMM{TPUT}{SE}\n");
		vxif_p("$COMM{TPUT}{CS}$COMM{TPUT}{SS}");
	}
	vxif_p("$ss${title}$es");
	if (defined($COMM{TPUT}{SE})) {
		vxif_pl("$COMM{TPUT}{SE}\n");
	} else {
		vxif_pl("\n");
	}
}

# 
# Check whether a regular expression is found within a member of an array.
#
# Input: 1) pattern to match;
#        2) the array;
#
# Return the member number, or -1 if it is not.
#
sub vxif_tlist ($@) {
  my(@arr,$match,$n);
  ($match,@arr) = (@_);
  @arr=split(/\s+/,$arr[0]) if (!$#arr);
  foreach $n(0..$#arr) { 
    return "$n" if ($match =~ /$arr[$n]/); 
  }
  return "-1";
}

#
# Filter all duplicate entries from a list.
#
# Input: 1) a array.
#
# Retun: the reference of an array which contains the unique elements of the input array.
#
sub vxif_uniq (@) {
  my ($element, %elements, @a);
  my ($index) = 0;
  
  foreach $element (@_) {
    if (!defined($elements{$element})) {
      $elements{$element} = $index;
      $index = $index + 1;
    }
  }
  
  foreach $element (keys(%elements)) {
    $a[$elements{$element}] = $element;
  }

  return \@a;
}

# 
# Write a string to a file or append a file with a string.
#
# Input: 1) localized message to append to the file;
#        2) complete path of the file;
#        3) indicate whether to append or overwrite the given file;
#        4) reference of the error handler subroutine;
#
sub vxif_writef ($$$;$) {
  my($msg,$file,$new,$rd,$error_handler);
  ($msg,$file,$new,$error_handler) = (@_);
  $rd = ($new) ? ">" : ">>";
  open(WF,"$rd $file") or ($error_handler && &$error_handler());
  print WF "$msg\n";
  close(WF);
}

#
# Check whether the current installation host is local host.
#
# Input: 1) installation context;
#
# Return: 1 if the currently installation host is local host; Otherwise, return 0.
#
sub vxif_localsys ($) {
	my ($ctx) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $verdict = vxif_is_localhost($trg_host, $ctx);

	return $verdict;
}

#
# Contruct the fully qualified domain name base of the information available.
#
# Input: 1) installation context; must not be null;
#
# Return: the fully qualified domain name.
#
sub vxif_fqdn ($) {
  my($context) = @_;
  
  return ($context->{LOCAL}{DOMAIN}) ? $context->{LOCAL}{HOST} . '.' . $context->{LOCAL}{DOMAIN} : $context->{LOCAL}{HOST};
}

#
# Check whether a given host is localhost.
#
# Input: 1) host name; Must not be null;
#        2) installation context; Must not be null;
#
# Return: 1 if the given host name is local host; 0 otherwise.
#
sub vxif_is_localhost ($$) {
	my ($hostname,$ctx) = @_;
	my $hostinfo = $ctx->get_hostinfo();
	my $verdict = $hostinfo->is_local_host($hostname);

	return $verdict;
}

# 
# Copy a file.
#
# Input: 1) source;
#        2) destination;
#        3) the reference to the installation context; Cannot not be null;
#        4) source host, or null if localhost;
#        5) destination host, or null if localhost;
#        6) reference of the error handler subroutine, or null;
#
sub vxif_copy ($$$$$;$) {
	my($cmd,$sf,$df,$ctx,$error_handler,$sh,$dh,$r2l,$l2r,$do);
	($sf,$df,$ctx,$sh,$dh,$error_handler,) = @_;
        # adding these lines due to Darwin (Mac OS X) needs a capital R for the copy command or it 
        # will fail.  This is the reason there is a test for the O/S added lower down. - Dec 01, 05 - jpk
        my $trg_host = $ctx->get_target_hostname();
        my $trg_os_name = $ctx->get_host_os_name($trg_host);

	$r2l = $l2r = 0;

	if ($sh && !vxif_is_localhost($sh, $ctx)) {
		$r2l = 1;
	}

	if ($dh && !vxif_is_localhost($dh, $ctx)) {
		$l2r = 1;
	}

	if ($r2l && $l2r) { # Obviously, this is not possible!
		($error_handler && &$error_handler());
		return 0;
	}

        # Check for IPv6 address.  If there is one the you need to put the IP inside brackets or
        # the scp will not work.  Error num 256 will indicate it tried to copy program to a host 
        # with the name of the IP.
        if ($$ctx{IPv6}) {
           $dh = "\[".$dh."\]";
           $sh = "\[".$sh."\]";
        }

#  $sf = $r2l ? "${sh}:${sf}" : $sf;
#  $df = $l2r ? "${dh}:${df}" : $df;
# HACK
	$sf = $r2l ? "root\@${sh}:${sf}" : $sf;
	$df = $l2r ? "root\@${dh}:${df}" : $df;
	my $local_cmds = $ctx->get_local_cmds();
        my $rcp_cmd = $local_cmds->{RCP};
	my $cp_cmd = $local_cmds->{CP};
        #  Testing for Darwin OS - jpk
        if ($trg_os_name eq "Darwin") {
           $cmd = ($r2l || $l2r) ? "${rcp_cmd} -rp" : "${cp_cmd} -Rp";
        } else {
           $cmd = ($r2l || $l2r) ? "${rcp_cmd} -rp" : "${cp_cmd} -rp";
        }

        # test if this is for an IPv6 address.  If so, then use scp -6  - jpk 07 Feb 06
        if ($$ctx{IPv6}) {
           $cmd = $cmd."6";
           $$ctx{LOGGER}->fine("Utils::vxif_copy This is an IPv6 address");
        } 
	$cmd .= " ${sf} ${df}";

	$$ctx{LOGGER}->fine($cmd);
        # added the second parameter equal to 1 so the status of the command is returned.
        #$do=vxif_dol($cmd);   #  <-- Original command
        $do=vxif_dol($cmd,1);
        $$ctx{LOGGER}->fine("Copy result = ${do}");
	((!$do) && $error_handler && &$error_handler());

	return $do;
}

#
# Execute a command on local system.
#
# Input: 1) the command;
#        2) indicate what to return;
#            $re=NULL - return output
#            $re=1 - return exit code
#            $re=2 - return both as an array reference
#        3) command handler, or null; There will be three parameters pass into the
#           command handler: the original command, the output, and the exit code ($?).
#
# Output: the output of the command, the exit code, or both as an array reference.
#
sub vxif_dol ($;$$$) {
  my ($cmd,$re,$command_handler, $redir) = @_;
  my ($doe,$doo,$rdo);
  my ($cmd_proc, $argv) = (undef, undef);

  if (running_on_windows()) {
	  # just do this on windows.
	  ($cmd_proc, $argv) = PerlUnixSh::parse_cmdline($cmd);
  }

  if ($cmd_proc) {
	  # our native perl supports this command
	  if ($redir) {
		  # had to do it
		  if (open(REDIR, "> $redir")) {
			  my $proc = sub {
				  my ($outp_rec) = @_;
				  print(REDIR "${outp_rec}\n");
			  };
			  $doo = PerlUnixSh::backquote($cmd_proc, $argv, $proc);
			  close(REDIR);
		  } else {
			  # but couldn't
			  $doo = "";
			  $? = 1;
		  }
	  } else {
		  # vanilla
		  $doo = PerlUnixSh::backquote($cmd_proc, $argv);
	  }
  } else {
	  # our native perl does not support this command,
	  # so do it the "old fashioned" way.
	  #
	  # stick on directive for stderr ==> stdout
	  $cmd .= " 2>&1" unless ($cmd =~ />/);

	  # bracket in parens so shell will redirect
	  # both stdout and stderr as we require.
	  if ($redir) {
		  $cmd = "(${cmd}) > $redir";
	  }

	  $doo=`$cmd`;
  }
  # either way.
  $doe = $?;
  chomp($doo);
  # now, $doo contains the command output, and $doe contains the return code.
  
  ($command_handler && &$command_handler($cmd, $doo, $doe));

  # TODO: clean this up (DE 4-30-05)
  return "$doo" if (!$re);
  return "$doe" if ($re==1);
  $rdo=["$doe","$doo"];
  return $rdo; 
}

#
# Execute a command on target system.
#
# Input: 1) the command;
#        2) indicate what to return;
#            $re=NULL - return output
#            $re=1 - return exit code
#            $re=2 - return both as an array reference
#        3) installation context; Must not be null;
#        4) command handler, or null; There will be three parameters pass into the
#           command handler: the original command, the output, and the exit code ($?).
#
# Output: the output of the command, the exit code, or both as an array reference.
#
sub vxif_dor ($$$;$$) {
	my ($cmd,$re,$ctx,$command_handler, $redir) = @_;
	my (@r,$q,$rsh,$rtn,$doe,$doo);
	my $trg_host = $ctx->get_target_hostname();
	my $local_host = $ctx->get_local_hostname();
	my $local_cmds = $ctx->get_host_os_cmds($local_host);
	my $local_rsh_cmd = $local_cmds->{RSH};

	if (vxif_localsys($ctx)) {
		($ctx->{LOGGER}) && ($ctx->{LOGGER}->fine($cmd));
		return vxif_dol($cmd, $re, $command_handler, $redir);
	}

	if ($re =~ /^\d+$/ && $re==2) {
		my $trg_os = $ctx->get_host_os_name($trg_host);
		my $os_cmds = $ctx->get_os_cmds($trg_os);
		my $rex_cmd = $os_cmds->{REX};
		my $cat_cmd = $os_cmds->{CAT};
		$doe = vxif_dol("$local_rsh_cmd " . $trg_host . " '" . ${rex_cmd} . " $cmd 2>&1'");
		$doo = vxif_dol("$local_rsh_cmd " . $trg_host . " " . ${cat_cmd} . " /tmp/rex");
		@r=($doe,$doo);
		
		($command_handler && &$command_handler($cmd, $doo, $doe));
		
		return \@r;
	}

	$rsh = sprintf
		(
		 "%s %s -l root ",
		 $local_rsh_cmd,
		 $trg_host,
		 );
	my $dq = '"';
	$q = $dq unless ($cmd =~ /^\s*${dq}/);
	my $eff_rsh_cmd = "${rsh}${q}${cmd}${q}";
	$rtn = vxif_dol($eff_rsh_cmd, $re, $command_handler, $redir);
	if ($ctx->{LOGGER}) {
		$ctx->{LOGGER}->fine("${eff_rsh_cmd}");
		$ctx->{LOGGER}->fine("rtn=${rtn}");
	}
	return "${rtn}";
}

#
# Copy a file and a backup with local host.
#
# Input: 1) source;
#        2) destination;
#        3) backup postfix;
#        4) reference of the error handler subroutine, or null;
#
sub vxif_copywbu ($$$$) {
  my ($sf,$df,$be,$error_handler) = (@_);
  
  #vxif_copy($sf, $df, "", "", "", $error_handler);
  #vxif_copy($sf,"${df}${be}", "", "", "", $error_handler);
}

#
# Directory check - remote must be done by rshing an ls command.
#
# Input: 1) complete directory path;
#        2) installation context; must not be null;
#
# Return: 1 if the directory exists; Otherwise, 0 is returned.
#
sub vxif_dc ($$) {
	my ($file, $ctx) = @_;
	my ($ls,$r);
	my $trg_host = $ctx->get_target_hostname();
	my $logger = $ctx->get_logger();
	my $rc;

	if ($logger) {
		# we may be creating/testing the logdir itself
		# and, so, the logger may not yet exist.
		$logger->fine(Utils::_tr("Checking directory, ${file}, exist on ${trg_host}.", 12, 1007, "${file}", "${trg_host}"));
	}

	if (vxif_localsys($ctx)) {
		$rc = (-d $file ? 1 : 0);
	} else {
		my $trg_os = $ctx->get_host_os_name($trg_host);
		my $os_cmds = $ctx->get_os_cmds($trg_os);
		my $ls_cmd = $os_cmds->{LS};
		my $eff_cmd = "${ls_cmd} -d ${file}";
		$logger->fine($eff_cmd);
		$ls = vxif_dor($eff_cmd, "", $ctx);
		$logger->fine("result ls = ${ls}");
		$rc = ($ls eq $file ? 1 : 0);
	}

	return $rc;
}

# 
# File check - remote must be done by rshing an ls command.
#
# Input: 1) complete file path;
#        2) installation context; must not be null;
#
# Return: 1 if the file exist; Otherwise, 0 is returned.
#
sub vxif_fc ($$) {
	my ($file, $ctx) = @_;
	my $rc;
	
	if (vxif_localsys($ctx)) {
		$rc = (-e $file ? 1 : 0);
	} else {
		my $trg_host = $ctx->get_target_hostname();
		my $trg_os = $ctx->get_host_os_name($trg_host);
		my $os_cmds = $ctx->get_os_cmds($trg_os);
		my $ls_cmd = $os_cmds->{LS};
		my $ls = vxif_dor("${ls_cmd} ${file}", "", $ctx);

		$rc = ($ls eq $file ? 1 : 0);
	}

	return $rc;
}

#
# Create a given directory on the system for which the installation occurs.
#
# Input: 1) the full path to the directory;
#        2) the installation context;
#
# Return: 1 if the directory is successfully create; 0 otherwise.
#
sub vxif_mkdir ($$) {
	my ($dirname,$ctx) = @_;
	my $logger = $ctx->get_logger();

	return 1 if (vxif_dc($dirname, $ctx));

	my $trg_host = $ctx->get_target_hostname();
	my $os_cmds = $ctx->get_host_os_cmds($trg_host);
	my $mkdir_cmd = $os_cmds->{MKDIR};
	my $eff_cmd = "${mkdir_cmd} -p ${dirname}";

	$logger->fine("${eff_cmd}");
	vxif_dor($eff_cmd, "", $ctx);

	my $rc = vxif_dc($dirname, $ctx);

	return $rc;
}

#
# Remove a given directory on the system for which the installation occurs.
#
# Input: 1) the full path to the directory;
#        2) the installation context;
#
# Return: 1 if the directory is successfully removed; 0 otherwise.
#
sub vxif_rmdir ($$) {
	my ($dirname, $ctx) = @_;
	my ($path,$filename) = 0;

	# get the path and directory name
	if ($dirname =~ /(.+)\/(.+)$/) {
		$path = $1;
		$filename = $2;
	}

	my $trg_host = $ctx->get_target_hostname();
	my $os_cmds = $ctx->get_host_os_cmds($trg_host);
	my $rmr_cmd = $os_cmds->{RMR};
	my $cmd = $path ? "cd ${path}; " : "";

	$cmd .= "${rmr_cmd} ${filename}";

	return 1 if (!vxif_dc($dirname, $ctx));

	vxif_dor("${cmd}", "", $ctx);

	my $rc = vxif_dc($dirname, $ctx);

	return (!$rc);
}

#
# Translate into locale
#
# Input: 1) the message to be translated;
#        2) the catalog ID;
#        3) the message ID;
#        4)* the message parameters;
#
# Return: the localized message.
#
sub _tr {
	if ($ENV{"VXIF_UTILS_NOLOCALIZE"}) {
		# since this method (and class) is static (:-(,
		# and since context is not passed, use an
		# environment varible.  Hopefully, set this
		# only in the outer program as a result of
		# the -nolocalize flag.
		# 
		# mostly for debugging, in situations where
		# localization is unnecessary and slow.
		return $_[0];
	}
	my (@l,@w,$a,$c,$line,$fc,$msg,$nm,$sp,$tm,$vxa,$word,$trmsg);
	if (defined($_[1]) && defined($_[2])) {
		#if (($_[1]>0) && ($_[2]>0) && (-x "$ENV{VXIF_HOME}/tools/$ENV{VXIF_LOCAL_OS}/vcsi18n/bin/vxgettext")) {
		if (($_[1]>0) && ($_[2]>0)) {
			foreach $a (@_) {
				$vxa .= " \"$a\"";
			}

                        my $osname = `uname -s`;
                        $osname =~ s/\s+$//; # Strip whitespaces
			
                        # This bit is to correct a bug?, strange occurence in HP-UX.  HP wants the locales
                        # with the first part in lower case when coming from Perl.  31 Aug 06 - jpk - ET 789715
                        if ($osname eq "HP-UX") {
				if ( $ENV{VXIF_ORIGENVLANG} ne "C" ) {
                          		my ($one, $two) = split(/\./, $ENV{VXIF_ORIGENVLANG});
                           		$ENV{LANG}=$ENV{LC_ALL}= lc($one).".".$two;
				}
                        } else {
                           $ENV{LANG}=$ENV{LC_ALL}=$ENV{VXIF_ORIGENVLANG};
                        }

        		#my ($lang, $country) = split(/\_/, $ENV{VXIF_ORIGENVLANG});
	        	#$ENV{VCSI18N_LANG}=$lang;

                        # if we are running on SunOS we need to be able to 
                        # determine which hardware platform we are running on
                        # for vxgettext purposes
			my $cmd="$ENV{VXIF_HOME}tools/$ENV{VXIF_LOCAL_OS}/vcsi18n/bin/vxgettext";
                        if ($osname eq "SunOS") {
                                my $hdwr_type = `uname -p`;
                                $hdwr_type =~ s/\s+$//; # Strip ending whitespaces
			        $cmd="$ENV{VXIF_HOME}tools/$ENV{VXIF_LOCAL_OS}/" . $hdwr_type . "/vcsi18n/bin/vxgettext";
                        }
                        my $cmdline="$cmd $vxa 2> /dev/null";
                        my ($ecode, $langtype) = split(/\./, $ENV{VXIF_ORIGENVLANG});

                        my ($lang, $country) = split(/\_/, $ecode);
                        if (($lang eq "zh") && ($country eq "TW")) {
                                $ENV{VCSI18N_LANG}="tw";
                        } else {
                                $ENV{VCSI18N_LANG}=$lang;
                        }
                        
                        if ($langtype ne "") {
                                        if (lc($langtype) eq "iso8859-1" ) { $lang = "iso-8859-1"  }
                                        elsif(lc($langtype) eq  "iso8859-15") { $lang = "iso-8859-15" }
                                        elsif(lc($langtype) eq  "iso88591"  ) { $lang = "iso-8859-1"  }
                                        elsif(lc($langtype) eq  "roman8"    ) { $lang = "hp-roman8"   }
                                        elsif(lc($langtype) eq  "gb18030"   ) { $lang = "gbk"         }
                                        elsif(lc($langtype) eq  "euccn"     ) { $lang = "euc-cn"      }
                                        elsif(lc($langtype) eq  "gb2312"    ) { $lang = "gb2312"      }
                                        elsif(lc($langtype) eq  "gbk"       ) { $lang = "gbk"         }
                                        elsif(lc($langtype) eq  "big5"      ) { $lang = "big5"        }
                                        elsif(lc($langtype) eq  "pck"       ) { $lang = "Shift_JIS"   }
                                        elsif(lc($langtype) eq  "sjis"      ) { $lang = "Shift_JIS"   }
                                        elsif(lc($langtype) eq  "eucjp"     ) { $lang = "euc-jp"      }
                                        elsif(lc($langtype) eq  "euckr"     ) { $lang = "euc-kr"      }
                                        elsif(lc($langtype) eq  "cp949"     ) { $lang = "euc-kr"      }
                                        else              { $lang = "utf8"       }
                        } else {
                                if (($osname eq "AIX") && ($ecode eq "ja_JP")) {
                                        $lang = "euc-jp";
                                } elsif (($osname eq "Darwin") && ($ecode eq "ko_KR")) {
                                        $lang = "euc-kr";
                                } elsif ($osname eq "AIX") {
                                        $lang = "iso-8859-1";
                                } elsif ($ecode eq "zh_CN") {
                                        $lang = "gb2312";
                                } elsif (($osname eq "Darwin") && ($ecode eq "ja_JP")) {
                                        $lang = "euc-jp";
                                } elsif (($osname eq "SunOS") && ($ecode eq "ko")) {
                                        $lang = "euc-kr";
                                } elsif (($osname eq "SunOS") && ($ecode eq "zh")) {
                                        $lang = "gb2312";
                                } elsif (($osname eq "SunOS") && (($ecode eq "fr") || ($ecode eq "fr_FR"))) {
                                        $lang = "iso-8859-1";
                                } elsif (($osname eq "SunOS") && (($ecode eq "de") || ($ecode eq "de_DE"))) {
                                        $lang = "iso-8859-1";
                                } else {
                                        $lang = "utf8";
                                }
                        }
			from_to( $msg = `$cmdline`, $lang, $lang);
			chomp($msg);
			$ENV{LANG}=$ENV{LC_ALL}="C";
		}
        }
	$msg=$_[0] if (!$msg);
	# return $msg auto-wrapping at $COMM{TERMX}-length($MSG{INDENT}) chars
	$trmsg="";
	@l=split(/^/,$msg);
	foreach $line (@l) {
		$fc=substr($line,0,1);
		@w=split(/ /,$line);
		$c = ($fc eq "\t") ? 7 : 0;
		foreach $word(@w) {
			$nm = ($word=~/\n/) ? 1 : 0;
			# $COMM{TPUT} characters are adding to the length()
			# stripping them is a mess so adding 5 to the screen
			# width is the best way to fake it out
			$tm+=5 if (($word=~/\p{IsC}/) && ($word!~/\n/));
			if (defined($tm)) {
				if (($c>15) && ($c+length($word)-$nm>=$COMM{TERMX}+$tm-length($MSG{INDENT}))) {
					$trmsg.="\n"; 
					$trmsg.="\t\t" if (($fc eq "\t") || ($FLAG{TABLB}));
					$c = (($fc eq "\t") || ($FLAG{TABLB})) ? 15 : 0;
					$tm=0;
				}
			}
			$word=" $word" if ($c>0);
			$c+=length($word);
			$trmsg.=$word;
		}
	}

	return "$trmsg";
}

#
# Get file size.
#
# Input: 1) the name of the file
#
# Return: the size of the file, or -1 if no can do
#
sub vxif_filesize ($) {
	my ($fnam) = @_;
#	my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
#		$atime,$mtime,$ctime,$blksize,$blocks)
#		= lstat($fq_fnam);
	my @stat_stuff = lstat($fnam);
	my $size_ix = 7;
	my $stat_stuff_size = scalar(@stat_stuff);
	my $size = -1;

	if ($stat_stuff_size > 0) {
		$size = $stat_stuff[$size_ix];
	}

	return $size;
}

#
# Get file size.
#
# Input: 1) the name of the file
#
# Return: the size of the file in Kb, or -1 if no can do
#
sub vxif_filesize_in_k ($) {
	my ($fnam) = @_;
	my $size = vxif_filesize($fnam);
	my $size_in_k;

	if ($size >= 0) {
		$size_in_k = int(($size / 1024) + 0.5);
	} else {
		$size_in_k = -1;
	}

	return $size_in_k;
}

1;

