
package TrainInstaller;

our @ISA = ("Installer");

require VxIF::Installer;
require VxIF::ThunkAll;
require VxIF::SunOS::ThunkSunOS;

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: UPIS          => the reference of the list of all UPIs that are part of the train;
#         AUPIS         => the reference of the list of IFRTA UPIs that are part of the train;
#         BUPIS         => the reference of the list of IFRTB* UPIs that are part of the train;
#         TRAINROOT     => the root train directory which contains all the IFRTC-compliant 
#                          product scripts for products on the train. VxIF will load the 
#                          IFRTC-compliant product scripts using this path: "<TRAINROOT>::<UPI>::<OS>::<UPI><OS>";
#         PRODUCT       => the official name of the train to install;
#         VERSION       => the official version of the product train to install;
#         PROGRAM       => program name to start this train installer;
#         TRAINID       => train identifier; For example, "SxRT", "LxRT", etc.
#         LOGGER        => reference to the Logger object; This is optional;
#         TITLE         => product title to display on the title bar; This is option; 
#                          The default title is <Product Name> <Product Version>;
#         SPLASHMSG     => localized splash screen message; This is optional;
#         COPYRIGHTH    => localized product copyright header; This is optional;
#         COPYRIGHTB    => localized product copyright message; This is optional;
#         TMPPATH       => the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY   => install without configure; This is optional;
#                          The default is install and configure;
#         INSTALLVLIC   => automatically install the VERITAS license package if it is not present on
#                          the target system; This is option; By default, VERITAS license will be added
#                          to the target host for the licensable products;
#         RESPONSEFILE  => response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH        => indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         LOGLEVEL      => indicate the log level; This is optional;
#                          The default is VxIF::Logger::LEVEL_INFO;
#         LOGDIR        => the complete path to the log directory; This is optional; The
#                          default is /var/tmp/<PROGRAM><TIME>;
#         LOGFILE       => the name of the log file; This is optional; The default is
#                          <PROGRAM><TIME>.log;
#         SUPPORTEDOS   => a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, Darwin, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#
# Return: an instance of the TrainInstaller object.
#
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = {TITLE         => "VERITAS Train Installer",
              PRODUCT       => 'cpi',
              PROGRAM       => 'vxif',
              TRAINROOT     => 'VxIF',
              INSTALLONLY   => 0,
              USESSH        => 0,
              TMPPATH       => '/var/tmp',
              LOGLEVEL      => VxIF::Logger::LEVEL_INFO,
              TIMEFILE      => '/tmp/vxif_time',
              SUPPORTEDOS   => [ qw(SunOS Linux Darwin AIX HPUX OSF1) ],
              CMDSEARCHPATHS=> [ qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin /local/bin) ],
              @_,
             };
      
  return bless $self, $class;
}

#
# Start the installation of the product(s) on this train.
#
# Return: 1 if the installation is successful. Otherwise, 0 is returned.
#
sub install {
  my ($self) = @_;
  my @upis = @{$self->{UPIS}};
  my @oss = @{$self->{SUPPORTEDOS}};
  my ($upi,$os,$path,$class);
  my (%thunkmap);
  
  $thunkmap{SunOS} = ThunkSunOS->new();
  $thunkmap{SunOS}->initialize();
  $thunkmap{AIX} = ThunkSunOS->new();
  $thunkmap{AIX}->initialize();
  $thunkmap{HPUX} = ThunkSunOS->new();
  $thunkmap{HPUX}->initialize();
  $thunkmap{OSF1} = ThunkSunOS->new();
  $thunkmap{OSF1}->initialize();
  $thunkmap{Linux} = ThunkSunOS->new();
  $thunkmap{Linux}->initialize();
  $thunkmap{ALL} = ThunkAll->new();
  $thunkmap{ALL}->initialize();
    
  if ($#upis < 0) {
    die Utils::_tr("Unable to install products. UPI list is empty. No product to install.", 15, 1000);
  }
  
  if ($#oss < 0) {
      die Utils::_tr("Unable to install products. Supported OS list is empty.", 15, 1001);
  }
  
  # now load the UPIS.
  foreach $os (@oss) {
    foreach $upi (@upis) {
      if (($self->{AUPIS}) && (Utils::vxif_list($upi, @{$self->{AUPIS}}) >= 0)) {
        #load the B stuff here
        $self->add_product($upi, $os, $thunkmap{$os});
      } elsif (($self->{BUPIS}) && (Utils::vxif_list($upi, @{$self->{BUPIS}}) >= 0)) {
        #load the B stuff here
        $self->add_product($upi, $os, $thunkmap{ALL});
      } else {
        #load the C stuff here
        #$path = "$self->{TRAINROOT}::${upi}::${os}::${upi}${os}";
		my $ret = eval "require ($self->{TRAINROOT}::${upi}::${os}::${upi}${os})";
        if (1 == $ret) {
          $class = "${upi}${os}";
          $self->add_product($upi, $os, $class->new());
        } else {
          die Utils::_tr("Unable to load $upi from $path.", 15, 1002, "$upi", "$path");
        }
      }
    }
  }
  
  $self->{RESPONSEFILE} || ($self->{DISPLAYRTI} = 1);
  $self->SUPER::install();
}

#############################################################
#
# Train-related code from Installer.pm (DE 8-29-05)
#
#############################################################

#
# Display the product copyright information.
#
# Input: 1) the reference to self;
#        2) the reference of the installation context;
#
sub display_product_copyright ($\%) {
	my $self = shift;
	my ($ctx) = @_;

	Utils::vxif_title($$ctx{INSTALLERTITLE});
	Utils::vxif_pl("$self->{COPYRIGHTH}\n");
	Utils::vxif_pl("$self->{COPYRIGHTB}\n");
}

#
# Check to see whether products to be installed had already installed on localhost 
# and get the version numbers of the installed products.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_for_installed_products (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($rc) = 1;

	$rc = $$ctx{INSTALLERS}{$$ctx{LOCAL}{OS}{NAME}}->check_installed_packages_versions($ctx);
	Utils::vxif_title($$ctx{INSTALLERTITLE});
	($rc) && ($$ctx{INSTALLERS}{$$ctx{LOCAL}{OS}{NAME}}->display_installed_product_status($ctx));

	return $rc;
}

#
# Ask user to select an installation task to perform.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
# Return: the selected task.
#
sub select_installation_task (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($menuitem,$m,@mku,@mkl,$helpmsg,$msg);
	my ($ident) = 0;
	my (@menuitems) = qw(INSTALL CONFIG LICENSE PRECHECK UNINSTALL DESC QUIT HELP);

	Utils::vxif_bpl(Utils::_tr("\nSelection Menu:\n", 15, 1003));
	foreach $menuitem (@menuitems) {
		($ident % 2)&&(Utils::vxif_p($self->{MSG}{IDENT}));
		Utils::vxif_bp($self->{MSG}{"${menuitem}KEY"}{U});
		push(@mku, $self->{MSG}{"${menuitem}KEY"}{U});
		push(@mkl, $self->{MSG}{"${menuitem}KEY"}{L});
		Utils::vxif_p(") $self->{MSG}{MENU}{$menuitem}");
		foreach $m (length($self->{MSG}{MENU}{$menuitem})..25) {Utils::vxif_p(" ");}
		($ident % 2)&&(Utils::vxif_pl(""));
		$ident++;
	}

	Utils::vxif_pl("");
	$m = join(", ", @mku);
	#$helpmsg = Utils::_tr("Please enter one of the following choices [${m}].");
	my $validator = sub ($) {
	#my $validator = sub {
		my ($answer) = @_;

		if ((Utils::vxif_list($answer, @mku) < 0)&&(Utils::vxif_list($answer, @mkl) < 0)) {
			Utils::vxif_pl(Utils::_tr("$answer is an invalid choice.", 15, 1004, "$answer"));
			return 0;
		}

		return 1;
	};
	$msg = Utils::_tr("Enter a Selection:", 15, 1005);
	$m = Utils::vxif_ask($msg, $self->{MSG}{INSTALLKEY}{U}, $helpmsg, $validator, 0, 0, $m);

	foreach $menuitem (@menuitems) {
		if (($m eq $self->{MSG}{"${menuitem}KEY"}{L}) || ($m eq $self->{MSG}{"${menuitem}KEY"}{U})) {
			return $self->{TASKS}{"$menuitem"};
		}
	}
}

#
# Display help. Describe the installation options.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
sub display_help (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my (@menuitems) = qw(INSTALL CONFIG LICENSE PRECHECK UNINSTALL DESC QUIT HELP);
	my ($menuitem);

	foreach $menuitem (@menuitems) {
		Utils::vxif_bp($self->{MSG}{"${menuitem}KEY"}{U});
		Utils::vxif_p(") $self->{MSG}{MENUHELP}{$menuitem}\n\n");    
	}

	Utils::vxif_prtc($ctx);
}

#
# Ask user to select a product to perform installation task.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
# Return: the UPI of the product to install or null.
#
sub select_product_to_install (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my (@selections,$product,$menutitle,@upis,$choice);
	my ($back_key_handler,$quit_key_handler);

	# go through all the products to check for existing main packages
	foreach $product (@{$self->{PRODUCT_ORDERING}{$$ctx{LOCAL}{OS}{NAME}}}) {
		$$ctx{TARGET}{UPI} = $product;
		push(@selections, $self->{PRODUCT}{$product}{$$ctx{LOCAL}{OS}{NAME}}->get_product_title($ctx));
		push(@upis, $product);
	}

	if ($$ctx{TASK} eq $self->{TASKS}{INSTALL}) {
		$menutitle = Utils::_tr("Select a product to install:", 15, 1006);
	} elsif ($$ctx{TASK} eq $self->{TASKS}{CONFIG}) {
		$menutitle = Utils::_tr("Select a product to configure:", 15, 1007);
	} elsif ($$ctx{TASK} eq $self->{TASKS}{LICENSE}) {
		$menutitle = Utils::_tr("Select a product to license:", 15, 1008);
	} elsif ($$ctx{TASK} eq $self->{TASKS}{PRECHECK}) {
		$menutitle = Utils::_tr("Select a product to perform a pre-installation check for:", 15, 1009);
	} elsif ($$ctx{TASK} eq $self->{TASKS}{DESC}) {
		$menutitle = Utils::_tr("Select a product to view a product description of:", 15, 1010);
	} elsif ($$ctx{TASK} eq $self->{TASKS}{UNINSTALL}) {
		$menutitle = Utils::_tr("Select a product to uninstall:", 15, 1011);
	}

	$back_key_handler = sub {
		$$ctx{TASK} = $self->{TASKS}{BACK};
	};
	$quit_key_handler = sub {
		$$ctx{TASK} = $self->{TASKS}{QUIT};
	};
	$choice = Utils::vxif_menu($menutitle, \@selections, "", 0, $back_key_handler, $quit_key_handler);

	if ($choice) {
		return $upis[$choice - 1];
	} else {
		return 0;
	}
}

#
# Display the product description.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
sub display_product_description (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($os);
	my ($desc) = Utils::_tr("$$ctx{TARGET}{UPI} has no product description.", 15, 1012, "$$ctx{TARGET}{UPI}");

	foreach $os (keys(%{$self->{PRODUCT}{$$ctx{TARGET}{UPI}}})) {
		if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}{$os}->can("get_product_description")) {
			$desc = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}{$os}->get_product_description($ctx);
			last;
		}
	}

	Utils::vxif_pl("\n${desc}\n");
	Utils::vxif_prtc($ctx);
}

#
# Display the Release Train Menu.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
sub display_train_menu (\%) {
  my $self = shift;
  my ($ctx) = @_;
  #display product copyright
  $self->display_product_copyright($ctx);
  
  while (1) {
    # check whether the products are install on localhost
    $self->check_for_installed_products($ctx);
    
    # ask user to select an installation task
    if ($$ctx{TASK} && $$ctx{RESPONSEFILE}) {
      $$ctx{LOGGER}->info("Task = $$ctx{TASK}");
    } elsif (!$$ctx{TASK}) {
      $$ctx{TASK} = $self->select_installation_task($ctx);
    }
    $self->cleanup($ctx) if ($$ctx{TASK} eq $self->{TASKS}{QUIT});
    
    Utils::vxif_title($$ctx{INSTALLERTITLE});
    if ($$ctx{TASK} eq $self->{TASKS}{HELP}) {
      $self->display_help($ctx); 
      next;
    }
  
    Utils::vxif_title($$ctx{INSTALLERTITLE});
    # ask user to select a product
    if (!$$ctx{RESPONSEFILE}) {
      my $upi = $self->select_product_to_install($ctx);
      if ((!$upi)&&($$ctx{TASK} ne $self->{TASKS}{BACK})&&($$ctx{TASK} ne $self->{TASKS}{QUIT})) {
        Utils::vxif_lpl("Unable to determine the product to install!", "error", $ctx);
        $self->cleanup($ctx);
      }
      $$ctx{TARGET}{UPI} = $upi;
    } else {
      $$ctx{TARGET}{UPI} = $$ctx{ORIGTARGET}{UPI};
    }
      
    if ($$ctx{TASK} eq $self->{TASKS}{DESC}) {
      $self->display_product_description($ctx);
      next;
    }
      
    $self->cleanup($ctx) if ($$ctx{TASK} eq $self->{TASKS}{QUIT});
    last if ($$ctx{TASK} ne $self->{TASKS}{BACK});
  }
}

#############################################################
#
# Train-related code from CPICommon.pm (DE 8-29-05)
#
#############################################################

#
# Get the version number for all the existing product packages on the local host.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if successfully determine the installed product versions; 0 otherwise.
#
sub check_installed_packages_versions (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($product,$pkg,$version);
	my $rc = 1;

	# make sure we are on local host
	$$ctx{TARGET}{HOST} = $$ctx{LOCAL}{HOST};
	$$ctx{HOST}{$$ctx{TARGET}{HOST}}{OS}{NAME} = $$ctx{LOCAL}{OS}{NAME};

	# go through all the products to check for existing main packages
	foreach $product (keys(%{$self->{PRODUCT}})) {
		$$ctx{TARGET}{UPI} = $product;
		$self->{$$ctx{TARGET}{HOST}}{$product}{IVER} = $self->check_product_main_package_version($ctx, $product, \$rc);
	}

	return $rc;
}

#
# Check to make sure the version installed on target host is valid.
#
# Input: 1) reference to the installation cotnext;
#
# Return: 1 if successful; 0 otherwise; 2 same product version already installed.
#
sub check_product_version (\%) {
  my ($self,$ctx) = @_;
  my ($version) = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_version($ctx);
  my ($iver) = $$ctx{HOST}{$$ctx{TARGET}{HOST}}{$$ctx{TARGET}{UPI}}{IVER};
  my ($upgradable) = 0;
  my ($downgradable) = 0;
  my ($configurable) = 0;
  
  $$ctx{LOGGER}->entering("CPICommon::check_product_version");
 
  $$ctx{LOGGER}->finer("iver=${iver}\nversion=${version}");
  
  return 1 if ((($$ctx{TASK} eq "install") || ($$ctx{TASK} eq "precheck")) && ($$ctx{HOST}{$$ctx{TARGET}{HOST}}{$$ctx{TARGET}{UPI}}{IVER} eq ""));
  return 1 if (($$ctx{TASK} eq "uninstall") && ($$ctx{HOST}{$$ctx{TARGET}{HOST}}{$$ctx{TARGET}{UPI}}{IVER} ne ""));
  
  if ($$ctx{TASK} eq "install") {
    ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("is_product_upgradable")) && ($upgradable = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->is_product_upgradable($ctx, $version));
    ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("is_product_downgradable")) && ($downgradable = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->is_product_downgradable($ctx, $version));
 
    $$ctx{LOGGER}->finer("upgradable=${upgradable}\ndowngradable=${downgradable}");
 
    my $msg = Utils::_tr("\n$$ctx{TARGET}{UPI} version $iver is already installed on $$ctx{TARGET}{HOST}.", 15, 1013, "$$ctx{TARGET}{UPI}", "$iver", "$$ctx{TARGET}{HOST}");
    if (Utils::vxif_cv($iver, $version) == 0) {
      Utils::vxif_lpl($msg, "error", $ctx);
      return 2;
    } elsif ((Utils::vxif_cv($iver, $version) == 1) && (!$downgradable)) {
      $msg .= " ". Utils::_tr("This version is not downgradable.");
      Utils::vxif_lpl($msg, "error", $ctx);
      return 0;
    } elsif (Utils::vxif_cv($iver, $version) == 2) {
		if ($upgradable) {
			# this is an upgrade
			$$ctx{TASK} = "upgrade";
			return 1;
		} else {
			$msg .= " ". Utils::_tr("This version is not upgradable.");
			Utils::vxif_lpl($msg, "error", $ctx);
			return 0;
		}
    }
  } elsif ($$ctx{TASK} eq "uninstall") {
    my $msg = Utils::_tr("\n$$ctx{TARGET}{UPI} is not installed on $$ctx{TARGET}{HOST}.", 15, 1014, "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}");
    Utils::vxif_lpl($msg, "error", $ctx);
    return 0;
  } elsif (($$ctx{TASK} eq "configure")||($$ctx{TASK} eq "license")){
    $$ctx{LOGGER}->finer("iver=${iver}\nversion=${version}");
    
    if ($iver eq "") {
      my $msg = Utils::_tr("Unable to configure $$ctx{TARGET}{UPI} since it does not exist on $$ctx{TARGET}{HOST}", 15, 1015, "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}");
	  Utils::vxif_lpl($msg, "warning", $ctx);

      #$$ctx{LOGGER}->error($msg); 
      $$ctx{LOGGER}->error("Unable to configure $$ctx{TARGET}{UPI} since it does not exist on $$ctx{TARGET}{HOST}");
      return 0;
    } elsif (Utils::vxif_cv($iver, $version) == 0) {
      return 1;
    } else {
      ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("is_product_configurable")) && ($configurable = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->is_product_configurable($ctx, $version));
      $$ctx{LOGGER}->finer("configurable=$configurable");
      return $configurable;
    }
  } else {
    return 1;
  }
}

#
# Determine the version of the main product package if it exist on the target host. Users
# must check $rc to make sure it is not 0.
#
# Input: 1) reference to the installation cotnext;
#        2) the product UPI;
#        3) the reference to the return status;
#
# Return: the main package version on target host if it exist; null otherwise.
#
sub check_product_main_package_version (\%$\$) {
  my ($self,$ctx,$product,$rc) = @_;
  my ($pkg,$pkgs,$osver,$osarch,$prodver);
  my ($version) = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_version($ctx);
    
  # get the product packages to install
  $prodver = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_version($ctx);
  $osver = $$ctx{HOST}{$$ctx{TARGET}{HOST}}{OS}{VERSION};
  $osarch = $$ctx{HOST}{$$ctx{TARGET}{HOST}}{OS}{ARCHITECTURE};
  $pkgs = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_packages($ctx, $prodver, $osver, $osarch);
#  ($version = $$pkgs{$pkg}{VERSION}) if ($$pkgs{$pkg}{VERSION}); 
  
  $$rc = 1;
  
  if ($self->{PRODUCT}{$product}->can("get_product_main_package")) {
    $pkg = $self->{PRODUCT}{$product}->get_product_main_package($ctx);
    #$$ctx{LOGGER}->info(Utils::_tr("Main package for ${product} is ${pkg}.", 15, 1016, "${product}", "${pkg}"));
    $$ctx{LOGGER}->info("Main package for ${product} is ${pkg}.");
    if ($pkg && ($pkg ne "")) {
      $version = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
    } else {
      $$rc = 0;
      #$$ctx{LOGGER}->error(Utils::_tr("Unable to determin the main package for ${product} because the package name return by get_product_main_package() is null.", 15, 1017, "${product}"));
      $$ctx{LOGGER}->error("Unable to determin the main package for ${product} because the package name return by get_product_main_package() is null.");
    }
  } else {
    #$$ctx{LOGGER}->error(Utils::_tr("Unable to determine the main package for ${product} because the get_product_main_package() interface is not implemented.", 15, 1018, "${product}"));
    $$ctx{LOGGER}->error("Unable to determine the main package for ${product} because the get_product_main_package() interface is not implemented.");
    $$rc = 0;
  }
  
  #($version) && ($$ctx{LOGGER}->info(Utils::_tr("${version} of ${pkg} found on $$ctx{LOCAL}{HOST}", 15, 1019, "${version}", "${pkg}", "$$ctx{LOCAL}{HOST}")));
  ($version) && ($$ctx{LOGGER}->info("${version} of ${pkg} found on $$ctx{LOCAL}{HOST}"));
  #($version) || ($$ctx{LOGGER}->info(Utils::_tr("${pkg} currently not installed on $$ctx{LOCAL}{HOST}", 15, 1020, "${pkg}", "$$ctx{LOCAL}{HOST}")));
  ($version) || ($$ctx{LOGGER}->info("${pkg} currently not installed on $$ctx{LOCAL}{HOST}"));
          
  $$ctx{HOST}{$$ctx{TARGET}{HOST}}{$$ctx{TARGET}{UPI}}{IVER} = $version;
  
  return $version;
}

#
# Display the installed product status on the local host.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if successfully determine the installed product versions; 0 otherwise.
#
sub display_installed_product_status (\%) {
  my ($self,$ctx) = @_;
  my ($product,$bs,$pl,$lic,$name,$installed);
  my $rc = 1;
  
  # print the header
  $bs = "%-50s%-16s%-s";
  $pl=sprintf("%-44s%-20s%-s", $self->{MSG}{VENDORPR}, $self->{MSG}{VERSINST}, $self->{MSG}{LICENSED});
  Utils::vxif_bpl($pl);
  print "===========================================================================\n";
  
  # go through all the products to check for existing main packages
  foreach $product (@{$self->{PRODUCT_ORDERING}}) { 
    $$ctx{TARGET}{UPI} = $product;
    $name = $self->{PRODUCT}{$product}->get_product_title($ctx);
    $lic = $self->{MSG}{YES};
    if (($self->{PRODUCT}{$product}->can("is_product_licensable"))&&(!$self->{PRODUCT}{$product}->is_product_licensable($ctx))) {
      $lic = $self->{MSG}{NA};
    }
    $installed = $self->{$$ctx{LOCAL}{HOST}}{$product}{IVER} ? $self->{$$ctx{LOCAL}{HOST}}{$product}{IVER} : $self->{MSG}{NO};
    $pl=sprintf($bs, $name, $installed, $lic);
    Utils::vxif_pl($pl);
  }
  
  return $rc;
}

1;

