package CPISunOS;

our @ISA = ("CPICommon");

require VxIF::CPICommon;

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#         SPACEWARNING	=> warning the users of low on space if the number of bytes on the 
#                          volume after the installation is less than this value; This is optional;
#                          The default is 5120 kilobytes;
#
sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $os = [qw(SunOS Linux HPUX AIX OSF1)];
	my $searchpaths = [qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin)];
	my $self = { 	
		USESSH		=> 0,
		IPv6		=> 0,
		TMPPATH		=> '/var/tmp/vxif',
		TIMEFILE	=> '/tmp/vxif_time',
		PKGSDIR		=> 'pkgs',
		PATCHESDIR	=> 'patches',
		OSINSTALLER	=> 'SunOS',
		SPACEWARNING	=> '5120',
		SUPPORTEDOS	=> $os,
		CMDSEARCHPATHS	=> $searchpaths,
		PKGPATH => '',
		@_,
	};

	return bless $self, $class;
}

#
# Set the common system commands.
#
sub set_commands {
  my ($self) = @_;
  
  $self->SUPER::set_commands();
  
  $self->{CMD}{DOMAINNAME}="/usr/bin/domainname";
  $self->{CMD}{PATCHADD}="/usr/sbin/patchadd";
}

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  $self->{MSG}{PDFREN} 	= "package";
  $self->{MSG}{PDFR} 	= Utils::_tr("package", 19, 1000);
  $self->{MSG}{PDFRS}	= Utils::_tr("packages", 19, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}

#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
  
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) extension;
#        4) the reference to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers (\%$;$) {
	my $self = shift;
	my ($ctx,$pkg,$ext,$pkgs) = @_;
	my ($iv,@pkgnames);
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	$$ctx{LOGGER}->entering("CPISunOS::get_instpkgvers");

	@pkgnames = keys(%{$pkgs});

	$$ctx{LOGGER}->finer("pkg=${pkg}");

	if ((Utils::vxif_list($pkg, @pkgnames) >= 0) && $upi_inst->can("get_installed_package_version")) {
		$iv = $upi_inst->get_installed_package_version($ctx, $pkg);
	} else {
		$iv = $self->get_instpkginfovalue($ctx, $pkg, "VERSION");
		if ($iv =~ /^(\d+(\-\d+)(\.\d+)+).*/) {
			$iv = $1;
		}
	}
	$$ctx{LOGGER}->exiting("CPISunOS::get_instpkgvers");

	return "$iv";
}

#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
  my ($self,$ctx,$pkg,$pkgs,$ext) = @_;
  
  return $self->get_mediapkginfovalue($ctx, $pkg, "VERSION");
}

# 
# Get information from the pkginfo file on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the property in the pkginfo file; Must be in upper case;
#
# Return: the property value of the given property in the given package.
#
sub get_mediapkginfovalue (\%$$) {
	my $self = shift;
	my ($ctx,$pkg,$param) = @_;
	my ($v) = "";
	my $trg_upi = $ctx->get_target_upi();
	my $os_cmds = $ctx->get_target_cmds();
	my $grep_cmd = $os_cmds->{GREP};

	$$ctx{LOGGER}->entering("CPISunOS::get_mediapkginfovalue");

	$param.="=";

	if (Utils::vxif_fc("$self->{$trg_upi}{$pkg}{TLOC}/${pkg}/pkginfo", $ctx)) {
		$$ctx{LOGGER}->info("${grep_cmd} '^$param' $self->{$trg_upi}{$pkg}{TLOC}/${pkg}/pkginfo");
		$v=Utils::vxif_dor("${grep_cmd} '^$param' $self->{$trg_upi}{$pkg}{TLOC}/${pkg}/pkginfo", "", $ctx);
	} elsif (Utils::vxif_fc("$self->{$trg_upi}{$pkg}{TLOC}/info/${pkg}", $ctx)) {
		$$ctx{LOGGER}->info("${grep_cmd} '^$param' $self->{$trg_upi}{$pkg}{TLOC}/info/${pkg}");
		$v=Utils::vxif_dor("${grep_cmd} '^$param' $self->{$trg_upi}{$pkg}{TLOC}/info/${pkg}", "", $ctx);
	}

	$v =~ s/($param)//;

	# Strip everything after the comma.
	if ($v =~ /^([^,]*)/) {
		$v = $1;
	}

	return $v;
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
  my ($self,$ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
  my($size) = 0;
  
  $$ctx{LOGGER}->entering("CPISunOS::get_mediapkgsize");
  
  if ($compressed) {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 19, 1002, "${pkg}"));
    if (-f "${loc}/${filename}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -skL ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -skL ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      # assume compression of 3x. multiple size by 4 to account for the compressed package itself
      $size *= 4;
    } else {
      $$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on $$ctx{LOCAL}{HOST}.", 19, 1003, "${pkg}", "${loc}", "${filename}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  } else {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is uncompressed", 19, 1004, "${pkg}"));
    if (-f "${loc}/${pkg}/pkginfo") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -skL ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -skL ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
    } else {
      $$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${pkg}/pkginfo does not exist on $$ctx{LOCAL}{HOST}.", 19, 1005, "${pkg}", "${loc}", "${pkg}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  }
  
  $$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} is ${size} kbs", 19, 1006, "${pkg}", "${size}"));
  
  return $size;
}

# 
# Get information from the pkginfo file.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the property in the pkginfo file; Must be in upper case;
#
# Return: the property value of the given property in the given package.
#
sub get_instpkginfovalue ($$$) {
  my ($self,$ctx,$pkg,$param) = @_;
  my ($v);
  
  return "" unless (Utils::vxif_fc("/var/sadm/pkg/$pkg/pkginfo", $ctx));
  $param.="=";
  $v=Utils::vxif_dor("$self->{CMD}{GREP} '^$param' /var/sadm/pkg/$pkg/pkginfo", "", $ctx);
  $v =~ s/($param)//;
 
  return $v;
}

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my ($q,$dep,$deplist,$deppkgs,@deps,@ideps);
	my $trg_upi = $ctx->get_target_upi();
	
	if ($upi_inst->can("check_for_package_dependents")) {
		$deppkgs = $upi_inst->check_for_package_dependents($ctx, $pkg);
		@deps = keys(%{$deppkgs});
		$deplist = join(" ", @deps);
		$deplist .= " ";
	}

	my $prefix = "/var/sadm/pkg";
	my $cmd = sprintf
		(
		 "%s -l 'P[ \t][ \t]*%s[ \t]' %s/VRTS*/install/depend 2> /dev/null",
		 $self->{CMD}{GREP},
		 $pkg,
		 $prefix,
		 );
	$dep=Utils::vxif_dor($cmd, "", $ctx);
	my @flist = split(/\n/, $dep);
	my $dep_plist = [];
	my $num_deppkgs = scalar(@flist);
	for (my $i = 0; $i < $num_deppkgs; $i++) {
		my $dep_pkg = $flist[$i];
		$dep_pkg =~ s/^${prefix}\///;
		$dep_pkg =~ s/\/.*//;
		push(@$dep_plist, $dep_pkg);
	}

	if ($deplist) {
		$deplist .= join(" ", @$dep_plist);
	} else {
		$deplist = join(" ", @$dep_plist);
	}

	return Utils::vxif_desp($deplist);
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%) {
  my ($self,$ctx,$nic) = @_;
  my ($na,$nm);
  
  # first check if the nic is actually plumbed.  It better be for anything
  # using it to work, but a user could still enter a dead one for testing
  $na = Utils::vxif_dor("$self->{CMD}{IFCONFIG} -a | $self->{CMD}{GREP} '^$nic: '", "", $ctx);
  return "" if (!$na);
  $nm = Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic | $self->{CMD}{GREP} netmask | $self->{CMD}{AWK} '{print \$4}'", "", $ctx);
  
  return "$nm";
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my (@pnics,$do);
  
  $do = Utils::vxif_dor("$self->{CMD}{IFCONFIG} -a | $self->{CMD}{GREP} UP | $self->{CMD}{SED} 's/:.*\$//' | $self->{CMD}{UNIQ} | $self->{CMD}{GREP} -v lo", "", $ctx);
  @pnics=split(/\s+/,$do);
  
  return \@pnics;
}

#
# Create the admin file on localhost.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if admin file is successfully created; 0 otherwise.
#
sub create_local_admin_file (\%) {
  my ($self,$ctx) = @_;
  my ($admfile) = "$$ctx{TMPPATH}/adminfile";
  
  if (!(-e $admfile)) {
    Utils::vxif_writef("mail=\ninstance=overwrite\npartial=nocheck\nrunlevel=quit\nidepend=quit\nrdepend=nocheck\nspace=quit\nsetuid=nocheck\nconflict=nocheck\naction=nocheck\nbasedir=default\n", $admfile, 1);
    if (!(-e $admfile)) {
      $$ctx{LOGGER}->error(Utils::_tr("Unable to create ${admfile} on $$ctx{LOCAL}{HOST}.", 19, 1007, "${admfile}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  }
  
  return 1;
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) the reference to the installation context;
#        2) the patch to be checked;
#        3) the reference to the existing patch on target OS;
#        2) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 the patch is obseleted; 
#         2 the same version or higher is installed.       
#
sub check_patch (\%$\$$) {
  my ($self,$ctx,$patch,$opatch,$refresh) = @_;
  my ($pn,$pv,$iv);
  
  $$ctx{LOGGER}->entering("CPISunOS::check_patch");
  
  if ($patch =~ /-/) {
    ($pn,$pv) = split(/-/,$patch,2);
    $$ctx{LOGGER}->finer("pn=${pn}\npv\${pv}");
  } else {
    $$ctx{LOGGER}->error(Utils::_tr("Invalid patch ID, $patch. The patch ID must be in ######-## format.", 19, 1008, "$patch"));
    return -1;
  }
  
  # get the OS patches for the installation host
  if ($refresh || (!defined($$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES}))) {
    $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES} = $self->get_patches($ctx);
  }
  
  $iv = $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES}{$patch};
  
  if ($iv) {
    $$opatch = $iv;
    return 1;
  }
  
  $iv = $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES}{$pn};
    
  if (!$iv) {
    return 0;
  } elsif ($iv < $pv) {
    $$opatch = "$pn-$iv";
  } else {
    $$opatch = "$pn-$iv";
    return 2;
  }
}

#
# Get all the patches for the given host.
#
# Input: 1) the reference of the installation context;
#
# Return: the reference to the patches hash.
#
sub get_patches ($) {
  my ($self,$ctx) = @_;
  my(%ph,@a,@op,$c,$j,$o,$p,$pkgs,$s,$v);
	
  $$ctx{LOGGER}->entering("CPISunOS::get_patches");

  $s=Utils::vxif_dor("$self->{CMD}{SHOWREV} -p", "", $ctx);
  @a=split(/^/,$s);
  foreach $c (@a) {
    ($j,$s,$j)=split(/\s/,$c,3);
    ($p,$v)=split(/-/,$s,2);
    # all patch keys are $p{######}=##
    $ph{$p}=$v unless ($v<$ph{$p});
    ($j,$o)=split(/Obsoletes: /,$c,2);
    ($o,$j)=split(/ Requires/,$o,2);
    @op=split(/, /,$o);
    for $o (@op) {
      # all obsoleted patch keys are $p{######-##}=######-##
      $ph{$o}="$p-$v";
    }
    # all package references are $p{$PKG} = [ ######, ###### ];
    chomp($c);
    ($j,$pkgs)=split(/ Packages: /,$c,2);
    if ($pkgs=~/VRTS/) {
      @op=split(/, /,$pkgs);
      for $o (@op) {
        $ph{$o}||=[];
	      push(@{$ph{$o}},$p) if (Utils::vxif_list($p,@{$ph{$o}})<0);
      }
    }
  }

  return \%ph;
}

#
# Check for and run post pkgadd operations.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
	my $self = shift;
	my ($upi,$pkg,$ctx) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my (@patchlist,$patch,$patchid);
	my ($rc,$installedpatchid,$msgl,$msgr);
	my ($result) = 1;
	
	if ($upi_inst->can("get_required_os_patches")) {
		@patchlist = $upi_inst->get_required_os_patches($pkg, $ctx);
	}
	
	if ($#patchlist >= 0 ) {
		Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 19, 1009, "$pkg"));
		  foreach $patch (@patchlist) {
			  $patchid = $patch;
			  ($upi_inst->can("get_patch_id")) && ($patchid = $upi_inst->get_patch_id($patch, $ctx));
			  $msgl = Utils::_tr("Checking OS patch ${patchid}", 19, 1010, "${patchid}");
			  Utils::vxif_pbl($msgl);
			  
			  $rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
			  if ($rc == -1) {
				  $msgr = Utils::_tr("invalid patch ID, ${patchid}", 19, 1011, "${patchid}");
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } elsif ($rc == 0) {
				  $msgr = Utils::_tr("not installed", 19, 1012);
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } elsif ($rc == 1) {
				  $msgr = Utils::_tr("${installedpatchid} installed", 19, 1013, "${installedpatchid}");
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } else {
				  $msgr = Utils::_tr("${installedpatchid} installed", 19, 1013, "${installedpatchid}");
				  Utils::vxif_pbr($msgl, $msgr);
			  }
		  }
	  }
	
	return 1;
}

#
# Copy the (presumably already existing) admin file to the target host.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub copy_admin_file_to_target_host (\%) {
	my ($self,$ctx) = @_;
	my $src_tmppath = $$ctx{TMPPATH};
	my $src_admfile = "${src_tmppath}/adminfile";
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $trg_admfile = "${trg_tmppath}/adminfile";

	if (-e $src_admfile) {
		if (($src_tmppath eq $trg_tmppath) && Utils::vxif_localsys($ctx)) {
			# doing local copy and, so, src == dst.
			# just leave file in place.
		} else {
			if (Utils::vxif_copy($src_admfile, "${trg_tmppath}", $ctx, "", $trg_host)) {
				$$ctx{LOGGER}->error(Utils::_tr("Unable to copy ${src_admfile} to ${trg_host}.", 19, 1014, "${src_admfile}", "${trg_host}"));
				return 0;
			}
		}
	} else {
		$$ctx{LOGGER}->error(Utils::_tr("Cannot find admin file ${src_admfile}.", 19, 1023, "${src_admfile}"));
		return 0;
	}

	# either copied or already there, but check anyway
	return Utils::vxif_fc($trg_admfile, $ctx);
}

#
# Create the response file and copy it to target host if necessary.
#
# Input: 1) the reference to the installation context;
#        2) the pkgadd response file content
#
# Return: 1 if successful; 0 otherwise.
#
sub create_response_file (\%$) {
	my $self = shift;
	my ($ctx,$pkg,$contents) = @_;
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my ($respfile) = "${trg_tmppath}/pkgresp.${trg_upi}.${pkg}.${trg_host}";

	$$ctx{LOGGER}->entering("CPISunOS::create_response_file");
	if ($contents ne "") {
		Utils::vxif_dor("$self->{CMD}{ECHO} '${contents}' >> ${respfile}", "", $ctx);
		if (!Utils::vxif_fc($respfile, $ctx)){
			$$ctx{LOGGER}->error(Utils::_tr("Unable to create ${respfile} on ${trg_host}.", 19, 1007, "${respfile}", "${trg_host}"));
			return 0;
		}
	} else {
		$$ctx{LOGGER}->info(Utils::_tr("Response file not created. Content is null.", 19, 1015));
		return 0;
	}

	return 1;
}

#
# Create the response file.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        2) the reference to the packages hash;
#        3) the package name;
#        4) the patch name;
#
# Return: 1 if successful; 0 otherwise.
#
sub create_and_copy_response_file (\%\%\%$$) {
	my $self = shift;
	my ($ctx,$cfg,$pkgs,$pkg,$patch) = (@_);
	my $trg_host = $ctx->get_target_hostname();
	my $trg_upi = $ctx->get_target_upi();

	$$ctx{LOGGER}->entering("CPISunOS::create_and_copy_response_file");

	# response file is only need for install (upgrade, too, I guess)
	return 1 if (($$ctx{TASK} ne "install") && ($$ctx{TASK} ne "upgrade"));

	my $package_info = $pkgs->{$pkg};
	my $contents = $package_info->search_responsefile_content() || "";

	if ($contents ne "") {
		return $self->create_response_file($ctx, $pkg, $contents);
	} else {
		$$ctx{LOGGER}->info(Utils::_tr("Response file not created. Content is null.", 19, 1015));
	}

	return 1;
}

#
# Check for patchadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the patch installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub patch_installerror (\%) {
	my $self = shift;
	my ($ctx,$patch) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $task_type = $ctx->get_task_type();
	my $trg_host = $ctx->get_target_hostname();
	my $logfile = "${logdir}/${task_type}.${pkg}.${trg_host}";
	my ($error);

	$$ctx{LOGGER}->entering("CPISunOS::patch_installerror");

	# need to fix this. the following statement doesn't seem to be giving the correct result.
	# for now checking for the string successfully installed in the log file.
	#$error = Utils::vxif_dol("$self->{CMD}{CAT} $logfile | $self->{CMD}{GREP} -v '^Checking' | $self->{CMD}{GREP} -v '^Patchadd'");
	#$$ctx{LOGGER}->info("$self->{CMD}{CAT} $logfile | $self->{CMD}{GREP} -v '^Checking' | $self->{CMD}{GREP} -v '^Patchadd'");

	$error = Utils::vxif_dol("$self->{CMD}{CAT} $logfile | $self->{CMD}{GREP}  'successfully installed'");
	if ($error ne "") {
		return 0;
	}
	$$ctx{LOGGER}->info($error);

	return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $task_type = $ctx->get_task_type();
	my $trg_host = $ctx->get_target_hostname();
	my $logfile = "${logdir}/${task_type}.${pkg}.${trg_host}";
	my ($error);

	# log is local, by design (DE 2-10-05, but it's been that way for awhile)
	my $local_cmds = $ctx->get_local_cmds();
	my $local_grep_cmd = $local_cmds->{GREP};
	$error = Utils::vxif_dol("${local_grep_cmd} ERROR: ${logfile}");

	if ($error eq "") {
		return 0;
	}

	$logger->info($error);

	return 1;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;

	return $self->pkg_installerror($ctx, $pkg);
}

#
# Add the native path to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the patch hash;
#        3) the patch to install;
#
# Return: 1 if successful; 0 otherwise.
# 
sub install_native_patch (\%\%$) {
	my $self = shift;
	my ($ctx,$patches,$patch) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $task_type = $ctx->get_task_type();
	my $trg_host = $ctx->get_target_hostname();
	my $logfile = "${logdir}/${task_type}.${patch}.${trg_host}";

	$$ctx{LOGGER}->entering("CPISunOS::install_native_patch");

	Utils::vxif_dor("$self->{CMD}{PATCHADD} $self->{$$ctx{TARGET}{UPI}}{$patch}{TLOC}/$patch 2>${logfile} 1>${logfile}"," ",$ctx);

	return $self->is_patch_install_successful($ctx, $patch, $patches);
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 2 if pkg already installed; 1 if install successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $trg_upi = $ctx->get_target_upi();
	my $pkgadd_dir = $self->{$trg_upi}{$pkg}{TLOC};
	my ($trg_respfile) = "${trg_tmppath}/pkgresp.${trg_upi}.${pkg}.${trg_host}";
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $pkgadd_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
	my $trg_admfile = "${trg_tmppath}/adminfile";

	# check to make sure the given package is not already exist on the target host
	my $result = $self->check_for_existing_package($ctx, $pkg, $pkgs);
	if (0 == $result) {
		return 0;
	} elsif (2 == $result) {
		return 2;
	}

	# make sure the response file exists
	if (Utils::vxif_fc($trg_respfile, $ctx)) {
		$trg_respfile = "-r " . $trg_respfile;
	} else {
		$trg_respfile = "";
	}

	# See incident 343075
	# Native Perl rsh does not (yet) support interaction.
	# Plus, pkgadd refers 1st to the admin/response files.
	# So, force non-interactive to avoid potential hangs.
	my $force_noninteractive = "-n";

	# The package itself may not be at the top of the "pkgadd" directory.
	# It may, rather, be one or more levels down.  Use the EXTRACTION_PATH
	# configuration variable (if present) to see where to dig.
	#
	# maybe put this in common code
	my $installer_info = $ctx->get_installer_info();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $pkginfo = $installer_info->get_platform_package_info($trg_os, $pkg);
	my $extra_path = $pkginfo->search_extraction_path();
	if ($extra_path) {
		$pkgadd_dir .= "/${extra_path}";
	}

	# now add the package
	my $pkgadd_cmd = "$self->{CMD}{PKGADD} ${force_noninteractive} -a ${trg_admfile} -d ${pkgadd_dir} ${trg_respfile} ${pkg}";
	Utils::vxif_dor($pkgadd_cmd, "", $ctx, undef, $pkgadd_logfile);

	# was the package added?
	$result = $self->is_package_install_successful($ctx, $pkg, $pkgs);

	return $result;
}

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my ($trg_admfile) = "${trg_tmppath}/adminfile";
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $pkgrm_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
    
	# now remove the package
	my $pkgrm_cmd = "$self->{CMD}{PKGRM} -n -a ${trg_admfile} ${pkg}";
	Utils::vxif_dor($pkgrm_cmd, "", $ctx, undef, $pkgrm_logfile);
	
	# if the package is successfully uninstalled, the version number null.
	if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg) eq "")) {
		$$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from ${trg_host}", 19, 1019, "${pkg}", "${trg_host}"));
		return 1;
	}
	
	$$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from ${trg_host}", 19, 1020, "${pkg}", "${trg_host}"));

	return 0;
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  # for SunOS, the package name is the native package name
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
  my ($self,$ctx,$pkg,$fod,$pkgs) = @_;

  $$fod = "d";
  return $pkg;
}

1;

