
package SpaceMgr;
use strict;

use constant ERR_PRODUCT_SPACE_CALCULATION => 1;
use constant ERR_REQUIRED_SPACE_UNKNOWN => 2;
use constant ERR_INSUFFICIENT_SPACE => 3;

sub new {
	my $pkg = shift;
	my $this = {};

	bless($this, $pkg);

	$this->init(@_);

	return $this;
}

sub init {
	my $this = shift;
	#
}

#
# Get the free space of a volume/partition.
#
# Input: 1) the reference to the installation context;
#        2) the volume to check;
#
# Return: the spaces available in kilobytes on the given volume.
#
sub get_volumespace {
	my $self = shift;
	my ($ctx,$vol) = @_;
	my (@a,@df,$f);
	my $logger = $ctx->get_logger();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_cmds = $ctx->get_host_os_cmds($trg_host);
	my $dfk_cmd = $trg_cmds->{DFK};

	$$ctx{HOST}{$trg_host}{DFK} = Utils::vxif_dor("${dfk_cmd}", "", $ctx) if (!$$ctx{HOST}{$trg_host}{DFK});
	@df = split(/\n/, $$ctx{HOST}{$trg_host}{DFK});
	foreach $f (@df) {
		@a = split(/\s+/, $f);
		
		$logger->info("return $a[3] if $a[5] equals ${vol}");
		return "$a[3]" if ($a[5] eq $vol);
	}
	#$logger->info(Utils::_tr("No ${vol} volume on ${trg_host}", 75, 1000, "${vol}", "${trg_host}"));
	$logger->info("No ${vol} volume on ${trg_host}");

	return "";
}

#
# Calculate the required spaces for the given product.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the volumes list to check;
#
# Return: a list of total required spaces for the given product.
#
sub calculate_required_spaces_for_product (\%\@) {
	my $self = shift;
	my ($ctx,$chkvols) = @_;
	my ($pkgs,$patches,$index);
	my (@pkgns,@patchns);
	my $pkgsrsp = [];
	my $patchesrsp = [];
	my $rsp = [];
	my $logger = $ctx->get_logger();
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $trg_mode = $ctx->get_host_inst_mode($trg_host);

	$logger->entering("SpaceMgr::calculate_required_spaces_for_product");

	# get the product packages to install
	my $osver = $ctx->get_host_os_version($trg_host);
	my $osarch = $ctx->get_host_os_architecture($trg_host);
        #Here is an idea, how about we send a list of what the product is we want to install? - adding upi_inst
        my $prodver = $upi_inst->get_product_version($ctx, $upi_inst);
	$pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch, $trg_mode);
         

	if ($upi_inst->can("get_product_patches")) {
		$patches = $upi_inst->get_product_patches($ctx, $prodver, $osver, $osarch, $trg_mode);
	}

	@pkgns = keys(%{$pkgs});
	if ($upi_inst->can("get_product_packages_ordering")) {
		@pkgns = $upi_inst->get_product_packages_ordering($ctx, $prodver, $osver, $osarch);
	}
	@pkgns = split(/ /, Utils::vxif_desp(join(" ", @pkgns)));

	@patchns = keys(%{$patches});
	@patchns = split(/ /, Utils::vxif_desp(join(" ", @patchns)));
	
	# calculate the required spaces for product packages
	$pkgsrsp = $self->calculate_required_spaces_for_product_packages($ctx, $pkgs, \@pkgns, $chkvols);
	
	# calculate the required spaces for product patches
	$patchesrsp = $self->calculate_required_spaces_for_product_patches($ctx, $patches, \@patchns, $chkvols);
	
	# sum up the spaces required for both packages and patches
	foreach $index (1..$#$chkvols) {
		$rsp->[$index] = $pkgsrsp->[$index] + $patchns[$index];
	}

	return $rsp;
}

#
# Calculate the required spaces for the given product packages.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the reference to the package names;
#
# Return: a list of total required spaces for the product packages.
#
sub calculate_required_spaces_for_product_packages (\%\%\@\@) {
	my $self = shift;
	my ($ctx,$pkgs,$pkgns,$chkvols) = @_;
	my $logger = $ctx->get_logger();
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $os_inst = $ctx->get_host_os_installer($trg_host);
	my $pkgsrsp = [];

	$logger->entering("SpaceMgr::calculate_required_spaces_for_product_packages");

	for my $pkg (@$pkgns) {
		my $package_info = $pkgs->{$pkg};
		my $space_aref = $package_info->search_space();

		if (defined($space_aref)) {
			my $num_vols = scalar(@$chkvols);
			for (my $n = 1; $n < $num_vols; $n++) {
				# @{$PKGI{$PKG}{SPACE}} does not define /tmp 
				# space so values are off by one
				$pkgsrsp->[$n] += $space_aref->[$n-1];
				my $space = $space_aref->[$n-1];
				my $volname = $$chkvols[$n];
				#$logger->fine(Utils::_tr("${pkg} requires ${space} kbs in ${volname}.", 75, 1001, "${pkg}", "${space}", "${volname}"));
				$logger->fine("${pkg} requires ${space} kbs in ${volname}.");
			}
		} else {
			#$logger->warning(Utils::_tr("SPACE requirements for ${pkg} not defined.", 75, 1002, "${pkg}"));
			$logger->warning("SPACE requirements for ${pkg} not defined.");
		}

		# calculate the space from media
		if (!defined($os_inst->{$trg_upi}{$pkg}{SPACE})) {
			$os_inst->check_package_on_media($ctx, $pkg, "", $pkgs);
		}
	}

	return $pkgsrsp;
}

#
# Calculate the required spaces for the given product patches.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the patches hash;
#        3) the reference to the patch names;
#        4) the reference to the volumes list to check;
#
# Return: a list of total required spaces for the product packages.
#
sub calculate_required_spaces_for_product_patches (\%\%\@\@) {
	my $self = shift;
	my ($ctx,$patches,$patchns,$chkvols) = @_;
	my $logger = $ctx->get_logger();
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $os_inst = $ctx->get_host_os_installer($trg_host);
	my ($patch,$n,$space,$volname);
	my $patchesrsp = [];

	$logger->entering("SpaceMgr::calculate_required_spaces_for_product_patches");

	foreach $patch (@$patchns) {
		if (defined($$patches{$patch}{SPACE})) {
			my $num_chkvols = scalar(@$chkvols);
			#foreach $n (1..$#$chkvols) {
			for (my $n = 1; $n < $num_chkvols; $n++) {
				# @{$PATCHES{$PATCH}{SPACE}} does not define /tmp 
				# space so values are off by one
				$patchesrsp->[$n] += $$patches{$patch}{SPACE}[$n-1];
				$space = $$patches{$patch}{SPACE}[$n-1];
				$volname = $$chkvols[$n];
				#$logger->fine(Utils::_tr("${patch} requires ${space} kbs in ${volname}", 75, 1003, "${patch}", "${space}", "${volname}"));
				$logger->fine("${patch} requires ${space} kbs in ${volname}");
			}
		} else {
			#$logger->warning(Utils::_tr("SPACE requirements for ${patch} not defined.", 75, 1002, "${patch}"));
			$logger->warning("SPACE requirements for ${patch} not defined.");
		}

		# calculate the space from media
		if (!defined($os_inst->{$trg_upi}{$patch}{SPACE})) {
			$os_inst->check_package_on_media($ctx, "", $patch, $patchns);
		}
	}

	return $patchesrsp;
}

#
# Check to make target host have the required spaces.
#
# Input: 1) the reference to the installation context;
#        2) an optional array of "low-space" warnings;
#
# Return: 0 if successful, error object if not.
#
sub check_for_required_spaces (\%;\@) {
	my $self = shift;
	my ($ctx, $warnings) = @_;
	my $rsp = [1000, 1000, 1000, 2000, 1000];
	my $chkvols = ["/tmp", "/opt", "/usr", "/var", "/"];
	my $ncv = $#$chkvols;
	my $ifmgr = $ctx->get_interface_manager();
	my $logger = $ctx->get_logger();
	my $error = undef;

	$logger->entering("SpaceMgr::check_for_required_spaces");

	eval {
		my $dc = [];
		my $extract_ix = 3;
		my $trg_host = $ctx->get_target_hostname();
		my $trg_spacevol = $ctx->get_host_spacevol($trg_host);

		if ($trg_spacevol) {
			$chkvols->[$extract_ix] = ${trg_spacevol};
		}

		# calculate the spaces needed for the product
		my $prodrsp = $self->calculate_required_spaces_for_product($ctx, $chkvols);
		#vikas 
		if ($$ctx{FATALERROR}) {
			my $err_msg = $$ctx{FATALERROR};
			$$ctx{FATALERROR}="";
			$ifmgr->throw(ERR_PRODUCT_SPACE_CALCULATION, $err_msg);
		}

		foreach my $n (1..$ncv) {
			$rsp->[$n] += $prodrsp->[$n];
		}

		my $os_inst = $ctx->get_host_os_installer($trg_host);
		my $trg_upi = $ctx->get_target_upi();

		if ($os_inst->{$trg_upi}{MAXPKGSPACE}) {
			my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
			my @a=split(/\//, $trg_tmppath);
			my $itv = Utils::vxif_list("/$a[1]",@{$chkvols});
			$rsp->[$itv] += $os_inst->{$trg_upi}{MAXPKGSPACE};
			my $max_space = $os_inst->{$trg_upi}{MAXPKGSPACE};
			my $volname = $$chkvols[$itv];
			my $pdfr = $os_inst->{MSG}{PDFR};
			#$logger->info(Utils::_tr("Raising ${volname} space by ${max_space} for largest copied ${pdfr}", 75, 1004, "${volname}", "${max_space}", "${pdfr}"));
			$logger->info("Raising ${volname} space by ${max_space} for largest copied ${pdfr}");
			$logger->info("${volname} + ${max_space} = $rsp->[$itv]");
		}

		my $ct = defined($os_inst->{$trg_upi}{MAXPKGSPACE}) ? 0 : 1;
		foreach my $n ($ct..$ncv) {
			next if (!$rsp->[$n]);
			my $vol=$$chkvols[$n];
			my $rspn = $rsp->[$n]; # for @#*&$ localization stuff

			#$logger->fine(Utils::_tr("Getting the spaces for ${vol}.", 75, 1005, "${vol}"));
			$logger->fine("Getting the spaces for ${vol}.");
			my $rvs = $self->get_volumespace($ctx, $vol);

			if ($rvs eq "") {
				if ($vol eq '/') {
					my $err_msg = $os_inst->{MSG}{ERROR};
					#$logger->error(Utils::_tr("Cannot determine space available in root (/) filesystem on ${trg_host}", 75, 1006, "${trg_host}"));
					$logger->error("Cannot determine space available in root (/) filesystem on ${trg_host}");
					$ifmgr->throw(ERR_REQUIRED_SPACE_UNKNOWN, $err_msg);
					return 0;
				}
				$rsp->[$ncv] += $rspn;
				my $rspncv = $rsp->[$ncv];
				#$logger->info(Utils::_tr("adding ${rspn} KB required to / = ${rspncv}", 75, 1007, "${rspn}", "${rspncv}"));
				$logger->info("adding ${rspn} KB required to / = ${rspncv}");
			} elsif ($rvs < $rspn) {
				#my $err_msg = Utils::_tr("${trg_upi} installation requires ${rspn} KB in the $vol volume and only $rvs KB is available on ${trg_host}", 75, 1008, "${trg_upi}", "${rspn}", "$vol", "$rvs", "${trg_host}");
				my $err_msg = "${trg_upi} installation requires ${rspn} KB in the $vol volume and only $rvs KB is available on ${trg_host}";
				$logger->error($err_msg);
				if ($n >= $extract_ix) {
					# either /var substitute or / itself (i.e. volume does not exist, so we are on the / volume)
					my $msg;
					if ($trg_spacevol) {
						#$msg = Utils::_tr("Consider revising your -hostspace ${trg_host}:${trg_spacevol} to -hostspace ${trg_host}:/<larger-volume>.", 75, 1009, "${trg_host}", "${trg_spacevol}", "${trg_host}");
						$msg = "Consider revising your -hostspace ${trg_host}:${trg_spacevol} to -hostspace ${trg_host}:/<larger-volume>.";
					} else {
						$msg = "Try including -hostspace ${trg_host}:/<large-volume> in your installer command line.";
					}
					$logger->info("${msg}");
				}
				$ifmgr->throw(ERR_INSUFFICIENT_SPACE, $err_msg);
				return 0;
			} else {
				#$logger->info(Utils::_tr("${vol} on ${trg_host} requires ${rspn} KB and has $rvs KB available", 75, 1011, "${vol}", "${trg_host}", "${rspn}", "$rvs"));
				$logger->info("${vol} on ${trg_host} requires ${rspn} KB and has $rvs KB available");
				my $kr = $rvs - $rspn; 
				if ($kr < $os_inst->{SPACEWARNING}) {
					push(@$dc,"${vol},${kr}"); 
				}
			}
		}

		if ($warnings) {
			push(@$warnings, @$dc);
		}
	};
	$error = $ifmgr->catch();

	return $error;
}

1;

