package Response;

use File::Basename;
use Text::ParseWords;
use VxIF::Utils;

#
# Read response file.
#
# Input: 1) The name of the file.
#
# Return: A hash of sections, each of which
#         is a hash of name/value pairs.
#
sub read_response_file {
	my ($resp_fnam) = @_;
	my $response = {
		values => {},
		sections => {
		},
	};
	my $currsect_name = "";
	my $hash_symbol = '#';

	my $rc = open(RESP, $resp_fnam);

	if ($rc) {
		while (my $rec = <RESP>) {
			# Well, chomp won't do this, and chop is
			# dangerous, especially between platforms.
			# And we *will* be across multiple platforms.
			# So, do the following chomp-like variation.
			$rec =~ s/[\r\n]+$//;

			if ($rec =~ /^\s*$/) {
				# spaces only
				next;
			} elsif ($rec =~ /^\s*${hash_symbol}/) {
				# a comment
				next;
			#} elsif ($rec =~ /^\s*([\.\w]+)\s*=\s*(.*\S)?\s*$/) {
			} elsif ($rec =~ /^\s*(.*\S)\s*=\s*(.*\S)?\s*$/) {
				# a key/value pair assignment
				my $key = $1;
				my $val = $2;
				if (!defined($val)) {
					$val = "";
				}
				my $asgn_href = undef;

				# create the entry in appropriate place
				if ($currsect_name) {
					my $currsect = $response->{sections}{$currsect_name};
					$asgn_href = $currsect;
				} else {
					$asgn_href = $response->{values};
				}
				#my @levels = split(/\./, $key);
				my @levels = Text::ParseWords::parse_line('\.', 0, $key);
				my $num_levels = scalar(@levels);
				my $h = $asgn_href;
				for (my $i = 0; $i < $num_levels; $i++) {
					my $sub_key = $levels[$i];
					if ($i < ($num_levels - 1)) {
						if (!exists($h->{$sub_key})) {
							$h->{$sub_key} = {};
						}
						$h = $h->{$sub_key};
					} else {
						$h->{$sub_key} = $val;
					}
				}
			} elsif ($rec =~ /^\s*\[\s*(.*)\s*\]\s*$/) {
				# a section
				$currsect_name = $1;
				if (!exists($response->{sections}{$currsect_name})) {
					$response->{sections}{$currsect_name} = {};
				}
			} else {
				# nothing we know about
				next;
			}
		}
		close(RESP);
	} else {
		die Utils::_tr("Cannot read response file ${resp_fnam}.", 37, 1000, "${resp_fnam}");
	}

	return $response;
}

#
# Dump the scalar value.
#
# Input: 1) the scalar value;
#
# Return: the scalar value itself.
#
sub print_scalar {
	my ($val) = @_;

	return "$val";
};

#
# Dump the list value.
#
# Input: 1) the list value;
#
# #Return: the list as a string, separated by commas.# NOT
# Return: the list as a string, separated by single blanks.
#
sub print_list {
	my (@l) = @_;
	my ($q,$val,$rv);

	#$rv = join(", ", @l);
	$rv = join(" ", @l);

	return $rv;
}

#
# Recursively serialized the hash array..
#
# Input: 1) the reference to the hash array;
#        2) header;
#        3) end of the file indicator;
#
# Return: the hash content in Perl format.
#
sub print_hash {
	my ($hr,$var,$en,$progname,$hck) = @_;
	my ($cv,$file,$key,$kq,$hash,$ref);
	if (!$hck) {
		$hck = [];
	}
	
	foreach $key (sort keys(%$hr)) {
		my $kq = ($key=~/\W/) ? "\"" : "";
		my $eff_key = "$kq$key$kq";
		push(@$hck, $eff_key);
		$ref=ref($$hr{$key});
		if (!$ref) {
			$hash .= join(".", @$hck);
			$hash .= (' = ' . print_scalar($$hr{$key}) . "\n");
		} elsif ($ref eq "ARRAY") {
			$hash .= join (".",@$hck);
			$hash .= (' = ' . print_list(@{$$hr{$key}}) . "\n");
		} elsif ($ref eq "HASH") {
			#$hash .= "$key\." . print_hash($$hr{$key}, "$var\{$key}", undef,undef, $hck);
			$hash .= print_hash($$hr{$key}, "$var\{$key}", undef,undef, $hck);
		} else {
			$hash.="#$var\{$key}: $ref is not a supported reference\n";
		}
		pop(@$hck);
	}
	if ($en) {
		$cv=Utils::_tr("configuration values:", 37, 1001);
		$hash = "#\n# ${progname} $cv\n#\n\n[config]\n\n$hash\n";
		#$hash =~s/@/\\@/g;
	}

	# just a newline at the end
	$hash =~ s/[\n\r]+$/\n/;
	
	return "$hash";
}

#
# Dump some of the required installation context properties to the responsefile.
#
# Input: 1) the reference to the installation context;
#
# Return: the required installation context hash in Perl format.
#
sub dump_installation_context ($) {
	my ($ctx) = @_;
	my ($hash, $cnt);
	my $trg_upi = $ctx->get_target_upi();

	$hash = "#\n# Installation Context\n#\n\n";
	$hash .= '[context]'."\n\n";
	$hash .= 'TASK = ' . print_scalar($$ctx{TASK}) . "\n";
	$hash .= 'PROGRAM = ' . print_scalar($$ctx{PROGRAM}) . "\n";
	$hash .= 'PRODUCT = ' . print_scalar($$ctx{PRODUCT}) . "\n";
	$hash .= 'USESSH = ' . print_scalar($$ctx{USESSH}) . "\n";
	$hash .= 'SYSTEMS = ' . print_list(@{$$ctx{SYSTEMS}}) . "\n";
	$hash .= 'NOCONFIG = ' . print_scalar($$ctx{NOCONFIG}) . "\n";

	$hash .= 'UPI = ' . print_scalar($trg_upi) . "\n";
	$hash .= 'HOST = ' . print_scalar($trg_upi) . "\n"; # ???

	if ($ctx->{HOSTSPACE}) {
		my $spec = "";
		my $spec_aref = [];
		my $hostspace = $ctx->{HOSTSPACE};
		for my $host (keys(%$hostspace)) {
			my $vol = $hostspace->{$host};
			push(@$spec_aref, "${host}:${vol}");
		}
		$spec = join(",", @$spec_aref);
		$hash .= 'HOSTSPACE = ' . $spec . "\n";
	}

	foreach $cnt (keys (%{$$ctx{DEPCHK}})) {
		$hash .= 'DEPCHK.'.$cnt.' = ' . print_scalar($$ctx{DEPCHK}{$cnt}) . "\n";	
	}

	$hash .= "\n###############################################\n\n";
}

#
# Dump the configuration hash to the responsefile.
#
# Input: 1) the reference to the configuration hash;
#
# Return: the content of the configuration hash in Perl format.
#
sub dump_configuration_hash ($$) {
	my ($cfg,$progname) = @_;
	my ($hash);
	
	$hash .= print_hash($cfg, "config", 1, $progname);
	
	return $hash;
}

sub compute_response_file_name {
	my ($ctx) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $response_basename = "$$ctx{PROGRAM}$$ctx{TIME}.response";
	# response file goes in same directory as log file
	my $responsefile = "${logdir}/${response_basename}";

	return $responsefile;
}

#
# Create a file which holds all configuration information of the install.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
sub write_response_file ($$) {
	my ($ctx,$cfg) = @_;
	my $logger = $ctx->get_logger();
	my $rc = 1; # benefit of doubt
	# need to write file if none to begin with, or upgrading from old format.
#	my $need_to_write_it =
#		(
#		 defined($$ctx{RESPONSEFILE}) && !defined($$ctx{OLDRESPONSEFILE})
#		 ? 0
#		 : 1
#		 );

	# let's always write it.  let's see how that works. (DE 6-13-05)
	$need_to_write_it = 1;
	if ($need_to_write_it) {
		my $responsefile = compute_response_file_name($ctx);

		my $respcontent = "";
		$respcontent .= dump_installation_context($ctx);
		$respcontent .= dump_configuration_hash($cfg, $$ctx{PROGRAM});

		$logger->fine("responsefile = ${responsefile}");
		$logger->finer("\n--- response file content ----\n\n${respcontent}\n-----------------------\n");

		my $error_handler = sub {
			$rc = 0;
		};

		Utils::vxif_writef($respcontent, $responsefile, 1, $error_handler);
	}

	return $rc;
}

1;

