package ProductHelper;
use strict;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;

	my $product_info = $self->get_product_info();
	if (!$product_info) {
		# if not passed to constructor,
		# explicity start out with none
		$self->set_product_info(undef);
	}
}

sub get_product_info {
	my $self = shift;
	my $product_info = $self->{product_info};

	return $product_info;
}

sub set_product_info {
	my $self = shift;
	my ($product_info) = @_;

	$self->{product_info} = $product_info;
}

sub assure_config_params_initialized {
	my $self = shift;
	my ($ctx) = @_;
	my $cfg = $ctx->get_config();

	# override in derived class, if desired.
}

sub obtain_product_info {
	my $self = shift;
	my ($ctx, $default_loc) = @_;
	my $product_info = $self->get_product_info();

	$self->assure_config_params_initialized($ctx);

	# Essentially a read accessor, but inputs from file the 1st time.
	if (!$product_info) {
		my $root_installer = $ctx->get_root_installer();
		my $task = $root_installer->{INTERFACE_TASK};

		my $product_path = undef;

		if ($task) {
			my $instinfo = $task->get_installer_info();
			my $trg_host = $ctx->get_target_hostname();
			my $trg_os = $ctx->get_host_os_name($trg_host);
			$product_info = $instinfo->get_platform_product_info($trg_os);
		}

		if (!$product_info) {
			if (!$product_path) {
				# couldn't find it ... revert to hack
				$product_path = $default_loc;
			}

			my $eff_fnam = $product_path;

			my $ifmgr = $ctx->get_interface_manager();
			my $class_factory = $ifmgr->get_class_factory();
			$product_info = $class_factory->create_product_info();
			$product_info->from_config_file($eff_fnam);
		}

		$self->set_product_info($product_info);
	}

	return $product_info;
}

#
# Get the localized product title for this product. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#
# Return: the complete localized title for this product.
#
sub get_product_title (\%;$) {
	my $self = shift;
	my ($ctx) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_product_title");
	
	my $product_info = $self->obtain_product_info($ctx);
	 #first see if it's in product info
	my $title = $product_info->{TITLE};
	if (!$title) {
		# nope.  see if it's in installer info.
		my $instinfo = $product_info->get_super_instinfo();
		if ($instinfo) {
			$title = $instinfo->get_title();
		}
	}
	if (!$title) {
		my $main_pkg = $self->get_product_main_package($ctx);
		my $ppkgs = $self->get_product_packages($ctx);
		my $pkginfo = $ppkgs->{$main_pkg};
		$title = $pkginfo->search_name($main_pkg);
	}
	if (!$title) {
		my $main_pkg = $self->get_product_main_package($ctx);
		my $version = $self->get_product_version($ctx, $main_pkg);
		$title = "Installer for ${version}";
	}

	return $title;
}

#
# Get the product version number (i.e. 4.1.3.10). For some products, the product version
# number may be different from the main package version number. The main package version 
# number is optional and will be absent for the current product version. If present, 
# the main package version will be different the current main package version. 
# The reason is that for product upgrade and product downgrade, VxIF needs a list of 
# packages associated with a particular product version so they can properly removed.
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the main package version; this parameter is optional;
#
# Return: product version number
#
sub get_product_version (\%;$) {
	my $self = shift;
	my ($ctx, $pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_product_version");
	
	my $product_info = $self->obtain_product_info($ctx);
	my $version = $product_info->{VERSION};
	if (!$version) {
		$version = "DEVELOPMENT_VERSION";
#jpk
	}
	$$ctx{LOGGER}->exiting("ProductHelper::get_product_version");

	return $version;
}

#
# Get the localized product description for this product. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: product description.
#
sub get_product_description (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_product_description");

	my $product_info = $self->obtain_product_info($ctx);
	my $dscr = $product_info->{DESCRIPTION};
	if (!$dscr) {
		my $main_pkg = $self->get_product_main_package($ctx);
		my $ppkgs = $self->get_product_packages($ctx);
		my $pkginfo = $ppkgs->{$main_pkg};
		$dscr = $pkginfo->{OPTIONAL};
	}

	return $dscr;
}

#
# Determine whether the destination host OS is supported.
#
# Input: 1) the reference to the installation context;
#        2) host name;
#        3) host OS;
#        4) host OS version number;
#        5) host OS architecture;
#
# Return: 1 if the given OS is supported. Otherwise, return 0;
#
sub is_os_supported (\%$$$$) {
	my $self = shift;
	my ($ctx,$host,$os,$version,$arch) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::is_os_supported");

	my $product_info = $self->obtain_product_info($ctx);

	my $rc = 0;
	my $os_name = $product_info->{OS_NAME};
	if (defined($os_name) && $os eq $os_name) {
		# os name must be in product info
		my $os_versions_href = $product_info->{OS_VERSIONS};
		if ($os_versions_href) {
			# there is a list of supported versions
			if (exists($os_versions_href->{$version})) {
				# it's in the list
				$rc = 1;
			}
		} else {
			# No list of supported versions.
			# Assume all versions supported.
			$rc = 1;
		}
	}

	return $rc;
}

#
# Get the name of the main package. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the selected installation mode if one is available;
#
# Return: the name of the main package.
# 
sub get_product_main_package (\%$) {
	my $self = shift;
	my ($ctx,$mode) = @_;
	my $this_package = __PACKAGE__;
  
	$$ctx{LOGGER}->entering("${this_package}::get_product_main_package");
  
	my $product_info = $self->obtain_product_info($ctx);
	my $main_package = undef;
	my @pkg_order_a = $self->get_product_packages_ordering($ctx, undef, undef,undef, $mode);
	my $pkg_order = \@pkg_order_a;
	if ($pkg_order) {
		# had best get here ... ordering should really return a reference
		my $num_pkgs = scalar(@$pkg_order);

		if ($ctx->{TASK} eq "uninstall") {
			# gosh, but I dislike "if's".
			$main_package = $pkg_order->[0];
		} else {
			$main_package = $pkg_order->[$num_pkgs - 1];
		}
	}

	return $main_package;
}

#
# Get the order in which the product packages will be installed. This interface
# is used for backward compatibility support only. Some old product scripts
# may not set the {DEPS} correctly. Basically, $PROD{<UPI>}{PKGS} instructs the
# installer the order in which to install the product packages.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the list of product packages in the order in which to be installed.
#
sub get_product_packages_ordering (\%$$$;$) {
	my $self = shift;
	my ($ctx,$pdver,$osver,$osarch,$mode) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_product_packages_ordering");
  
	my $cfg = $ctx->get_config();
	my $product_info = $self->obtain_product_info($ctx);
	my $dictionary = $product_info->get_dictionary();
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_mode = $ctx->get_host_inst_mode($trg_host);
	my $pkg_list = undef;
	if ($trg_mode) {
		$pkg_list = $dictionary->expand($trg_mode);
	} else {
		$pkg_list = $dictionary->expand($trg_upi);
	}
	if (!$pkg_list) {
		my $ifmgr = $ctx->get_interface_manager();
		my $localizer = $ifmgr->get_localizer();
		my $fq_fnam = $product_info->get_fq_fnam();
		my $task_type = $ctx->{TASK};
		my $msg = "${fq_fnam} contains no packages to {$task_type}!\n";
		my $lmsg = $localizer->translate($msg);
		die "$lmsg";
	}
	my $eff_pkg_list = undef;

	if ($ctx->{TASK} eq "uninstall") {
		$eff_pkg_list = [reverse @$pkg_list]
	} else {
		$eff_pkg_list = $pkg_list;
	}

	return @$eff_pkg_list;
}

#
# Return a list of required packages for a given version of this product. The given 
# version may not be the current version. The reason is that for product upgrade 
# and product downgrade, VxIF needs a list of packages associated with a particular 
# version so they can properly removed.
#
# This interface is mandatory for package install. This interface is optional 
# for patch-only install.
#
# This interface shall return the reference of a hash with package names as the keys
# with the following properties.
#
# {NAME}    	- the package name; This property is mandatory;
# {VERSION} 	- the package version; This property is mandatory;
# {SPACE}	- a list which specified the spaces, in kilobytes, needed for this
#                 package in /opt, /usr, /var, and /; this property is mandatory;
# {REQREBOOT}	- require system reboot after adding this package? This property is
#                 optional;
# {RELOC}	- is this package relocatable; This property is optional;
# {DEPS}	- a list packages, within this product, that this package depended on;
#                 If a dependent package is not part of this product, it must be
#                 included in get_external_dependents(); This property is optional;
# {VRTSPATCHES}	- a list of patches to be installed with this package; This property is
#                 optional;
# {OPTIONAL}	- localized package description; This property is optional;
#
# For example,
#
# $pkgs{VRTSfoo}{NAME} = "VERITAS Foo Package";
# $pkgs{VRTSfoo}{VERSION} = "1.2.3.4";
# $pkgs{VRTSfoo}{SPACE} = [100, 200, 100, 10];
# $pkgs{VRTSfoo}{DEPS} = ["VRTSbar"];
# $pkgs{VRTSfoo}{DESC} = "This is the foo package of product foobar.";
# $pkgs{VRTSfoo}{VRTSPATCHES} = ["123456", "434200"];
#
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the reference of the hash which contains the packages to be installed.
#
sub get_product_packages (\%$$$;$) {
	my $self = shift;
	my ($ctx,$prodver,$osver,$osarch,$mode) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_product_packages");

	my $product_info = $self->obtain_product_info($ctx);
	my $packages = $product_info->get_packages();

	return $packages;
}

#
# For a given package, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: Return the directory path where the given package is located.
#
sub get_package_location (\%) {
	my $self = shift;
	my ($ctx, $pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_package_location");

	my $product_info = $self->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $package_location = $package_info->search_media_location($pkg);

	return $package_location;
}

#
# Indicate whether a given package is in a compressed file. 
#
# This interface is optional. By default, VxIF assumes a given package is not compressed.
#
# Input: 1) the reference to the installation context;
#        2) package name; 
#
# Return: 0 if the given package is not compressed;
#         1 if the given package is compressed using gunzip (.tar.gz);
#         2 if the given package is compressed in other format; If the package
#           is compressed in other format, product must implement get_package_decompressor() 
#           method to provide a custom decompressor;
#
sub get_package_compression_type (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_package_compression_type");

	my $product_info = $self->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $compression_type = $package_info->search_compressed($pkg);

	return $compression_type;
}

#
# Return the compressed filename, without the path, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed filename, without the path, for the given package.
#
sub get_package_compressed_filename (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_package_compressed_filename");

	my $product_info = $self->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $compressed_file = $package_info->search_compressed_file($pkg);

	return $compressed_file;
}

#
# Return the compressed file size, in kilobytes, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed file size, in kilobytes, for the given package.
#
sub get_package_compressed_filesize (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_package_compressed_filesize");

	my $package_location = $self->get_package_location($ctx, $pkg);
	my $compressed_filename = $self->get_package_compressed_filename($ctx, $pkg);
	my $fq_fnam = "${package_location}/${compressed_filename}";
	my $size_in_k = Utils::vxif_filesize_in_k($fq_fnam);

	return $size_in_k;
}

sub fill_in_dynamic_content {
	my $self = shift;
	my ($ctx, $content) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $trg_host_os_version = $ctx->get_host_os_version($trg_host);
	my $trg_host_os_architecture = $ctx->get_host_os_architecture($trg_host);
	my $dynamic_info = {
		'<<TARGET_OS_NAME>>' => $trg_os,
		'<<TARGET_HOSTNAME>>' => $trg_host,
		'<<TARGET_VERSION>>' => $trg_host_os_version,
		'<<TARGET_ARCHITECTURE>>' => $trg_host_os_architecture,
	};
	my $pattern = join('|', keys(%{$dynamic_info}));
	my $eff_content = $content;

	$eff_content =~ s/($pattern)/$dynamic_info->{$1}/g;

	return $eff_content;
}

#
# Get the content of the response file for a given package or patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        3) the package to be installed;
#        4) the name of the host for which this package will be installed;
#        5) host OS version;
#        6) host OS architecture;
#
# Return: the content of the response file for the given package or patch ID.
#
sub get_responsefile_content (\%\%$$$$) {
	my $self = shift;
	my ($ctx,$cfg,$pkg,$host,$osver,$osarch) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::get_responsefile_content");

	my $product_info = $self->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $content = $package_info->search_responsefile_content($pkg);
	my $eff_content = undef;

	if ($content) {
		$eff_content = $self->fill_in_dynamic_content($ctx, $content);
	} else {
		$eff_content = $content;
	}

	$$ctx{LOGGER}->info("Response file content for ${pkg}: \n${eff_content}\n");

	return $eff_content;
}

sub package_base_installer {
	my $self = shift;
	my ($ctx, $pkg) = @_;
	my $this_package = __PACKAGE__;

	$$ctx{LOGGER}->entering("${this_package}::package_base_installer");

	my $product_info = $self->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $base_installer = $package_info->search_source_product_info($pkg);

	return $base_installer;
}

1;

