package PerlUnixCmds;
use strict;

use File::Spec;
use File::Copy;
use Getopt::Long;
use POSIX ();
use VxIF::NativePerl::Net::Rsh();
use VxIF::NativePerl::Net::Rcp();

BEGIN {Getopt::Long::Configure("bundling");}

# dispatch table for native perl unix commands.
my $cmd_dispatch = {
	cp    => \&PerlUnixCmds::cp,
	domainname => \&PerlUnixCmds::domainname,
	grep  => \&PerlUnixCmds::grep,
	mkdir => \&PerlUnixCmds::mkdir,
	touch => \&PerlUnixCmds::touch,
	uname => \&PerlUnixCmds::uname,
	rm    => \&PerlUnixCmds::rm,
	rsh   => \&PerlUnixCmds::rsh,
	rcp   => \&PerlUnixCmds::rcp,
	ssh   => \&PerlUnixCmds::ssh,
	scp   => \&PerlUnixCmds::scp,
};

########################################
# Essentially a dispatch table lookup.
########################################
sub get_cmd_proc {
	my ($cmd) = @_;
	my $cmd_proc = $cmd_dispatch->{$cmd};

	return $cmd_proc;
}

########################################
# By default, print to stdout.
########################################
my $default_output_proc = sub {
	# if no filter proc is passed in,
	# print chomped record to stdout.
	my ($chomped_output) = @_;
	print(STDOUT "$chomped_output\n");
};

########################################
# Essentially a wrapper for Getopt.
# Push local ARGV (Getopt uses that).
# Invoke and populate/modify things
########################################
sub get_args ($$) {
	my ($argv, $opts) = @_;
	local @ARGV = @$argv;
	my $go_rc = Getopt::Long::GetOptions(%$opts);

	if ($go_rc) {
		# changed ARGV, reflect (back) in argv.
		@$argv = @ARGV;
	}

	return $go_rc;
}

########################################
# No flags currently supported.
########################################
sub domainname ($;$) {
	my ($argv, $proc) = @_;
	my $hostname = undef;
	my $uname_args = ["-n"];
	my $uname_proc = sub{
		my ($h) = @_;
		$hostname = $h;
	};
	my $rc = uname($uname_args, $uname_proc);

	if (!$rc) {
		# rc == 0 ==> uname successful.  assume hostname set.
		my @x=gethostbyname($hostname);
		my $y=$x[0];
		$y =~ s/^([^\.])*\.//;
		my $domain = $y;

		if ($proc) {
			$proc->($domain);
		} else {
			$default_output_proc->($domain);
		}
	}

	return $rc;
}

########################################
# like in unix, return codes mean:
# 0: found at least one match
# 1: found no match
# 2: failure
########################################
sub grep ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 0;
	# set up hash to contain arg values
	my $grep_info = {
		l => 0,
		v => 0,
	};
	my $flist = [];
	my $pattern = undef;
	my $num_cant_open = 0;
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			l => \$grep_info->{l},
			v => \$grep_info->{v},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			# have specified flags.  parse remaining args.
			$pattern = shift(@$argv);
			if ($pattern) {
				my $numargs = scalar(@$argv);
				for (my $i = 0; $i < $numargs; $i++) {
					my $arg = $argv->[$i];

					# any args are file specs (or filespec)
					my $fspec = $arg;
					push(@$flist, glob($fspec));
				}

				if (scalar(@$flist) > 0) {
					# there was a pattern and at least one file
					$pa_rc = 1;
				}
			}
		}

		return $pa_rc;
	};
	my $run_filter = sub {
		my $eff_proc = ($proc ? $proc : $default_output_proc);
		my $numfiles = scalar(@$flist);
		for (my $i = 0; $i < $numfiles; $i++) {
			my $fnam = $flist->[$i];
			if (open(INPT, "$fnam")) {
				while (my $rec = <INPT>) {
					chomp $rec;

					if ($rec =~ /$pattern/) {
						# have a match
						if (!($grep_info->{v})) {
							# -v was not specified
							if ($grep_info->{l}) {
								$eff_proc->($fnam);
								last; # done with file
							} else {
								my $eff_rec =
									(
									 $numfiles > 1
									 ? "$fnam:$rec"
									 : $rec
									 );
								$eff_proc->($eff_rec);
							}
						}
					} else {
						# not a match
						if ($grep_info->{v}) {
							# -v was specified
							if ($grep_info->{l}) {
								$eff_proc->($fnam);
								last; # done with file
							} else {
								my $eff_rec =
									(
									 $numfiles > 1
									 ? "$fnam:$rec"
									 : $rec
									 );
								$eff_proc->($eff_rec);
							}
						}
					}
				}
				close(INPT);
			} else {
				$num_cant_open++;
			}
		}
	};

	# process command line args
	my $args_ok = $parse_args->();

	if ($args_ok) {
		# process files
		$run_filter->();

		# grep sets rc to 1, even if only one of many
		# files could not open (according to manpage,
		# but that is a bit vague ... probably close
		# enough here).
		if ($num_cant_open > 0 && $rc == 0) {
			$rc = 1;
		}
	} else {
		# invalid args (no pattern, or no file(s))
		$rc = 2;
	}

	return $rc;
}

########################################
# Like in unix, return codes mean:
# 0: successful directory creation
# 1: unsuccessful directory creation
# Supported flags:
# -m <mode>
#    defulat mode == 0777
# -p
#
# Returns:
#  1: success
#  0: otherwise
#
########################################
sub mkdir ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 0;
	# set up hash to contain arg values
	my $mkdir_info = {
		m => 0777,
		p => 0,
	};
	my $dirlist = [];
	# specified flags already obtained. get dirs.
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up and invoke utility method to populate args
		# (and depopulate argv).
		#
		# 1st set up hash pointing getopts to args.
		my $opts = {
			'm=s' => \$mkdir_info->{m},
			p     => \$mkdir_info->{p},
		};
		# invoke utility method.
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			# have specified flags.  parse remaining args.
			my $numargs = scalar(@$argv);
			for (my $i = 0; $i < $numargs; $i++) {
				my $arg = $argv->[$i];

				# for now, if it start with a dash, then it's a flag.
				if ($arg =~ /^-(\w+)/) {
					# an unspecified flag. ignore
				} else {
					# dirname
					my $dirname = $arg;
					push(@$dirlist, $dirname);
				}
			}

			if (scalar(@$dirlist) > 0) {
				# if at least one directory specified.
				# good enough for arg parsing.
				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};
	my $create_dirs = sub {
		my $ret_rc = 0;
		my $eff_perm = $mkdir_info->{m};

		my $numdirs = scalar(@$dirlist);
		for (my $i = 0; $i < $numdirs; $i++) {
			my $dir = $dirlist->[$i];
			if ($mkdir_info->{p}) {
				my $dir_elts = [];
				my $accum_dir = "";
				my $device_specd = "";
				my $is_absolute = File::Spec->file_name_is_absolute($dir);
				@$dir_elts = File::Spec->splitdir($dir);
				if ($is_absolute) {
					# if starts with '/', then shift empty string.
					# othewise will get device (c:, foo: ???).
					$device_specd = shift(@$dir_elts);
				}
				my $numelts = scalar(@$dir_elts);
				for (my $i = 0; $i < $numelts; $i++) {
					my $dir_elt = $dir_elts->[$i];
					# build up to full directory
					if ($accum_dir) {
						$accum_dir .= "/$dir_elt";
					} else {
						if ($is_absolute) {
							$accum_dir = "$device_specd/$dir_elt";
						} else {
							$accum_dir = $dir_elt;
						}
					}
					if ( ! -d $accum_dir ) {
						$rc = mkdir($accum_dir, $eff_perm);
						if (!$rc) {
							# first sign of trouble, bail
							last;
						}
					}
				}
			} else {
				$rc = mkdir($dir, $eff_perm);
			}

			if (!$rc) {
				# mkdir returns 1 if success.
				# we return 0 if success.
				# return error rc (i.e. nonzero) if any errors.
				$ret_rc = 1;
			}
		}

		return $ret_rc;
	};

	# process command line args
	my $args_ok = $parse_args->();

	if ($args_ok) {
		# got enough info.  start creating.
		$rc = $create_dirs->();
	} else {
		$rc = 1;
	}

	return $rc;
}

########################################
# No flags supported (yet) (DE 1-26-05).
# Modifies times if files exists,
# creates the files if they don't.
# Returns 0 on success, 1 if any error.
########################################
sub touch ($;$) {
	my ($argv, $proc) = @_;
	# all args are files (or file specs)
	my $flist = [];
	my $ret_rc = 0;
	my $parse_args = sub {
		my $pa_rc = 0;
		my $numargs = scalar(@$argv);
		for (my $i = 0; $i < $numargs; $i++) {
			my $arg = $argv->[$i];
			push(@$flist, glob($arg));
		}

		# expect at least one file to be specified.
		# unix gives usage otherwise.
		if (scalar(@$flist) > 0) {
			$pa_rc = 1;
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $numfiles = scalar(@$flist);
		for (my $i = 0; $i < $numfiles; $i++) {
			my $fnam = $flist->[$i];
			if (-e $fnam) {
				# exists.  modify info.
				my ($atime, $mtime);
				$atime = $mtime = time();
				my $num_files_changed = utime($atime, $mtime, $fnam);
				if (!$num_files_changed && !$ret_rc) {
					$ret_rc = 1;
				}
			} else {
				# does not exist.  create it.
				if (open(F, "> $fnam")) {
					close(F);
				} else {
					# could not create.
					# set error.
					if (!$ret_rc) {
						$ret_rc = 1;
					}
				}
			}
		}
	} else {
		# invalid command line.
		$ret_rc = 2;
	}

	return $ret_rc;
}

########################################
# Flags supported:
#   a, m, n, r, s, v
# No flags behaves like -s.
# Returns:
#  0: success
#  1: could not get info
#  2: invalid command line (i.e. argv)
########################################
sub uname ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 0;
	# set up hash to contain arg values
	my $uname_info = {
		a => 0,
		m => 0,
		n => 0,
		r => 0,
		s => 0,
		v => 0,
	};
	my $parse_args = sub {
		my $pa_rc = 0;

		# set up hash for getopts
		my $opts = {
			a => \$uname_info->{a},
			m => \$uname_info->{m},
			n => \$uname_info->{n},
			r => \$uname_info->{r},
			s => \$uname_info->{s},
			v => \$uname_info->{v},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			# specified args obtained.
			# do remaining work.
			my $some_arg_set = 0;
			for my $key (keys(%$opts)) {
				my $val = $uname_info->{$key};
				if ($val) {
					$some_arg_set = 1;
					last;
				}
			}

			if (!$some_arg_set) {
				# no args set.  behave as if -s were specified.
				$uname_info->{s} = 1;
			}

			$pa_rc = 1;
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		# this saves some work
		my ($sysname, $nodename, $release, $version, $machine) =
			POSIX::uname();
		# yet another hack
		$sysname =~ s/ /_/g;
		# yet another hash
		my $uname_values = {
			s => $sysname,
			n => $nodename,
			r => $release,
			v => $version,
			m => $machine,
		};

		# start small
		my $output_string = "";

		# they get output in this order (if at all)
		my $opt_order = [qw(s n r v m)];

		for my $opt (@$opt_order) {
			# options in standard uname order
			if ($uname_info->{$opt} || $uname_info->{a}) {
				my $uname_value = $uname_values->{$opt};
				if ($output_string) {
					$output_string .= " $uname_value";
				} else {
					$output_string = $uname_value;
				}
			}
		}

		if ($proc) {
			$proc->($output_string);
		} else {
			$default_output_proc->("$output_string");
		}
	} else {
		# 2 if invalid command line options
		$rc = 2;
	}

	return $rc;
}

sub compute_local_user {
	my $local_user = getlogin() || getpwuid($<) || "root";

	return $local_user;
}

sub compute_local_userdomain {
	my $local_user = compute_local_user();
	my $local_userdomain = $local_user;

	if ($^O eq "MSWin32") {
		# trouble (:-)
		if ($ENV{USERDOMAIN}) {
			my $ud = $ENV{USERDOMAIN};
			$local_userdomain = "${ud}\\${local_user}";
		}
	}

	return $local_userdomain;
}

#
# For any given host, we will always call rsh before rcp.
# Since rsh "knows" the hostname (rcp has it embedded in
# file names, maybe), rsh can use a domain-qualified
# local username 1st, then try just the local username
# if the longer one fails (AIX requires the longer one,
# Solaris seems to reject it).
#
# Once, for a give host, rsh determines which form to
# use, it put that information into an embedded hash
# (keyed by hostname, for its own reference), and into
# another location in the outer hash, where (subsequent)
# rcp calls can consult it.
#
# There certainly must be a better way, and we should
# figure out what it is before too long. (DE 5-19-05).
#
my $rsh_rcp_stuff = {
	# rcp consumes this
	use_local_user_domain => undef,
	# rsh produces this, and the above
	hosts => {
	},
};

#
# Run a remote shell command.
#
# Input: 1) an array of arguments to the command
#        2) an optional proc to run perl output line.
#
# Return: 0 if connected, 1 if could not connect.
#
sub rsh ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 1;
	my $local_user = compute_local_user();
	# set up hash to contain arg values
	my $rsh_info = {
		remote_host => undef,
		local_user  => $local_user,
		remote_user => "root",
		remote_cmd  => undef,
	};
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			'l=s' => \$rsh_info->{remote_user},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			my $numargs = scalar(@$argv);

			if ($numargs >= 2) {
				$rsh_info->{remote_host} = $argv->[0];
				$rsh_info->{remote_cmd} = $argv->[1];

				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $host = $rsh_info->{remote_host};
		my $local_user = $rsh_info->{local_user};
		my $remote_user = $rsh_info->{remote_user};
		my $rsh_cmd = $rsh_info->{remote_cmd};
		my $output = "";
		my $eff_rsh_cmd = sprintf('( %s ) 2>&1', $rsh_cmd);

		eval {
			my $netrsh = Net::Rsh->new();
			my $local_user_domain = compute_local_userdomain();
			my $rsh_rc = undef;
			my $result = undef;

			if (exists($rsh_rcp_stuff->{hosts}{$host})) {
				# heuristic os-specific determination made
				my $host_info = $rsh_rcp_stuff->{hosts}{$host};
				if ($host_info->{use_local_user_domain}) {
					$result = $netrsh->rsh
						(
						 $host,
						 $local_user_domain,
						 $remote_user,
						 $eff_rsh_cmd
						 );
				} else {
					$result = $netrsh->rsh
						(
						 $host,
						 $local_user,
						 $remote_user,
						 $eff_rsh_cmd
						 );
				}
				$rsh_rc = substr($result->[0], 0, 1);
				$result->[0] = substr($result->[0], 1);
			} else {
				# heuristic os-specific determination still to be made
				my $host_info = {};
				$result = $netrsh->rsh
					(
					 $host,
					 $local_user_domain,
					 $remote_user,
					 $eff_rsh_cmd
					 );
				########################################################
				# Apparently (i.e. after a bunch of googling),
				# the 1st byte of the output from the socket
				# is a nul byte if connection was successful,
				# or a nonzero-byte (usually '\1') otherwise.
				#
				# Split it off.
				#
				# Neither do I know (nor could I find on CPAN or
				# Google) under what circumstances more than one
				# element of the array is populated via socket.
				#
				# Join it with an empty string for now. (DE 1-26-05)
				########################################################
				$rsh_rc = substr($result->[0], 0, 1);
				$result->[0] = substr($result->[0], 1);

				if ($rsh_rc eq "\0") {
					# works with qualified local user
					$host_info->{use_local_user_domain} = 1;
				} elsif ($rsh_rc eq "\1" &&
						 $local_user ne $local_user_domain &&
						 ($result->[0] =~ /remuser too long/i)) {
					# probably not AIX and, therefore, don't use userdomain
					$result = $netrsh->rsh
						(
						 $host,
						 $local_user,
						 $remote_user,
						 $eff_rsh_cmd
						 );
					$rsh_rc = substr($result->[0], 0, 1);
					$result->[0] = substr($result->[0], 1);

					if ($rsh_rc eq "\0") {
						# works with unqualified local user
						$host_info->{use_local_user_domain} = 0;
					} else {
						# don't work 'tall
						$host_info->{use_local_user_domain} = -1;
					}
				} else {
					# don't work 'tall
					$host_info->{use_local_user_domain} = -1;
				}

				$rsh_rcp_stuff->{hosts}{$host} = $host_info;
			}
			$rsh_rcp_stuff->{use_local_user_domain} =
				$rsh_rcp_stuff->{hosts}{$host}{use_local_user_domain};

			# one way or 'tother
			$rc = ($rsh_rc eq "\0") ? 0 : 1;
			# maybe should be newline
			$output = join("", @$result);
			# always do this
			chomp $output;
		};
		if ($@) {
			if (!$output) {
				$output = "$!: $@\n";
			}
			$rc = 2;
		}

		if ($proc) {
			# an output method was passed
			$proc->($output);
		} else {
			# no output method was passed, so use default.
			$default_output_proc->($output);
		}
	}

	return $rc;
}

#
# Perform a remote copy.
#
# Input: 1) a source file (optional platform)
#        2) a target file (optional platform)
#
# Flags supported: -p, -r
#
# Return: 0 if connected, 1 if could not connect.
#
sub rcp ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 1;
	my $local_user = compute_local_user();
	# set up hash to contain arg values
	my $rcp_info = {
		p => 0,
		r => 0,
		src_flist => [],
		trg_fnam => undef,
		local_user  => $local_user,
		remote_user => "root",
	};
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			p => \$rcp_info->{p},
			r => \$rcp_info->{r},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			if (scalar(@$argv) >= 2) {
				# at least one source file.
				# assume target file is last one.
				$rcp_info->{trg_fnam} = pop(@$argv);

				my $numargs = scalar(@$argv);

				for (my $i = 0; $i < $numargs; $i++) {
					my $arg = $argv->[$i];
					push(@{$rcp_info->{src_flist}}, $arg);
				}

				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $output = "";
		my $local_user = $rcp_info->{local_user};
		my $remote_user = $rcp_info->{remote_user};
		my $src_flist = $rcp_info->{src_flist};
		# no longer used
		#my $from = join(" ", @$src_flist);
		my $to = $rcp_info->{trg_fnam};
		my $amp = '@';
		if ($to =~ /(\w+)${amp}(.*)/) {
			$remote_user = $1;
			$to = $2;
		} elsif ($src_flist->[0] =~ /(\w+)${amp}(.*)/) {
			$remote_user = $1;
			$src_flist->[0] = $2;
		}
		my $sd_flist = [];
		push(@$sd_flist, @$src_flist);
		push(@$sd_flist, $to);
		eval {
			#
			# Turn off warnings.  we'll get info in $! (???!!!).
			#
			# Note: Should probably do something like redirect
			# stderr to stdout, and tie to some method that
			# streams output onto the end of a given string.
			#
			local $^W = 0;
			local $! = "";
			my $netrcp = Net::Rcp->new();
			my $local_user_domain = compute_local_userdomain();
			my $result = undef;
			# netrcp->rcp now takes an flist parameter as a reference
			# to an array of all files involved (including, possibly,
			# multiple source and optional target files.
			if ($rsh_rcp_stuff->{use_local_user_domain}) {
				$result = $netrcp->rcp
					(
					 $local_user_domain,
					 $remote_user,
					 $sd_flist
					 );
			} else {
				$result = $netrcp->rcp
					(
					 $local_user,
					 $remote_user,
					 $sd_flist
					 );
			}
			my $rcp_rc = $result;
			if ($!) {
				# Still alive, but some warning.
				# Don't know how to "fatalize",
				# so throw it.
				die "Error in rcp: $!\n";
			} elsif ($rcp_rc) {
				# we found some error, possibly before system did
				die "Unknown error in rcp: ${rcp_rc}\n";
			}
		};
		if ($@) {
			# This should catch any death or warning
			# from netrcp implementation of rcp.
			$output = $@;
			$rc = 2;
		} else {
			$rc = 0;
		}

		chomp $output;

		if ($proc) {
			# an output method was passed
			$proc->($output);
		} else {
			# no output method was passed, so use default.
			$default_output_proc->($output);
		}
	}

	return $rc;
}

#
# Run a secure remote shell command.
#
# Input: 1) an array of arguments to the command
#        2) an optional proc to run perl output line.
#
# Flags/Args supported: -i, -l
#
# Return: 0 if connected, 1 if could not connect.
#
sub ssh ($;$) {
	my ($argv, $proc) = @_;
	# do not give benefit of doubt.
	my $rc = 1;
	# set up hash to contain arg values
	my $ssh_info = {
		remote_host => undef,
		remote_user => "root",
		remote_cmd  => undef,
	};
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			'l=s' => \$ssh_info->{remote_user},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			my $numargs = scalar(@$argv);

			# after stripping flags and args, there
			# should be at least (and, preferably,
			# at most :-) two args left over.
			if ($numargs >= 2) {
				$ssh_info->{remote_host} = $argv->[0];
				$ssh_info->{remote_cmd} = $argv->[1];

				$pa_rc = 1;
			}
		}

		# return success of arg parsing
		return $pa_rc;
	};

	# see if args correctly specified
	my $args_ok = $parse_args->();

	if ($args_ok) {
		# args good enough.  munge on.
		my $host = $ssh_info->{remote_host};
		my $remote_user = $ssh_info->{remote_user};
		my $cmd = $ssh_info->{remote_cmd};
		my $ssh_exe = undef;

		if ($ENV{VXIF_SSH_EXE}) {
			$ssh_exe = $ENV{VXIF_SSH_EXE};
		} else {
			if ($^O eq "MSWin32") {
				# on windows, default is to use whatever is there
				$ssh_exe = "ssh";
			} else {
				# on other-than-windows, assum in /usr/bin
				$ssh_exe = "/usr/bin/ssh";
			}
		}

		my $ssh_cmd = $ssh_exe;
		{
			my $dq = '"';
			# Assume that if cmd is not double quoted
			# then it needs to be, and that the first
			# non-space being a double quote (or not)
			# determines aforementioned quotedness.
			#
			# "Unless" is *soooo* perl, but what else?
			$cmd = "${dq}${cmd}${dq}" unless $cmd =~ /^\s*${dq}/;
		}
		$ssh_cmd .= " -l ${remote_user} ${host} ${cmd}";
		my $eff_ssh_cmd = sprintf('( %s ) 2>&1', $ssh_cmd);
		my $output = qx{$eff_ssh_cmd};
		$rc = $?;
		chomp $output;

		if ($proc) {
			# an output method was passed
			$proc->($output);
		} else {
			# no output method was passed, so use default.
			$default_output_proc->($output);
		}
	}

	return $rc;
}

#
# Perform a secure remote copy.
#
# Input: 1) a source file (optional platform)
#        2) a target file (optional platform)
#
# Flags supported: -p, -r
#
# Return: 0 if connected, 1 if could not connect.
#
sub scp ($;$) {
	my ($argv, $proc) = @_;
	my $rc = 1;
	# set up hash to contain arg values
	my $scp_info = {
		p => 0,
		q => 0,
		r => 0,
		src_flist => [],
		trg_fnam => undef,
	};
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			p => \$scp_info->{p},
			q => \$scp_info->{q},
			r => \$scp_info->{r},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			if (scalar(@$argv) >= 2) {
				# at least one source file.
				# assume target file is last one.
				$scp_info->{trg_fnam} = pop(@$argv);

				my $numargs = scalar(@$argv);

				for (my $i = 0; $i < $numargs; $i++) {
					my $arg = $argv->[$i];
					push(@{$scp_info->{src_flist}}, $arg);
				}

				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $output = "";
		my $scp_exe = undef;

		if ($ENV{VXIF_SCP_EXE}) {
			$scp_exe = $ENV{VXIF_SCP_EXE};
		} else {
			if ($^O eq "MSWin32") {
				# on windows, default is to use whatever is there
				$scp_exe = "scp";
			} else {
				# on other-than-windows, assum in /usr/bin
				$scp_exe = "/usr/bin/scp";
			}
		}

		my $scp_cmd = $scp_exe;
		if ($scp_info->{p}) {
			$scp_cmd .= " -p";
		}
		if ($scp_info->{q}) {
			$scp_cmd .= " -q";
		}
		if ($scp_info->{r}) {
			$scp_cmd .= " -r";
		}
		my $num_srcfiles = scalar(@{$scp_info->{src_flist}});
		for (my $i = 0; $i < $num_srcfiles; $i++) {
			my $srcfile = $scp_info->{src_flist}[$i];
			$scp_cmd .= " $srcfile";
		}
		my $trgfnam = $scp_info->{trg_fnam};
		$scp_cmd .= " $trgfnam";
		my $eff_scp_cmd = sprintf('( %s ) 2>&1', $scp_cmd);
		$output = qx{$eff_scp_cmd};
		$rc = $?;
		chomp $output;

		if ($proc) {
			# an output method was passed
			$proc->($output);
		} else {
			# no output method was passed, so use default.
			$default_output_proc->($output);
		}
	}

	return $rc;
}

########################################
# Flags supported:
#   None.  Behaves like -f (i.e. force).
#          Can pass -f, but it's a no-op.
# Args:
#   2 files, source and target.
# Returns:
#  0: success
#  1: couldn't do it
#  2: invalid command line,
#     or no file(s) specified.
########################################
sub cp ($;$) {
	my ($argv, $proc) = @_;
	my $ret_rc = 1;
	# set up hash to contain flags
	my $cp_info = {
		f => 0,
		src => undef,
		trg => undef,
	};
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			f => \$cp_info->{f},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			# have specified flags.  parse remaining args.

			my $numargs = scalar(@$argv);

			if ($numargs >= 2) {
				$cp_info->{src} = $argv->[0];
				$cp_info->{trg} = $argv->[1];

				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $src = $cp_info->{src};
		my $trg = $cp_info->{trg};
		my $cp_rc = copy($src, $trg);

		if ($cp_rc) {
			# success
			$ret_rc = 0;
		}
	} else {
		# invalid command line.
		$ret_rc = 2;
	}

	return $ret_rc;
}

########################################
# Flags supported:
#   None.  Behaves like -f (i.e. force).
#          Can pass -f, but it's a no-op.
# Args:
#   A list of files.
# Returns:
#  0: success
#  1: couldn't do it
#  2: invalid command line,
#     or no file(s) specified.
########################################
sub rm ($;$) {
	my ($argv, $proc) = @_;
	my $ret_rc = 0;
	# set up hash to contain flags
	my $rm_info = {
		f => 0,
	};
	# all args are files (or file specs)
	my $flist = [];
	my $parse_args = sub {
		my $pa_rc = 0;
		# set up hash for getopts
		my $opts = {
			f => \$rm_info->{f},
		};
		# invoke utility method to populate opts (and depopulate argv)
		my $go_rc = get_args($argv, $opts);

		if ($go_rc) {
			# have specified flags.  parse remaining args.

			my $numargs = scalar(@$argv);
			for (my $i = 0; $i < $numargs; $i++) {
				my $arg = $argv->[$i];
				push(@$flist, glob($arg));
			}

			# expect at least one file to be specified.
			# unix gives usage otherwise.
			if (scalar(@$flist) > 0) {
				$pa_rc = 1;
			}
		}

		return $pa_rc;
	};

	my $args_ok = $parse_args->();

	if ($args_ok) {
		my $numfiles = scalar(@$flist);
		my $numdel = unlink(@$flist);

		if ($numfiles != $numdel) {
			# did not delete all files
			$ret_rc = 1;
		}
	} else {
		# invalid command line.
		$ret_rc = 2;
	}

	return $ret_rc;
}

1;

