package Net::SocketCmd;
use strict;

use IO::Socket;
use Carp;
use Errno;

sub new {
	my $pkg = shift;
	my $this = {};

	bless($this, $pkg);

	$this->init(@_);

	return $this;
}

sub init {
	my $this = shift;

	# for message localization
	$this->set_loc_proc(undef);

	# need to set this in derived class
	$this->set_peer_port(undef);
}

sub get_loc_proc {
	my $this = shift;
	my $loc_proc = $this->{loc_proc};

	return $loc_proc;
}

sub set_loc_proc {
	my $this = shift;
	my ($loc_proc) = @_;

	$this->{loc_proc} = $loc_proc;
}

sub get_peer_port {
	my $this = shift;
	my $peer_port = $this->{peer_port};

	return $peer_port;
}


sub set_peer_port {
	my $this = shift;
	my ($peer_port) = @_;

	$this->{peer_port} = $peer_port;
}

# die/croak, w/optional localization
sub loc_croak {
	my $this = shift;
	my ($msg) = @_;
	my $loc_proc = $this->get_loc_proc();
	my $loc_msg = $loc_proc ? $loc_proc->($msg) : $msg;

	croak $loc_msg;
}

# comes up with a socket.
sub get_socket {
	my $this = shift;
	my ($host) = @_;
	my $peer_port = $this->get_peer_port();

	if (!defined($peer_port)) {
		$this->loc_croak("Need to set peer port (e.g. 514, for rsh\n");
	}

	my $start_port = 512;
	my $end_port = 1023;

	my $port=$end_port;                                                                                                                   
	my $socket = undef;

	# if connection refused, try again, up to 10 times.
	# problem seems to go away on the 2nd try.
	my $tries = 0;
	my $max_tries = 10;

	# initialize to loop
	my $try_some_more = 1;

	while ($try_some_more) {
		if($port<$start_port) {$this->loc_croak("All ports in use.\n");}

		# Set Reuse=>1 to allow reuse of the port after the program exits.
		# This is to ensure that if our program exits abnormally and does not properly close the socket,
		# running it again will allow opening a new socket on the same port.
		$! = 0;
		$socket = IO::Socket::INET->new(PeerAddr=>$host,
                                		PeerPort=>$peer_port,
                                		LocalPort=>$port,
                                		Proto=>"tcp",
										Reuse=>1);

		if (!defined $socket) {
			if ($!{EADDRINUSE}) {
				$port-=1;
			} elsif ($!{ECONNREFUSED}) {
				#print(STDERR "Connection refused.\n");
				$tries+=1;
				if ($tries > $max_tries) {
					$this->loc_croak("Couldn't connect to rsh server.\nMax tries exceeded.\n");
				} else {
					#print(STDERR "Will try again.\n");
				}
			} else {
				#print(STDERR "Will croak!\n");
				my $msg = "Unhandled error: $!\n";
				for my $key (keys(%!)) {
					my $val = $!{$key};
					if ($val) {
						$msg .= "$key ==> $val\n";
					}
				}
				$this->loc_croak($msg);
			}
		} else {
			$try_some_more = 0;
		}
	}

	return $socket;
}

sub recycle_socket {
	my $this = shift;
	my ($socket) = @_;
	# For some reason (TBD :-), a close on a socket
	# sets $! to "invalid file descriptor", even if
	# everything is apparently valid.
	# So, use a local descriptor here.
	local $!;

	$socket->close();
}

1;

