package Net::Rcp;
use strict;

#
# Implements an rcp (remote copy) client.
#

# rcp works by talking to rshd. This is how rshd works
# -----------------------------------------------------
# rshd provides remote execution facilities with authentication based on privileged port numbers from trusted hosts. 
# rshd listens for service requests at port 514. When a service request is received, the following protocol is initiated: 
# The service checks the client's source port. 
# If the port is not in the range 512-1023 then the service aborts the connection. 
# The service reads characters from the socket up to a NUL byte. 
# The resultant string is interpreted as an ASCII number, base 10. 
# If the number received in step 1 is non-zero, 
# it is interpreted as the port number of a secondary stream to be used for stderr. 
# If the port is not in the range 512-1023 then the service aborts the connection. 
# A second connection is then created to the specified port on the client's machine. 
# A NUL terminated user name of at most 256 characters is retrieved on the initial socket. 
# This is the user name (also known as remuser) on the client machine. 
# A NUL terminated user name of at most 256 characters is retrieved on the initial socket. 
# This is the user name (also known as locuser on the machine running the rshd service 
# A NUL terminated command to be passed to a shell is retrieved on the initial socket. 
# The length of the command is limited to 8192 bytes. 
# rshd then attempts to retrieve the user's password from the LSA database (see the rsetup reference page). 
# If this fails the connection is aborted with a diagnostic message returned. 
# rshd then validates the user as is done at login time. 
# If this fails the connection is aborted with a diagnostic message returned. 
# rshd then validates the host/client user name by checking the ROOTDIR/etc/hosts.equiv and ~/.rhosts files. 
# If this fails the connection is aborted and a diagnostic message is returned. 
# A NUL byte is returned on the initial socket. 
# rshd loads the user's profile and runs the command specified in the user's home directory. 

use IO::Socket;
use Carp;
use Errno;
use bytes ();
use File::Find;
use File::stat;

require Exporter;

use vars qw($VERSION @ISA @EXPORT);

$VERSION=0.02;

use VxIF::NativePerl::Net::SocketCmd;
@Net::Rcp::ISA = qw(Net::SocketCmd);

@EXPORT = qw(&rcp);

my %FILENAME;
my @files;

# one megabyte, for now (DE, 4-14-05)
use constant DEFAULT_CHUNK_SIZE => 1048576;

sub new {
	my $pkg = shift;
	my $this = new Net::SocketCmd();

	bless($this, $pkg);

	$this->init(@_);

	return $this;
}

sub init {
	my $this = shift;
	$this->set_peer_port(514);
}

# returns either "from" or "to", based on target file.
sub determine_direction {
	my ($dest) = @_;

	# KLUDGE (but a reasonly well thought-through kludge):
	# if length(<to>host)==1, assume Win32 device (and, therefore, win32 "from" target
	my $direction = "from";
	if ($dest =~ /:/) {
		my ($host, $destination) = split(/:/,$dest,2);
		if (length($host) > 1) {
			# see comments before this computation
			$direction = "to";
		}
	}

	return $direction;
}

sub rcp {
	my ($self,$local_user,$remote_user,$flist) = @_;
	croak("Usage: \$c->rsh(\$local_user,\$remote_user,\$flist)") unless @_ == 4;
	my ($src, $dst);
	my $rc = 0;

	# destination file is always the last file in the list
	$dst = pop(@$flist);

	# compute direction of copy based on (syntax of) destination file.
	my $direction = determine_direction($dst);

	if ($direction eq "to") {
		# direction is rcp -t
		my ($host, $destination) = split(/:/,$dst,2);

		foreach $src (@$flist) {
			$rc = $self->rcp_to($host, $local_user, $remote_user, $src, $destination);
			if ($rc) {
				last;
			}
		}
		
	} else { # direction is rcp -f
		foreach $src (@$flist) {
			if ($src =~ /:/) {
				my ($host, $source) = split(/:/,$src,2);
				$rc = $self->rcp_from($host, $local_user, $remote_user, $source, $dst);
				if ($rc) {
					last;
				}
			} else {
				#crib
				croak("Please specify a host for remote copy, or use cp for local copy\n");
			}
		}
	}

	return $rc;
}

#
# rcp -t (recursive)
# read the directory tree specified and stream it to the target host on the rshd port (514).
sub rcp_to {
	my ($self,$host,$local_user,$remote_user,$from, $to)=@_;
	croak("Usage: \$c->rsh(\$host,\$local_user,\$remote_user,\$from,\$to)") unless @_ == 6;
	my $rc = 0;

	# make sure the source file exists
	if (! (-e $from)) {
		return 1;
	}

	my $socket = $self->get_socket($host);

	print $socket "0\0";
	print $socket "$local_user\0";
	print $socket "$remote_user\0";
	print $socket "rcp -r -p -t $to\0";

	my $ret;
	my $n = sysread($socket,$ret,1);

	if ($n) {
		find(\&wanted,$from);
		$self->write_data($socket, $from);

		# empty the files array after the job is done.
		for my $i(1..scalar(@files)) {
			pop(@files);
		}
	} else {
		$rc = 1;
	}

	$self->recycle_socket($socket);

	return $rc;
}

# wanted function for the File::find method.
# Do a DFS on a directory and push the files in a hash %FILENAME
# FILENAME{/full/path/of/a/file} = filename
# Also the files  are stored in an array @files to retain the DFS order
# which is required while writing files to the socket.
sub wanted {
	push(@files, $File::Find::name);
	$FILENAME{$File::Find::name} = $_;
}

# Given a path, determine it's depth (in the tree sense)
sub dir_depth {
	my ($dir) = @_;

	my @tarr = split(/\//, $dir);
	return scalar(@tarr);
}

# Write file data to the socket for rcp -t.
# The function begins by reading the @files array and
# write it to the socket in DFS order.
#
# For directories write "D%04o %d %s\n" (mode, 0, dirname)
# For files write "C%04o %jd %s\n" (mode, size, filename)
# End directories with E\n
sub write_data {
	my ($self, $socket, $dir) = @_;
	my ($cwd, $top, $ret, $n, $st, $mode);

	if (-d $dir) {
		$cwd = $top = $dir;
	} else {
		#probably nothing here
	}

	foreach my $item (@files) {
		if (-d $item) {
			$st = stat($item);
			$mode = sprintf("%04o", $st->mode & 07777);

			if ((dir_depth($item) - 1) > dir_depth($cwd)) {
				print $socket "D$mode 0 $FILENAME{$item}\n";
				$n = sysread($socket, $ret, 1);
				$cwd = $item;
			} elsif ($item eq $cwd) {
				$item =~s/.*\///;
				print $socket "D$mode 0 $item\n";
				$n = sysread($socket, $ret, 1);
			} else { # sizeof ($item) < $cwd
				my $diff = dir_depth($cwd) - (dir_depth($item) - 1);
				foreach my $i (1..$diff) {
					print $socket "E\n";
					$n = sysread($socket, $ret, 1);
				}
				print $socket "D$mode 0 $FILENAME{$item}\n";
				$n = sysread($socket, $ret, 1);
				$cwd = $item;
			}
		} else {
			$st = stat($item);
			$mode = sprintf("%04o", $st->mode & 07777);
			my ($fbuf);
			open(FILE, "<$item") or die "can't open $item";
			binmode FILE;
			
			my $fsize = $st->size;

			print $socket "C$mode $fsize $FILENAME{$item}\n";
			$n = sysread($socket, $ret, 1);

			my $errmsg = undef;
			my $num_bytes_read = 0;
			my $num_bytes_written = 0;
			my $chunk_size = DEFAULT_CHUNK_SIZE;

			while ($num_bytes_read = read(FILE,$fbuf,$chunk_size)) {
				$num_bytes_written = syswrite($socket, $fbuf, $num_bytes_read);
				if (!defined($num_bytes_written)) {
					$errmsg = $!;
				}
			}

			print $socket "\0";
			$n = sysread($socket, $ret, 1);
			close FILE;
		}
	}

	my $diff = 0;
	if (defined($cwd) && defined($top)) {
		$diff = dir_depth($cwd) - dir_depth($top);
	}
	foreach my $i (1..$diff) {
		print $socket "E\n";
		$n = sysread($socket, $ret, 1);
	}
}

#
# rcp -f (recursive)
# ask rshd to send a directory tree and write the tree onto the local host.
sub rcp_from {
	my ($self,$host,$local_user,$remote_user,$from, $to)=@_;
	croak("Usage: \$c->rsh(\$host,\$local_user,\$remote_user,\$from,\$to)") unless @_ == 6;

	my $socket = $self->get_socket($host);

	print $socket "0\0"; 
	print $socket "$local_user\0";
        print $socket "$remote_user\0";
	print $socket "rcp -r -p -f $from\0";

	my $WORKINGDIR = $to;
	my $line = "some data"; #initialise, so the while tester is happy
	my $data;
	my $rc = 0;

	while ($line) {
		print $socket "\0"; 
		$line=<$socket>;
		chomp($line);

		# if line starts with a T, it's the time stamp
		#if ($line =~ /^T/) {
		if ($line =~ /0$/) {
			print $socket "\0"; 
			$line=<$socket>;

			if ($line =~ /^C/)  { #this is a file
				my ($permissions, $size, $filename) = split(/\s/, $line,3);
				my $chunk;
				
				open(FILE, ">$WORKINGDIR/$filename") or die "can't open $WORKINGDIR/$filename";
				binmode FILE;
				
					print $socket "\0"; 
				while ($size > 0) {
					$chunk = sysread($socket, $data, $size);
					print FILE $data;
					$chunk = bytes::length($data);
					$size = $size - $chunk;
				}
				close FILE;
			} elsif ($line =~ /^D/) { #this is a directory
				my ($permissions, $size, $dirname) = split(/\s/, $line,3);
				chomp($dirname);
				$WORKINGDIR = $WORKINGDIR . "/". $dirname;
				system (mkdir $WORKINGDIR);
			} else {
				#Problem! should not get here!
			}
		} elsif ($line =~ /E$/) { #directory ends here
			my (@path) = split(/\//,$WORKINGDIR);
			pop (@path);

			#$WORKINGDIR = "";
			$WORKINGDIR = join "/", @path
		}
	}
	
	$self->recycle_socket($socket);

	return $rc;
}

END { } 

1;

