
package Logger;

use File::Basename;

###############################
# Messages                    #
###############################

$MSG_ENTERING 		= "Entering";
$MSG_EXITING  		= "Exiting";
$MSG_ERROR		= "Error";
$MSG_WARNING		= "Warning";
$MSG_CONFIG		= "Config";
$MSG_INFO		= "Info";
$MSG_FINE		= "Fine";
$MSG_FINER		= "Finer";
$MSG_FINEST		= "Finest";
$MSG_UNOPEN_LOGFILE	= "Unable to open log file";

###############################
# Public Members              #
###############################

use constant LEVEL_ALL 		=> 0;
use constant LEVEL_ERROR	=> 1000;
use constant LEVEL_WARNING	=> 900;
use constant LEVEL_INFO		=> 800;
use constant LEVEL_CONFIG	=> 700;
use constant LEVEL_FINE		=> 500;
use constant LEVEL_FINER	=> 400;
use constant LEVEL_FINEST	=> 300;
use constant LEVEL_OFF		=> 5000;

###############################
# Private Members             #
###############################
my %msgmap = (
	&LEVEL_ERROR 	=> $MSG_ERROR,
	&LEVEL_WARNING 	=> $MSG_WARNING,
	&LEVEL_CONFIG	=> $MSG_CONFIG,
	&LEVEL_INFO	=> $MSG_INFO,
	&LEVEL_FINE	=> $MSG_FINE,
	&LEVEL_FINER	=> $MSG_FINER,
	&LEVEL_FINEST	=> $MSG_FINEST,
);

###############################
# Private Methods             #
###############################

#
# Default log writer.
#
my $file_writer = sub ($$$) {
  my ($self,$log_level,$msg,$umi) = @_;
  
  if ($self->{LOGAPPEND} > 0) {
    open(LOGFILE, ">>:utf8", "$self->{LOGFILE}") || die "${MSG_ERROR}: ${MSG_UNOPEN_LOGFILE}: $self->{LOGFILE}\n";
  } else {
    open(LOGFILE, ">:utf8", "$self->{LOGFILE}") || die "${MSG_ERROR}: ${MSG_UNOPEN_LOGFILE}: $self->{LOGFILE}\n";
  }
  $self->{LOGAPPEND} =  1;
  print LOGFILE $self->{LOGFORMATTER}($self, $log_level, $msg, $umi);
  close(LOGFILE);
};

my $default_formatter = sub ($$$) {
  my ($self,$log_level,$msg,$umi) = @_;
  return ($umi ? $umi . $self->{SEPARATOR} : "" ) . $self->{LOGAPPLICATION} . $self->{SEPARATOR} . (($msgmap{$log_level} ne '') ?  $msgmap{$log_level} : $log_level) . $self->{SEPARATOR} . " " . $msg . "\n";
};

###############################
# Public Methods       	      #
###############################
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = {
  	LOGLEVEL 	=> LEVEL_ERROR,
  	LOGORIGINATORID	=> 1,
  	LOGWRITER 	=> \&$file_writer,
  	LOGFORMATTER	=> \&$default_formatter,
  	LOGFILE 	=> './vxif.log',
  	LOGAPPEND  	=> 1,
  	LOGAPPLICATION 	=> '',
  	SEPARATOR 	=> '::',
  	@_,
  };
  return bless $self, $class;
}

sub get_logfile {
	my $self = shift;
	my $logfile = $self->{LOGFILE};

	return $logfile;
}

sub get_logdir {
	my $self = shift;
	my $logfile = $self->get_logfile();
	my $logdir = dirname($logfile);

	return $logdir;
}

sub entering ($) {
  my ($self,$subname) = @_;
  $self->info("${MSG_ENTERING} ${subname}");
}

sub exiting ($) {
  my ($self,$subname) = @_;
  $self->info("${MSG_EXITING} ${subname}");
}

sub log ($$$) {
  my ($self,$severity,$msg,$umi) = @_;
  if ($severity >= $self->{LOGLEVEL}) {
    $self->{LOGWRITER}($self, $severity, $msg, $umi);
  }
}

sub logp {
  my ($self) = shift;
  my ($param, $params);
  
  foreach $param (@_) {
    $params .= $self->{SEPARATOR} . ${param};
  }
  
  $self->log(LEVEL_FINE, $params);
}

sub error ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_ERROR, $msg, $umi);
}

sub warning ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_WARNING, $msg, $umi);
}

sub info ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_INFO, $msg, $umi);
}

sub config ($$) {
  my ($self,$msg,$umi) = @_;
  Logger::log(LEVEL_CONFIG, $msg, $umi);
}

sub fine ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_FINE, $msg, $umi);
}

sub finer ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_FINER, $msg, $umi);
}

sub finest ($$) {
  my ($self,$msg,$umi) = @_;
  
  $self->log(LEVEL_FINEST, $msg, $umi);
}

sub use_default_formatter {
  my ($self) = shift;
  
  $self->{LOGFORMATTER} = \&$default_formatter;
}

sub use_default_writer {
  my ($self) = shift;
  
  $self->{LOGWRITER} = \&$file_writer;
}

1;
