package CPILinux;

use VxIF::CPICommon;
@CPILinux::ISA = qw(CPICommon);

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#	IPv6		=> This indicates if the address is in IPv6 format.  If so, the SSH must be
#			   used.
#
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my @os = qw(SunOS Linux HPUX AIX OSF1);
  my @searchpaths = qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin);
  my $self = { 	
                USESSH		=> 0,
		IPv6		=> 0,
  	  	TMPPATH		=> '/var/tmp/vxif',
  	 	TIMEFILE	=> '/tmp/vxif_time',
  	 	PKGSDIR		=> 'rpms',
  	 	PATCHSDIR	=> 'patches',
  	 	OSINSTALLER	=> 'Linux',
  	 	SUPPORTEDOS	=> \@os,
  	 	CMDSEARCHPATHS	=> \@searchpaths,
          	@_,
  	     };

  return bless $self, $class;
}

#
# Set the common system commands.
#
# Input: 1) the reference of the installation context;
#
sub set_commands ($) {
  my ($self) = @_;

  $self->SUPER::set_commands();
  $self->{CMD}{AWK}="/bin/awk";
  $self->{CMD}{CAT} = "/bin/cat";
  $self->{CMD}{CHMOD}="/bin/chmod";
  $self->{CMD}{CLEAR}="/usr/bin/clear";
  $self->{CMD}{CP}="/bin/cp -L";
  $self->{CMD}{DFK}="/bin/df -kl";
  $self->{CMD}{DOMAINNAME} = "/usr/bin/domainname";
  $self->{CMD}{DU}="/usr/bin/du";
  $self->{CMD}{ECHO}="/bin/echo";
  $self->{CMD}{GREP}="/bin/grep";
  $self->{CMD}{GUNZIP}="/bin/gunzip";
  $self->{CMD}{ID}="/usr/bin/id";
  $self->{CMD}{IFCONFIG}="/sbin/ifconfig";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  $self->{CMD}{KILL}="/bin/kill";
  $self->{CMD}{LS}="/bin/ls";
  $self->{CMD}{MKDIR} = "/bin/mkdir";
  $self->{CMD}{MOUNT}="/bin/mount";
  $self->{CMD}{MV}="/bin/mv";
  $self->{CMD}{PING}="/bin/ping";
  $self->{CMD}{PS}="/bin/ps -w";
  $self->{CMD}{RCP}="/usr/bin/rcp";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{RMR} = "/bin/rm -rf";
  $self->{CMD}{RSH}="/usr/bin/rsh";
  $self->{CMD}{SED}="/bin/sed";
  $self->{CMD}{SHUTDOWN}="/sbin/shutdown -r now";
  $self->{CMD}{SORT}="/bin/sort";
  $self->{CMD}{TAR}="/bin/tar";
  $self->{CMD}{TOUCH}="/bin/touch";
  $self->{CMD}{TPUT}="/usr/bin/tput";
  $self->{CMD}{UNAME}="/bin/uname";
  $self->{CMD}{UNIQ}="/usr/bin/uniq";
  $self->{CMD}{WC}="/usr/bin/wc";
  $self->{CMD}{WHICH}="/usr/bin/which";
  $self->{CMD}{SSH6}="/usr/bin/ssh -6";
 
  $self->{CMD}{RPM}=$self->{CMD}{RMPOBJ}="/bin/rpm";
  $self->{CMD}{MODINFO}="/sbin/lsmod";
  $self->{CMD}{MODUNLOAD}="/sbin/rmmod";

  # Added for Linux Debian Pkg Support
  $self->{CMD}{DPKG}=$self->{CMD}{DPKGOBJ}="/usr/bin/dpkg";
  
  $self->{CMD}{LSBR}="/usr/bin/lsb_release -a";
}


#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  $self->{MSG}{PDFREN}  = "package";
  $self->{MSG}{PDFR}  = Utils::_tr("package", 22, 1000);
  $self->{MSG}{PDFRS} = Utils::_tr("packages", 22, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}



#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
 
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
# Etrack 3983297 - on ubuntu or debian sometimes customers update /etc/issue to use personal messages and in such scenarios we should not use the file to determine vendor 
# We alternatively should use lsb_release -a command which is available on both debian and ubuntu to find the distribution name which helps us to determine whether to use dpkg or rpm
#
sub is_debian {
  my ($self, $ctx) = @_;
  
  my $deb_ver_file = "/etc/issue";
  my $results;

  my $found = Utils::vxif_fc($deb_ver_file, $ctx);
  if ( $found ) {
        my $cat_cmd = "$self->{CMD}{CAT} /etc/issue";
        $results = Utils::vxif_dor($cat_cmd, "", $ctx);
        my @deb_version_line = split( " ", $results );
        foreach $item(@deb_version_line) {
                if (($item =~ m/debian/i) || ($item =~ m/ubuntu/i)) {
                        return 1;
                }
        }
  }   
  
  #alternatively use lsb_release command to determine the vendor distribution if we cannot determine it from the /etc/issue file.
  
  my $lsb_cmd = "$self->{CMD}{LSBR}";
  $results = Utils::vxif_dor($lsb_cmd, "", $ctx);
  my @lsb_version_line = split( " ", $results );
  foreach $item(@lsb_version_line) {
          if (($item =~ m/debian/i) || ($item =~ m/ubuntu/i)) {
                  return 1;
         }
  }
  return 0;
}

#
#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) extension;
#        4) the reference to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers ($\%;$) {
  my $self = shift;
  my ($ctx,$pkg,$ext,$pkgs) = @_;
  my ($iv,@pkgnames,$trg_host,$upi_inst,$trg_upi);
  $trg_host = $ctx->get_target_hostname();
  $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
  $trg_upi = $ctx->get_target_upi();
  $$ctx{LOGGER}->entering("CPILinux::get_instpkgvers");
  
  @pkgnames = keys(%{$pkgs});

  $$ctx{LOGGER}->finer("pkg=${pkg}");

  if ((Utils::vxif_list($pkg, @pkgnames) >= 0) && $upi_inst->can("get_installed_package_version")) {
    $iv = $upi_inst->get_installed_package_version($ctx, $pkg);
  } elsif ( is_debian($self, $ctx) ) {
        # Assumes version is embedded in the .deb name
        # Use dpkg -s | grep Version if this is not true
        $$ctx{LOGGER}->info("$self->{CMD}{DPKG} -s ${pkg} | grep Version 2> /dev/null");
        my $version_str = Utils::vxif_dor("$self->{CMD}{DPKG} -s ${pkg} | grep Version 2> /dev/null", "", $ctx);
        my $Version_Key = "Version:";
        my $position = rindex($version_str, $Version_Key) + length($Version_Key)+1; 
        $iv = substr($version_str, $position);
  } else {
        # Assumes version is embedded in the name
        # Use rpm -q -i if this is not true
        $$ctx{LOGGER}->info("$self->{CMD}{RPM} -q --queryformat '%{VERSION}-%{RELEASE}' ${pkg} 2> /dev/null");
        $iv = Utils::vxif_dor("$self->{CMD}{RPM} -q --queryformat '%{VERSION}-%{RELEASE}' ${pkg} 2> /dev/null", "", $ctx);
  }
  
  return "" if $iv =~ /not installed/;

  return "$iv";
}

#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs,$ext) = @_;
	my $rpmfile;
	my $v = "";

	$$ctx{LOGGER}->entering("CPILinux::get_mediapkgvers");

	my $trg_upi = $ctx->get_target_upi();
	my $loc = $self->{$trg_upi}{$pkg}{TLOC};

	# get the product packages to install
	$pkgs = $self->{PRODUCT}{$trg_upi}->get_product_packages($ctx, $prodver, $osver, $osarch);
	$rpmfile = $$pkgs{$pkg}{RPM};

        # Replace the RPM extension with DEB
        if ( is_debian($self, $ctx) ) {
                $rpmfile =~ s/rpm$/deb/i;
                
        	if ($rpmfile && Utils::vxif_fc("${loc}/${rpmfile}", $ctx)) {
	        	$$ctx{LOGGER}->info("$self->{CMD}{DPKG} -f ${loc}/${rpmfile} Version");
		        $v = Utils::vxif_dor("$self->{CMD}{DPKG} -f ${loc}/${rpmfile} Version", "", $ctx);
                } else {
		        #$$ctx{LOGGER}->error(Utils::_tr("Unable to locate the DEB file, ${rpmfile}, associated with ${pkg} in ${loc} on $$ctx{TARGET}{HOST}.", 22, 1002, "${rpmfile}", "${pkg}", "${loc}", "$$ctx{TARGET}{HOST}"));
		        $$ctx{LOGGER}->error("Unable to locate the DEB file, ${rpmfile}, associated with ${pkg} in ${loc} on $$ctx{TARGET}{HOST}.");
                }
        } else {

        	if ($rpmfile && Utils::vxif_fc("${loc}/${rpmfile}", $ctx)) {
	        	$$ctx{LOGGER}->info("$self->{CMD}{RPM} -qp --queryformat '%{VERSION}' ${loc}/${rpmfile}");
		        $v = Utils::vxif_dor("$self->{CMD}{RPM} -qp --queryformat '%{VERSION}' ${loc}/${rpmfile}", "", $ctx);
	        } else {
		        #$$ctx{LOGGER}->error(Utils::_tr("Unable to locate the RPM file, ${rpmfile}, associated with ${pkg} in ${loc} on $$ctx{TARGET}{HOST}.", 22, 1003, "${rpmfile}", "${pkg}", "${loc}", "$$ctx{TARGET}{HOST}"));
		        $$ctx{LOGGER}->error("Unable to locate the RPM file, ${rpmfile}, associated with ${pkg} in ${loc} on $$ctx{TARGET}{HOST}.");
	        }
        }

	return "$v";
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
  my ($self,$ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
  my($size) = 0;
  
  $$ctx{LOGGER}->entering("CPILinux::get_mediapkgsize");
  
  if ($compressed) {
    #$$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 22, 1004, "${pkg}"));
    $$ctx{LOGGER}->fine("${pkg} is compressed");
    if (-f "${loc}/${filename}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      # assume compression of 3x. multiple size by 4 to account for the compressed package itself
      $size *= 4;
    } else {
      #$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on $$ctx{LOCAL}{HOST}.", 22, 1005, "${pkg}", "${loc}", "${filename}", "$$ctx{LOCAL}{HOST}"));
      $$ctx{LOGGER}->warning("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on $$ctx{LOCAL}{HOST}.");
      return 0;
    }
  } else {
    #$$ctx{LOGGER}->fine(Utils::_tr("${pkg} is uncompressed", 22, 1006, "${pkg}"));
    $$ctx{LOGGER}->fine("${pkg} is uncompressed");
    if (-f "${loc}/${pkg}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
    } else {
      #$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${pkg}/pkginfo does not exist on $$ctx{LOCAL}{HOST}.", 22, 1007, "${pkg}", "${loc}", "${pkg}", "$$ctx{LOCAL}{HOST}"));
      $$ctx{LOGGER}->warning("Unable to determine the size of ${pkg} because ${loc}/${pkg}/pkginfo does not exist on $$ctx{LOCAL}{HOST}.");
      return 0;
    }
  }
  
  #$$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} is ${size} kbs", 22, 1008, "${pkg}", "${size}"));
  $$ctx{LOGGER}->info("Size of ${pkg} is ${size} kbs");
  
  return $size;
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%) {
  my ($self,$ctx,$nic) = @_;
  
  # special case for localhost. death to lunix.
  return "255.0.0.0" if $nic eq 'lo';

  #
  # run ifconfig and grab the lines with 'Mask:' in them
  # implicitly ifconfig for all interfaces if $nic is undefined                   
  #
  my @all_netmasks = split ( "\n", Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic", "", $ctx) );

  # in the case where we have multiple interfaces, we only need                   
  # the top line, and we assume it's the default
  my @def_netmask_line = split( " ", $all_netmasks[1] );

  # grab the last end of that line
  my @def_netmask = split( ":", $def_netmask_line[-1] );

  # split it at the : and take the end
  my $def_netmask = $def_netmask[-1];

  # return that as the netmask string
  return $def_netmask;
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my @all_interfaces;
  my @all_interface_lines = split( "\n",  Utils::vxif_dor("$self->{CMD}{IFCONFIG}", "", $ctx)) ;

  foreach (@all_interface_lines) {
    if ( grep /Link/i, $_ ) {
      my @temp = split( " ", $_);
      if ( ($temp[0] ne "lo")  && ($temp[0] !~ /:\d+$/)) {
        push( @all_interfaces, $temp[0] );
      }
    }
  }   

  return \@all_interfaces;
}   

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
  my ($self,$ctx,$pkg) = @_;
  my ($dep,$deplist,$deprpms,$deppkgs,@deps);
  my $trg_upi = $ctx->get_target_upi();

  if ($trg_upi->can("check_for_package_dependents")) {
    $deppkgs = $trg_upi->check_for_package_dependents($ctx, $pkg);
    @deps = keys(%{$deppkgs});
    $deprpms = join(" ", @deps);
    $deprpms .= " ";
  }
  
  # Check for Debian system when querying for Dependencies
  if ( is_debian($self, $ctx) ) {
        @deplist = split("\n", Utils::vxif_dor("$self->{CMD}{DPKG} -s $pkg | grep ^Depends:", "", $ctx) );
  } else {
        @deplist = split("\n", Utils::vxif_dor("$self->{CMD}{RPM} -q --whatrequires $pkg", "", $ctx) );
  }
  # Display the list of dependencies if available
  foreach $dep(@deplist) {
    if (!($dep =~ 'no package')) {
      $dep =~ (s/-.*$//);
      if (Utils::vxif_list($dep, @deps) < 0) {
        $deprpms .= "$dep ";
      }
    }
  }    
  
  return Utils::vxif_desp($deprpms);
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) UPI;
#        2) the patch name;
#        3) the reference to the installation context;
#        4) the reference to the installed patch;
#        5) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 if older version is installed; 
#         2 if version or higher is installed.       
#
sub check_patch ($$$$$) {
  my ($self,$upi,$patch,$ctx,$installedpatch,$refresh) = @_;
  my ($pn,$pv,$iv);
  my ($rc) = 0;
  
  ($self->{PRODUCT}{$upi}->can("get_patch_id")) && ($patch = $self->{PRODUCT}{$upi}->get_patch_id($patch, $ctx));
  if ($patch =~ /-/) {
    ($pn,$pv) = split(/-/,$patch,2);
  } else {
    #$$ctx{LOGGER}->severe(Utils::_tr("Invalid patch ID, $patch. The patch ID must be in ######-## format.", 22, 1009, "$patch"));
    $$ctx{LOGGER}->severe("Invalid patch ID, $patch. The patch ID must be in ######-## format.");
    return -1;
  }
  
  # get the OS patches for the installation host
  if ($refresh || (!$$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES})) {
    $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES} = $self->get_patches($ctx);
  }
  
  $iv = $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES}{$pn};
  if ($iv) {
    $$installedpatch = "${pn}-${iv}";
    ($iv < $pv) && ($rc = 1);
    ($iv >= $pv) && ($rc = 2);
  }
  
  return $rc;
}

#
# Get all the patches for the given host.
#
# Input: 1) the reference of the installation context;
#
# Return: the reference to the patches hash.
#
sub get_patches ($) {
  my ($self,$ctx) = @_;
  my(%ph,@a,@op,$c,$j,$o,$p,$pkgs,$s,$v);
	

  $s=Utils::vxif_dor("$self->{CMD}{SHOWREV} -p", "", $ctx);
  @a=split(/^/,$s);
  foreach $c (@a) {
    ($j,$s,$j)=split(/\s/,$c,3);
    ($p,$v)=split(/-/,$s,2);
    # all patch keys are $p{######}=##
    $ph{$p}=$v unless ($v<$ph{$p});
    ($j,$o)=split(/Obsoletes: /,$c,2);
    ($o,$j)=split(/ Requires/,$o,2);
    @op=split(/, /,$o);
    for $o (@op) {
      # all obsoleted patch keys are $p{######-##}=######-##
      $ph{$o}="$p-$v";
    }
    # all package references are $p{$PKG} = [ ######, ###### ];
    chomp($c);
    ($j,$pkgs)=split(/ Packages: /,$c,2);
    if ($pkgs=~/VRTS/) {
      @op=split(/, /,$pkgs);
      for $o (@op) {
        $ph{$o}||=[];
	push(@{$ph{$o}},$p) if (_list($p,@{$ph{$o}})<0);
      }
    }
  }

  return \%ph;
}

#
# Check for patch dependencies.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
  my ($self,$upi,$pkg,$ctx) = (@_);
  my (@patchlist,$patch,$patchid);
  my ($rc,$installedpatchid,$msgl,$msgr);
  my ($result) = 1;
  
  if ($self->{PRODUCT}{$upi}->can("get_required_os_patches")) {
    @patchlist = $self->{PRODUCT}{$upi}->get_required_os_patches($pkg, $ctx);
  }
  
  if ($#patchlist >= 0 ) {
    Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 22, 1010, "$pkg"));
    foreach $patch (@patchlist) {
      $patchid = $patch;
      ($self->{PRODUCT}{$upi}->can("get_patch_id")) && ($patchid = $self->{PRODUCT}{$upi}->get_patch_id($patch, $ctx));
      $msgl = Utils::_tr("Checking OS patch ${patchid}", 22, 1011, "${patchid}");
      Utils::vxif_pbl($msgl);
      
      $rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
      if ($rc == -1) {
        $msgr = Utils::_tr("invalid patch ID, ${patchid}", 22, 1012, "${patchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 0) {
        $msgr = Utils::_tr("not installed", 22, 1013);
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 1) {
        $msgr = Utils::_tr("${installedpatchid} installed", 22, 1014, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } else {
        $msgr = Utils::_tr("${installedpatchid} installed", 22, 1014, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
      }
    }
  }
  
  return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;

	return 0;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;

	return 0;
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $rpm_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
	my $trg_upi = $ctx->get_target_upi();

	$logger->entering("CPILinux::install_native_package");

	# check to make sure the given package is not already exist on the target host
	my $result = $self->check_for_existing_package($ctx, $pkg, $pkgs);
	if (0 == $result) {
		return 0;
	} elsif (2 == $result) {
		return 1;
	}

	my $xloc;
	if (Utils::vxif_localsys($ctx) && !$self->{${trg_upi}}{$pkg}{COMPRESSED}) {
		# if local install, and package not compressed (or archived),
		# install directly from source location.
		$xloc = "SLOC";
	} else {
		$xloc = "TLOC";
	}

	# get rpm file to install
	my $package_info = $pkgs->{$pkg};
	my $rpmfile = $package_info->search_rpm();

        my $isdebian_os = is_debian($self, $ctx);
        if ( $isdebian_os ) {
                # Debian/Ubuntu filesystem we need to reference
                # the DEB package instead of the RPM
                $rpmfile =~ s/rpm$/deb/i;
        }

	# The package itself may not be at the top of the "pkgadd" directory.
	# It may, rather, be one or more levels down.  Use the EXTRACTION_PATH
	# configuration variable (if present) to see where to dig.
	#
	# maybe put this in common code
	my $install_dir = $self->{$trg_upi}{$pkg}{$xloc};
	my $extra_path = $package_info->search_extraction_path();
	if ($extra_path) {
		$install_dir .= "/${extra_path}";
	}

	# now add the package
        my $rpm_cmd = "";
        if ( $isdebian_os ) {
	        $rpm_cmd = "$self->{CMD}{DPKG} -i -G ${install_dir}/${rpmfile}";
        } else {
	        $rpm_cmd = "$self->{CMD}{RPM} -i --force --nodeps -v ${install_dir}/${rpmfile}";
        }
	Utils::vxif_dor($rpm_cmd, "", $ctx, undef, $rpm_logfile);

	my $rc = $self->is_package_install_successful($ctx, $pkg, $pkgs);

	return $rc;
}

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $rpm_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";

	$logger->entering("CPILinux::uninstall_native_package");

	# now remove the package
        if ( is_debian($self, $ctx) ) {
	        Utils::vxif_dor("$self->{CMD}{DPKG} -r ${pkg}", "", $ctx, undef, $rpm_logfile);
        } else {
	        Utils::vxif_dor("$self->{CMD}{RPM} -e  --nodeps -v ${pkg}", "", $ctx, undef, $rpm_logfile);
        }

	# if the package is successfully uninstalled, the version number null.
	if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg) eq "")) {
		#$$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from ${trg_host}", 22, 1015, "${pkg}", "${trg_host}"));
		$$ctx{LOGGER}->info("${pkg} successfully uninstalled from ${trg_host}");
		return 1;
	}

	#$$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from ${trg_host}", 22, 1016, "${pkg}", "${trg_host}"));
	$$ctx{LOGGER}->info("Unable to uninstall ${pkg} from ${trg_host}");

	return 0;
}

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%) {
  my ($self,$ctx) = @_;
  my ($msgl,$msgr);

  # this has become moot.  Leaving in for compatability. - jpk 10 Feb 06
  #$msgl = Utils::_tr("Checking for external dependencies");
  #Utils::vxif_pbl($msgl);
  #$msgr = Utils::_tr("all external dependencies satisfied");
  #Utils::vxif_pbr($msgl, $msgr);
  return 1; # stub this out for now
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
	my $self = shift;
	my ($ctx,$pkg,$fod,$pkgs) = @_;
	my $package_info = $pkgs->{$pkg};
	my $rpm = $package_info->search_rpm();
	
        if ( is_debian($self, $ctx) ) {
                $rpm =~ s/rpm$/deb/i;
        }

	$$ctx{LOGGER}->entering("CPILinux::get_native_package_file");
	$$ctx{LOGGER}->fine("RPM file for ${pkg} = ${rpm}");
	
	$$fod = "f";
	return $rpm;
}

1;