
package License;

require VxIF::Utils;

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->{MSG}{SITELIC}   = Utils::_tr("Site License", 13, 1000);
  $self->{MSG}{DEMO}      = Utils::_tr("Demo", 13, 1001);
  $self->{MSG}{PERMANENT} = Utils::_tr("Permanent", 13, 1002);
  $self->{MSG}{NFR}       = Utils::_tr("NFR", 13, 1003);
  $self->{MSG}{EXTENSION} = Utils::_tr("Extension", 13, 1004);
  $self->{MSG}{NODELOCK}  = Utils::_tr("Node Lock", 13, 1005);
  $self->{MSG}{UNKNOWN}   = Utils::_tr("Unknown", 13, 1006);
  $self->{MSG}{YESKEY}{L} = Utils::_tr("y", 13, 1010);
  $self->{MSG}{YESKEY}{U} = Utils::_tr("Y", 13, 1011);
  $self->{MSG}{NOKEY}{L}  = Utils::_tr("n", 13, 1012);
  $self->{MSG}{NOKEY}{U}  = Utils::_tr("N", 13, 1013);
}

#
# Get the mode of a mode based product.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the key hash;
#
# Return: the product key mode.
#
sub get_keymode (\%\%) {
  my ($self,$ctx,$rk) = @_;
  my ($mode) = $$ctx{TARGET}{UPI};
  
  $$ctx{LOGGER}->entering("License::get_keymode");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_key_mode")) {
    $mode = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_key_mode($ctx, $rk);
  }
  
  return "$mode";
}

#
# Set the license flags.
#
# Input: 1) the reference to the installation context;
#
sub set_license_flags (\%) {
  my ($self,$ctx) = @_;
  
  $$ctx{LOGGER}->entering("License::set_product_license_flags");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("set_product_license_flags")) {
    $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->set_product_license_flags($ctx);
  }
}

#
# Determine if special product license available.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if special product license available; 0 otherwise.
#
sub special_license_available (\%) {
  my ($self,$ctx) = @_;
  
  $$ctx{LOGGER}->entering("License::special_license_available");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("is_special_product_license_available")) {
    return $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->is_special_product_license_available($ctx);
  }
  
  return 0;
}

#
# Get the unsupported key modes.
#
# Input: 1) the reference to the installation context;
#
# Return: the product key mode.
#
sub get_unsupported_keymodes (\%\%) {
  my ($self,$ctx) = @_;
  my ($umodes);
  
  $$ctx{LOGGER}->entering("License::get_supported_keymodes");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_unsupported_product_key_modes")) {
    $umodes = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_unsupported_product_key_modes($ctx);
  }
  
  return $umodes;
}

#
# Get the product license modes.
#
# Input: 1) the reference to the installation context;
#
# Return: the reference to the list of product license modes..
#
sub get_license_modes (\%\%) {
  my ($self,$ctx) = @_;
  my ($pmodes);
  
  $$ctx{LOGGER}->entering("License::get_license_modes");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_license_modes")) {
    $pmodes = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_license_modes($ctx);
  }
  
  return $pmodes;
}

#
# Get the name of the product key mode.
#
# Input: 1) the reference to the installation context;
#        2) the product key mode;
#
# Return: the name of the product key mode.
#
sub get_keymode_name (\%$) {
  my ($self,$ctx,$mode) = @_;
  my ($name) = $$ctx{TARGET}{UPI};
  
  $$ctx{LOGGER}->entering("License::get_keymode_name");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_key_mode_name")) {
    $name = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_key_mode_name($ctx, $mode);
  }
  
  return "$name";
}

#
# Validate product key.
#
# Input: 1) the reference to the installation context;
#        2) the product key;
#
# Return: 1 if the given registration key is a valid product registration key;
#         -1 if the validate_product_key() interface is not implemented;
#         0 otherwise.
#
sub validate_product_key (\%\%) {
  my ($self,$ctx,$key) = @_;
  my ($pk) = -1;
  
  $$ctx{LOGGER}->entering("License::validate_product_key");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("is_product_key")) {
    $pk = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->is_product_key($ctx, $key);
  }
  
  return $pk;
}

#
# Get the product license feature.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license feature.
#
sub get_license_feature (\%) {
  my ($self,$ctx) = @_;
  my ($feature);
  
  $$ctx{LOGGER}->entering("License::get_license_feature");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_license_feature")) {
    $feature = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_license_feature($ctx);
  }
  
  return $feature;
}

#
# Get the product license ID.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license ID.
#
sub get_license_id (\%) {
  my ($self,$ctx) = @_;
  my ($id);
  
  $$ctx{LOGGER}->entering("License::get_license_id");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_license_id")) {
    $id = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_license_id($ctx);
  }
  
  return $id;
}

#
# Determine if the given license mode and license type are valide.
#
# This interface is optional. By default, VxIF assumes the given license mode and 
# license type are valid.
#
# Input: 1) the reference to the installation context;
#        2) license mode;
#        3) license type;
#
# Return: 1 if the given license mode and license type are valid; 0 otherwise.
#
sub check_proudct_license (\%$$) {
  my ($self,$ctx,$mode,$type) = @_;
  
  $$ctx{LOGGER}->entering("License::check_proudct_license");
    
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("check_proudct_license")) {
    return $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->check_proudct_license($ctx);
  }
    
  return 1;
}

#
# Get the product license UPI.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license UPI.
#
sub get_license_upi (\%) {
  my ($self,$ctx) = @_;
  my ($upi);
  
  $$ctx{LOGGER}->entering("License::get_license_upi");
  
  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("get_product_license_upi")) {
    $upi = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_license_upi($ctx);
  }
  
  return $upi;
}

#
# Determine the key type (PERMANENT, PERMANENT_NODE_LOCK, DEMO, SITELIC, NFR)
#
# Input: 1) the reference to the installation context;
#        2) the reference to the key hash;
#        2) the product key mode;
#
# Return: the product key type.
# 
sub get_keytype (\%\%;$) {
  my ($self,$ctx,$rk,$mode) = @_;
  my ($ayn,$lt,$type,$quit_key_handler,$msg);
  my ($upi) = $$ctx{TARGET}{UPI};
  
  $$ctx{LOGGER}->entering("License::get_keytype");
  
  $lt=$$rk{"License Type"};
  $$ctx{LOGGER}->finer("License Type = ${lt}");
  $type = ($lt=~/SITE/) ? $self->{MSG}{SITELIC} :
          ($lt=~/DEMO/) ? $self->{MSG}{DEMO} :
          ($lt=~/PERMANENT/) ? $self->{MSG}{PERMANENT} :
          ($lt=~/NFR/) ? $self->{MSG}{NFR} : $self->{MSG}{UNKNOWN};
  $type .= " $self->{MSG}{EXTENSION}" if ($lt=~/EXTENSION/);
  $type .= " $self->{MSG}{NODELOCK}" if ($lt=~/NODE_LOCK/);
  $$ctx{LOGGER}->finer("type = ${type}");
  # need feature/product key check here

  if (($mode) && (($$rk{'License Type'} eq "SITE") ||
      ($$rk{'License Type'} eq "DEMO"))) {
    push(@{$$ctx{LICENSING}{ALLSYSKEYS}},  $$rk{'EnteredKey'});
  } elsif ($mode && ($lt !~ / $self->{MSG}{PERMANENT}/) && ($lt !~ /$self->{MSG}{NODE_LOCK}/) && ($lt !~ /$self->{MSG}{UNKNOWN}/)) {
    Utils::vxif_pl();
    $quit_key_handler = sub {
      $self->cleanup($ctx);
    };
    $upi = $self->get_keymode_name($ctx, $mode);
    $msg = Utils::_tr("Do you want to register this ${upi} ${type} key on all systems?", 13, 1007, "${upi}", "${type}");
    $ayn = Utils::vxif_ayny($msg, "", "", $quit_key_handler);
    push(@{$$ctx{LICENSING}{ALLSYSKEYS}}, $$rk{'EnteredKey'}) if ($ayn eq $self->{MSG}{YESKEY}{U});
  }
  
  return "$type";
}

# 
# Register a license key.
#
# Input: 1) the reference to the installation context;
#        2) the license key;
#        3) the reference to the registration keys hash;
#
# Return: 1 if successful; 0 otherwise.
#
sub register_key (\%$;\%) {
  my ($self,$ctx,$key,$rk) = (@_);
  my ($host) = $$ctx{TARGET}{HOST};
  my ($regk);
  
  $$ctx{LOGGER}->entering("License::register_key");
  
  push(@{$$ctx{HOST}{$host}{KEYS}}, $key) if (!$$ctx{RESPONSEFILE});
  if ($rk) {
    Utils::vxif_pl(Utils::_tr("Registering $$rk{'Product Name'} $$rk{'License Type'} key on ${host}", 13, 1014, "$$rk{'Product Name'}", "$$rk{'License Type'}", "${host}"));
  } else {
    Utils::vxif_pl(Utils::_tr("Registering ${key} on ${host}", 13, 1015, "${key}", "${host}"));
  }
  $regk = Utils::vxif_dor("$$ctx{TARGET}{CMD}{VXLICINST} -k $key", "", $ctx);
  if ($regk=~/ successfully /) {
    $$ctx{LICENSING}{KEYREG}++;
    $$ctx{LICENSING}{RKEY}{$host}++;
  } else {
    if ($regk =~ / Duplicate /) {
      Utils::vxif_pl(Utils::_tr("Duplicate License key detected", 13, 1016));
    } else {
      Utils::vxif_pl("${regk}");
      $$ctx{LOGGER}->warning(Utils::_tr("license key ${key} did not successfully validate", 13, 1017, "${key}"));
    }
  }
  
  return 1;
}

#
# Determine whether a feature is Enabled within a key.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the registration keys hash;
#        3) the license feature;
#
# Return: 1 if the given feature is Enabled; 0 otherwise.
#
sub license_feature (\%\%$) {
  my ($self,$ctx,$rf,$feat) = @_;
  my ($k);
  
  $$ctx{LOGGER}->entering("License::license_feature");
  
  foreach $k (keys(%$rk)) {
    $$ctx{LOGGER}->finer("k=${k}");
    return 1 if ((($k eq $feat) || ($k=~/^$feat BASE/i)) && ($$rk{$k} eq "Enabled"));
  }
  
  # Need to accommodate automatic licenses for VM for PHOTON and SSA
  $$ctx{LOGGER}->finer("UPI=$$ctx{TARGET}{UPI}");
  $$ctx{LOGGER}->finer("PHOTON=$$rk{PHOTON}");
  $$ctx{LOGGER}->finer("SSA=$$rk{SSA}");
  return 1 if (($$ctx{TARGET}{UPI} eq "VM") && ( $$rk{PHOTON} eq "Enabled" || $$rk{SSA} eq "Enabled"));
  
  return 0;
}

#
# Determine whether a license identifier exists in a key as a feature value.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the registration keys hash;
#        3) the license identifier;
#
# Return: 1 if the given key identifiere exist; 0 otherwise.
#
sub license_id (\%\%$$) {
  my ($self,$ctx,$rk,$id,$licupi) = @_;
  my ($k);
  
  $$ctx{LOGGER}->entering("License::license_id");
  
  return 0 if ($$rk{UPI} ne $licupi);
  $id = $licupi if (!$id);
  
  $$ctx{LOGGER}->finer("id=${id}");
  foreach $k (keys(%$rk)) {
    $$ctx{LOGGER}->finer("$$rk{$k}=${id}");
    return 1 if ($$rk{$k} eq $id);
  }
  
  return 0;
}

#
# This is for the rare case where there are more than one valid product keys
# on the system.  Dominant key is ranked by Entered over Found,
# then by Version of key
# then SITE over PERMANENT over NFR over DEMO,
# then MODE ranking defined by the product team
#
# Input: 1) the reference to the installation context;
#        2) the reference to the registration keys hash;
#
# Return: 1 if successful; 0 otherwise.
#
sub rank_pkeys (\%\%) {
  my ($self,$ctx,$k1) = @_;
  my ($k2,$m1,$m2,$r1,$r2);
  my ($umodes) = $self->get_unsupported_keymodes($ctx);
  my ($pmodes) = $self->get_license_modes($ctx);
  my ($ltrank) = [ qw(DEMO NFR PERM SITE) ]; # rank order
  
  $$ctx{LOGGER}->entering("License::rank_pkeys");
  
  return 1 if (!$$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}});
  $k2 = $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}};
  # First rank by entered/found, can't go from Entered to Found
  return 1 if (($$k1{EnteredKey}) && (!$$k2{EnteredKey}));
  # Then product mode, if one is now unsupported
  $m1 = $self->get_keymode($ctx, $k1);
  $m2 = $self->get_keymode($ctx, $k2);
  $r1 = Utils::vxif_tlist($m1,@{$umodes});
  $r2 = Utils::vxif_tlist($m2,@{$umodes});
  $$ctx{LOGGER}->finer("usmode key1=$r1 key2=$r2");
  return 1 if ($r2>=0);
  return 0 if ($r1>=0);
  # Then rank by version
  $$ctx{LOGGER}->finer("version key1=$$k1{'Version'} key2=$$k2{'Version'}");
  return 1 if ($$k1{"Version"}>$$k2{"Version"});
  return 0 if ($$k2{"Version"}>$$k1{"Version"});
  # Then rank by license type
  $r1 = Utils::vxif_tlist($$k1{"License Type"}, @{$ltrank});
  $r2 = Utils::vxif_tlist($$k2{"License Type"}, @{$ltrank});
  $$ctx{LOGGER}->finer("lictype $$k1{'License Type'}=$r1 $$k2{'License Type'}=$r2");
  return 1 if ($r1>$r2);
  return 0 if ($r2>$r1);
  # Then finally by product mode
  $m1 = $self->get_keymode($ctx, $k1);
  $m2 = $self->get_keymode($ctx, $k2);
  $r1 = Utils::vxif_list($m1,@{$pmodes});
  $r2 = Utils::vxif_list($m2,@{$pmodes});
  $$ctx{LOGGER}->finer("pmode $$k1{Mode} $m1=$r1 $$k2{Mode} $m2=$r2");
  return 0 if ($r2>$r1);
  return 1;
}

#
# Push the key into @{$COMM{KEYS}{$SYS}} (all keys)
# and set $COMM{PKEY}{$SYS} it is a valid product key
#
# Input: 1) the reference to the installation context;
#        2) the registration key;
#
# Return: 1 if successful; 0 otherwise.
#
sub push_key {
  my ($self,$ctx,$rk) = @_;
  my ($pk,$lf,$lid,$lupi);
  
  $$ctx{LOGGER}->entering("License::push_key");
  
  return 0 if (!$$rk{'License Key'});
  $pk = $self->validate_product_key($ctx, $rk);
  $lf = $self->get_license_feature($ctx);
  $lid = $self->get_license_id($ctx);
  $lupi = $self->get_license_upi($ctx);
  $pk = ($pk >= 0) ? $pk : 
        $lf ? $self->license_feature($ctx, $rk, $lf) :
        $lid ? $self->license_id($ctx, $rk, $lid, $lupi) :
        ($$rk{UPI} eq $lupi) ? 1 : 0;
  $$ctx{LOGGER}->finer("pk=${pk}\nlf=${lf}\nlid=${lid}\nlupi=${lupi}");
  push(@{$$ctx{LICENSING}{KEYS}{$$ctx{TARGET}{HOST}}}, Utils::vxif_duph($rk));
  push(@{$$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}}}, Utils::vxif_duph($rk)) if (($pk) && ($self->rank_pkeys($ctx, $rk)));
  $self->{$$ctx{TARGET}{UPI}}{MLIC} = 1;     
  
  return 1;
}

#
# Determine the UPI associated with the given product name. Hopefully, the product name is not
# localized.
#
# Input: 1) the reference to the installation context;
#        2) the product name;
#
# Return: the UPI associated with the given product name, or "UNKNOWN" if no UPI match the 
#         given product name.
#
sub get_upifromproductname {
  my ($self,$ctx,$pn) = @_;
  my ($upi);
  
  foreach $upi (keys(%{$self->{PRODUCT}})) {
    $$ctx{TARGET}{UPI} = $upi;
    return $upi if ($self->{PRODUCT}{$upi}->can("is_product_name") && $self->{PRODUCT}{$upi}->is_product_name($ctx, $pn));
  }
  
  $$ctx{LOGGER}->error(Utils::_tr("Cannot determine UPI from Product Name: ${pn}", 13, 1018, "${pn}"));
  return "UNKNOWN";
}

#
# Check for license on target host.
#
# Input: 1) the reference to the installation context;
#        2) the license key;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_license (\%;$) {
  my ($self,$ctx,$key) = @_;
  my (%k,$v,$va,$vb,$vc,$lr,$f,$i,@a,$mode,$type,$msgl,$msgr,$name);
  
  $$ctx{LOGGER}->entering("License::check_license");
  
  $msgl = Utils::_tr("Checking $$ctx{TARGET}{UPI} license key on $$ctx{TARGET}{HOST}", 13, 1008, "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}");
  (!$key)&&(Utils::vxif_pbl($msgl));
  
  $v = Utils::vxif_dor("$$ctx{TARGET}{CMD}{VXLICREP} -v", "", $ctx);
  $$ctx{LOGGER}->info("vxlicrep version = ${v}\n");
  ($va,$vb,$vc)=split(/\./,$v);
  $v="$va.$vb";
  $$ctx{LOGGER}->finer("v = ${v}");
  
  $lr = ($key) ? Utils::vxif_dor("$$ctx{TARGET}{CMD}{VXLICREP} -k ${key}", "", %$ctx) : ($v >= 3.02) ? Utils::vxif_dor("$$ctx{TARGET}{CMD}{VXLICREP} -i", "", %$ctx) : Utils::vxif_dor("$self->{CMD}{VXLICREP}", "", $ctx);
  $$ctx{LOGGER}->finer("\n${lr}\n");
  if (($key) && (($lr !~ /Product Name/) || ($lr =~ /ERROR V-/))) {
    $msgl = Utils::_tr("$key is not a valid $$ctx{TARGET}{UPI} license key for system $$ctx{TARGET}{HOST}.", 13, 1009, "$key", "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}");
    Utils::vxif_pl($msgl);
    $$ctx{LOGGER}->error($msgl);
    return 0;
  }
  
  (@a) = split(/^/,$lr);
  foreach $lr (@a) {
    chomp($lr);
    $$ctx{LOGGER}->finer("lr = ${lr}");
    next unless ($lr =~ / = /);
    ($f,$i)=split(/= /, $lr, 2);
    $f=Utils::vxif_desp($f);
    $f=~s/#.*$//;
    $i=Utils::vxif_desp($i);
    $$ctx{LOGGER}->finer("k{$f}=$i");
    if ($f eq "License Key") {
      if (($key) && ($k{'License Type'})) {
        $mode = $self->get_keymode($ctx, \%k);
        $type = $self->get_keytype($ctx, \%k, $mode);
        $self->register_key($ctx, $key, \%k);
        $key="";
      }
      $self->push_key($ctx, \%k);
      %k=( $f => $i );
      $k{EnteredKey}=$key if ($key);
    } elsif ($f eq "Product Name") {
      next if ($k{'Product Name'});
      $k{$f}=$i;
      $i=~s/VERITAS //;
      $upi = $self->get_upifromproductname($ctx, $i);
      $k{UPI} = $upi;
    } elsif (($f eq "Site License") && ($i eq "YES")) {
      $k{"License Type"}="SITE";
    } else {
      $k{$f}||=$i;
      # handling the two exceptions here now
      # should make this scalable for others
      # It would need LICFEATURE and LICID to be defined in reverse
      $self->{VVR}{MLIC}=1 if (($f eq "VVR") && ($i eq "Enabled"));
      $self->{SPCQS}{MLIC}=1 if (($f =~ /SPC Lite/) && ($i eq "Enabled"));
      $self->{TD}{MLIC}=1 if (($f eq "Mode") && ($i eq "TRAFFIC_DIRECTOR"));
    }
  }
  if ($key) {
    $mode = $self->get_keymode($ctx, \%k);
    $type = $self->get_keytype($ctx, \%k, $mode);
    $self->register_key($ctx, $key, \%k);
  }
  $self->push_key($ctx, \%k);
  if (!$rkey) {
    if ($$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}}) {
      $type = $self->get_keytype($ctx, $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}});
      $mode = $self->get_keymode($ctx, $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}});
      $name = $self->get_keymode_name($ctx, $mode);
      
      Utils::vxif_pbr($msgl, "$name $type");
      if (!$self->check_proudct_license($ctx, $mode, $type)) {
        $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}} = "";
      }
    } else {
      $msgr = Utils::_tr("not licensed", 13, 1019);
      Utils::vxif_pbr($msgl, $msgr);
    }
  } 
}

#
# Ask for system license keys until system is licensed and they say no more.
#
# Input: 1) the reference to the installation context;
#
sub license_system (\%$) {
  my ($self,$ctx,$first) = @_;
  my ($ayn,$key,$mode,$helpmsg,$modename,$mname,$question,$ask_license,$validator);
  my ($prodname) = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->get_product_title($ctx);
  
  $$ctx{LOGGER}->entering("License::license_system");
  
  $ask_license = sub (\%) {
    my ($ctx,$prodname) = @_;
    my ($question,$helpmsg);
    
    $helpmsg = Utils::_tr("${prodname} is already licensed on this system.  Entering a new license key is only necessary if you want to install a different version of $$ctx{TARGET}{UPI} or license additional features.", 13, 1020, "${prodname}", "$$ctx{TARGET}{UPI}");
    $question = Utils::_tr("Do you want to enter another license key for $$ctx{TARGET}{HOST}?", 13, 1021, "$$ctx{TARGET}{HOST}");
    return Utils::vxif_aynn($question, $helpmsg, "", "");
  };
  
  while (1) {
    if ($$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}}) {
      return if (($$ctx{TASK} eq "configure") || ($$ctx{TASK} eq "upgrade"));
      Utils::vxif_pl();
      # we should check all keys here, just in case
      $mode = (!$first) ? $self->get_keymode($ctx, $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}}) : $$ctx{LICENSING}{PMODE};
      if ($mode ne $$ctx{LICENSING}{PMODE}) {
        $modename = $self->get_keymode_name($ctx, $mode);
        $mname = $self->get_keymode_name($ctx, $$ctx{LICENSING}{PMODE});
        $$ctx{LOGGER}->warning(Utils::_tr("The ${modename} key entered for $$ctx{TARGET}{HOST} does not match the ${mname} keys used with the rest of the cluster.", 13, 1022, "${modename}", "$$ctx{TARGET}{HOST}", "${mname}"));
      } else {
        return 1 if (&$ask_license($ctx, $prodname) eq $self->{MSG}{NOKEY}{U});
      }
    } elsif ($self->special_license_available($ctx)) {
      return 1 if (&$ask_license($ctx, $prodname) eq $self->{MSG}{NOKEY}{U});
    } else {
      $helpmsg = Utils::_tr("${prodname} is a product that requires a license key to be registered on each system for it to operate.  Feature keys may also be added to provide additional functionality.", 13, 1023, "${prodname}");
    }
    
    if (($$ctx{LICENSING}{RKEY}{$$ctx{TARGET}{HOST}}) &&
        (!$$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}}) &&
        (!$$ctx{LICENSING}{FLAG}{VM_SUN_LIC_GRANTED}{$$ctx{TARGET}{HOST}})) {
      Utils::vxif_pl(Utils::_tr("$$ctx{TARGET}{UPI} is still unlicensed on $$ctx{TARGET}{HOST}\n", 13, 1024, "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}"));
    }
    $validator = sub {
      my ($ans) = @_;
      
      if ($ans eq "") {
        return 1 if ($$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}});
      }
      
      return 1;
    };
    $question = Utils::_tr("Enter a $$ctx{TARGET}{UPI} license key for $$ctx{TARGET}{HOST}:", 13, 1025, "$$ctx{TARGET}{UPI}", "$$ctx{TARGET}{HOST}");
    $key = Utils::vxif_ask($question, "", $helpmsg, $validator, "", "", "");
    $self->check_license($ctx, $key) if ($key);
  }
}

#
# Check licensing on target host.
#
# Input: 1) the reference to the installation context;
#        2) first of multiple systems?
#
# Return: 1 if licensing check is successful; 0 otherwise.
#
sub check_product_licensing_on_target_host {
  my ($self,$ctx,$first) = @_;
  my ($key);
  
  $$ctx{LOGGER}->entering("License::check_product_licensing_on_target_host");
  
  foreach $key (@{$$ctx{LICENSING}{ALLSYSKEYS}}) {
    $self->register_key($ctx, $key);
  }
  $self->check_license($ctx);
  $self->license_system($ctx,$first);
  if ($first) {
    $$ctx{LICENSING}{PMODE} = $self->get_keymode($ctx, $$ctx{LICENSING}{PKEY}{$$ctx{TARGET}{HOST}});
  }
  
  $self->set_license_flags($ctx);
  
  return 1;
}

1;
