package VxIF::Interface::Utils::Dictionary;
use strict;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;

	$self->set_defs_aref([]);
	$self->set_defs_href({});
}

sub get_defs_aref {
	my $self = shift;
	my $defs_aref = $self->{defs_aref};

	return $defs_aref;
}

sub set_defs_aref {
	my $self = shift;
	my ($defs_aref) = @_;

	$self->{defs_aref} = $defs_aref;
}

sub get_defs_href {
	my $self = shift;
	my $defs_href = $self->{defs_href};

	return $defs_href;
}

sub set_defs_href {
	my $self = shift;
	my ($defs_href) = @_;

	$self->{defs_href} = $defs_href;
}

sub add_definition {
	my $self = shift;
	my ($definition) = @_;
	my $defs_aref = $self->get_defs_aref();
	my $defs_href = $self->get_defs_href();
	my $name = $definition->{name};

	$defs_href->{$name} = $definition;
	push(@$defs_aref, $name);
}

sub add_definitions {
	my $self = shift;
	my ($definitions) = @_;
	my $defs_aref = $self->get_defs_aref();
	my $defs_href = $self->get_defs_href();

	for my $definition (@$definitions) {
		$self->add_definition($definition);
	}
}

sub from_href {
	my $self = shift;
	my ($href) = @_;

	for my $lhs_name (keys(%$href)) {
		my $definition = {
			name => $lhs_name,
			sequence => [],
		};
		my $rhs_val = $href->{$lhs_name};
		my @seq_arr = split(/\s+/, $rhs_val);
		for my $rhs_elt (@seq_arr) {
			my $ref = {
				name => $rhs_elt,
				type => undef,
			};
			push(@{$definition->{sequence}}, $ref);
		}
		$self->add_definition($definition);
	}
}

sub lookup {
	my $self = shift;
	my ($name) = @_;
	my $defs_href = $self->get_defs_href();
	my $definition = $defs_href->{$name};

	return $definition;
}

# expand while type == definition (i.e. type == package ==> stop)
sub expand {
	my $self = shift;
	my ($name) = @_;
	my $expanded = undef; # no benefit of doubt
	my $seen = {}; # no recursive definitions
	my $stack = []; # you know what a stack is
	my $def = $self->lookup($name); # see if it's in the dictionary.

	if ($def) {
		my $seq = $def->{sequence};

		# it's in the dictionary.
		# push onto stack and away we go.
		unshift(@$stack, {ix => 0, seq => $seq});

		$expanded = [];
	}

	while (scalar(@$stack)) {
		my $seq_exp = $stack->[0];
		my $seq = $seq_exp->{seq};
		my $ref = $seq->[$seq_exp->{ix}];
		my $name = $ref->{name};

		my $new_def = $self->lookup($name);
		#if ($ref->{type} eq "definition") {
		if ($new_def) {
			if ($seen->{$name}) {
				die "Recursive definitions not allowed (${name})\n";
			}

			$seen->{$name} = 1;

			# another sequence to expand
			#my $new_def = $self->lookup($name);
			my $new_seq = $new_def->{sequence};
			unshift(@$stack, {ix => 0, seq => $new_seq});
		} else {
			# type == package
			# type == section, if anything
			push(@$expanded, $ref->{name});
			$seq_exp->{ix}++;
			while ($seq_exp->{ix} >= scalar(@{$seq_exp->{seq}})) {
				# done with this one, so pop stack
				shift(@$stack);
				if (!scalar(@$stack)) {
					last;
				}
				$seq_exp = $stack->[0];
				$stack->[0]{ix}++;
			}
		}
	}

	return $expanded;
}

sub list {
	my $self = shift;
	my ($indent) = @_;
	my $defs_aref = $self->get_defs_aref();
	my $defs_href = $self->get_defs_href();
	my $ind = " " x $indent;

	print("\n");

	my $title = "Definitions";
	my $underline = "-" x length($title);
	print("${ind}${title}\n");
	print("${ind}${underline}\n");

	print("\n");

	for my $name (@$defs_aref) {
		my $definition = $defs_href->{$name};
		print("${ind}  ${name} =");
		my $sequence = $definition->{sequence};
		for my $reference (@$sequence) {
			my $ref_name = $reference->{name};
			print(" ${ref_name}");
		}
		print("\n");

		my $expanded = $self->expand($name);
		my $exp_str = join(" ", @$expanded);
		print("${ind}    ${exp_str}\n");
	}

	print("\n");
}

sub compute_symbol_status {
	my $self = shift;
	my $symstat = {};

	for my $def_name (@{$self->{defs_aref}}) {
		my $def = $self->{defs_href}{$def_name};
		# defined name encountered
		if (!exists($symstat->{$def_name})) {
			# first encounter.  catalog it.
			$symstat->{$def_name} = {};
		}
		# either way, it's defined
		$symstat->{$def_name}{defined} = 1;
		# traverse (rhs of) definition
		my $seq = $def->{sequence};
		for my $ref (@$seq) {
			my $elt_name = $ref->{name};
			# referenced name encountered
			if (!exists($symstat->{$elt_name})) {
				# first encounter.  catalog it.
				$symstat->{$elt_name} = {
					# only a reference (so far)
					defined => 0,
				};
			}
		}
	}

	# result of analysis
	return $symstat;
}

1;

