package VxIF::Interface::Utils::Config;
use strict;

use Text::ParseWords;
use VxIF::Utils;

use constant LINE_TYPE_UNKNOWN => "unknown";
use constant LINE_TYPE_SPACES => "spaces";
use constant LINE_TYPE_COMMENT => "comment";
use constant LINE_TYPE_KEYVAL_ASGN => "keyval_asgn";
use constant LINE_TYPE_HEREDOC_ASGN => "heredoc_asgn";
use constant LINE_TYPE_HEREDOC_LINE => "heredoc_line";
use constant LINE_TYPE_SECTION => "section";

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;

	$self->set_values({});
	$self->set_sections({});
	$self->set_keep_lines(0);
	$self->set_line_infos([]);
	$self->set_do_quotes(1);
}

sub get_line_infos {
	my $self = shift;
	my $line_infos = $self->{line_infos};

	return $line_infos;
}

sub set_line_infos {
	my $self = shift;
	my ($line_infos) = @_;

	$self->{line_infos} = $line_infos;
}

sub get_keep_lines {
	my $self = shift;
	my $keep_lines = $self->{keep_lines};

	return $keep_lines;
}

sub set_keep_lines {
	my $self = shift;
	my ($keep_lines) = @_;

	$self->{keep_lines} = $keep_lines;
}

sub get_do_quotes {
	my $this = shift;
	my $do_quotes = $this->{do_quotes};

	return $do_quotes;
}

sub set_do_quotes {
	my $this = shift;
	my ($do_quotes) = @_;

	$this->{do_quotes} = $do_quotes;
}

sub get_values {
	my $self = shift;
	my $values = $self->{values};

	return $values;
}


sub set_values {
	my $self = shift;
	my ($values) = @_;

	$self->{values} = $values;
}
sub get_sections {
	my $self = shift;
	my $sections = $self->{sections};

	return $sections;
}


sub set_sections {
	my $self = shift;
	my ($sections) = @_;

	$self->{sections} = $sections;
}

#
# Read config file.
#
# Input: 1) The name of the file.
#
# Return: 1 if successful, 0 if error.
#
sub from_file {
	my $self = shift;
	my ($resp_fnam) = @_;
	my $values = $self->get_values();
	my $sections = $self->get_sections();
	my $hash_symbol = '#';
	my $keep_lines = $self->get_keep_lines();
	my $line_infos = $self->get_line_infos();
	my $do_quotes = $self->get_do_quotes();
	my $current = {
		currsect_name => "",
		heredoc_key => "",
		heredoc_contents => undef,
	};

	my $rc = open(CONF, $resp_fnam);

	if (!$rc) {
		die Utils::_tr("Cannot read response file ${resp_fnam}.", 48, 1000, "${resp_fnam}");
	}

	while (1) {
		my $new_rec = <CONF>;
		my $line_info = {
			rec => $new_rec,
			type => LINE_TYPE_UNKNOWN,
		};
		last unless $line_info->{rec};
		# Well, chomp won't do this, and chop is
		# dangerous, especially between platforms.
		# And we *will* be across multiple platforms.
		# So, do the following chomp-like variation.
		$line_info->{rec} =~ s/[\r\n]+$//;

		if ($keep_lines) {
			push(@$line_infos, $line_info);
		}

		if ($line_info->{rec} =~ /^\s*$/) {
			# spaces only
			$line_info->{type} = LINE_TYPE_SPACES;
			next;
		} elsif ($line_info->{rec} =~ /^\s*${hash_symbol}/) {
			# a comment
			$line_info->{type} = LINE_TYPE_COMMENT;
			next;
		} elsif ($line_info->{rec} =~ /^\s*(.*\S)\s*=\s*(.*\S)?\s*$/) {
			# a key/value pair assignment
			my $key = $1;
			my $val = $2;
			if (!defined($val)) {
				$val = "";
			}

			if ($keep_lines) {
				$line_info->{type} = LINE_TYPE_KEYVAL_ASGN;
				$line_info->{key} = $key;
				$line_info->{currsect_name} = $current->{currsect_name};
			}

			# see if "here" document
			if ($val =~ /<<\s*(\S+)/) {
				# start of "here" document
				$current->{heredoc_key} = $1;
				$current->{heredoc_contents} = "";

				if ($keep_lines) {
					$line_info->{type} = LINE_TYPE_HEREDOC_ASGN;
				}

				while (my $hd_rec = <CONF>) {
					# same big chomp
					$hd_rec =~ s/[\r\n]+$//;

					if ($keep_lines) {
						$line_info = {
							rec => $hd_rec,
							type => LINE_TYPE_HEREDOC_LINE
							};
						push(@$line_infos, $line_info);
					}

					# process "here" document
					if ($hd_rec =~ /^$current->{heredoc_key}\s*$/) {
						# end of "here" document
						last;
					} else {
						# more to "here" document
						# accumulate into string
						if ($current->{heredoc_contents}) {
							$current->{heredoc_contents} .= "\n${hd_rec}";
						} else {
							$current->{heredoc_contents} = "${hd_rec}";
						}
					}
				}

				$val = $current->{heredoc_contents};
			} else {
				# not a "here" document
				my $dq = '"';
				if ($do_quotes) {
					# values can be in double quotes
					if ($val =~ /^${dq}(.*)${dq}$/) {
						$val = $1;
					}
				}
			}

			my $asgn_href = undef;

			# create the entry in appropriate place
			if ($current->{currsect_name}) {
				my $currsect = $sections->{$current->{currsect_name}};
				$asgn_href = $currsect;
			} else {
				$asgn_href = $values;
			}
			#my @levels = split(/\./, $key);
			my @levels = Text::ParseWords::parse_line('\.', 0, $key);
			my $num_levels = scalar(@levels);
			my $h = $asgn_href;
			for (my $i = 0; $i < $num_levels; $i++) {
				my $sub_key = $levels[$i];
				if ($i < ($num_levels - 1)) {
					if (!exists($h->{$sub_key})) {
						$h->{$sub_key} = {};
					}
					$h = $h->{$sub_key};
				} else {
					$h->{$sub_key} = $val;
				}
			}
		} elsif ($line_info->{rec} =~ /^\s*\[\s*(.*)\s*\]\s*$/) {
			# a section
			$line_info->{type} = LINE_TYPE_SECTION;
			$current->{currsect_name} = $1;
			if (!exists($sections->{$current->{currsect_name}})) {
				$sections->{$current->{currsect_name}} = {};
			}
		} else {
			# nothing we know about
			next;
		}
	}

	close(CONF);
}

1;

