package VxIF::Interface::Manager;
use strict;

use VxIF::Interface::Object;
@VxIF::Interface::Manager::ISA = qw(VxIF::Interface::Object);

#
# this has some "top level" objects available,
# such as:
# - Localizer
# - Logger
# - Class Factory
# - Events
#

sub new {
	my $pkg = shift;
	my $this = {};

	bless($this, $pkg);

	$this->init(@_);

	return $this;
}

sub init {
	my $this = shift;
	my ($class_factory) = @_;

	$this->VxIF::Interface::Object::init();

	$this->set_class_factory($class_factory);

	my $events = $class_factory->create_events();
	$this->set_events($events);

	my $status_bar = $class_factory->create_status_bar("ID_STATUS");
	$this->set_status_bar($status_bar);

	my $localizer = $class_factory->create_localizer();
	$this->set_localizer($localizer);

	my $logger = $class_factory->create_logger();
	$this->set_logger($logger);

	my $text_input = $class_factory->create_text_input("ID_TEXT_INPUT");
	$this->set_text_input($text_input);

	my $product_ops = $class_factory->create_product_ops();
	$this->set_product_ops($product_ops);

	my $package_ops = $class_factory->create_package_ops();
	$this->set_package_ops($package_ops);

	my $platform_info = $class_factory->create_platform_info();
	$this->set_platform_info($platform_info);

	my $system_info = $class_factory->create_system_info($platform_info);
	$this->set_system_info($system_info);

	# 
	$this->set_registered_interfaces({});
	$this->set_registered_classes({});
}

sub get_vxif_context {
	my $this = shift;
	my $vxif_context = $this->{vxif_context};

	return $vxif_context;
}

sub set_vxif_context {
	my $this = shift;
	my ($vxif_context) = @_;

	$this->{vxif_context} = $vxif_context;
}

sub get_product_ops {
	my $this = shift;
	my $product_ops = $this->{product_ops};

	return $product_ops;
}

sub set_product_ops {
	my $this = shift;
	my ($product_ops) = @_;

	$this->{product_ops} = $product_ops;
}

sub get_package_ops {
	my $this = shift;
	my $package_ops = $this->{package_ops};

	return $package_ops;
}

sub set_package_ops {
	my $this = shift;
	my ($package_ops) = @_;

	$this->{package_ops} = $package_ops;
}

sub get_system_info {
	my $this = shift;
	my $system_info = $this->{system_info};

	return $system_info;
}

sub set_system_info {
	my $this = shift;
	my ($system_info) = @_;

	$this->{system_info} = $system_info;
}

sub get_platform_info {
	my $this = shift;
	my $platform_info = $this->{platform_info};

	return $platform_info;
}

sub set_platform_info {
	my $this = shift;
	my ($platform_info) = @_;

	$this->{platform_info} = $platform_info;
}

sub get_client_data {
	my $this = shift;
	my $client_data = $this->{client_data};

	return $client_data;
}

sub set_client_data {
	my $this = shift;
	my ($client_data) = @_;

	$this->{client_data} = $client_data;
}

sub get_localizer {
	my $this = shift;
	my $localizer = $this->{localizer};

	return $localizer;
}

sub set_localizer {
	my $this = shift;
	my ($localizer) = @_;

	$this->{localizer} = $localizer;
}

sub get_logger {
	my $this = shift;
	my $logger = $this->{logger};

	return $logger;
}

sub set_logger {
	my $this = shift;
	my ($logger) = @_;

	$this->{logger} = $logger;
}

sub get_events {
	my $this = shift;
	my $events = $this->{events};

	return $events;
}

sub set_events {
	my $this = shift;
	my ($events) = @_;

	$this->{events} = $events;
}

sub get_status_bar {
	my $this = shift;
	my $status_bar = $this->{status_bar};

	return $status_bar;
}

sub set_status_bar {
	my $this = shift;
	my ($status_bar) = @_;

	$this->{status_bar} = $status_bar;
}

sub get_text_input {
	my $this = shift;
	my $text_input = $this->{text_input};

	return $text_input;
}

sub set_text_input {
	my $this = shift;
	my ($text_input) = @_;

	$this->{text_input} = $text_input;
}

sub get_registered_interfaces {
	my $this = shift;
	my $registered_interfaces = $this->{registered_interfaces};

	return $registered_interfaces;
}

sub set_registered_interfaces {
	my $this = shift;
	my ($registered_interfaces) = @_;

	$this->{registered_interfaces} = $registered_interfaces;
}

sub get_registered_classes {
	my $this = shift;
	my $registered_classes = $this->{registered_classes};

	return $registered_classes;
}

sub set_registered_classes {
	my $this = shift;
	my ($registered_classes) = @_;

	$this->{registered_classes} = $registered_classes;
}

sub register_interface {
	my $this = shift;
	my ($key, $obj) = @_;
	my $registered_interfaces = $this->get_registered_interfaces();

	$registered_interfaces->{$key} = $obj;
}

sub query_interface {
	my $this = shift;
	my ($key) = @_;
	my $registered_interfaces = $this->get_registered_interfaces();
	my $obj = $registered_interfaces->{$key};

	return $obj;
}

##########################################################
#
# Purpose:
#
#   Register a class to be dynamically created and,
#   if necessary, dynamically subclassed as a vxif
#   interface object.
#
# Inputs: 1) name by which an application addresses
#            the class
#         2) name of the (perl) class
#         3) what perl must 'require' to create
#            establish linkage to the class code.
#
# Returns: undefined if o.k., error object otherwise.
#
##########################################################
sub register_class {
	my $this = shift;
	my ($key, $class_name, $req) = @_;
	my $registered_classes = $this->get_registered_classes();
	my $error = undef;

	if (!$req) {
		# req not specified, so class name == require name
		$req = $class_name;
	}

	eval {
		if (!$req) {
			my $errmsg = "invalid class specification";
			$this->throw(1, $errmsg);
		}
		##########################################################
		# Do "require" here so that, when asked
		# to be created, it will be found.
		#
		# Perhaps should be in a separate container,
		# and consulted only when created, however
		# doing it here will detect absence early (DE 9-18-05)
		##########################################################
		eval "require ${req};";
		if ($@) {
			# uh-oh, can't find it, or syntax error,
			# or something.
			my $errmsg = $@;
			$this->throw(1, $errmsg);
		}
		# This class may already be a vxif interface object
		# and, as such, is responsible for initializing the
		# base class.
		my $dynamically_subclassed = 0;
		# hackery, plain and simple.
		my $dyna_base_class = 'VxIF::Interface::Object';
		if (!UNIVERSAL::isa($class_name, $dyna_base_class)) {
			my $isa_izer = "push(\@${class_name}::ISA, '${dyna_base_class}');";
			eval $isa_izer;
			if ($@) {
				# Probably class doesn't exist
				# and, so, has no "ISA" array.
				my $errmsg = $@;
				$this->throw(1, $errmsg);
			}
			##########################################################
			# We have dynamically subclassed the registered class
			# as a vxif interface object.
			# Remember for this class, so that when an object of
			# this class is created, we will know to dynamically
			# invoke base class initialization.
			##########################################################
			$dynamically_subclassed = 1;
		}
		my $eff_dyna_base_class =
			(
			 $dynamically_subclassed
			 ? $dyna_base_class
			 : undef
			 );
		my $class_info = {
			class_name => $class_name,
			dynamic_base_class => $eff_dyna_base_class,
		};
		$registered_classes->{$key} = $class_info;
	};
	$error = $this->catch();

	return $error;
}

sub query_class {
	my $this = shift;
	my ($key) = @_;
	my $registered_classes = $this->get_registered_classes();
	my $class_info = $registered_classes->{$key};

	return $class_info;
}

1;

