package VxIF::Interface::Generic::SystemInfo;
use strict;

use VxIF::Interface::Object;
@VxIF::Interface::Generic::SystemInfo::ISA = qw(VxIF::Interface::Object);

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;
	my ($class_factory, $platform_info) = @_;

	$self->VxIF::Interface::Object::init();
	$self->set_class_factory($class_factory);

	$self->{host_os_infos_aref} = [];
	$self->{host_os_infos_href} = {};

	$self->compute_local_host_osinfo($platform_info);
}

sub get_bootstrap_searchpaths {
	my $self = shift;
	my $bootstrap_searchpaths = [ qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin /local/bin) ];

	return $bootstrap_searchpaths;
}

sub get_bootstrap_command {
	my $self = shift;
	my ($cmd) = @_;
	my $searchpaths = $self->get_bootstrap_searchpaths();
	my $fq_cmd = undef;

	# search for command on the "known" paths
	for my $path (@$searchpaths) {
		my $search_cmd = "${path}/${cmd}";
		if (-x "${search_cmd}") {
			# found one.  go with it.
			$fq_cmd = $search_cmd;
			last;
		}
	}

	return $fq_cmd;
}

#
# Determine the localhost OS so VxIF can launch the correct platform installer.
#
sub get_local_os {
	my $self = shift;
	my $local_os = undef;

	if (Utils::running_on_windows()) {
		$local_os = "NativePerl";
	} else {
		my $uname_cmd = "uname";
		my $fq_uname_cmd = $self->get_bootstrap_command($uname_cmd);

		if (!defined($fq_uname_cmd)) {
			die Utils::_tr("Unable to find the ${uname_cmd} command. Unable to determine the localhost OS type.", 73, 1000, "${uname_cmd}");
		}

		my $os = qx{${fq_uname_cmd}};
		chomp($os);
		$os =~ s/-//; # get rid of any - character. Mainly for HP-UX.

		$local_os = $os;
	}

	return $local_os;
}

sub compute_local_host_osinfo {
	my $self = shift;
	my ($platform_info) = @_;
	my $local_host_os_name = $self->get_local_os();
	my $local_os_installer = $platform_info->get_os_installer($local_host_os_name);
	my $local_cmds = $local_os_installer->get_commands();
	my $uname_cmd = $local_cmds->{UNAME};
	my $fqdn = Utils::vxif_dol("${uname_cmd} -n");

	chomp $fqdn;
	$fqdn =~ s/\..*$//;

	my $local_hostname = $fqdn;
	$local_host_os_name =~ s/-//; # get rid of any - character. Mainly for HP-UX.
	chomp $local_host_os_name;

	my $local_host_os_version = Utils::vxif_dol("${uname_cmd} -r");
	chomp $local_host_os_version;
	if ($local_host_os_name eq "AIX") {
		my $maj_ver = Utils::vxif_dol("${uname_cmd} -v");
		chomp $maj_ver;
		$local_host_os_version = $maj_ver.".".$local_host_os_version;
	}

	my $local_host_os_architecture = Utils::vxif_dol("${uname_cmd} -m");
	chomp $local_host_os_architecture;

	my $cf = $self->get_class_factory();
	my $local_host_os_info = $cf->create_host_os_info();

	$local_host_os_info->set_name($local_host_os_name);
	$local_host_os_info->set_version($local_host_os_version);
	$local_host_os_info->set_architecture($local_host_os_architecture);

	$self->set_local_hostname($local_hostname);
	$self->set_host_os_info($local_hostname, $local_host_os_info);
}

sub get_local_hostname {
	my $self = shift;
	my $local_hostname = $self->{local_hostname};

	return $local_hostname;
}

sub set_local_hostname {
	my $self = shift;
	my ($local_hostname) = @_;

	$self->{local_hostname} = $local_hostname;
}

sub get_valid_hosts {
	my $self = shift;
	my $valid_hosts = $self->{valid_hosts};

	return $valid_hosts;
}

sub set_valid_hosts {
	my $self = shift;
	my ($valid_hosts) = @_;

	$self->{valid_hosts} = $valid_hosts;
}

sub get_invalid_hosts {
	my $self = shift;
	my $invalid_hosts = $self->{invalid_hosts};

	return $invalid_hosts;
}

sub set_invalid_hosts {
	my $self = shift;
	my ($invalid_hosts) = @_;

	$self->{invalid_hosts} = $invalid_hosts;
}

sub validate_host {
	my $self = shift;
	my ($host) = @_;
}

sub get_host_os_info {
	my $self = shift;
	my ($trg_host) = @_;
	my $host_os_infos_href = $self->{host_os_infos_href};
	my $host_os_info = $host_os_infos_href->{$trg_host};

	return $host_os_info;
}

sub set_host_os_info {
	my $self = shift;
	my ($trg_host, $host_os_info) = @_;
	my $host_os_infos_aref = $self->{host_os_infos_aref};
	my $host_os_infos_href = $self->{host_os_infos_href};

	if (!defined($host_os_infos_href->{$trg_host})) {
		push(@$host_os_infos_aref, $trg_host);
	}
	$host_os_infos_href->{$trg_host} = $host_os_info;
}

#
# Check for ssh and scp availability on localhost.
# If there, use them (instead of rsh and rcp).
#
# Input: 1) reference to self;
#
# Return: 0 if successful, error object if not.
#
sub setup_secure_remote_ops {
	my $self = shift;
	my $ifmgr = $self->get_interface_manager();
	my $platform_info = $ifmgr->get_platform_info();
	my $local_host = $self->get_local_hostname();
	my $local_os_info = $self->get_host_os_info($local_host);
	my $local_os_name = $local_os_info->get_name();
	my $local_os_installer = $platform_info->get_os_installer($local_os_name);
	my $local_cmds = $local_os_installer->get_commands();
	my $cf = $self->get_class_factory();
	my $error = undef;
	
	# check for ssh and scp binaries on local system
	if (Utils::running_on_windows()) {
		$local_cmds->{RSH} = "ssh";
		$local_cmds->{RCP} = "scp -q";
	} else {
		my $which_cmd = $local_cmds->{WHICH};
		my $cmd;

		$cmd = Utils::vxif_dol("${which_cmd} ssh");
		chomp $cmd;
		if ($cmd !~ /\/ssh/) {
			$cmd = $self->get_bootstrap_command("ssh");

			if (!$cmd) {
				$error = $cf->create_error();
				$error->set_rc(1);
				$error->set_msg("cannot find ssh");

				return $error;
			}
		}
                # adding check to find if using IPv6 - jpk Feb 07,06
                if ($self->{IPv6}) {
                   $local_cmds->{RSH} = $cmd." -6";
                } else {
                   $local_cmds->{RSH} = $cmd;
                }
		

		$cmd = Utils::vxif_dol("${which_cmd} scp");
		chomp $cmd;
		if ($cmd !~ /\/scp/) {
			$cmd = $self->get_bootstrap_command("scp");

			if (!$cmd) {
				$error = $cf->create_error();
				$error->set_rc(1);
				$error->set_msg("cannot find scp");

				return $error;
			}
		}
                # adding check to find if using IPv6.  scp needs IP address put into brackets[] with escapes - jpk Feb 07,06
                if ($self->{IPv6}) { 
                   $local_cmds->{RSH} = $cmd." -6";
                } else {
                   $local_cmds->{RCP} = $cmd;
                }
	}

	return $error;
}

#
# Obtain the host information for the target host.
#
# Input: 1) the reference to the installation context;
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub compute_target_host_osinfo ($) {
	my $self = shift;
	my ($trg_host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $platform_info = $ifmgr->get_platform_info();
	my $uname_cmd = "uname";
	my $searchpaths = $self->get_bootstrap_searchpaths();
	my $cf = $ifmgr->get_class_factory();
	my $ctx = $ifmgr->get_vxif_context();
	my $error = undef;

	# Stick entire thing into an eval block,
	# to catch any VxIF-generated fatalities.
	# Can use for locally generated errors
	# as well.
	eval {
		$ctx->set_target_hostname($trg_host);

		my $unamecmd = undef;

		# search on paths for uname on target host
		for my $path (@$searchpaths) {
			my $fq_uname_cmd = "${path}/${uname_cmd}";
			my $search_unamecmd = Utils::vxif_dor("ls ${fq_uname_cmd}", "", $ctx);
			chomp $search_unamecmd;
			if ($search_unamecmd eq "${fq_uname_cmd}") {
				$unamecmd = $search_unamecmd;
				last;
			}
		}

		if ($unamecmd) {
			# o.k., we have a uname command.
			# compute host os name.
			my $os = Utils::vxif_dor("${unamecmd}", "", $ctx);
			chomp($os);
			$os =~ s/-//; # get rid of any - character. Mainly for HP-UX.

			my $platform_info = $ifmgr->get_platform_info();
			my $os_valid = $platform_info->os_supported($os);
			if (!$os_valid) {
				my $msg = Utils::_tr("OS, ${os}, for host, ${trg_host}, is not a supported platform.", 73, 1001, "${os}", "${trg_host}");
				$self->throw(1, $msg);
			}

			# compute host os version
			my $version = Utils::vxif_dor("$unamecmd -r", "", $ctx);
			chomp $version;
			if ($os eq "AIX"){
				my $maj_ver = Utils::vxif_dor("$unamecmd -v", "",$ctx);
				chomp $maj_ver;
				$version = $maj_ver.".".$version;
			}

			# compute host os architecture
			my $arch = Utils::vxif_dor("$unamecmd -m", "", $ctx);
			chomp($arch);

			my $host_os_info = $cf->create_host_os_info();
			$host_os_info->set_name($os);
			$host_os_info->set_version($version);
			$host_os_info->set_architecture($arch);

			$self->set_host_os_info($trg_host, $host_os_info);

		} else {
			# can't find uname anywhere (would have returned, after computation, if we could)
			my $msg = Utils::_tr("Unable to obtain host information for ${trg_host}. It may be inaccessable.", 73, 1002, "${trg_host}");
			$self->throw(1, $msg);
		}
	};
	$error = $self->catch();

	return $error;
}

1;

