package VxIF::Interface::Generic::ProductOps;
use strict;

use VxIF::SpaceMgr;
use VxIF::Interface::Object;
@VxIF::Interface::Generic::ProductOps::ISA = qw(VxIF::Interface::Object);

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;
	my ($product_info) = @_;

	$self->VxIF::Interface::Object::init();
}

#
# Invoke a product installer method on a host
#
# Input: 1) the product
#        2) the host
#        3) the method name
#
# Note: this assumes the classic VxIF signature
#       for a product installer ... that is, you
#       pass in ctx and cfg, and get back 1 if o.k.,
#       0 if not o.k..
#
# Return: 0 if successful, error object if not.
#
sub invoke_method {
	my $self = shift;
	my ($product_info, $host, $method_name) = @_;
	my $installer_object = $product_info->get_installer_object();
	my $error = undef;

	if ($installer_object) {
		my $method = $installer_object->can($method_name);
		if ($method) {
			my $ifmgr = $self->get_interface_manager();
			my $ctx = $ifmgr->get_vxif_context();
			my $cfg = $ctx->get_config();

			$ctx->set_target_hostname($host);

			eval {
				my $rc = $installer_object->$method($ctx, $cfg);

				if (!$rc) {
					my $msg = "Error in ${method_name}, target host == ${host}.";
					$self->throw(1, $msg);
				}
			};
			$error = $self->catch();
		}
	}

	return $error;
}

#
# Pre install product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub pre_install_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "pre_product_install";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

#
# Check installation requirements on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub check_installation_requirements_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# Keep this in CPICommon just for now.
		# Invoke it here, however.
		#my $rc = $os_inst->check_installation_requirements_on_host($ctx, $host);

		my $rc = $os_inst->check_for_external_dependencies($ctx);

		if (!$rc) {
			my $msg = "Error in checking external dependencies, target host == ${host}.";
			$self->throw(1, $msg);
		}

		my $space_mgr = new SpaceMgr();
		my $warnings = [];
		$error = $space_mgr->check_for_required_spaces($ctx, $warnings);

		if ($error) {
			#my $msg = "Error in checking installation requirements, target host == ${host}.";
			$self->throw($error);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Install product packages on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub silent_install_packages_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $package_ops = $ifmgr->get_package_ops();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $logger = $ctx->get_logger();
	my $cfg = $ctx->get_config();
	my $error = undef;

	# yecchhh! (DE 9-19-05)
	$ctx->set_task_type("install");

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# o.k., let's give this a try
		my $pkgsinfo = $product_info->get_packages();
		my $inst_order = $os_inst->get_effective_package_order($ctx);
		for my $pkgname (@$inst_order) {
			my $package_info = $product_info->get_package_info($pkgname);
			$error = $package_ops->install_on_host($package_info, $host);
			if ($error) {
				# take a licking
				my $err_rc = $error->get_rc();
				my $err_msg = $error->get_msg();
				$logger->error("${err_rc}: ${err_msg}");
				# and keep on ticking
				$error = undef;
			}
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Install product packages on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub install_packages_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $package_ops = $ifmgr->get_package_ops();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $logger = $ctx->get_logger();
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# Keep this in CPICommon just for now.
		# Invoke it here, however.
		my $rc = $os_inst->install_product_packages($ctx, $cfg);

		if (!$rc) {
			my $msg = "Error in install_product_packages, target host == ${host}.";
			$self->throw(1, $msg);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Post install product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub post_install_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "post_product_install";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

#
# Install product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub install_on_host {
	my $self = shift;
	my ($product_info, $trg_host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $events = $ifmgr->get_events();
	my $error = undef;

	# pre product install callbacks
	$error = $self->pre_install_on_host($product_info, $trg_host);
	if ($error) {
		return $error;
	}

	$events->pre_install_packages();

	# install the product packages
	$error = $self->install_packages_on_host($product_info, $trg_host);
	if ($error) {
		$events->install_packages_fail($product_info);
		return $error;
	}
	$events->install_packages_pass($product_info);

	# pre product install callbacks
	$error = $self->post_install_on_host($product_info, $trg_host);
	if ($error) {
		return $error;
	}

	return $error;
}

#
# Pre uninstall product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub pre_uninstall_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "pre_product_uninstall";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

sub check_uninstallation_requirements_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# Nothing (yet :-).
		# Some per-package checks, but do
		# these during the actual uninstall
		# process, since package dependencies
		# may change dynamically, depending on
		# order of package uninstallation.
	};
	$error = $self->catch();

	return $error;
}

#
# Uninstall product packages on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub silent_uninstall_packages_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $package_ops = $ifmgr->get_package_ops();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $logger = $ctx->get_logger();
	my $cfg = $ctx->get_config();
	my $error = undef;

	# yecchhh! (DE 9-29-05)
	$ctx->set_task_type("uninstall");

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# o.k., let's give this a try
		my $pkgsinfo = $product_info->get_packages();
		my $uninst_order = $os_inst->get_effective_package_order($ctx);
		for my $pkgname (@$uninst_order) {
			my $iver = $os_inst->get_instpkgvers($ctx, $pkgname, "", $pkgsinfo);

			if (!$iver) {
				my $msg = "Package ${pkgname} is not installed.";
				$logger->info($msg);
				next;
			}

			my $package_info = $product_info->get_package_info($pkgname);

			# see if any installed packages depend on this one
			my $depstr = $os_inst->get_pkgdep($ctx, $pkgname);
			if ($depstr) {
				my $deps = [ split(/\s+/, $depstr) ];
				my $depstr2 = join(", ", @$deps);
				my $msg = "Leaving ${pkgname} installed, since it is used by {${depstr2}}.";
				$logger->info($msg);
				next;
			}

			$error = $package_ops->uninstall_on_host($package_info, $host);
			if ($error) {
				# take a licking
				my $err_rc = $error->get_rc();
				my $err_msg = $error->get_msg();
				$logger->error("${err_rc}: ${err_msg}");
				# and keep on ticking
				$error = undef;
			}
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Uninstall product packages on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub uninstall_packages_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# Keep this in CPICommon just for now.
		# Invoke it here, however.
		my $rc = $os_inst->uninstall_product_packages($ctx, $cfg);

		if (!$rc) {
			my $msg = "Error in uninstall_product_packages, target host == ${host}.";
			$self->throw(1, $msg);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Post uninstall product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub post_uninstall_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "post_product_uninstall";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

#
# Uninstall product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub uninstall_on_host {
	my $self = shift;
	my ($product_info, $trg_host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $events = $ifmgr->get_events();
	my $error = undef;

	# pre product uninstall callbacks
	$error = $self->pre_uninstall_on_host($product_info, $trg_host);
	if ($error) {
		return $error;
	}

	$events->pre_uninstall_packages($product_info, $trg_host);

	# uninstall the product packages
	$error = $self->uninstall_packages_on_host($product_info, $trg_host);
	if ($error) {
		$events->uninstall_packages_fail($product_info);
		return $error;
	}
	$events->uninstall_packages_pass($product_info);

	# pre product uninstall callbacks
	$error = $self->post_uninstall_on_host($product_info, $trg_host);
	if ($error) {
		return $error;
	}

	$events->product_uninstall_successful($product_info, $trg_host);

	return $error;
}

#
# Pre configure product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub pre_configure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "preconfigure_product";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

#
# Unconfigure (specifically) product on a host.
# Do so with *no* input or output, except,
# possibly returning an error object if an
# error occurs.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub silent_plain_unconfigure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# the ability to do this depends on the underlying
		# product installer implementation.
		my $upi_installer = $ctx->get_host_os_upi_installer($host);

		# need to be able to remove any existing configuration
		my $unconfig_method = $upi_installer->can("silent_unconfigure");
		if (!$unconfig_method) {
			my $errmsg = "Unable to unconfigure silently, prior to configuration.";
			$self->throw(1, $errmsg);
		}

		# Remove current configuration.
		$error = $upi_installer->$unconfig_method($ctx);
		if ($error) {
			$self->throw($error);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Configure (specifically) product on a host.
# Do so with *no* input or output, except,
# possibly returning an error object if an
# error occurs.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub silent_plain_configure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# the ability to do this depends on the underlying
		# product installer implementation.
		my $upi_installer = $ctx->get_host_os_upi_installer($host);

		# Of course, need to be able to configure.
		# Not configuring at all does not count as a silent configure.
		my $config_method = $upi_installer->can("silent_configure");
		if (!$config_method) {
			my $errmsg = "Unable to configure silently.";
			$self->throw(1, $errmsg);
		}

		# So, we are at least able to do a configuration.
		# First, however, remove current configuration.
		$error = $self->silent_plain_unconfigure_on_host($product_info, $host);
		if ($error) {
			# nope
			$self->throw($error);
		}

		# Put up a (registered) configuration.
		$error = $upi_installer->$config_method($ctx);
		if ($error) {
			$self->throw($error);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Configure (specifically) product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub plain_configure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $os_inst = $ctx->get_host_os_installer($host);
	my $cfg = $ctx->get_config();
	my $error = undef;

	# still have to do this.
	$ctx->set_target_hostname($host);

	eval {
		# Keep this in CPICommon just for now.
		# Invoke it here, however.
		my $rc = $os_inst->configure_product_on_target_host($ctx, $cfg);

		if (!$rc) {
			my $msg = "Error in configure_product_on_target_host, target host == ${host}.";
			$self->throw(1, $msg);
		}
	};
	$error = $self->catch();

	return $error;
}

#
# Post configure product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub post_configure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $method_name = "postconfigure_product";
	my $error = $self->invoke_method($product_info, $host, $method_name);

	return $error;
}

#
# Configure (overall) product on a host.
#
# Input: 1) the product
#        2) the host
#
# Return: 0 if successful, error object if not.
#
sub configure_on_host {
	my $self = shift;
	my ($product_info, $host) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $ctx = $ifmgr->get_vxif_context();
	my $trg_upi = $ctx->get_target_upi();
	my $logger = $ctx->get_logger();
	my $error = undef;

	# do preconfigure
	#$logger->info(Utils::_tr("Pre Configuring ${trg_upi} on ${host}...\n", 71, 1000, "${trg_upi}", "${host}"));
	$logger->info("Pre Configuring ${trg_upi} on ${host}...\n");
	$error = $self->pre_configure_on_host($product_info, $host);
	if ($error) {
		#$logger->info(Utils::_tr("Unable to pre-configure ${trg_upi} on ${host}.", 71, 1001, "${trg_upi}", "${host}"));
		$logger->info("Unable to pre-configure ${trg_upi} on ${host}.");
		return $error;
	}
	#$logger->info(Utils::_tr("${trg_upi} has been successfully pre-configured on ${host}.", 71, 1002, "${trg_upi}", "${host}"));
	$logger->info("${trg_upi} has been successfully pre-configured on ${host}.");

	# do configure
	#$logger->info(Utils::_tr("Configuring ${trg_upi} on ${host}...\n", 71, 1003, "${trg_upi}", "${host}"));
	$logger->info("Configuring ${trg_upi} on ${host}...\n");
	$error = $self->plain_configure_on_host($product_info, $host);
	if ($error) {
		#$logger->info(Utils::_tr("Unable to configure ${trg_upi} on ${host}.", 71, 1004, "${trg_upi}", "${host}"));
		$logger->info("Unable to configure ${trg_upi} on ${host}.");
		return $error;
	}
	#$logger->info(Utils::_tr("${trg_upi} has been successfully configured on ${host}.", 71, 1005, "${trg_upi}", "${host}"));
	$logger->info("${trg_upi} has been successfully configured on ${host}.");

	# do postconfigure
	#$logger->info(Utils::_tr("Post Configuring ${trg_upi} on ${host}...\n", 71, 1006, "${trg_upi}", "${host}"));
	$logger->info("Post Configuring ${trg_upi} on ${host}...\n");
	$error = $self->post_configure_on_host($product_info, $host);
	if ($error) {
		#$logger->info(Utils::_tr("Unable to post-configure ${trg_upi} on ${host}.", 71, 1007, "${trg_upi}", "${host}"));
		$logger->info("Unable to post-configure ${trg_upi} on ${host}.");
		return $error;
	}
	#$logger->info(Utils::_tr("${trg_upi} has been successfully post-configured on ${host}.", 71, 1008, "${trg_upi}", "${host}"));
	$logger->info("${trg_upi} has been successfully post-configured on ${host}.");

	return $error;
}

1;

