package VxIF::Interface::Generic::ProductInfo;
use strict;

use Cwd;
use File::Basename;
use File::Spec;

use VxIF::Interface::Generic::ConfigInfo;
@VxIF::Interface::Generic::ProductInfo::ISA = qw(VxIF::Interface::Generic::ConfigInfo);

use VxIF::Interface::Utils::Config;
use VxIF::Interface::Utils::Dictionary;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;
	my ($super_instinfo) = @_;

	$self->VxIF::Interface::Generic::ConfigInfo::init();

	$self->set_super_instinfo($super_instinfo);
	$self->set_sub_instinfos({});
	$self->set_section_info({});
	$self->set_packages({});

	my $dictionary = new VxIF::Interface::Utils::Dictionary();
	$self->set_dictionary($dictionary);
}

sub get_super_instinfo {
	my $self = shift;
	my $super_instinfo = $self->{super_instinfo};

	return $super_instinfo;
}

sub set_super_instinfo {
	my $self = shift;
	my ($super_instinfo) = @_;

	$self->{super_instinfo} = $super_instinfo;
}

sub get_sub_instinfos {
	my $self = shift;
	my $sub_instinfos = $self->{sub_instinfos};

	return $sub_instinfos;
}

sub set_sub_instinfos {
	my $self = shift;
	my ($sub_instinfos) = @_;

	$self->{sub_instinfos} = $sub_instinfos;
}

sub get_installer_object {
	my $self = shift;
	my $installer_object = $self->{installer_object};

	return $installer_object;
}

sub set_installer_object {
	my $self = shift;
	my ($installer_object) = @_;

	$self->{installer_object} = $installer_object;
}

sub get_packages {
	my $this = shift;
	my $packages = $this->{packages};

	return $packages;
}

sub set_packages {
	my $this = shift;
	my ($packages) = @_;

	$this->{packages} = $packages;
}

sub get_section_info {
	my $self = shift;
	my $section_info = $self->{section_info};

	return $section_info;
}

sub set_section_info {
	my $self = shift;
	my ($section_info) = @_;

	$self->{section_info} = $section_info;
}

sub get_dictionary {
	my $self = shift;
	my $dictionary = $self->{dictionary};

	return $dictionary;
}

sub set_dictionary {
	my $self = shift;
	my ($dictionary) = @_;

	$self->{dictionary} = $dictionary;
}

sub get_package_info {
	my $self = shift;
	my ($pkgname) = @_;
	my $packages = $self->get_packages();
	my $package_info = $packages->{$pkgname};

	return $package_info;
}

sub get_installation_modes {
	my $self = shift;
	my $modes = $self->{modes};

	return $modes;
}

sub get_installation_modes_ordering {
	my $self = shift;
	my $modes = $self->{MODES_ORDERING};

	return $modes;
}

sub resolve_external_reference {
	my $self = shift;
	my ($section_name, $fnam, $os_name) = @_;
	# this should go away soon
	my $sect_href_aref = [];
	# should have been passed as absolute, but ...
	my $fq_fnam = $self->absolute_fnam($fnam);

	if (! -f "${fq_fnam}") {
		die "Cannot find nested installer in ${fq_fnam}.";
	}

	# populate section from nested installer
	my $ni_dir = dirname($fq_fnam);
	my $ifmgr = $self->get_interface_manager();
	my $cf = $ifmgr->get_class_factory();
	my $instinfo = $cf->create_installer_info();
	$instinfo->from_config_file($fq_fnam);

	# maintain {hash of) nested installer references,
	# keyed on fully-qualified file name of top level
	# installer config file.
	my $sub_instinfos = $self->get_sub_instinfos();
	if (!exists($sub_instinfos->{$fq_fnam})) {
		$sub_instinfos->{$fq_fnam} = $instinfo;
	}

	# grab os-specific product info
	my $product_os = $os_name;
	my $sub_product_info = $instinfo->get_platform_product_info($product_os);

	if ($sub_product_info) {
		# might be a package, or might be a definition.
		# if a definition, first expand, then check packages.
		my $sub_dictionary = $sub_product_info->get_dictionary();
		my $sub_def = $sub_dictionary->expand($section_name);
		my $sect_list;
		if ($sub_def) {
			$sect_list = $sub_def;
		} else {
			# not in dictionary.
			# must be a package.
			# make an array of 1.
			$sect_list = [$section_name];
		}

		my $sub_packages = $sub_product_info->get_packages();
		for my $final_sect (@$sect_list) {
			if (exists($sub_packages->{$final_sect})) {
				my $data_href = {};
				my $pkg_data = $sub_packages->{$final_sect};
				my $sect_href = {
					final_sect => $final_sect,
					data_href => $data_href,
				};
				push(@$sect_href_aref, $sect_href);
			} else {
				die "Can't find nested ${product_os} installer for ${final_sect}.\n";
			}
		}
	}

	my $extref_info = {
		src_product_info => $sub_product_info,
		sect_href_aref => $sect_href_aref,
	};

	return $extref_info;
}

############################################################
#
# Read a basic config file, then interpret this flavor.
#
# There are:
#   - some values at the global level
#   - a data dictionary ([Dictionary])
#   - some package information (all remaining [...]'s)
#
# If a section contains 'IMPORT = <file-name>', then
# information for that section is obtained from the
# specified file, which is in InstallerInfo format,
# and which then further points to an os-specific
# ProductInfo config file (which contains the
# actual data).
#
# It should probably be considered an error if
# a section name is defined in the data dictionary.
# However, this may indicate that the symbol (in its
# section name incarnation) is to be resolved in the
# scope of the imported product info.
#
############################################################
sub from_config_file {
	my $self = shift;
	my ($config_fnam) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $cf = $ifmgr->get_class_factory();
	my $reserved_section_names = {
		Dictionary => 1,
	};
	# read config file, and get its fully-qualified name
	my $basic_config = $self->obtain_config($config_fnam);
	my $fq_config_fnam = $self->get_fq_fnam();

	# get global data (outside any sections)
	my $global_values = $basic_config->get_values();
	for my $key (keys(%$global_values)) {
		my $val = $global_values->{$key};
		my $eff_val;
		if ($key eq "INSTALL_ORDER") {
			my @order = split(/\s+/, $val);
			$eff_val = \@order;
		} elsif ($key eq "PRODUCTS") {
			my @products = split(/\s+/, $val);
			$eff_val = \@products;
		} elsif ($key eq "OS_VERSIONS") {
			my $os_versions_href = {};
			my $os_versions_str = $val;
			my @os_versions = split(/\s+/, $os_versions_str);
			for my $os_version (@os_versions) {
				$os_versions_href->{$os_version} = 1;
			}
			# replace version string by its hash entity
			$eff_val = $os_versions_href;
		} elsif ($key eq "MODES_ORDERING") {
			$eff_val = [ split(/\s+/, $val) ];
		} else {
			$eff_val = $val;
		}
		$self->{$key} = $eff_val;
	}

	# get sections, will have been filled in
	# when config file was read.
	my $sections = $basic_config->get_sections();

	# populate section_info from the sections.
	my $section_info = $self->get_section_info();

	my $packages = $self->get_packages();

	######################################################
	# First resolve imports.
	#
	# This involves reading any imported installers,
	# and sticking them into "SOURCE_PRODUCT_INFO"
	# name/value pairs.
	#
	# This can also involve "pre-importing" the
	# appropriate sections in the externally
	# referenced installer product.
	######################################################
	for my $section_name (keys(%$sections)) {
		$section_info->{$section_name} = {};
		my $data = $sections->{$section_name};
		if (exists($data->{IMPORT})) {
			my $loc = $data->{IMPORT};
			my $fq_loc;
			if (File::Spec->file_name_is_absolute($loc)) {
				$fq_loc = $loc;
			} else {
				my $product_location = dirname($fq_config_fnam);
				$fq_loc = "${product_location}/${loc}";
			}
			# call this with an absolute file name
			my $os_name = $self->{OS_NAME};
			my $extref_info = $self->resolve_external_reference($section_name, $fq_loc, $os_name);
			my $src_product_info = $extref_info->{src_product_info};
			my $sect_href_aref = $extref_info->{sect_href_aref};
			my $section_name_def = {
				name => $section_name,
				sequence => [],
			};
			for my $sect_href (@$sect_href_aref) {
				my $final_sect = $sect_href->{final_sect};
				my $data_href = $sect_href->{data_href};
				# if import was of a symbol containing multiple sections,
				# as defined in the source product, then some of them
				# might not already exist here, so create.
				if (!exists($section_info->{$final_sect})) {
					$section_info->{$final_sect}= {};
				}
				# Always put this in an imported section,
				# whether or not the section was already there.
				$section_info->{$final_sect}{'SOURCE_PRODUCT_INFO'} = $src_product_info;

				my $elt = {
					name => $final_sect,
					type => undef,
				};
				push(@{$section_name_def->{sequence}}, $elt);
			}
			if (scalar(@$sect_href_aref) == 1 && $sect_href_aref->[0]{final_sect} eq $section_name) {
				# nothing
				#
				# Actually, I *hate* if's like these.  And it was I who did this one.
				# Now I have to figure it out and then comment it properly.
			} else {
				my $dictionary = $self->get_dictionary();
				$dictionary->add_definition($section_name_def);
			}
		}
	}

	######################################################
	# Next copy in local values.
	#
	# Since imported values have already been placed
	# (assuming these values are actually pre-placed),
	# locally defined values override imported values.
	#
	# If the imported values are not pre-placed, then
	# they can be searched for dynamically via the
	# SOURCE_PRODUCT_INFO field, for a given package.
	######################################################
	for my $section_name (keys(%$sections)) {
		if ($section_name eq 'Dictionary') {
			my $dictionary = $self->get_dictionary();
			$dictionary->from_href($sections->{$section_name});
		} else {
			my $section_data = $sections->{$section_name};
			# must be a package
			for my $datum (keys(%$section_data)) {
				my $value = $section_data->{$datum};
				my $eff_value;

				if ($datum eq "SPACE") {
					my @numbers = split(/\s*[,\s]\s*/, $value);
					$eff_value = \@numbers;
				} else {
					$eff_value = $value;
				}

				$section_info->{$section_name}{$datum} = $eff_value;
			}
		}
	}

	# see what we have in the dictionary
	my $dictionary = $self->get_dictionary();
	my $symstat = $dictionary->compute_symbol_status();
	# to find out which symbols are packages

	# walk status table
	for my $section_name (keys(%$symstat)) {
		my $status = $symstat->{$section_name};
		if (!$status->{defined}) {
			# if a symbol is referenced in the dictionary,
			# but not defined in the dictionary, then it's
			# a package.
			my $package_info = $cf->create_package_info($self, $section_name);
			my $section_data = $section_info->{$section_name};
			$package_info->from_href($section_data);

			if (!defined($section_data->{IMPORT})) {
				if (!defined($section_data->{PACKAGE_LOCATION})) {
					my $package_location = dirname($fq_config_fnam);
					$package_info->set_media_location($package_location);
				}
			}

			$packages->{$section_name} = $package_info;
		}
	}

	# fill in package locations (packages and symbols, from dictionary)
	for my $section_name (keys(%$symstat)) {
		if ($section_name eq 'Dictionary') {
			next;
		}
		my $section = $section_info->{$section_name};
		if (!exists($section->{IMPORT})) {
			# this entry is added.  this may change, pending incremental improvement.
			# if the section was "IMPORT"d, then it will already be there.
			if (!exists($section_info->{$section_name}{PACKAGE_LOCATION})) {
				my $package_location = dirname($fq_config_fnam);
				$section_info->{$section_name}{PACKAGE_LOCATION} = $package_location;
			}
		}
	}

	$self->set_section_info($section_info);
}

1;

