package VxIF::Interface::Generic::InstallerInfo;
use strict;

use File::Basename;
use File::Spec;

use VxIF::Interface::Generic::ConfigInfo;
@VxIF::Interface::Generic::InstallerInfo::ISA = qw(VxIF::Interface::Generic::ConfigInfo);

use VxIF::Interface::Utils::Config;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;

	$self->VxIF::Interface::Generic::ConfigInfo::init();

	my $pgm = $self->get_program();
	if (!$pgm) {
		# no "program" attribute set.
		# initialize with prefix of
		# base running perl program.
		my $pgm = basename($0);
		$pgm =~ s/\.pl$//;
		# stick this in.  can be overridden
		$self->set_program($pgm);
	}

	# start off empty
	$self->set_product_infos({});
	$self->set_product_paths({});
	$self->set_source_paths({});
}

sub get_product {
	my $self = shift;
	my $product = $self->{product};

	return $product;
}

sub set_product {
	my $self = shift;
	my ($product) = @_;

	$self->{product} = $product;
}

sub get_upi {
	my $self = shift;
	my $upi = $self->{upi};

	return $upi;
}

sub set_upi {
	my $self = shift;
	my ($upi) = @_;

	$self->{upi} = $upi;
}

sub get_upi_long {
    my $self = shift;
    my $upi_long = $self->{upi_long};
                                                                                                    
    return $upi_long;
}
                                                                                                    
sub set_upi_long {
    my $self = shift;
    my ($upi_long) = @_;
                                                                                                    
    $self->{upi_long} = $upi_long;
}


sub get_version {
	my $self = shift;
	my $version = $self->{version};

	return $version;
}

sub set_version {
	my $self = shift;
	my ($version) = @_;

	$self->{version} = $version;
}

sub get_title {
	my $self = shift;
	my $title = $self->{title};

	return $title;
}

sub set_title {
	my $self = shift;
	my ($title) = @_;

	$self->{title} = $title;
}

sub get_supported_os {
	my $self = shift;
	my $supported_os = $self->{supported_os};

	return $supported_os;
}

sub set_supported_os {
	my $self = shift;
	my ($supported_os) = @_;

	$self->{supported_os} = $supported_os;
}

sub get_product_paths {
	my $self = shift;
	my $product_paths = $self->{product_paths};

	return $product_paths;
}

sub set_product_paths {
	my $self = shift;
	my ($product_paths) = @_;

	$self->{product_paths} = $product_paths;
}

sub get_source_paths {
	my $self = shift;
	my $source_paths = $self->{source_paths};

	return $source_paths;
}

sub set_source_paths {
	my $self = shift;
	my ($source_paths) = @_;

	$self->{source_paths} = $source_paths;
}

sub get_program {
	my $self = shift;
	my $program = $self->{program};

	return $program;
}

sub set_program {
	my $self = shift;
	my ($program) = @_;

	$self->{program} = $program;
}

#
# Scan directory structure for SUPPORTED_OS.
#
sub scan_for_os {
	my $self = shift;
	my ($os, $upi) = @_;
	my $lc_upi = lc($upi);
	my $uc_upi = uc($upi);
	my $section = undef;
	my $conf_fnam =
		sprintf
		(
		 'pkgs/%s/%sinst.conf',
		 $os,
		 $lc_upi,
		 );
	my $src_path =
		sprintf
		(
		 "src/%s",
		 $os,
		 );
	my $prim_src_base =
		sprintf
		(
		 "%s%s.pm",
		 $uc_upi,
		 $os,
		 );
	my $prim_src =
		sprintf
		(
		 "%s/%s",
		 $src_path,
		 $prim_src_base,
		 );

	if (-f $conf_fnam && -f $prim_src) {
		$section = {
			PRODUCT_PATH => $conf_fnam,
			SRC_PATH => $src_path,
		};
	}

	return $section;
}

#
# Scan directory structure for all vxif-supported os.
# If any there that are not in config, use as though
# they were.
#
sub scan_directories_for_supported_os {
	my $self = shift;
	my ($sections) = @_;
	my $supported_os = $self->get_supported_os();
	my $all_supported_os = [ qw(AIX Darwin HPUX Linux OSF1 SunOS) ];
	my $upi = $self->get_upi();

	# fill in (temporary) hash with supported os from config,
	# for convenient (and, hopefully, quick) lookup.
	my $os_from_conf = {};
	for my $os (@$supported_os) {
		$os_from_conf->{$os} = 1;
	}

	# for each supported os discovered on disk, if not present
	# in config (config file overrides), add to (now in memory)
	# config sections as though it had been present in config file.
	for my $os (@$all_supported_os) {
		if (defined($os_from_conf->{$os})) {
			next;
		}
		my $section_from_dir = $self->scan_for_os($os, $upi);
		if ($section_from_dir) {
			# supported os in directory, and not in config file.
			$sections->{$os} = $section_from_dir;
			push(@$supported_os, $os);
		}
	}
}

# this will be a wall of code for awhile.
# hopefully the comments will mitigate.
sub from_config_file {
	my $self = shift;
	my ($config_fnam) = @_;

	# read raw config file
	my $config_info = $self->obtain_config($config_fnam);
	# get fully qualified name
	my $fq_config_fnam = $self->get_fq_fnam();
	# and, while we're at it ...
	my $fq_config_dir = dirname($fq_config_fnam);

	# get global values from config
	my $values = $config_info->get_values();
	# and sections as well
	my $sections = $config_info->get_sections();

	# get the usual (:-) global values
	my $product = $values->{PRODUCT} || die "Cannot find product name.\n";
	my $upi = $values->{UPI} || die "Cannot find product upi.\n";
	my $upi_long = $values->{UPI_LONG};
	my $version = $values->{VERSION} || die "Cannot find installer version.\n";
	my $title = $values->{TITLE} || die "Cannot find installer title.\n";
	# and stick 'em into this object via accessors
	$self->set_product($product);
	$self->set_upi($upi);
	$self->set_upi_long($upi_long);
	$self->set_version($version);
	$self->set_title($title);
	# list of supported os's is a space-separated list.
	# can be blank/empty, since we will (additionally)
	# scan for supported os.
	my $supported_os_str = $values->{SUPPORTED_OS} || "";
	my $supported_os = [ split(/\s+/, $supported_os_str) ];
	$self->set_supported_os($supported_os);

	# we'll need class factory to generate
	# vxif interface objects for supported os's.
	my $cf = $self->get_class_factory();
	# we'll be adding one of each per supported os
	my $product_infos = $self->get_product_infos();
	my $product_paths = $self->get_product_paths();
	my $source_paths = $self->get_source_paths();

	# set up addressing for product installer package
	my $use_lib = qq{use lib "${fq_config_dir}"};
	# Dynamically set up package addressability
	# for product installer code.
	eval "$use_lib";
	if ($@) {
		# if some error, print to stderr.
		print STDERR "$@\n";
		# should probably return here
	}

	# Also scan directories for all vxif-supported platforms.
	# If any there that are not in config, use as though they
	# were.
	$self->scan_directories_for_supported_os($sections);

	# now should be ready to go
	for my $os (@$supported_os) {
		my $section = $sections->{$os};
		# require a section per supported os
		if (!$section) {
			die "Expected configuration section for ${os}.\n";
		}
		my $product_path = $section->{PRODUCT_PATH};
		my $src_path = $section->{SRC_PATH};
		# leave src_path relative, since it will be used
		# to generate source code addressability logic.
		$source_paths->{$os} = $src_path;
		my $fq_product_path =
			$self->absolute_fnam
			(
			 $product_path,
			 $fq_config_dir,
			 );
		$product_paths->{$os} = $fq_product_path;

		# read and set product info for platform
		my $product_info = $cf->create_product_info();
		$product_info->from_config_file($fq_product_path);
		$product_infos->{$os} = $product_info;

		# set up addressing for, and load product installer code
		my $prod_os = sprintf("%s%s", $upi, $os);
		# Construct package import statement dynamically.
		# Should look something like:
		#   require VxSS::SunOS::ATSunOS;
		# and there should be one per supported platform
		# for this product.
		my $source_path = $source_paths->{$os};
		my $pkg_path = sprintf("%s/%s", $source_path, $prod_os);
		# make sure path is converted to perl package syntax
		$pkg_path =~ s/\//::/g;

		# specify addressability
		my $req = "require ${pkg_path};";

		# Dynamically execute platform package addressability statement.
		eval "$req";
		if ($@) {
			# if some error, print to stderr.
			print STDERR "$@\n";
			# should probably return or even die here
		} else {
			# We were able to import the package.
			# Create an instance of same.
			my $prod_obj = $prod_os->new();

			# point these at each other.
			# this may mean something.
			$product_info->set_installer_object($prod_obj);
			$prod_obj->set_product_info($product_info);
		}
	}

	return undef;
}

sub get_product_infos {
	my $self = shift;
	my $product_infos = $self->{product_infos};

	return $product_infos;
}

sub set_product_infos {
	my $self = shift;
	my ($product_infos) = @_;

	$self->{product_infos} = $product_infos;
}

sub get_platform_product_info {
	my $self = shift;
	my ($product_os) = @_;
	my $product_infos = $self->get_product_infos();
	my $product_info = $product_infos->{$product_os};

	return $product_info;
}

sub get_platform_packages_info {
	my $self = shift;
	my ($product_os) = @_;
	my $packages = undef;
	my $product_info = $self->get_platform_product_info($product_os);

	if ($product_info) {
		$packages = $product_info->get_packages();
	}

	return $packages;
}

sub get_platform_package_info {
	my $self = shift;
	my ($product_os, $pkg) = @_;
	my $product_info = $self->get_platform_product_info($product_os);
	my $package_info = undef;

	if ($product_info) {
		$package_info = $product_info->get_package_info($pkg);
	}

	return $package_info;
}

sub query {
	my $self = shift;
	my $product = $self->get_product();
	my $upi = $self->get_upi();
	my $version = $self->get_version();
	my $title = $self->get_title();
	my $supported_os = $self->get_supported_os();
	my $supported_os_str = join(" ", @$supported_os);

	printf("Product: %s\n", $product);
	printf("UPI: %s\n", $upi);
	printf("Version: %s\n", $version);
	printf("Title: %s\n", $title);
	printf("Supported OS: %s\n", $supported_os_str);

	for my $product_os (@$supported_os) {
		printf("  OS: %s\n", $product_os);
		my $packages = $self->get_platform_packages_info($product_os);
		for my $pkgnam (sort(keys(%$packages))) {
			printf("    Package: %s\n", $pkgnam);
			my $package_info = $packages->{$pkgnam};
			$package_info->list(6);
			next;
			for my $key (sort(keys(%$package_info))) {
				my $val = $package_info->{$key};
				if ($key eq "RESPONSEFILE") {
					printf("      %s\n", $key);
					my @lines = split(/\n/, $val);
					for my $line (@lines) {
						chomp $line;
						printf("        %s\n", $line);
					}
				} elsif ($key eq "SPACE") {
					my $val_str = join(" ", @$val);
					printf("      %s ==> [%s]\n", $key, $val_str);
				} else {
					printf("      %s ==> %s\n", $key, $val);
				}
			}
		}
	}
}

1;

