package VxIF::Interface::Generic::ClassFactory;
use strict;

use VxIF::Interface::Manager;
use VxIF::Interface::Generic::InstallerInfo;
use VxIF::Interface::Generic::ProductInfo;
use VxIF::Interface::Generic::PackageInfo;
use VxIF::Interface::Generic::SystemInfo;
use VxIF::Interface::Generic::HostOSInfo;
use VxIF::Interface::Generic::PlatformInfo;
use VxIF::Interface::Generic::ProductOps;
use VxIF::Interface::Generic::PackageOps;
use VxIF::Interface::Generic::Task;
use VxIF::Interface::Generic::Localizer;
use VxIF::Interface::Generic::Events;
use VxIF::Interface::Generic::Modes;
use VxIF::Interface::Generic::Logger;
use VxIF::Interface::Generic::Error;

use VxIF::Interface::Generic::UI::Menu;
use VxIF::Interface::Generic::UI::MenuItem;
use VxIF::Interface::Generic::UI::Button;
use VxIF::Interface::Generic::UI::TextInput;
use VxIF::Interface::Generic::UI::OptionsMenu;
use VxIF::Interface::Generic::UI::Message;
use VxIF::Interface::Generic::UI::LabeledInfo;
use VxIF::Interface::Generic::UI::StatusBar;
use VxIF::Interface::Generic::UI::Progress;
use VxIF::Interface::Generic::UI::FormField;
use VxIF::Interface::Generic::UI::Form;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;
	my $interface_mgr = $self->create_interface_manager();

	$self->set_interface_manager($interface_mgr);
}

sub get_interface_manager {
	my $self = shift;
	my $interface_manager = $self->{interface_manager};

	return $interface_manager;
}

sub set_interface_manager {
	my $self = shift;
	my ($interface_manager) = @_;

	$self->{interface_manager} = $interface_manager;
}

sub create_interface_manager {
	my $self = shift;
	my $interface_mgr = new VxIF::Interface::Manager($self);

	return $interface_mgr;
}

##########################################################
#
# Purpose:
#
#   Create an object of a registered class and,
#   if necessary, dynamically initialize a vxif
#   interface object base class.
#
#   Classes to be created in this manner must expect
#   the first arg passed to be a reference to vxif
#   interface initialization code, and should invoke
#   it before using any vxif interface functionality
#   (i.e., right away (well, best to bless first).
#
# Example of use:
#
# package MyClass;
#
# sub new {
#     my $pkg = shift;
#     my ($init_dynamic_base) = shift;
#     my $self = {};
#     bless($self, $pkg);
#     # after bless, but before anything else
#     $self->$init_dynamic_base();
#     # other stuff ... can now use vxif interface
#     ...
#     return $self;
# }
#
# Inputs: 1) name by which an application addresses
#            the class
#         2) any args required by the class constructor.
#
# Returns: undefined if o.k., error object otherwise.
#
##########################################################
sub create_client_object {
	my $self = shift;
	my $key = shift;
	my $ifmgr = $self->get_interface_manager();
	my $class_info = $ifmgr->query_class($key);
	my $obj = undef;

	# to create an object of this class, the class
	# must have been successfully registered.
	if ($class_info) {
		# return an object.  error contained here.
		my $error = undef;
		# o.k., it was registered.  get the class name.
		my $class_name = $class_info->{class_name};
		# there may be errors that need to be caught.
		eval {
			my $init_dynamic_base = sub {
				my $new_obj = shift;
				my $dyna_base_class = $class_info->{dynamic_base_class};
				if ($dyna_base_class) {
					# Object was dynamically subclassed.
					# Invoke base class initialization.
					if ($dyna_base_class->can("init")) {
						# had better be true
						my $str1 = "\$new_obj->${dyna_base_class}::init();";
						eval $str1;
						if ($@) {
							my $errmsg = $@;
							$ifmgr->throw(1, $errmsg);
						}
					}
				}
				eval {
					$new_obj->set_class_factory($self);
				};
				if ($@) {
					my $errmsg = $@;
					$ifmgr->throw(1, $errmsg);
				}
			};
			# pass initialization code as first arg.
			$obj = $class_name->new($init_dynamic_base, @_);
		};
		$error = $ifmgr->catch();
		if ($error) {
			my $logger = $ifmgr->get_logger();
			my $rc = $error->get_rc();
			my $msg = $error->get_msg();
			my $fmsg = sprintf
				(
				 "%s: %s",
				 $rc,
				 $msg,
				 );
			$logger->error($fmsg);
		}
	}

	return $obj;
}

sub create_installer_info {
	my $self = shift;
	my $installer_info = new VxIF::Interface::Generic::InstallerInfo();

	$installer_info->set_class_factory($self);

	return $installer_info;
}

sub create_product_info {
	my $self = shift;
	my $product_info = new VxIF::Interface::Generic::ProductInfo();

	$product_info->set_class_factory($self);

	return $product_info;
}

sub create_package_info {
	my $self = shift;
	my ($parent_product_info, $section_name) = @_;
	my $package_info =
		new VxIF::Interface::Generic::PackageInfo
		(
		 $parent_product_info,
		 $section_name,
		 );

	$package_info->set_class_factory($self);

	return $package_info;
}

sub create_system_info {
	my $self = shift;
	my ($platform_info) = @_;
	my $system_info = new VxIF::Interface::Generic::SystemInfo($self, $platform_info);

	#$system_info->set_class_factory($self);

	return $system_info;
}

sub create_host_os_info {
	my $self = shift;
	my $host_os_info = new VxIF::Interface::Generic::HostOSInfo();

	$host_os_info->set_class_factory($self);

	return $host_os_info;
}

sub create_platform_info {
	my $self = shift;
	my $platform_info = new VxIF::Interface::Generic::PlatformInfo();

	$platform_info->set_class_factory($self);

	return $platform_info;
}

sub create_product_ops {
	my $self = shift;
	my $product_ops = new VxIF::Interface::Generic::ProductOps();

	$product_ops->set_class_factory($self);

	return $product_ops;
}

sub create_package_ops {
	my $self = shift;
	my $package_ops = new VxIF::Interface::Generic::PackageOps();

	$package_ops->set_class_factory($self);

	return $package_ops;
}

sub create_task {
	my $self = shift;
	my ($installer_info) = @_;
	my $task = new VxIF::Interface::Generic::Task($installer_info);

	$task->set_class_factory($self);

	return $task;
}

sub create_localizer {
	my $self = shift;
	my $localizer = new VxIF::Interface::Generic::Localizer();

	$localizer->set_class_factory($self);

	return $localizer;
}

sub create_events {
	my $self = shift;
	my $events = new VxIF::Interface::Generic::Events();

	$events->set_class_factory($self);

	return $events;
}

sub create_modes {
	my $self = shift;
	my $modes = new VxIF::Interface::Generic::Modes();

	$modes->set_class_factory($self);

	return $modes;
}

sub create_logger {
	my $self = shift;
	my $logger = new VxIF::Interface::Generic::Logger();

	$logger->set_class_factory($self);

	return $logger;
}

sub create_error {
	my $self = shift;
	my $error = new VxIF::Interface::Generic::Error();

	$error->set_class_factory($self);

	return $error;
}

# UI

sub create_menu {
	my $self = shift;
	my ($item_id) = @_;
	my $menu = new VxIF::Interface::Generic::UI::Menu($item_id);

	$menu->set_class_factory($self);

	return $menu;
}

sub create_menu_item {
	my $self = shift;
	my ($item_id) = @_;
	my $menu_item = new VxIF::Interface::Generic::UI::MenuItem($item_id);

	$menu_item->set_class_factory($self);

	return $menu_item;
}

sub create_button {
	my $self = shift;
	my ($item_id) = @_;
	my $button = new VxIF::Interface::Generic::UI::Button($item_id);

	$button->set_class_factory($self);

	return $button;
}

sub create_text_input {
	my $self = shift;
	my ($item_id) = @_;
	my $text_input = new VxIF::Interface::Generic::UI::TextInput($item_id);

	$text_input->set_class_factory($self);

	return $text_input;
}

sub create_options_menu {
	my $self = shift;
	my $options_menu = new VxIF::Interface::Generic::UI::OptionsMenu($self);

	#$options_menu->set_class_factory($self);

	return $options_menu;
}

sub create_message {
	my $self = shift;
	my ($msg_id) = @_;
	my $message = new VxIF::Interface::Generic::UI::Message($msg_id);

	$message->set_class_factory($self);

	return $message;
}

sub create_labeled_info {
	my $self = shift;
	my ($msg_id) = @_;
	my $labeled_info = new VxIF::Interface::Generic::UI::LabeledInfo($msg_id);

	$labeled_info->set_class_factory($self);

	return $labeled_info;
}

sub create_status_bar {
	my $self = shift;
	my ($msg_id) = @_;
	my $status_bar = new VxIF::Interface::Generic::UI::StatusBar($msg_id);

	$status_bar->set_class_factory($self);

	return $status_bar;
}

sub create_progress {
	my $self = shift;
	my ($item_id) = @_;
	# set class factory in constructor, so can create sub_widgets
	my $progress = new VxIF::Interface::Generic::UI::Progress($item_id, $self);

	return $progress;
}

sub create_form_field {
	my $self = shift;
	my ($item_id) = @_;
	my $form_field = new VxIF::Interface::Generic::UI::FormField($item_id);

	$form_field->set_class_factory($self);

	return $form_field;
}

sub create_form {
	my $self = shift;
	my ($item_id) = @_;
	my $form = new VxIF::Interface::Generic::UI::Form($item_id);

	$form->set_class_factory($self);

	return $form;
}

1;

