package Installer;
use strict;

use VxIF::Utils;

BEGIN {
	my $min_req_version = 5.008; # 5.8.0
	my $this_perl_version = $];

	if ($this_perl_version < $min_req_version) {
		my $msg = "Running perl ${this_perl_version} and, so, cannot handle ${min_req_version}.";
		my $loc_msg = Utils::_tr($msg);
		print(STDERR "${loc_msg}\n");
		require POSIX;
		# Just a mere perl exit won't do it for legacy (pre 5.6?) perls.
		# Apparently, though, the posix rendition does.  Blast out of here.
		POSIX::_exit(1);
	}
}

use VxIF::Context;
use VxIF::Logger;
use VxIF::Response;

require VxIF::SunOS::CPISunOS;
require VxIF::Linux::CPILinux;
require VxIF::HPUX::CPIHPUX;
require VxIF::AIX::CPIAIX;
require VxIF::OSF1::CPIOSF1;
require VxIF::Darwin::CPIDarwin;
require VxIF::IRIX64::CPIIRIX64;
require VxIF::FreeBSD::CPIFreeBSD;
require VxIF::NativePerl::CPINativePerl;

###############################
# Messages                    #
###############################

my %COMM;

# figure out why this needs to exist
$COMM{COPYRIGHTYEAR} = 2005;

#
# Set the global messages used by this object.
#
# Input: 1) reference to the installation context;
#
sub set_msgs (\%) {
	my $self = shift;
	my ($ctx) = @_;

	$self->{MSG}{INDENT}="    ";
	$self->{MSG}{PDFRS}=Utils::_tr("components", 14, 1090);
	$self->{MSG}{YESKEY}{L}=Utils::_tr("y", 14, 1000);
	$self->{MSG}{YESKEY}{U}=Utils::_tr("Y", 14, 1091);
	$self->{MSG}{NOKEY}{L}=Utils::_tr("n", 14, 1001);
	$self->{MSG}{NOKEY}{U}=Utils::_tr("N", 14, 1092);
	$self->{MSG}{BACKKEY}{L}=Utils::_tr("b", 14, 1002);
	$self->{MSG}{BACKKEY}{U}=Utils::_tr("B", 14, 1093);
	$self->{MSG}{QUITKEY}{L}=Utils::_tr("q", 14, 1003);
	$self->{MSG}{QUITKEY}{U}=Utils::_tr("Q", 14, 1094);
	$self->{MSG}{HELPKEY}{L}=Utils::_tr("?", 14, 1004);
	$self->{MSG}{HELPKEY}{U}=Utils::_tr("?", 14, 1004);
	$self->{MSG}{INSTALLKEY}{L}=Utils::_tr("i", 14, 1005);
	$self->{MSG}{INSTALLKEY}{U}=Utils::_tr("I", 14, 1095);
	$self->{MSG}{CONFIGKEY}{L}=Utils::_tr("c", 14, 1006);
	$self->{MSG}{CONFIGKEY}{U}=Utils::_tr("C", 14, 1096);
	$self->{MSG}{LICENSEKEY}{L}=Utils::_tr("l", 14, 1007);
	$self->{MSG}{LICENSEKEY}{U}=Utils::_tr("L", 14, 1097);
	$self->{MSG}{PRECHECKKEY}{L}=Utils::_tr("p", 14, 1008);
	$self->{MSG}{PRECHECKKEY}{U}=Utils::_tr("P", 14, 1098);
	$self->{MSG}{UNINSTALLKEY}{L}=Utils::_tr("u", 14, 1009);
	$self->{MSG}{UNINSTALLKEY}{U}=Utils::_tr("U", 14, 1099);
	$self->{MSG}{DESCKEY}{L}=Utils::_tr("d", 14, 1010);
	$self->{MSG}{DESCKEY}{U}=Utils::_tr("D", 14, 1100);
	$self->{MSG}{VENDOR}=Utils::_tr("Veritas", 14, 1101);
	$self->{MSG}{VENDORPR}=Utils::_tr("Veritas Product", 14, 1102);
	$self->{MSG}{VERSINST}=Utils::_tr("Version Installed", 14, 1013);
	$self->{MSG}{LICENSED}=Utils::_tr("Licensed", 14, 1014);
	$self->{MSG}{MENU}{INSTALL}=Utils::_tr("Install/Upgrade a Product", 14, 1015);
	$self->{MSG}{MENU}{CONFIG}=Utils::_tr("Configure an Installed Product", 14, 1016);
	$self->{MSG}{MENU}{LICENSE}=Utils::_tr("License a Product", 14, 1017);
	$self->{MSG}{MENU}{PRECHECK}=Utils::_tr("Perform a Preinstallation Check", 14, 1018);
	$self->{MSG}{MENU}{UNINSTALL}=Utils::_tr("Uninstall a Product", 14, 1019);
	$self->{MSG}{MENU}{DESC}=Utils::_tr("View a Product Description", 14, 1020);
	$self->{MSG}{MENU}{QUIT}=Utils::_tr("Quit", 14, 1021);
	$self->{MSG}{MENU}{HELP}=Utils::_tr("Help", 14, 1022);
	$self->{MSG}{MENUHELP}{INSTALL}=Utils::_tr("Installs packages and patches for an initial installation or upgrade.  The product is also configured during installation, but you have the option to install without configuration.", 14, 1023);
	$self->{MSG}{MENUHELP}{CONFIG}=Utils::_tr("Configures the product after installing a product without configuration.", 14, 1024);
	$self->{MSG}{MENUHELP}{LICENSE}=Utils::_tr("Installs or updates a product license key. This option can be used to update an evaluation key to a permanent key, or to license a feature in a previously installed product.", 14, 1025);
	$self->{MSG}{MENUHELP}{PRECHECK}=Utils::_tr("Performs a preinstallation check to determine if systems meet all installation requirements. No products are installed.", 14, 1026);
	$self->{MSG}{MENUHELP}{UNINSTALL}=Utils::_tr("Removes the selected product's $self->{MSG}{PDFRS}, patches, files, and directories from the system providing they are not used by another Veritas product.", 14, 1103, "$self");
	$self->{MSG}{MENUHELP}{DESC}=Utils::_tr("Displays a brief overview of the selected product.", 14, 1028);
	$self->{MSG}{MENUHELP}{QUIT}=Utils::_tr("Exits the Installation Menu.", 14, 1029);
	$self->{MSG}{MENUHELP}{HELP}=Utils::_tr("Display this help.", 14, 1104);
	$self->{MSG}{COPYRIGHTB}=Utils::_tr("Veritas, the Veritas Logo and all other Veritas product names and slogans are trademarks or registered trademarks of Veritas Technologies LLC. Veritas and the Veritas Logo Reg. U.S. Pat. & Tm. Off. Other product names and/or slogans mentioned herein may be trademarks or registered trademarks of their respective companies.", 14, 1105);
	$self->{MSG}{COPYRIGHTH}=Utils::_tr("Copyright (c) $COMM{COPYRIGHTYEAR} Veritas Technologies LLC. All rights reserved.", 14, 1106, "$COMM{COPYRIGHTYEAR}");


	$self->{SUPPORTEDTASKS} = [ qw(install uninstall configure query license precheck describe rti quit back help) ];
	$self->{TASKS}{INSTALL} = "install";
	$self->{TASKS}{UNINSTALL} = "uninstall";
	$self->{TASKS}{CONFIG} = "configure";
	$self->{TASKS}{QUERY} = "query";
	$self->{TASKS}{LICENSE} = "license";
	$self->{TASKS}{PRECHECK} = "precheck";
	$self->{TASKS}{DESC} = "describe";
	$self->{TASKS}{RTI} = "rti";
	$self->{TASKS}{QUIT} = "quit";
	$self->{TASKS}{BACK} = "back";
	$self->{TASKS}{HELP} = "help";
}

###############################
# Private Methods             #
###############################

sub set_vxif_log_names {
	my $self = shift;
	my ($ctx) = @_;

	# If the environment variable "CPI_LOGLEVEL" is set, use it.
	($ENV{CPI_LOGLEVEL}) && ($self->{LOGLEVEL} = $ENV{CPI_LOGLEVEL});
	(!$self->{LOGDIR}) && ($self->{LOGDIR} = "$self->{TMPPATH}/$self->{PROGRAM}$$ctx{TIME}");
	(!$self->{LOGFILE}) && ($self->{LOGFILE} = "$self->{LOGDIR}/$self->{PROGRAM}.log");
}

#
# Create the CPI Logger.
#
# Input: 1) the reference of the installation context;
#
sub create_logger ($) {
	my $self =shift;
	my ($ctx) = @_;

	# compute logger file names, in the
	# context of [this] installer object.
	$self->set_vxif_log_names($ctx);

	# keep commands explicitly local, until target hosts are validated
	my $local_cmds = $ctx->get_local_cmds();
	my $mkdir_cmd = $local_cmds->{MKDIR};
	my $logger = $self->{LOGGER};

	if ($logger) {
		# already exists.  just stick in name,
		# and assume (for now) that any (other)
		# required fields are set correctly.
		my $method_name = "attach";
		my $method = $logger->can($method_name);
		if ($method) {
			# if the logger has an attach method defined,
			# use it.
			$logger->$method($self->{LOGFILE});
		} else {
			# otherwise just shove value into hash.
			$logger->{LOGFILE} = $self->{LOGFILE};
		}
	} else {
		$self->{LOGGER} =
			Logger->new
			(
			 LOGLEVEL         => $self->{LOGLEVEL},
			 LOGORIGINATORID  => 1,
			 LOGFILE          => $self->{LOGFILE},
			 LOGAPPLICATION   => "VxIF"
			 );
		$logger = $self->{LOGGER};
	}

	# o.k., have logger, with log file set.
	#
	# check local file system access by writing to $self->{LOGDIR}
	my $logdir = $logger->get_logdir();
	my $eff_mkdir_cmd = "${mkdir_cmd} -p ${logdir}";
	my $rtn = Utils::vxif_dol($eff_mkdir_cmd);
	if (! -d "${logdir}") {
		my $local_host = $ctx->get_local_hostname();
		die Utils::_tr("Cannot create ${logdir} directory on ${local_host}.", 14, 1034, "${logdir}", "${local_host}");
	}
	my $touch_cmd = $local_cmds->{TOUCH};
	my $logfile = $logger->get_logfile();
	my $rc = Utils::vxif_dol("${touch_cmd} ${logfile}", 1);
	if ($rc) {
		my $local_host = $ctx->get_local_hostname();
		die Utils::_tr("Cannot write to ${logfile} on ${local_host}", 14, 1035, "${logfile}", "${local_host}");
	}

	# still alive.
	# stick it into context.
	$ctx->set_logger($logger);
}

# set interrupt routine
sub set_interrupt ($) {
	my $self = shift;
	my ($ctx) = @_;

	$SIG{INT} = sub { 
		# TBD:
		#if ($FLAG{INSTALLINGPKG}) {
		#  $PKG=$FLAG{INSTALLINGPKG};
		#  _pl("Removing uncompleted installation of $PKG on $SYS");
		#  _plat_sub_target("uninstall_pkg");
		#  _dol("$CMD{CAT} $COMM{TLOG}") if ($FLAG{FORK});
		#}

		exit 0 if ($$ != $$ctx{PARENTPID});
		my $ifmgr = $ctx->get_interface_manager();
		my $events = $ifmgr->get_events();
		$events->interrupt_received();
		sleep 3;
		$self->cleanup($ctx);
	}
}

#
# Perform basic script initiation tasks
#
sub installer_initiation {
	my $self = shift;
	my $ctx = $self->get_context();

	$$ctx{LOGGER}->fine("Entering installer_initiation");
	# set program vars
	$| = 1;
	$$ctx{PARENTPID} = $$;

	($self->{INSTALLONLY} && (!$$ctx{INSTALLONLY})) && ($$ctx{INSTALLONLY} = $self->{INSTALLONLY});

	# set the interrupt callback
	$self->set_interrupt($ctx);

	# display title and copyright info.
	$$ctx{INSTALLERTITLE} = "$self->{TITLE} $self->{VERSION}";
	(!$self->{COPYRIGHTH}) && ($self->{COPYRIGHTH} = $self->{MSG}{COPYRIGHTH});
	(!$self->{COPYRIGHTB}) && ($self->{COPYRIGHTB} = $self->{MSG}{COPYRIGHTB});

	my $local_host = $ctx->get_local_hostname();
        if (!$local_host) {
           #$$ctx{LOGGER}->error(Utils::_tr("Installer::installer_initiation Local Host name is not defined in /etc/hosts file or /etc/resolv.conf correctly.", 14, 1107));
           $$ctx{LOGGER}->error("Installer::installer_initiation Local Host name is not defined in /etc/hosts file or /etc/resolv.conf correctly.");
           Utils::vxif_bp(Utils::_tr("FATAL ERROR - Local host name is not defined correctly.\n Please check the /etc/hosts or /etc/resolv.conf file for an incorrect entry.", 14, 1108));
	   $$ctx{LOGGER}->fine("Exiting installer_initiation with error");
           return 0;
        } else {
           #$$ctx{LOGGER}->info(Utils::_tr("Local System: ${local_host}\n", 14, 1037, "${local_host}"));
           $$ctx{LOGGER}->info("Local System: ${local_host}\n");
	   $$ctx{LOGGER}->fine("Exiting installer_initiation successfully");
           return 1;
        }
}

#
# Self initialization.
#
# Input: 1) the reference to self;
#        2) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize {
	my $self = shift;
	my $ctx = $self->get_context();
	my (@lt) = localtime();
        my ($rc);

	# can check this early on.
	# ... NOT!!!
#	if (scalar(keys(%{$self->{PRODUCT}})) < 1) {
#		die Utils::_tr("No products to install.");
#	}
	# should probably require presence in config file,
	# and enforce when reading

	# if response file specified, read it
	$self->check_response_file($ctx);

	# see if alternate extraction volume specified
	$self->check_hostspace($ctx);

	# get the local host OS info
	my $ifmgr = $self->get_interface_manager();
#	my $system_ops = $ifmgr->get_system_ops();
#	$system_ops->compute_local_host_osinfo();
	my $system_info = $ifmgr->get_system_info();
	my $local_hostname = $system_info->get_local_hostname();
	my $local_host_os_info = $system_info->get_host_os_info($local_hostname);
	my $local_os = $local_host_os_info->get_name();

	my $local_cmds = $ctx->get_os_cmds($local_os);
	my $local_mkdir_cmd = $local_cmds->{MKDIR};
	my $local_tmppath = $ctx->get_host_tmppath($local_hostname);

	# make sure local working directory exists
	if (! -d $local_tmppath) {
		Utils::vxif_dol("${local_mkdir_cmd} -p ${local_tmppath}");
	}
	# make sure local working directory is writable by all
	chmod 0777, $local_tmppath;

	# set the environment variables
	$ENV{VXIF_LOCAL_OS} = $local_os;
	$ENV{VX_BMCMNEM}="CPI";
	$ENV{VX_BMCORIGID}=9;
	$ENV{VX_BMCDOMAINDIR}=$ENV{VXIF_HOME};
	$ENV{VXIF_ORIGENVLANG}=$ENV{LANG} if (!$ENV{VXIF_ORIGENVLANG});
	$ENV{VXIF_ORIGENVLANG}="C" if (($ENV{CPIEN}) || (!$ENV{LANG}));
	$ENV{LANG}="C";
	$ENV{LC_ALL}="C";

	my $platform_info = $ifmgr->get_platform_info();
	my $local_os_installer = $platform_info->get_os_installer($local_os);
#	$local_os_installer->get_local_host($ctx);
	$local_os_installer->get_local_domain($ctx);

	# set the global messages
	$self->set_msgs($ctx);

	# setup time
	if (!$$ctx{TIME}) {
		$$ctx{TIME}=sprintf("%d%02d%02d%02d%02d", $lt[4]+1, $lt[3], $lt[2], $lt[1], $lt[0]);
	}

	# setup the tmp directory
	$$ctx{TMPPATH} = $self->{TMPPATH};

	# store the program name (and some other stuff) in the context
	(!defined($$ctx{PROGRAM})) && ($$ctx{PROGRAM} = $self->{PROGRAM});
	(!defined($$ctx{USESSH})) && ($$ctx{USESSH} = $self->{USESSH});
        (!defined($$ctx{IPv6})) && ($$ctx{IPv6} = $self->{IPv6});
	(!defined($$ctx{PASSWORD})) && ($$ctx{PASSWORD} = $self->{PASSWORD});

	# create the logger
	$self->create_logger($ctx);

	# for each product/supported os we added, transfer some info to context
	foreach my $upi (keys(%{$self->{PRODUCT}})) {
		foreach my $os (keys(%{$self->{PRODUCT}{$upi}})) {
			my $os_installer = $platform_info->get_os_installer($os);
			$os_installer->{PRODUCT_ORDERING} = $self->{PRODUCT_ORDERING}{$os};
			$os_installer->add_product($upi, $self->{PRODUCT}{$upi}{$os});
		}
	}

	# more initialization
	$rc = $self->installer_initiation();

	return $rc;
}

#
# Check for ssh/rsh availability on target host.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is successful; 0 otherwise.
#
sub check_system_ssh (\%$) {
	my $self = shift;
	my ($ctx, $rsh) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $os_cmds = $ctx->get_os_cmds($trg_os);
	my $mkdir_cmd = $os_cmds->{MKDIR};
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();

	# create log directory over rsh/ssh on target host
	$events->try_create_target_logdir($trg_host);
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $trg_logdir = "${trg_tmppath}/$self->{PROGRAM}$$ctx{TIME}";
	if (Utils::vxif_dor("${mkdir_cmd} -p ${trg_logdir}", "", $ctx)) {
		$events->create_target_logdir_fail($trg_host);
		return 0;
	}
	$events->create_target_logdir_pass($trg_host);

	return 1;
}

#
# Check for scp/rcp availability on target host.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is successful; 0 otherwise.
#
sub check_system_scp ($$) {
	my $self = shift;
	my ($ctx,$rcp) = @_;
	my ($error_handler,$do,$msgl,$msgr);
	my ($src_timefile) = $self->{TIMEFILE};
	my $local_host = $ctx->get_local_hostname();
	my $trg_host = $ctx->get_target_hostname();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $os_cmds = $ctx->get_os_cmds($trg_os);
	my $cat_cmd = $os_cmds->{CAT};
	my $rmr_cmd = $os_cmds->{RMR};

        $$ctx{LOGGER}->fine("Entered Installer::check_system_scp");
	
	# create the time file for rcp test
	$error_handler = sub {
		#$$ctx{LOGGER}->error(Utils::_tr("Unable to create ${src_timefile} on ${local_host} for ${rcp} testing.", 14, 1047, "${src_timefile}", "${local_host}", "${rcp}"));
		$$ctx{LOGGER}->error("Unable to create ${src_timefile} on ${local_host} for ${rcp} testing.");
	};
	Utils::vxif_writef($$ctx{TIME}, $src_timefile, 1, $error_handler);
	if (!(-e $src_timefile)) {
		#$$ctx{LOGGER}->error(Utils::_tr("Unable to create ${src_timefile} on ${local_host}\n", 14, 1048, "${src_timefile}", "${local_host}"));
		$$ctx{LOGGER}->error("Unable to create ${src_timefile} on ${local_host}\n");
		return 0;
	}
	
	$events->test_rcp($rcp, $trg_host);

	my $os_installer = $ctx->get_os_installer($trg_os);
	my $trg_timefile = $os_installer->{TIMEFILE};

        $do = Utils::vxif_copy(${src_timefile}, ${trg_timefile}, $ctx, $local_host, $trg_host);
	
	if ($do) {
		$events->rcp_test_fail1($rcp, $trg_host);
		return 0;
	} else {
		$do = Utils::vxif_dor("${cat_cmd} ${trg_timefile}", "", $ctx);
		#chomp($do);
		# chomp won't do it.  now that we're going between various systems,
		# we need a slightly more powerful chomp.  maybe there's a way to
		# override it in perl.  maybe we shouldn't do that, though.
		# unix  ==> "\n"
		# win32 ==> "\r\n"
		# mac   ==> "\r"
		$do =~ s/[\r\n]+$//;
		if ($do ne $$ctx{TIME}) {
			$events->rcp_test_fail2($rcp, $trg_host);
			return 0;
		}
	}
	$events->rcp_test_pass($rcp, $trg_host);

	if (Utils::vxif_dor("${rmr_cmd} ${trg_timefile}", "", $ctx)) {
		#$$ctx{LOGGER}->warning(Utils::_tr("Unable to remove ${trg_timefile} on ${trg_host}.", 14, 1050, "${trg_timefile}", "${trg_host}"));
		$$ctx{LOGGER}->warning("Unable to remove ${trg_timefile} on ${trg_host}.");
	}
	$$ctx{LOGGER}->fine("Exited Installer::check_system_scp");
	return 1;
}

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	      => the official name of the product/bundle/train/whatever to install;
#         VERSION	      => the official version of the product/bundle/train/whatever to install;
#         PROGRAM	      => program name to start this installer;
#         LOGGER 	      => reference to the Logger object; This is optional;
#         TITLE		      => product title; This is option; The default title is
#                          "Veritas Installer";
#         COPYRIGHTH    => product copyright header; This is optional;
#         COPYRIGHTB	  => product copyright message; This is optional;
#         TMPPATH 	    => the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         INSTALLVLIC	  => automatically install the Veritas license package if it is not present on
#                          the target system; This is option; By default, VERITAS license will be added
#                          to the target host for the licensable products;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	      => indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         IPv6             => indicates if the address is IPv6 format;
#         LOGLEVEL	    => indicate the log level; This is optional;
#                          The default is VxIF::Logger::LEVEL_INFO;
#         LOGDIR        => the complete path to the log directory; This is optional; The
#                          default is /var/tmp/<PROGRAM>;
#         LOGFILE       => the name of the log file; This is optional; The default is
#                          <PROGRAM><TIME>.log;
#         SUPPORTEDOS	  => a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, Darwin, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#		  SYSTEMS		=> list of target systems.
#
# Return: an instance of the Installer object.
#
sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $tmpdir = "/tmp";
	my $tmppath = "/var/tmp/vxif";
	if (Utils::running_on_windows()) {
		$tmpdir = "c:$tmpdir";
		$tmppath = "c:$tmppath";
	}
	my $initial_loglevel = eval Logger::LEVEL_INFO;

	my $self = {
		TITLE          => "VxIF",
		PRODUCT        => 'cpi',
		PROGRAM        => 'vxif',
		TASK           => 'install',
		INSTALLONLY    => 0,
		USESSH         => 0,
                IPv6           => 0,
		TMPPATH        => $tmppath,
		LOGLEVEL       => $initial_loglevel,
		TIMEFILE       => "$tmpdir"."/vxif_time",
		SUPPORTEDOS    => [ qw(SunOS Linux Darwin HPUX AIX OSF1 IRIX64 FreeBSD) ],
		CMDSEARCHPATHS => [ qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin /local/bin) ],
		@_,
	};

	bless($self, $class);

	# set up interface manager
	my $ifmgr = $self->get_interface_manager();
	if (!$ifmgr) {
		require VxIF::Interface::Classic::ClassFactory;
		my $class_factory = new VxIF::Interface::Classic::ClassFactory();
		$ifmgr = $class_factory->get_interface_manager();

		$self->set_interface_manager($ifmgr);
	}
	# create context, asap
	my $ctx = new Context($self);
	$self->set_context($ctx);

	# it's used this way everywhere but in the construction.
	# fix here.
	$self->{PRODUCT} = {};

	$self->process_args(@ARGV);

	# initialization, in its varied and wondrous forms
	$self->initialize();

	return $self;
}

sub get_interface_task {
	my $self = shift;
	my $interface_task = $self->{INTERFACE_TASK};

	return $interface_task;
}

sub get_interface_manager {
	my $self = shift;
	my $interface_manager = $self->{INTERFACE_MGR};

	return $interface_manager;
}

sub set_interface_manager {
	my $self = shift;
	my ($interface_manager) = @_;

	$self->{INTERFACE_MGR} = $interface_manager;
}

sub get_context {
	my $self = shift;
	my $context = $self->{context};

	return $context;
}


sub set_context {
	my $self = shift;
	my ($context) = @_;

	$self->{context} = $context;
}

sub get_task_proc ($) {
	my $self = shift;
	my ($task_type) = @_;
	my $dispatch_table = {
		'query' => sub {return $self->perform_query();},
		'precheck' => sub {return 1;}, # done during host validation
		'install' => sub {return $self->install_product_on_target_hosts();},
		'uninstall' => sub {return $self->uninstall_product_on_target_hosts();},
		'configure' => sub {return $self->configure_product_on_target_hosts();},
		'upgrade' => sub {return $self->upgrade_product_on_target_hosts();},
	};
	my $task_proc = $dispatch_table->{$task_type};

	return $task_proc;
}

sub exec ($) {
	my $self = shift;
	my ($task_type) = @_;
	my $ctx = $self->get_context();
	my $task_proc = $self->get_task_proc($task_type);

	# make sure requested task type is supported
	if (!$task_proc) {
		# invalid task type, don't even bother to initialize
		my $eff_task_type = (defined($task_type) ? $task_type : "Unknown");
		die Utils::_tr("Unsupported task type: ${eff_task_type}", 14, 1109, "${eff_task_type}");
	}

	# task type is valid.  stick it in.
	$self->{TASK} = $task_type;
	$ctx->set_task_type($task_type);

	# I *hate* to do this 'if' here, but one of the more obvious
	# hack alternatives is to do it "in-line" in all methods that
	# require host names.  Fix soon! (DE 9-8-05)
	if ($task_type ne 'query') {
		# make sure can do operation on specified hosts
		$self->validate_target_hosts();
	}

	# system initialized, hosts validated.  life is good.
	# do it.
	my $rc = $task_proc->();
	if (!$rc) {
		my $logger = $ctx->get_logger();
		my $trg_upi = $ctx->get_target_upi();
		#$logger->error(Utils::_tr("Unable to ${task_type} ${trg_upi} on the target systems.", 14, 1110, "${task_type}", "${trg_upi}"));
		$logger->error("Unable to ${task_type} ${trg_upi} on the target systems.");
		$self->cleanup($ctx);
	}

	$rc = $self->installer_completion();

	return $rc;
}

sub validate_target_hosts {
	my $self = shift;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $rc = 1;

        # jpk - Adding logging messages.  Not knowing where I am when this blows up is bloody well annoying on every level.
        $$ctx{LOGGER}->fine("Entering Installer::validate_target_hosts");
	$events->pre_target_systems();

	my $systems = $ctx->get_target_systems();
	my $num_systems = scalar(@$systems);

	if (!$num_systems) {
		# get the list of systems to perform product installation task
		$rc = $self->prompt_for_target_systems($ctx);
		if (!$rc) {
			#$$ctx{LOGGER}->error(Utils::_tr("Unable to get the systems to perform product installation and configration operations.", 14, 1111));
			$$ctx{LOGGER}->fine("Unable to get the systems to perform product installation and configration operations.");
			$self->cleanup($ctx);
		}
	}

	# have at least one hostname.
        $$ctx{LOGGER}->fine("Checking system for IPv6 address");

	# perform initial system check.
        $rc = $self->initial_target_systems_check();
        $rc = 1;
	if ($rc) {
		# all's well
		$events->target_system_check_pass();
	} else {
		# uh-oh, problem(s).
		#my $msg = Utils::_tr("Initial system check failed.", 14, 1086);
		$$ctx{LOGGER}->error("Initial system check failed.");
		$events->target_system_check_fail();
		$self->cleanup($ctx, 0, 0);
	}

        $$ctx{LOGGER}->fine("Exiting Installer::validate_target_hosts");
	return $rc;
}

sub perform_query {
	my $self = shift;
	my $task_obj = $self->get_interface_task();

	if ($task_obj) {
		my $installer_info = $task_obj->get_installer_info();
		$installer_info->query();
	} else {
		die Utils::_tr("Cannot obtain interface task for query.", 14, 1112);
	}

	return 1;
}

#
# Add a product to the installer. Users must specify the OS this product is intended for.
# If the product support multiple OSs (i.e. cross OS installation), users must call
# add_product() multiple times with the supported OSs respectively.
#
# Input: 1) unique product identifier (UPI) of the product; This must not be null;
#        2) the OS this product is intended for; This must not be null; The allow
#           values are "SunOS", "HPUX", "Linux", "AIX", and "OSF1";
#        3) the reference to the product object; This must no be null;
#
# Return: 1 if the product is successfully added; Otherwise, 0 is returned.
#
sub add_product ($$$) {
	my $self = shift;
	my ($upi,$os,$product) = @_;
	my $ifmgr = $self->get_interface_manager();
	my $events = $ifmgr->get_events();

	if ($upi && $os && $product) {
		my $platform_info = $ifmgr->get_platform_info();
		my $os_valid = $platform_info->os_supported($os);
		if (!$os_valid) {
			$events->os_not_supported($os);
			return 0;
		} else {
			$self->{PRODUCT}{$upi}{$os} = $product;
			push(@{$self->{PRODUCT_ORDERING}{$os}}, $upi);
		}
	} else {
		($upi) || ($events->missing_upi());
		($os) || ($events->missing_os());
		($product) || ($events->missing_product());
		
		return 0;
	}
	
	return 1;
}

#
# Start the installation of the product(s).
#
# Read response file if specified.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the CFG hash;
#
# Return: void
#
sub check_response_file (\%\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $cfg = $ctx->get_config();
	my $response_file = $self->{RESPONSEFILE};

	if ($response_file) {
		# load the response file
		my $response_info = Response::read_response_file($response_file);

		# get data
		my $sections = $response_info->{sections};

		# populate ctx from the context section
		my $context = $sections->{context};
		if ($context) {
			for my $key (keys(%$context)) {
				my $val = $context->{$key};
				if ($key eq "SYSTEMS") {
					$ctx->add_target_systems($context->{SYSTEMS});
				} elsif ($key =~ /^(UPI|HOST|MODE)$/) {
					if (!exists($ctx->{TARGET})) {
						$ctx->{TARGET} = {};
					}
					$ctx->{TARGET}{$key} = $val;
				} elsif ($key eq "HOSTSPACE") {
					# command line, if specified (and,
					# therefore, already processed),
					# overrides response file
					if (!$self->{HOSTSPACE}) {
						$self->{HOSTSPACE} = $val;
					}
				} else {
					$ctx->{$key} = $val;
				}
			}
		}

		# populate cfg from the config section
		my $config = $sections->{config};
		if ($config) {
			for my $key (keys(%$config)) {
				my $val = $config->{$key};
				$cfg->{$key} = $val;
			}
		}

		# post-process

		$ctx->{RESPONSEFILE} = $self->{RESPONSEFILE};
		$ctx->{PASSWORD} = $self->{PASSWORD};
	} elsif ($self->{OLDRESPONSEFILE}) {
		if (-f $self->{OLDRESPONSEFILE}) {
			my %context;
			my %config;
			require "$self->{OLDRESPONSEFILE}";

			# copy values from context hash into Context object.
			for my $key (keys(%context)) {
				my $val = $context{$key};
				$ctx->{$key} = $val;
			}

			# populate cfg from the config hash
			for my $key (keys(%config)) {
				my $val = $config{$key};
				$cfg->{$key} = $val;
			}

			# pretend we've read the new format
			$ctx->{RESPONSEFILE} = $self->{OLDRESPONSEFILE};
			# but don't forget where we got it
			$ctx->{OLDRESPONSEFILE} = $self->{OLDRESPONSEFILE};
			$ctx->{PASSWORD} = $self->{PASSWORD};
		} else {
			die Utils::_tr("Legacy format response file '$self->{OLDRESPONSEFILE}' does not exist.", 14, 1113, "$self");
		}
	} else {
		# no response file.  start with empty hashes.
	}
}

sub check_hostspace (\%) {
	my $self = shift;
	my ($ctx) = @_;

	if ($self->{HOSTSPACE}) {
		my $hostspace_spec = $self->{HOSTSPACE};
		my $hostspace_info = {};
		# entry is <hostname> = <volname> [, ...]
		my @specs = split(/\s*,\s*/, $hostspace_spec);
		for my $spec (@specs) {
			if ($spec =~ /([\.\-\w]+)\s*:\s*([\/\w]+)/) {
				my $host = $1;
				my $vol = $2;
				$hostspace_info->{$host} = $vol;
			}
		}
		$ctx->{HOSTSPACE} = $hostspace_info;
	}
}

#
# Start the installation of the product(s).
#
# Return: 1 if the installation is successful. Otherwise, 0 is returned.
#
sub install {
	my ($self) = @_;

	my $msg = Utils::_tr("install() method no longer supported.  Use exec('install|uninstall|...') instead.\n", 14, 1114);
	die $msg;
}

#
# Check to make sure the system names are valid.
#
# Input: 1) reference to self;
#        2) the reference to the installation context;
#
# Return: 1 if the system names are valid; 0 otherwise.
#
sub check_system_names ($) {
   my $self = shift;
   my ($ctx) = @_;
   my ($host,$sname,$upi,$os, $test_ip, $ip_format);
   my ($rc) = 1;
   my ($idx) = 0;
   my $hosts = $ctx->get_target_systems();
   my $num_hosts = scalar(@$hosts);
   my $ifmgr = $ctx->get_interface_manager();
   my $events = $ifmgr->get_events();

   $$ctx{LOGGER}->entering("Installer::check_system_names");
   # make sure we have at least one system to install the stuff
   if ($num_hosts < 1) {
      $events->missing_hostnames();
      return 0;
   }

   for my $host (@$hosts) {
      $sname = $host;
      $sname =~ s/[A-Za-z0-9_-]//g;
      # jpk - modification for ET 520347 adding support for IPv6 addresses added the ":::::"
      # so the IPv6 addresses are recognized.  This will NOT recognize the IPv6 localhost address of 
      # ::1/128
      if (($sname ne "") && ($sname ne "..") && ($sname ne "...") && ($sname != /\:\:\:/)) {
         $events->invalid_hostname($host);
         $rc = 0;
      }
      if ($sname =~ /\:/){
         ($test_ip, $ip_format) = Utils::vxif_ipcheck($host);
         if (($test_ip == 1) && ($ip_format = 6)) {
            # Set global variable indicating this is an IPv6 address
            $$ctx{IPv6} = $self->{IPv6} = 1;
            $$ctx{USESSH} = $self->{USESSH} = 1;
         }
      }
      if (($test_ip == 0) && (Utils::vxif_list($host,@$hosts) != $idx)) {
         $events->duplicate_hostname($host);
         $rc = 0;
      }
      ++$idx;
   }
   $$ctx{LOGGER}->exiting("Installer::check_system_names");
   return $rc;
}
#
# Prompt user for the systems to be tasked.
#
# Input: 1) reference to self;
#        2) the reference to the installation context;
#
# Return: 1 if everything is successful; 0 otherwise.
#
sub prompt_for_target_systems (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $logger = $ctx->get_logger();
        my ($tsl, $rc, $local_os);

	$logger->entering("Installer::prompt_for_target_systems");

	# see if any here ... probably from command line
	if ($self->{SYSTEMS} && scalar(@{$self->{SYSTEMS}}) > 0) {
		# yup have some.
		# override context (probably from response file)
		$ctx->clear_target_systems();
		for my $system (@{$self->{SYSTEMS}}) {
			$ctx->add_target_system($system);
                        # Check if the address is valid, and what the format is
                        my ($rc, $format) = Utils::vxif_ipcheck($system);
                        if (($rc == 1) && ($format = 6)) {
                           # Set global variable indicating this is an IPv6 address
                           $$ctx{IPv6} = $self->{IPv6} = 1;
                           $$ctx{USESSH} = $self->{USESSH} = 1;
                        }
		}
	}

	my $trg_systems = $ctx->get_target_systems();
	my $num_trg_systems = scalar(@$trg_systems);
	my $trg_upi = $ctx->get_target_upi();

	# return if we already have a list of systems to be tasked.
	return 1 if ($num_trg_systems > 0);

	if (!$trg_upi) {
		#$$ctx{LOGGER}->error(Utils::_tr("No target UPI specified.", 14, 1115));
		$$ctx{LOGGER}->error("No target UPI specified.");
		return 0;
	}

	my $task_type = $ctx->get_task_type();

	if (!defined($task_type)) {
		#$$ctx{LOGGER}->error(Utils::_tr("No task specified.", 14, 1116));
		$$ctx{LOGGER}->error("No task specified.");
		return 0;
	}

	if (Utils::vxif_list($task_type, @{$self->{SUPPORTEDTASKS}}) < 0) {
		#$$ctx{LOGGER}->error(Utils::_tr("${task_type} is not one of the supported tasks.", 14, 1117, "${task_type}"));
		$$ctx{LOGGER}->error("${task_type} is not one of the supported tasks.");
		return 0;
	}

        # jpk - 23 Mar 03 ET 578866
        $local_os = $ctx->get_local_os_name();

        # jpk - Jan 16, 06 attempting to wire in parameter -forcelocal.  Checking for variable set.
        # If the variable is not set, go forward with the prompt.
        if ($self->{LOCALINSTALL}) {
           $tsl = $$ctx{hostinfo}{hostname};
        }  elsif (($self->{RESPONSEFILE}) && (!$$ctx{SERVER})){
           # Changing this for ET 537285 to add the local host if the response
           # file does not have a server listed - jpk 11 Feb 06
           $tsl = $$ctx{hostinfo}{hostname};
           Utils::vxif_desp($tsl);
        }  elsif ("$local_os" =~ "NativePerl") {
           # jpk - 23 Mar 06 - ET 578866 - if the host is windows do not list it as the default system to install.
           $tsl = $events->input_system_names($task_type, $trg_upi);
        } else {
           # jpk - Jan 11, 06 changing this for ET 510871 to add the local host as the default answer.
           $tsl = $events->input_system_names($task_type, $trg_upi, "$$ctx{hostinfo}{hostname}");
           Utils::vxif_desp($tsl);
        }
        $ctx->add_target_systems($tsl);
        
	# check to make sure the system names are valid
	$rc = $self->check_system_names($ctx);
	return $rc if ($rc == 0);

	$events->post_system_names();

	return $rc;
}  ### end of prompt_for_target_systems

#
# Determine which local commands to use for
# remote execution/copy, if any non-local
#
# Input: 1) reference to self;
#        2) reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub setup_remote_exec {
	my $self = shift;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $hosts = $ctx->get_target_systems();
	my $rc = 1;

	if (!$ctx->{USESSH}) {
		return 1;
	}

	# if at least one remote host, set up ssh/scp
	for my $host (@$hosts) {
		# can't set current target host,
		# since os info is not yet known.
		# just set hostname.  may be enough. (even unnecessary?)
		$ctx->set_target_hostname($host);
		my $hostinfo = $ctx->get_hostinfo();
		my $is_local = $hostinfo->is_local_host($host);

		if (!$is_local) {
			# found a remote system.  find rsh/rcp commands.
			my $ifmgr = $ctx->get_interface_manager();
			my $system_info = $ifmgr->get_system_info();
			my $error = $system_info->setup_secure_remote_ops();
			if ($error) {
				my $msg = $error->get_msg();
				if ($msg =~ /ssh/) {
					$events->cannot_find_ssh();
				} elsif ($msg =~ /scp/) {
					$events->cannot_find_scp();
				}
				$rc = 0; # die???
			} else {
				$events->ssh_scp_info();
			}
			# use these for all remote commands.
			# may change eventually???
			last;
		}
	}

	return $rc;
}

#
# Obtain system information for a given target host.
#
# Input: 1) reference to self;
#        2) a target host
#
# Return: 1 if successful; 0 otherwise.
#
sub validate_target_host {
	my $self = shift;
	my ($host) = @_;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $task_type = $ctx->get_task_type();
	my $logger = $ctx->get_logger();
	my $rc = 1;

	my $local_cmds = $ctx->get_local_cmds();
	my $rsh_str = $local_cmds->{RSH};
	my $rsh_aref = [ split(/\s+/, $rsh_str) ];
	my $rsh = $rsh_aref->[0];
	my $rcp_str = $local_cmds->{RCP};
	my $rcp_aref = [ split(/\s+/, $rcp_str) ];
	my $rcp = $rcp_aref->[0];

        $$ctx{LOGGER}->entering("Entered Installer::validate_target_host");
	
	if (Utils::vxif_localsys($ctx)) {
	} else {
		$events->test_rsh($rsh, $host);
	}

	# first "real" action ... populate this object per this host
	my $system_info = $ifmgr->get_system_info();
	my $error = $system_info->compute_target_host_osinfo($host);
        $$ctx{LOGGER}->finer("Entered Installer::validate_target_host Checking error code ${error} #");
	if ($error) {
		my $msg = $error->get_msg();
		$logger->error($msg);
		$rc = 0;
	}

	if ($rc) {
                $$ctx{LOGGER}->fine("Entered Installer::validate_target_host Checking return code ${rc}  #1");
		$ctx->set_target_hostname($host);

		if (Utils::vxif_localsys($ctx)) {
		} else {
			$events->rsh_test_pass($rsh, $host);
			# next "real" action ...
			# create a log dir (unnecessary,
			#        since log local, and rcs already assured
			#   and
			# copy a time file (probably o.k.)
			$rc = $self->check_communication($ctx, $rsh, $rcp);
                        $$ctx{LOGGER}->fine("Entered Installer::validate_target_host Checking return code ${rc}  #2");
		}
	} else {
		if (!Utils::vxif_localsys($ctx)) {
			$events->rsh_test_fail($rsh, $host);
		}
	}

	if ($rc) {
		# check for product (seems like we could not
		# have gotten this far if this were to fail)
		my $host_os_inst = $ctx->get_host_os_installer($host);
		if (!$host_os_inst) {
			$events->configuration_not_supported($host);
			$rc = 0;
		}
	}

	if ($rc) {
		$events->check_os_version($host);

		# check for system support
		if ($task_type eq "install" || $task_type eq "precheck") {
			$events->test_system_support($host);
			# last "real" action ... make sure product
			# support this version/architecture of the os
			$rc = $self->check_system_support($ctx);
			if ($rc) {
				$events->system_support_pass($host);
			} else {
				$events->system_support_fail($host);
			}
		}
	}

        $$ctx{LOGGER}->exiting("Installer::validate_target_host rc  ${rc}");
	return $rc;
}


#
# Initial target systems check.
#
# Input: 1) reference to self;
#        2) reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub initial_target_systems_check {
	my $self = shift;
	my $ctx = $self->get_context();
	my $task_type = $ctx->get_task_type();
	my $hosts = $ctx->get_target_systems();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $ncs = [];
	my $ccs = [];
	my $rc = 1;

        $$ctx{LOGGER}->entering("Installer::initial_target_systems_check");

        $$ctx{LOGGER}->fine("Check that the system name is valid.  This will also find if it is an IPv6 or IPv4");
        # Check if the address is valid, and what the format is
        foreach my $host (@$hosts) {
           my ($rc2, $format) = Utils::vxif_ipcheck($host);
           if (($rc2 == 1) && ($format = 6)) {
              # Set global variable indicating this is an IPv6 address
              $$ctx{IPv6} = $self->{IPv6} = 1;
              $$ctx{USESSH} = $self->{USESSH} = 1;
              $$ctx{LOGGER}->info("$host is in IPv6 format");
           }
        }
        
	$rc = $self->setup_remote_exec();
	if (!$rc) {
		return 0;
	}

	# have necessary rsh/ssh/rcp/scp.
	# so, we will proceed.
        
	$events->pre_check_syscomm();

	foreach my $host (@$hosts) {
           $ctx->set_target_hostname($host);

           # ask the host about its os information
           $$ctx{LOGGER}->finer("Installer::initial_target_systems_check going to validate_target_host");
           $rc = $self->validate_target_host($host);
           $$ctx{LOGGER}->finer("Installer::initial_target_systems_check BACK from validate_target_host RC = ${rc}");

           # check communication
           if ($rc) {
              push(@$ccs, $host);
           } else {
              push(@$ncs, $host);
              next;
           }

           # check installation requirements for precheck phase
           if ($task_type eq "precheck") {
              my $rc = 1;

              # check for external dependencies
              my $host_os_inst = $ctx->get_host_os_installer($host);
              $rc &= $host_os_inst->check_for_external_dependencies($ctx);

              # check for require spaces
              $rc &= $host_os_inst->check_for_required_spaces($ctx);

              # bail out if unsuccessful
              return 0 if (!$rc);
           }
        }

	# there should be some upgrade checks here
	# upgrading all systems from the same version
	# all systems are in the same cluster
	# any others?
	if (scalar(@$ncs) > 0) {
		my $ncl = join(" ",@$ncs);

		return 0 if ((scalar(@$ccs) <= 0) || ($$ctx{RESPONSEFILE}));

		my $ccl = join(" ",@$ccs);
		if ($$ctx{RESPFILE}) {
			$events->install_valid_only($$ctx{PRODUCT}, $ccs);
		} else {
			my $ask = $events->input_install_valid_only($$ctx{PRODUCT}, $ccs);
			$self->cleanup($ctx) if (!defined($ask));
			return 0 if (!$ask);

			$ctx->set_target_systems($ccs);
		}
	}

	# do "sanity" checking on system names
	# (a little late, though, isn't it?)
	for my $host (@$ccs) {
		$ctx->set_target_hostname($host);
		my $upi_inst = $ctx->get_host_os_upi_installer($host);

		if ($upi_inst->can("check_system_names")) {
			# make array of one.  asap, create single host check
			$rc = $upi_inst->check_system_names($ctx, [$host]);
			if (!$rc) {
				return 0;
			}
		}
	}

        $$ctx{LOGGER}->exiting("Installer::initial_target_systems_check");
	return 1;
}

#
# Check whether local system has ssh/rsh permissions with a list of systems.
#
# Input: 1) reference to self;
#        2) the reference of the installation context;
#        3) ssh/rsh indicator;
#        4) scp/rcp indicator;
#
# Return: 1) if the communication check is successful; 0 otherwise.
#
sub check_communication ($$$) {
	my $self = shift;
	my ($ctx,$rsh,$rcp) = @_;
	my $rc = 1;

        $$ctx{LOGGER}->fine("Entered Installer::check_communication");
	# if local install, then trivially o.k.
	if (!Utils::vxif_localsys($ctx)) {
		# check ssh on target host
		$rc = $self->check_system_ssh($ctx, $rsh);

		if ($rc) {
			# rsh works.
			# try a basic rcp (might be scp)
			$rc = $self->check_system_scp($ctx, $rcp);
		}
	}

        $$ctx{LOGGER}->fine("Exited Installer::check_communication");
	return $rc;
}

#
# Check to make sure the target host is support by the target product.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is successful; 0 otherwise.
#
sub check_system_support ($) {
	my $self = shift;
	my $ctx = $self->get_context();
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $rc = 1;

	if ($upi_inst) {
		if ($upi_inst->can("is_os_supported")) {
			my $trg_os_name = $ctx->get_host_os_name($trg_host);
			my $trg_os_version = $ctx->get_host_os_version($trg_host);
			my $trg_os_architecture = $ctx->get_host_os_architecture($trg_host);

			$rc = $upi_inst->is_os_supported($ctx, $trg_host, $trg_os_name, $trg_os_version, $trg_os_architecture);
		} else {
			# if no method, assum o.k.
			$rc = 1;
		}
	} else {
		# the product does support the given system OS
		$rc = 0;
	}

	return $rc;
}

#
# Check product licensing on the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) before installation?;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_licensing_on_target_hosts (\%$) {
	my $self = shift;
	my ($ctx,$preinstall) = @_;
	my ($os);
	my ($rc) = 1;
	my $licensable = 0;
	my $licafter = 0;
	my $hosts = $ctx->get_target_systems();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_upi = $ctx->get_target_upi();
	my $system_info = $ifmgr->get_system_info();

	# was by target os.
	# make this by host for now.
	# we can sort out later, if we land on licensince.
	for my $host (@$hosts) {
		$ctx->set_target_hostname($host);
		my $os = $system_info->get_host_os_info($host);
		my $upi_inst = $ctx->get_host_os_upi_installer($host);
		my $trg_mode = $ctx->get_host_inst_mode($host);

		# check to see if this product is a licensable product
		if ($upi_inst->can("is_product_licensable")) {
			$licensable = $upi_inst->is_product_licensable($ctx, $trg_mode);
		}

		# check to see if need to check for licensing after the installation
		# check for product licensing at post-install
		if ($upi_inst->can("check_licensing_after_install")) {
			$licafter = $upi_inst->check_licensing_after_install($ctx, $trg_mode);
		}
		last;
	}

	if (!$licensable) {
		#$$ctx{LOGGER}->info(Utils::_tr("${trg_upi} is not licensable.", 14, 1118, "${trg_upi}"));
		$$ctx{LOGGER}->info("${trg_upi} is not licensable.");
		return 1;
	}

	$$ctx{LOGGER}->finer("preinstall=${preinstall}\nlicafter=${licafter}");
	return 1 if ($preinstall && $licafter);
	return 1 if ((!$preinstall) && (!$licafter));

	$events->pre_check_licensing($$ctx{INSTALLERTITLE}, $trg_upi, $hosts);

	foreach $os (keys(%{$$ctx{INSTALLERS}})) {
		$rc &= $$ctx{INSTALLERS}{$os}->check_product_licensing($ctx);
	}

	if ($rc) {
		if ($$ctx{FLAG}{KEYREG}) {
			$events->check_licensing_completed($trg_upi);
		} else {
			$events->check_licensing_verified($trg_upi);
		}
	} else {
		$events->check_licensing_fail($trg_upi);
	}

	return $rc;
}

#
# Install the select product to the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to %cfg;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_product_on_target_hosts {
	my $self = shift;
	my $ctx = $self->get_context();
	my $cfg = $ctx->get_config();
	my $trg_upi = $ctx->get_target_upi();
	my $task_type = $ctx->get_task_type();
	my $ifmgr = $ctx->get_interface_manager();
	my $product_ops = $ifmgr->get_product_ops();
	my $task_obj = $self->get_interface_task();
	my $installer_info = $task_obj->get_installer_info();
	my $logger = $ctx->get_logger();
	my $rc = 1;

	$rc =$self->check_licensing_on_target_hosts($ctx, 1);
	if (!$rc) {
		#$logger->error(Utils::_tr("Licensing check failed.", 14, 1119));
		$logger->error("Licensing check failed.");
		$self->cleanup($ctx);
	}

    if (!$$ctx{RESPONSEFILE}) {
		# maybe there should be an actual test inside the following code
		$rc = $self->build_product_config_on_target_hosts($ctx, $cfg);
		if (!$rc) {
			#$logger->error(Utils::_tr("Host configuration failed.", 14, 1120));
			$logger->error("Host configuration failed.");
			return 0;
		}
	}

	# Display and get a list of optional packages.
	# The user can select the packages to install.
	my $optpkgs = $self->get_optional_packages($ctx);
  	
	# display a list of packages to install for each platform
	$self->display_packages_to_be_installed($ctx);
    
	# check installation requirements
	if (!$self->check_system_installation_requirements($ctx)) {
		#$logger->error(Utils::_tr("Installation requirement check failed.", 14, 1080));
		$logger->error("Installation requirement check failed.");
		return 0;
	}
    
	# loop over all target hosts.
	my $hosts = $ctx->get_target_systems();
	for my $host (@$hosts) {
		#$logger->info(Utils::_tr("Installing ${trg_upi} on ${host}...\n", 14, 1121, "${trg_upi}", "${host}"));
		$logger->info("Installing ${trg_upi} on ${host}...\n");

		# still have to do this
		$ctx->set_target_hostname($host);

		my $os_name = $ctx->get_host_os_name($host);
		my $product_info = $installer_info->get_platform_product_info($os_name);

		my $common = $ctx->get_host_os_installer($host);

		# create the admin file on localhost.
		if (($common->can("create_local_admin_file"))&&(!$common->create_local_admin_file($ctx))) {
			$rc = 0;
		}

		my $error = $product_ops->install_on_host($product_info, $host);
		if ($error) {
			my $msg = $error->get_msg();
			$logger->error($msg);
			$rc = 0;
			next;
		}

		# now configure the product
		if (!$$ctx{INSTALLONLY}) {
			my $error = $product_ops->plain_configure_on_host($product_info, $host);
			if ($error) {
				my $msg = $error->get_msg();
				$logger->error($msg);
				$rc = 0;
				next;
			}
		}

		#$logger->info(Utils::_tr("${trg_upi} successfully installed on ${host}\n", 14, 1122, "${trg_upi}", "${host}"));
		$logger->info("${trg_upi} successfully installed on ${host}\n");
	}

	if ($rc) {
		#$logger->info(Utils::_tr("${trg_upi} has been successfully installed on all target hosts.", 14, 1123, "${trg_upi}"));
		$logger->info("${trg_upi} has been successfully installed on all target hosts.");
	} else {
		#$logger->info(Utils::_tr("Unable to install ${trg_upi} on all target hosts.", 14, 1124, "${trg_upi}"));
		$logger->info("Unable to install ${trg_upi} on all target hosts.");
	}

	return $rc;
}

#
# Specify product configuration parameters on the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to %cfg;
#
# Return: 1 if successful; 0 otherwise.
#
sub build_product_config_on_target_hosts {
	my $self = shift;
	my ($ctx, $cfg) = @_;
	my $trg_upi = $ctx->get_target_upi();
	my $trg_hosts = $ctx->get_target_systems();
	my $alt_choice = "";
	my $can_do_hosts = [];
	my $can_do_ixs = [];
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $rc = 0;

	# for now, just do this if we can for a given host/os
	my $num_trg_hosts = scalar(@$trg_hosts);
	for (my $i = 0; $i < $num_trg_hosts; $i++) {
		my $trg_host = $trg_hosts->[$i];
		my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
		if ($upi_inst->can("build_product_config_on_target_host")) {
			push(@$can_do_hosts, $trg_host);
			push(@$can_do_ixs, $i);
		}
	}

	my $num_can_do_hosts = scalar(@$can_do_hosts);
	if (!$num_can_do_hosts) {
		# here, assume will happen some other way
		return 1;
	}

    #if there is only one host that needs to be installed on, skip the host selecting screen.
    #the change is supposed to fix the defect 763196.
    if ($num_can_do_hosts == 1) {
        my $only_one_trg_host = $can_do_hosts->[0];
        my $only_one_upi_inst = $ctx->get_host_os_upi_installer($only_one_trg_host);
        $only_one_upi_inst->build_product_config_on_target_host($ctx, $cfg, $only_one_trg_host);
        return 1;
    }
    
	my $text_input = $ifmgr->get_text_input();
	my $status_bar = $ifmgr->get_status_bar();

	my $master_hosts_configured = {};

	while (1) {
		my $hosts_configured = $events->select_host_to_configure($can_do_hosts);
		for my $host (keys(%$hosts_configured)) {
			$master_hosts_configured->{$host} = 1;
		}
		my $num_hosts_configured = scalar(keys(%$master_hosts_configured));
		# if a host *can* be configured, then it *must* be configured.
		if ($num_hosts_configured >= $num_can_do_hosts) {
			# got 'em all
			my $msg = Utils::_tr("All hosts were configured.", 14, 1125);
			$status_bar->display_status($msg);
			$$ctx{LOGGER}->info("All hosts were configured.");
			$rc = 1;
			last;
		}

		# missed some
		my $errmsg = Utils::_tr("Not all hosts were configured.", 14, 1126);
		my $prompt = Utils::_tr("  Configure remaining hosts? (y) ", 14, 1127);
		my $eff_prompt = $errmsg . $prompt;
		my $ans = $text_input->get_line($eff_prompt);
		if ($ans =~ /n/i) {
			# user opted to not configure.  bail.
			$$ctx{LOGGER}->error("Not all hosts were configured.");
			$rc = 0;
			last;
		}
	}

	return $rc
}

#
# Configure the select product on the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to %cfg;
#
# Return: 1 if successful; 0 otherwise.
#
sub configure_product_on_target_hosts {
	my $self = shift;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $product_ops = $ifmgr->get_product_ops();
	my $task_obj = $self->get_interface_task();
	my $installer_info = $task_obj->get_installer_info();
	my $trg_upi = $ctx->get_target_upi();
	my $rc = 1;

        # jpk

	# check installation requirements
	if (!$self->check_system_installation_requirements($ctx)) {
		#$$ctx{LOGGER}->error(Utils::_tr("Installation requirement check failed.", 14, 1080));
		$$ctx{LOGGER}->error("Installation requirement check failed.");
		return 0;
	}

	# get list of all target hosts
    my $hosts = $ctx->get_target_systems();
	# loop through 'em, configuring each.
	for my $host (@$hosts) {
		$ctx->set_target_hostname($host);

		my $os_name = $ctx->get_host_os_name($host);
		my $product_info = $installer_info->get_platform_product_info($os_name);

		#$$ctx{LOGGER}->info(Utils::_tr("Configuring ${trg_upi} on ${host}...\n", 14, 1128, "${trg_upi}", "${host}"));
		$$ctx{LOGGER}->info("Configuring ${trg_upi} on ${host}...\n");
		my $error = $product_ops->configure_on_host($product_info, $host);
		if ($error) {
			my $msg = $error->get_msg();
			$$ctx{LOGGER}->error($msg);
			$rc = 0;
			#$$ctx{LOGGER}->info(Utils::_tr("Unable to configure ${trg_upi} on ${host}.", 14, 1129, "${trg_upi}", "${host}"));
			$$ctx{LOGGER}->info("Unable to configure ${trg_upi} on ${host}.");
			next;
		}
		#$$ctx{LOGGER}->info(Utils::_tr("${trg_upi} has been successfully configured on ${host}.", 14, 1130, "${trg_upi}", "${host}"));
		$$ctx{LOGGER}->info("${trg_upi} has been successfully configured on ${host}.");
	}

	return $rc;
}

#
# Upgrade the select product to the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to %cfg;
#
# Return: 1 if successful; 0 otherwise.
#
sub upgrade_product_on_target_hosts {
	my $self = shift;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $product_ops = $ifmgr->get_product_ops();
	my $task_obj = $self->get_interface_task();
	my $installer_info = $task_obj->get_installer_info();
	my $cfg = $ctx->get_config();
	my ($os, $optpkgs);
	my $target_os_list = [];
	my $trg_upi = $ctx->get_target_upi();
	my $upgrade_by_uninstall_install = 1;
	my $rc = 1;

	die Utils::_tr("upgrade method currently disabled.", 14, 1131);

	# loop over all target hosts
	my $hosts = $ctx->get_target_systems();
	for my $host (@$hosts) {
		# fix this asap (and everywhere)
		$ctx->set_target_hostname($host);

		my $upi_inst = $ctx->get_host_os_upi_installer($host);

		my $os_name = $ctx->get_host_os_name($host);
		my $product_info = $installer_info->get_platform_product_info($os_name);

		if ($upi_inst->can("pre_product_upgrade")) {
			$rc = $upi_inst->pre_product_upgrade($ctx, $cfg);
		}

		# Awww ... do we *have* to ???
		# Actually, this should go in common.
		# and the upgrade (uninst/inst) should be
		# done per package, and only if the versions differ,
		# and then only if they differ in the proper direction.
		#
		# do it here now, then stick it in common.
		if ($upgrade_by_uninstall_install) {
			my $common = $ctx->get_host_os_installer($host);

			# should have method to display packages to be upgraded.
			# this should be only packages that will change to a
			# higher revision (unless some sort of downgrade is
			# allowed).

			# duplicate some uninstall logic.
			# uninstall top level first checks req's on all hosts,
			# but just do it per host first, in this loop.
			$common->check_uninstall_requirements_on_host($ctx, $host);
			if ($common->can("create_local_admin_file")) {
				$rc = $common->create_local_admin_file($ctx);
			}
			$rc &= $common->uninstall_product_on_target_host($ctx, $cfg);

			# now duplciate some install logic.
			# install top level first checks req's on all hosts,
			# but just do it per host first, in this loop.
			$common->check_installation_requirements_on_host($ctx, $host);
			if ($common->can("create_local_admin_file")) {
				$rc = $common->create_local_admin_file($ctx);
			}
			$rc &= $product_ops->install_on_host($product_info, $host);
		}

		if ($upi_inst->can("post_product_upgrade")) {
			$rc = $upi_inst->post_product_upgrade($ctx, $cfg);
		}
	}

	return $rc;

	for my $os (keys(%{$$ctx{INSTALLERS}})) {
		if (exists($self->{PRODUCT}{$trg_upi}{$os})) {
			# rules out "pseudo-installers", such as NativePerl.
			# eventually, may become "non-pseudo".
			push(@$target_os_list, $os);
		}
	}

	# Pre/Post upgrade, infact the whole of upgrade should probably be relegated to CPICommon
	# Which is the real installer. Installer.pm is just a driver.
	# That will be inline with the way VxIF code is organised.

	#preupgrade goes here
	for my $os (@$target_os_list) {
		if ($self->{PRODUCT}{$trg_upi}{$os}->can("pre_product_upgrade")) {
			$rc &= $self->{PRODUCT}{$trg_upi}{$os}->pre_product_upgrade($ctx, $cfg);
		}
	}

	if ($upgrade_by_uninstall_install) {
		# save current task ... probably "upgrade"
		my $save_curr_task = $$ctx{TASK};

		# fake vxif into thinking it is uninstalling
		$$ctx{TASK} = "uninstall";

		#Uninstall first  
		# check installation requirements
		if (!$self->check_system_uninstall_requirements($ctx)) {
			#$$ctx{LOGGER}->error(Utils::_tr("Upgrade requirement check failed.", 14, 1132));
			$$ctx{LOGGER}->error("Upgrade requirement check failed.");
			$rc = 0;
		} 

		if (!$rc) {
			# restore task
			$$ctx{TASK} = $save_curr_task;
			return 0;
		}
		
		for my $os (@$target_os_list) {
			$rc &= $$ctx{INSTALLERS}{$os}->uninstall_product($ctx, $cfg);
		}
		
		#Install next  

		# fake vxif into thinking it is installing
		$$ctx{TASK} = "install";

		# display and get a list of optional packages. The user can select the packages he wants to install.
		$optpkgs = $self->get_optional_packages($ctx);
		
		# display a list of packages to install for each platform
		$self->display_packages_to_be_installed($ctx);
		
		# check installation requirements
		if (!$self->check_system_installation_requirements($ctx)) {
			#$$ctx{LOGGER}->error(Utils::_tr("Installation requirement check failed.", 14, 1080));
			$$ctx{LOGGER}->error("Installation requirement check failed.");
			# restore task
			$$ctx{TASK} = $save_curr_task;
			return 0;
		}
		
		for my $os (@$target_os_list) {
			$rc &= $$ctx{INSTALLERS}{$os}->install_product($ctx, $cfg);
		}

		# restore task
		$$ctx{TASK} = $save_curr_task;
	}

	#postupgrade goes here
	for my $os (@$target_os_list) {
		if ($self->{PRODUCT}{$trg_upi}{$os}->can("post_product_upgrade")) {
			$rc &= $self->{PRODUCT}{$trg_upi}{$os}->post_product_upgrade($ctx, $cfg);
		}
	}

	return $rc;
}

#
# Uninstall the select product to the target systems.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to %cfg;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_product_on_target_hosts {
	my $self = shift;
	my $ctx = $self->get_context();
	my $cfg = $ctx->get_config();
	my $trg_upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $trg_upi = $trg_upi_long;
    }
	my $ifmgr = $ctx->get_interface_manager();
	my $product_ops = $ifmgr->get_product_ops();
	my $task_obj = $self->get_interface_task();
	my $installer_info = $task_obj->get_installer_info();
	my $events = $ifmgr->get_events();
	my $logger = $ctx->get_logger();
	my $rc = 1;

	# check installation requirements
	if (!$self->check_system_uninstall_requirements($ctx)) {
		#$logger->error(Utils::_tr("Uninstallation requirement check failed.", 14, 1133));
		$logger->error("Uninstallation requirement check failed.");
		$rc = 0;
	} else {
		if ($$ctx{TASK} ne "upgrade") {
			Utils::vxif_bpl(Utils::_tr("\n$$ctx{PROGRAM} is now ready to uninstall ${trg_upi} packages.", 14, 1081, "$$ctx{PROGRAM}", "${trg_upi}"));
		}
	}
	Utils::vxif_prtc($ctx) if ($$ctx{TASK} ne "upgrade");
	return 0 if (!$rc);

	# loop over all hosts (not over os, then host).
	my $hosts = $ctx->get_target_systems();
	for my $host (@$hosts) {
		#$logger->info(Utils::_tr("Uninstalling ${trg_upi} on ${host}...\n", 14, 1134, "${trg_upi}", "${host}"));
		$logger->info("Uninstalling ${trg_upi} on ${host}...\n");

		$ctx->set_target_hostname($host);

		my $os_name = $ctx->get_host_os_name($host);
		my $product_info = $installer_info->get_platform_product_info($os_name);

		my $common = $ctx->get_host_os_installer($host);

		# create the admin file on localhost.
		if (($common->can("create_local_admin_file"))&&(!$common->create_local_admin_file($ctx))) {
			$rc = 0;
			next;
		}

		# let 'em make sure they really want to do the uninstall
		my $opt = $events->confirm_product_uninstall($host);
		if (!$opt) {
			$logger->error("Uninstallation cancelled on target host == ${host}.");
			$rc = 0;
			# ain't gonna do it for this host
			next;
		}

		# stop the product processes that are currently running.
		my $os_inst = $ctx->get_host_os_installer($host);
		Utils::vxif_bpl(Utils::_tr("All ${trg_upi} processes that are currently running on ${host} will be stopped.", 14, 1135, "${trg_upi}", "${host}"));
		if (!$os_inst->stop_product_processes($ctx, $cfg)) {
			my $errmsg = Utils::_tr("Unable to stop ${trg_upi} processes on ${host}.", 14, 1136, "${trg_upi}", "${host}");
			Utils::vxif_lpl($errmsg, "error", $ctx);
			#$logger->error(Utils::_tr("Unable to stop processes, prior to uninstall, on target host == ${host}", 14, 1137, "${host}"));
			$logger->error("Unable to stop processes, prior to uninstall, on target host == ${host}");
			$rc = 0;
			next;
		}

		my $error = $product_ops->uninstall_on_host($product_info, $host);
		if ($error) {
			my $msg = $error->get_msg();
			$logger->error($msg);
			$rc = 0;
			next;
		}
	}

	return $rc;
}

#
# Display and get the optional packages for the target UPI to be install on all the target hosts.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
# Return: reference to an array of optional packages
#
sub get_optional_packages (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($os, $optpkgs);
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_upi = $ctx->get_target_upi();

	foreach $os (keys(%{$self->{PRODUCT}{$trg_upi}})) {
		if ($self->{PRODUCT}{$trg_upi}{$os}->can("get_optional_packages")) {
		    my $installer_title = $$ctx{INSTALLERTITLE};
		    $events->pre_os_optional_packages($os, $installer_title);
			$optpkgs = $self->{PRODUCT}{$trg_upi}{$os}->get_optional_packages($ctx);
		}
	}

	return Utils::vxif_uniq(@$optpkgs);
}

#
# Display the packages for the target UPI to be install on all the target hosts.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
sub display_packages_to_be_installed (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($os, $pkg, $n);
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $hosts = $ctx->get_target_systems();

	for my $host (@$hosts) {
		# should eliminiate these kinds of states.
		$ctx->set_target_hostname($host);

		my $common = $ctx->get_host_os_installer($host);

		$common->display_product_packages_to_install($ctx);
	}
}

#
# Perform installation requirement checks on all the target hosts.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
sub check_system_installation_requirements (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $hosts = $ctx->get_target_systems();
	my $trg_upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $trg_upi = $trg_upi_long;
    }
	
	my $sc = 0;
	my $rc = 1;

	my $hosts_str = join(' ', @$hosts);
	my $installer_title = $$ctx{INSTALLERTITLE};
	$events->begin_checking_sysinst_reqs($installer_title);

	for my $host (@$hosts) {
		# should eliminiate these kinds of states.
		$ctx->set_target_hostname($host);
		my $os = $ctx->get_host_os_name($host);

		if ($sc == 0) {
			$events->begin_checking_hostinst_reqs($installer_title, $trg_upi, $os, $hosts_str);
		} elsif (($sc > 0) && ($sc % 3 == 0)) {
			$events->continue_checking_hostinst_reqs($installer_title, $trg_upi, $os, $hosts_str);
		}

		my $common = $ctx->get_host_os_installer($host);
		$rc = $common->check_installation_requirements_on_host($ctx, $host);

		++$sc;
	}

	if ($rc) {
		$events->sysinst_reqs_pass();
	} else {
		#$$ctx{LOGGER}->error(Utils::_tr("Installation requirements not satisfied on all target systems.", 14, 1138));
		$$ctx{LOGGER}->error("Installation requirements not satisfied on all target systems.");
		$events->sysinst_reqs_fail();
	}
	
	return $rc;
}

#
# Perform uninstall requirement checks on all the target hosts.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#
sub check_system_uninstall_requirements (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $hosts = $ctx->get_target_systems();
	my $trg_upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $trg_upi = $trg_upi_long;
    }
	my $sc = 0;
	my $rc = 1;

	my $hosts_str = join (' ', @$hosts);
	my $installer_title = $$ctx{INSTALLERTITLE};
	$events->begin_checking_sysuninst_reqs($installer_title);

	for my $host (@$hosts) {
		$ctx->set_target_hostname($host);
		my $os = $ctx->get_host_os_name($host);

		if ($sc == 0) {
			$events->begin_checking_hostuninst_reqs($installer_title, $trg_upi, $os, $hosts_str);
		} elsif (($sc > 0) && ($sc % 3 == 0)) {
			$events->continue_checking_hostuninst_reqs($installer_title, $trg_upi, $os, $hosts_str);
		}

		my $common = $ctx->get_host_os_installer($host);
		$rc = $common->check_uninstall_requirements_on_host($ctx, $host);

		++$sc;
	}

	return $rc;
}

#
# Perform completion tasks.
#
# Input: 1) the reference to self;
#        2) the reference to the installation context;
#        3) the reference to the configuration hash;
#
# Return: 1 if successful; 0 otherwise.
#
sub installer_completion {
	my $self = shift;
	my $ctx = $self->get_context();
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $cfg = $ctx->get_config();
	my $indent_str = $self->{MSG}{INDENT};

	# create response file
	my $rc = Response::write_response_file($ctx, $cfg);
	if (!$rc) {
		#$$ctx{LOGGER}->error(Utils::_tr("Unable to create the response file.", 14, 1139));
		$$ctx{LOGGER}->error("Unable to create the response file.");
		return 0;
	}

	my $respfilename = Response::compute_response_file_name($ctx);
	$events->inform_response_file($respfilename, $indent_str);

	my $program = $$ctx{PROGRAM};
	my $logfile = $$ctx{LOGGER}->get_logfile();
	$events->inform_log_file($program, $logfile, $indent_str);

	return 1;
}

#
# Cleanup.
#
# Input: 1) the reference of the installation context;
#        2) display response file location?;
#        3) exit code;
#        3) display summary;
#
sub cleanup (\%;$$$) {
	my $self = shift;
	my ($ctx, $respfile, $exit_code, $sum) = @_;
	my $responsefile = Response::compute_response_file_name($ctx);
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $title = $$ctx{INSTALLERTITLE};
	my $indent_str = $self->{MSG}{INDENT};

	$events->announce_cleanup($title);
	if ($respfile) {
		$events->inform_response_file($responsefile, $indent_str);
	} else {
		$exit_code = 1 if ($#_ < 3);
	}

	if (-f "$$ctx{LOGGER}->{LOGFILE}") {
		my $program = $$ctx{PROGRAM};
		my $logfile = $$ctx{LOGGER}->{LOGFILE};
		$events->inform_log_file($program, $logfile, $indent_str);
	}

	exit($exit_code);
}

#
# Print usage.
#
# Input: 1) the reference to self;
#
sub print_usage ($) {
	my $self = shift;
	my $ctx = $self->get_context();
	my ($exit) = @_;
	my ($program) = $self->{PROGRAM};
	my $tmppath = $self->{TMPPATH};
	my $trg_upi = $ctx->get_target_upi();
	my $ifmgr = $self->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $sq = "'";

	if ($self->{TASK} eq "uninstall") {
		my $usage_str = "${program} [<system1> <system2>...]\n";
		$events->print_usage($usage_str);
		exit $exit;
	}

	# can't do pkgpath with Language pack because there are many
	# pkgs directories and mapping them out would be crazy
	my $lp_usage = "";
	my $pr_usage = "";
	if ("${trg_upi}" ne "LP") {
		$lp_usage = "\n\t[ -pkgpath <pkg_path> ]";
	}
	if ($self->{PATCHREL}) {
		$pr_usage = "\n\t[ -patchpath <patch_path> ]";
	}

	my $lp_msg = "";
	my $pr_msg = "";
	if ("${trg_upi}" ne "LP") {
		$lp_msg = "\nThe -pkgpath option is used to eliminate the step of copying packages to each system.  <pkg_path> is the complete path of a directory, usually NFS mounted, which contains all packages to be installed by ${program} on all systems.";
	}
	if ($COMM{PATCHREL}) {
		$pr_msg = "\nThe -patchpath option is used to eliminate the step of copying patches to each system or when more recent versions of patches have been downloaded.  <patch_path> is the complete path of a directory which contains all patches to be installed by ${program} on all systems.";
	}

	my $brief_msg = <<"EOMB";
Usage: ${program} [ <system1> <system2>... ]
	[ -installonly ]
        [ -configure ]
	[ -responsefile <response_file> ]
        [ -forcelocal ]
	[ -logdir <log-and-response-file-directory> ]${lp_usage}${pr_usage}
        [ -logfile <log-file-location-and-name> ]
	[ -tmppath <tmp_path> ]
	[ -usessh ]
        [ -hostspace <hostname>:/<larger_volume> ]
EOMB

	$events->print_usage_brief($brief_msg);

	my $details_msg = << "EOMD";
The -installonly option is used to install product packages on systems without configuration.
The -configure option is used to configure a product following use of the -installonly option.
The -responsefile option is used to perform an automated installations which install and configure using information stored in a file rather than prompting for information. <response_file> is the full path of the file which contains configuration information.
The -logdir option permits specification of a particular directory where VxIF is to place the log and response files.  For example: -logdir /tmp/log${lp_msg}${pr_msg}
The -logfile option permits the designation of the name and location of the file to contain the log file.  For example: -logfile /tmp/stuff/install.log
The -tmppath option is used to select a directory other than ${tmppath} as the working directory for ${program}.  This destination is where initial logging is performed and where $self->{MSG}{PDFRS} are copied on remote systems before installation.
The -usessh option is used when ssh and scp are to be used for communication between systems instead of rsh and rcp. The -usessh option requires that systems be pre-configured such that ssh commands between systems execute without prompting for passwords or confirmations.
The -forcelocal option is used to install the software on the local machine only.
The -hostspace option provides specification of specific target volumes for extracting archives.  For example: -hostspace hostname:/larger_volume.
EOMD

	# more usage
	$events->print_usage_details($details_msg);
        # The lines below was removed as the features do not work.
        # The -precheck option is used to confirm that systems meet the product${sq}s install requirements before installing.
        # The -hostspace option provides specification of specific target volumes for extracting archives.  For example: -hostspace my_small_host:/humongous_volume.
        # The -oldresponsefile option is similar to the -responsefile option, but reads the old (perl) format.  This will be unsupported in future releases.
        # The -useftp option is used when ftp is to be used to push packages to remote systems.
        # The -nolic option is used to install product packages on systems without licensing or configuration.  License based features are not installed when using this option.
        # The -configure option is used to configure a product following use of the -installonly option.
        # The -license option is used to update product licenses.
	exit $exit;
}

#
# Process the command line arguements. This routine should be called before calling install().
#
# Input: 1) the reference to self;
#        2) the command line arguements;
#
sub process_args (@) {
	my $self = shift;
	my $unknown_flags = [];

	while (@_) {
		my $a = shift(@_);
		$a = Utils::vxif_desp($a);
		$a = lc($a);
		
		if ($a eq '-installonly') {
			$self->{INSTALLONLY} = 1;
		} elsif ($a eq '-pkgpath') {
			$self->{PKGPATH} = shift;
		} elsif ($a eq '-pathcpath') {
			$self->{PATCHPATH} = shift;
		} elsif ($a eq '-tmppath') {
			$self->{TMPPATH} = shift;
		} elsif ($a eq '-usessh') {
			$self->{USESSH} = 1;
#		} elsif ($a eq '-useftp') {
#			$self->{USEFTP} = 1;
		} elsif ($a eq '-nolocalize') {
			# using environment only because
			# happens in a static method to
			# which state does not get passed.
			$ENV{"VXIF_UTILS_NOLOCALIZE"} = 1;
		} elsif ($a eq '-responsefile') {
			$self->{RESPONSEFILE} = shift;
#		} elsif ($a eq '-oldresponsefile') {
#			$self->{OLDRESPONSEFILE} = shift;
                } elsif ($a eq '-hostspace') {
                        $self->{HOSTSPACE} = shift;
		} elsif ($a eq '-logdir') {
			$self->{LOGDIR} = shift;
		} elsif ($a eq '-logfile') {
			$self->{LOGFILE} = shift;
		} elsif ($a eq '-password') {
			$self->{PASSWORD} = shift;
		} elsif (($a eq '-configure') || ($a eq '-license') || ($a eq '-precheck')) {
			$a =~ /-(.+)/;
			#$self->{TASK} = $1;  #  changing this to TASKS
                        $self->{TASKS} = $1;
# jpk 3/31/06
print "Value $self->{TASKS}\n";
#		} elsif ($a eq '-nolic') {
#			$self->{NOLIC} = 1;
                } elsif ($a eq '-forcelocal') {
                        $self->{LOCALINSTALL} = 1;
		} elsif ($a eq '-h' || $a eq '-help') {
			$self->print_usage(0);
		} else {
			if ($a =~ /^-/) {
				push(@$unknown_flags, $a);
			} else {
				push(@{$self->{SYSTEMS}},$a);
			}
		}
	}

	my $num_unknown_flags = scalar(@$unknown_flags);
	if ($num_unknown_flags > 0) {
		my ($noun, $verb);
		if ($num_unknown_flags > 1) {
			$noun = "Flags";
			$verb = "are";
		} else {
			$noun = "Flag";
			$verb = "is";
		}
		my $unkflag_str = join(", ", @$unknown_flags);
		my $msg = Utils::_tr("****Error: ${noun} (${unkflag_str}) ${verb} not recognized by VxIF.", 14, 1140, "${noun}", "${unkflag_str}", "${verb}");
		Utils::vxif_pl("");
		Utils::vxif_pl($msg);
		Utils::vxif_pl("");
		$self->print_usage(1);
	}
}

1;

