package HostInfo;
use strict;

use VxIF::Utils;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;
	my $cmd_searchpaths = {
		ifconfig => ["/usr/sbin", "/sbin", "/usr/bin"],
		netstat => ["/usr/bin", "/usr/sbin", "/sbin"],
	};

	$self->set_cmd_searchpaths($cmd_searchpaths);

	# set up localhost info
	$self->init_localhost();
}

sub get_hostname {
	my $self = shift;
	my $hostname = $self->{hostname};

	return $hostname;
}

sub set_hostname {
	my $self = shift;
	my ($hostname) = @_;

	$self->{hostname} = $hostname;
}

sub get_ipaddrs {
	my $self = shift;
	my $ipaddrs = $self->{ipaddrs};

	return $ipaddrs;
}

sub set_ipaddrs {
	my $self = shift;
	my ($ipaddrs) = @_;

	$self->{ipaddrs} = $ipaddrs;
}

sub get_cmd_searchpaths {
	my $self = shift;
	my $cmd_searchpaths = $self->{cmd_searchpaths};

	return $cmd_searchpaths;
}

sub set_cmd_searchpaths {
	my $self = shift;
	my ($cmd_searchpaths) = @_;

	$self->{cmd_searchpaths} = $cmd_searchpaths;
}

sub compute_local_os {
	my $self = shift;
	my $os = Utils::vxif_dol("uname -s");

	return $os;
}

sub compute_hostname {
	my $self = shift;
	my $hostname = Utils::vxif_dol("uname -n");

	return $hostname;
}

sub compute_host_ipaddrs {
	my $self = shift;
	my ($hostname) = @_;
	my $ipaddrs_aref = [];
	my $ipaddrs_href = {};
	my $ipaddrs = {
		aref => $ipaddrs_aref,
		href => $ipaddrs_href,
	};
	my ($name,$aliases,$addrtype,$length,@addrs) =
		gethostbyname
		(
		 $hostname
		 );
	my $num_addrs = scalar(@addrs);

	for (my $i = 0; $i < $num_addrs; $i++) {
		my $addr = $addrs[$i];
		my ($a, $b, $c, $d) = unpack('C4', $addr);
		my $ipaddr = "$a.$b.$c.$d";

		$ipaddrs_href->{$ipaddr} = 1;
		push(@$ipaddrs_aref, $ipaddr);
	}

	return $ipaddrs;
}

sub add_net_ipaddrs {
	my $self = shift;
	my $this_ipaddrs = $self->get_ipaddrs();
	my $this_ipaddrs_aref = $this_ipaddrs->{aref};
	my $this_ipaddrs_href = $this_ipaddrs->{href};
	my $more_ipaddrs_href = $self->compute_local_ips();

	for my $ipaddr (keys(%$more_ipaddrs_href)) {
		if (!exists($this_ipaddrs_href->{$ipaddr})) {
			$this_ipaddrs_href->{$ipaddr} = 1;
			push(@$this_ipaddrs_aref, $ipaddr);
		}
	}
}

sub init_localhost {
	my $self = shift;
	my $this_hostname = $self->compute_hostname();
	my $this_ipaddrs = $self->compute_host_ipaddrs($this_hostname);

	$self->set_hostname($this_hostname);
	$self->set_ipaddrs($this_ipaddrs);

	$self->add_net_ipaddrs();
}

sub is_local_host {
	my $self = shift;
	my ($test_hostname) = @_;
	my $verdict = 0;

	my $test_ipaddrs = $self->compute_host_ipaddrs($test_hostname);
	my $test_ipaddrs_aref = $test_ipaddrs->{aref};            # this will never find a local host using IPv6 or if there is a strange entry for 
	my $num_test_ipaddrs = scalar(@$test_ipaddrs_aref);       # in /etc/hosts

	if ($num_test_ipaddrs) {
		my $test_ipaddr = $test_ipaddrs_aref->[0];
		my $this_ipaddrs = $self->get_ipaddrs();
		my $this_ipaddrs_href = $this_ipaddrs->{href};

		if ($this_ipaddrs_href->{$test_ipaddr}) {
			$verdict = 1;
		}
	}

	return $verdict;
}

sub get_fq_cmd {
	my $self = shift;
	my ($base_cmd) = @_;
	my $cmd_searchpaths = $self->get_cmd_searchpaths();
	my $searchpaths = $cmd_searchpaths->{$base_cmd};
	my $fq_cmd = undef;

	if ($searchpaths) {
		for my $path (@$searchpaths) {
			my $path_cmd = "${path}/${base_cmd}";

			if (-x $path_cmd) {
				$fq_cmd = $path_cmd;
			}
		}
	} else {
		$fq_cmd = $base_cmd;
	}

	return $fq_cmd;
}

sub get_netstat_cmd {
	my $self = shift;
	my $netstat_cmd = "netstat";
	my $fq_netstat_cmd = $self->get_fq_cmd($netstat_cmd);

	return $fq_netstat_cmd;
}

sub get_ifconfig_cmd {
        my $self = shift;
        my $ifconfig_cmd;
        if(-f "/usr/bin/ip" || -f "/usr/sbin/ip" || -f "/sbin/ip" || -f "/bin/ip") {
         $ifconfig_cmd = "ip";
        }
        else {
        $ifconfig_cmd = "ifconfig";
        }
		
        my $fq_ifconfig_cmd = $self->get_fq_cmd($ifconfig_cmd);

        return $fq_ifconfig_cmd;
}

sub netstat_local_ips {
	my $self = shift;
	my $ipaddrs = {};
	my $numerrs = 0;
	my $netstat_cmd = $self->get_netstat_cmd();
        my ($addr);

	open(NETSTAT, "${netstat_cmd} -in |") or die "Cannot run netstat.\n";

	my $rec = <NETSTAT>;

        if (grep(/IPv6/i, $rec) || grep(/IPv4/i, $rec)) {
           # If there is a header tag showing this is IPvN type address, get the next line.
           # This is seen on HPUX. ET 546119 - jpk - Mar 7, 06
           $rec = <NETSTAT>;
        }

	if ($rec) {
		# mega chomp
		$rec =~ s/[\r\n]+$//;
		my $name_col = undef;
		my $addr_col = undef;

                if (grep(/IPv6/i, $rec) || grep(/IPv4/i, $rec)) {
                   $rec = <NETSTAT>;
                }

		my @hdrs = split(/\s+/, $rec);
		my $num_hdrs = scalar(@hdrs);

		for (my $i = 0; $i < $num_hdrs; $i++) {
			my $hdr = $hdrs[$i];
			if ($hdr =~ /^name$/i) {
				$name_col = $i;
			} elsif ($hdr =~ /^address$/i) {
				$addr_col = $i;
			}
		}

		if (!defined($name_col)) {
			print(STDERR "Name column expected but not found.\n");
			$numerrs++;
		}

		if (!defined($addr_col)) {
			print(STDERR "Address column expected but not found.\n");
			$numerrs++;
		}

		if (!$numerrs) {
			while ($rec = <NETSTAT>) {
				# mega chomp
				$rec =~ s/[\r\n]+$//;

                                #Check if there is a tag for the IP type
                                if (grep(/IPv6/i, $rec) || grep(/IPv4/i, $rec)) {
                                   $rec = <NETSTAT>;
                                   $rec = <NETSTAT>;
                                   $rec =~ s/[\r\n]+$//;
                                } # end of if

				my @fields = split(/\s+/, $rec);
                                # Check if this is an IPv6 address.  If so, then move the counter one column to the left
                                if (grep(/:/, @fields)) {
                                   $addr = $fields[$addr_col - 1];
                                   $addr =~ s/\/..$//;
                                } else {
                                   $addr = $fields[$addr_col];
                                }
                                my $name = $fields[$name_col];

				if ($addr && (($addr =~ /^(\d+\.\d+\.\d+\.\d+)$/) || ($addr =~ /:/))) {
					my $ipaddr = $1;
					$ipaddrs->{$ipaddr} = 1;
				}
			}
		}
	} else {
		die "Netstat: 1st record expects column headers.\n";
	}

	close(NETSTAT);

	return $ipaddrs;
} # netstat_local_ips

sub ifconfig_local_ips {
	my $self = shift;
	my ($ipaddr_key, $ipv6addr_key) = @_;
	my $ipaddrs = {};
	my $numerrs = 0;
	my $ifconfig_cmd = $self->get_ifconfig_cmd();

	if($ifconfig_cmd eq "ip")	{
	open(IFCONFIG, "${ifconfig_cmd} a |") or die "Cannot run ip.\n";
	}
	else {
	open(IFCONFIG, "${ifconfig_cmd} -a |") or die "Cannot run ifconfig.\n";
	}

	while (my $rec = <IFCONFIG>) {
		# mega chomp
		$rec =~ s/[\r\n]+$//;

		if ($rec =~ /\s+${ipaddr_key}(\d+\.\d+\.\d+\.\d+)\s+/) {
			my $ipaddr = $1;
			$ipaddrs->{$ipaddr} = 1;
		} elsif ($rec =~ /\s+${ipv6addr_key}\s/ ) {
                   # jpk
                   my ($blank,$temp) = split(/${ipv6addr_key}/,$rec);
                   # this is added so that addresses from Darwin using a %network_port are split correctly - Feb 2, 06 jpk
                   $temp =~ s/\%/\//;
                   my ($ipaddr) = split(/\//,$temp);
                   $ipaddrs->{$ipaddr} = 1;
                }
	}

	close(IFCONFIG);

	return $ipaddrs;
}

sub AIX_local_ips {
	my $self = shift;
	my $ipaddrs = $self->netstat_local_ips();

	return $ipaddrs;
}

sub HPUX_local_ips {
	my $self = shift;
	my $ipaddrs = $self->netstat_local_ips();

	return $ipaddrs;
}

sub Linux_local_ips {
	my $self = shift;
	my $ipaddr_key = "inet\\s+addr:";
        my $ipv6addr_key = "inet6\\s+addr:";
	my $ipaddrs = $self->ifconfig_local_ips($ipaddr_key, $ipv6addr_key);

	return $ipaddrs;
}

sub Darwin_local_ips {
   my $self = shift;
   my $ipaddr_key = "inet\\s+";
   my $ipv6addr_key = "inet6\\s+";
   my $ipaddrs = $self->ifconfig_local_ips($ipaddr_key, $ipv6addr_key);

   return $ipaddrs;
}

sub SunOS_local_ips {
	my $self = shift;
	#my $ipaddrs = $self->netstat_local_ips();
	# On solaris, there can be virtual ip addresses that
	# netstat will miss.  However, ifconfig gets them.
	my $ipaddr_key = "inet\\s+";
        my $ipv6addr_key = "inet6\\s+";
	my $ipaddrs = $self->ifconfig_local_ips($ipaddr_key, $ipv6addr_key);

	return $ipaddrs;
}

sub OSF1_local_ips {
	# Added for v4.4 compliance - jpk 10 Feb 06
        my $self = shift;
        my $ipaddr_key = "inet\\s+";
        my $ipv6addr_key = "inet6\\s+";
        my $ipaddrs = $self->ifconfig_local_ips($ipaddr_key, $ipv6addr_key);

        return $ipaddrs;
}

sub win32_local_ips {
	my $self = shift;
	my $ipaddrs = {
		"127.0.0.1" => 1,
	};
	my $curr_stanza = undef;

	open(IPCONFIG, "ipconfig /all |") or die "Cannot run ipconfig.\n";

	while (my $rec = <IPCONFIG>) {
		# mega chomp
		$rec =~ s/[\r\n]+$//;

		if ($rec =~ /^\S/) {
			# start of line is not a space, so begin a stanza
			$curr_stanza = $rec;
		} elsif ($rec =~ /ip\s+address/i) {
			# IP Address property
			if ($curr_stanza && $curr_stanza =~ /ethernet\s+adapter/i) {
				# within an ethernet adapter stanza
				$rec =~ s/.*:\s+//;
				if ($rec =~ /^(\d+\.\d+\.\d+\.\d+)\s*$/) {
					# have ip addres syntax.  stick it in.
					my $ipaddr = $1;
					$ipaddrs->{$ipaddr} = 1;
				}
			}
		}
	}

	close(IPCONFIG);

	return $ipaddrs;
}

sub compute_local_ips {
	#  added OSF_local_ips call for v4.4 - jpk 10 Feb 06
	my $self = shift;
	my $ipaddrs = undef;
	my $os = $self->compute_local_os();

	if ($os eq 'AIX') {
		$ipaddrs = $self->AIX_local_ips();
	} elsif ($os eq 'HP-UX') {
		$ipaddrs = $self->HPUX_local_ips();
	} elsif ($os eq 'Linux') {
		$ipaddrs = $self->Linux_local_ips();
	} elsif ($os eq 'Windows_NT') {
		$ipaddrs = $self->win32_local_ips();
	} elsif ($os eq 'SunOS') {
		$ipaddrs = $self->SunOS_local_ips();
	} elsif ($os eq 'Darwin') {
		$ipaddrs = $self->Darwin_local_ips();
        } elsif ($os eq 'OSF1') {
                $ipaddrs = $self->OSF1_local_ips();
	} else {
		print("Not yet.\n");
	}

	return $ipaddrs;
}

1;

