package CPIHPUX;

use VxIF::CPICommon;
@CPIHPUX::ISA = qw(CPICommon);

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#         SPACEWARNING	=> warning the users of low on space if the number of bytes on the 
#                          volume after the installation is less than this value; This is optional;
#                          The default is 5120 kilobytes;
#
sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $self = { 	
		USESSH		=> 0,
		IPv6            => 0,
		TMPPATH		=> '/var/tmp/vxif',
		TIMEFILE	=> '/tmp/vxif_time',
		PKGSDIR		=> 'pkgs',
		PATCHSDIR	=> 'patches',
		OSINSTALLER	=> 'HPUX',
		SPACEWARNING	=> '5120',
		SUPPORTEDOS => [ qw(SunOS Linux HPUX AIX OSF1) ],
		CMDSEARCHPATHS  => [ qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin) ],
		@_,
	};
    
	return bless $self, $class;
}

#
# Set the common system commands.
#
sub set_commands {
  my ($self) = @_;
  
  $self->SUPER::set_commands();
  $self->{CMD}{DFK}="/usr/bin/bdf -l";
  $self->{CMD}{GUNZIP}="/usr/contrib/bin/gunzip";
  $self->{CMD}{MV}="/usr/bin/mv";
  $self->{CMD}{NETSTAT}="/usr/bin/netstat";
  $self->{CMD}{PWD}="/usr/bin/pwd";
  $self->{CMD}{RSH}="/usr/bin/remsh";
  $self->{CMD}{SHUTDOWN}="/usr/sbin/shutdown";
   
  # Define unique platform specific commands
  $self->{CMD}{KMADMIN}="/usr/sbin/kmadmin";
  $self->{CMD}{LANSCAN}="/usr/sbin/lanscan";
  $self->{CMD}{SWAGENTD}="/usr/sbin/swagentd";
  $self->{CMD}{SWLIST}="/usr/sbin/swlist";
  $self->{CMD}{SWINSTALL}="/usr/sbin/swinstall";
  $self->{CMD}{SWREMOVE}=$self->{CMD}{RMPOBJ}="/usr/sbin/swremove";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
}

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  
  $self->{MSG}{PDFREN} 	= "depot";
  $self->{MSG}{PDFR} 	= Utils::_tr("depot", 23, 1000);
  $self->{MSG}{PDFRS}	= Utils::_tr("depots", 23, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}

#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
  
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) extension;
#        4) the reference to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers (\%$\%;$) {
	my $self = shift;
	my ($ctx,$pkg,$ext,$pkgs) = @_;
	my ($iv,@pkgnames);
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	$$ctx{LOGGER}->entering("CPIHPUX::get_instpkgvers");

	@pkgsnames = keys(%{$pkgs});

	$$ctx{LOGGER}->finer("pkg=${pkg}");

	if ((Utils::vxif_list($pkg, @pkgnames) >= 0) && $upi_inst->can("get_installed_package_version")) {
		$iv = $upi_inst->get_installed_package_version($ctx, $pkg);
	} else {
		if (Utils::vxif_localsys($ctx)) {
			$iv = Utils::vxif_dor("$self->{CMD}{SWLIST} -l product -a revision $pkg 2> /dev/null | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \$2}'", "", $ctx);
		} else {
			$iv = Utils::vxif_dor("\"$self->{CMD}{SWLIST} -l product -a revision $pkg 2> /dev/null | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \\\$2}'\"", "", $ctx);
		}

	}

	return "$iv";
}

#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
  my ($self,$ctx,$pkg,$pkgs,$ext) = @_;
  my ($v,$prefix);
  my ($path) = $self->{$$ctx{TARGET}{UPI}}{$pkg}{TLOC};
    
  # make sure the path to the package is a complete path
  if ($path =~ /^\.(.*)/) {
    # this is a local installation
    $path = $1;
    $prefix = Utils::vxif_dol("$self->{CMD}{PWD}");
    $path = $prefix . $path;
  }
  
  $$ctx{LOGGER}->entering("CPIHPUX::get_mediapkgvers");
  
  $v = Utils::vxif_dor("$self->{CMD}{SWLIST} -s ${path}/$pkg -l product -a revision $pkg | $self->{CMD}{GREP} $pkg | $self->{CMD}{GREP} -v 'Target' | $self->{CMD}{GREP} -v '^#' | $self->{CMD}{AWK} '{print \$2}'", "", $ctx);

  $v =~s/\n//;
  chomp($v);

  return $v;
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
	my $self = shift;
	my ($ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
	my $local_host = $ctx->get_local_hostname();
	my $size = 0;

	$$ctx{LOGGER}->entering("CPIHPUX::get_mediapkgsize");

	if ($compressed) {
		$$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 23, 1002, "${pkg}"));
		if (-f "${loc}/${filename}") {
			$$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
			$size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
			# assume compression of 3x. multiple size by 4 to account for the compressed package itself
			$size *= 4;
		} else {
			$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on ${local_host}.", 23, 1003, "${pkg}", "${loc}", "${filename}", "${local_host}"));
			return 0;
		}
	} else {
		$$ctx{LOGGER}->fine(Utils::_tr("${pkg} is uncompressed", 23, 1004, "${pkg}"));
		if (-f "${loc}/${pkg}") {
			$$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
			$size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
		} else {
			$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${pkg}/pkginfo does not exist on ${local_host}.", 23, 1005, "${pkg}", "${loc}", "${pkg}", "${local_host}"));
			return 0;
		}
	}

	$$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} is ${size} kbs", 23, 1006, "${pkg}", "${size}"));

	return $size;
}

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my ($dep,$deplist,$deppkgs,@deps,@ideps);

	$$ctx{LOGGER}->entering("CPIHPUX::get_pkgdep");

	if ($upi_inst->can("check_for_package_dependents")) {
		$deppkgs = $upi_inst->check_for_package_dependents($ctx, $pkg);
		@deps = keys(%{$deppkgs});
		$deplist = join(" ", @deps);
		$deplist .= " ";
	}

	if (Utils::vxif_localsys($ctx)) {
		$dep = Utils::vxif_dor("$self->{CMD}{SWLIST} -v $pkg | $self->{CMD}{GREP} requisites | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \$2}' | $self->{CMD}{AWK} -F. '{print \$1}' | $self->{CMD}{SORT} | $self->{CMD}{UNIQ} | $self->{CMD}{GREP} -v  $pkg", "", $ctx);
	} else {
		$dep = Utils::vxif_dor("\"$self->{CMD}{SWLIST} -v $pkg | $self->{CMD}{GREP} requisites | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \$2}' | $self->{CMD}{AWK} -F. '{print \$1}' | $self->{CMD}{SORT} | $self->{CMD}{UNIQ} | $self->{CMD}{GREP} -v  $pkg\"", "", $ctx);
	}

	$$ctx{LOGGER}->finer("dep=$dep");

	@ideps = split(" ", $dep);
	foreach $dep (@ideps) {
		if (Utils::vxif_list($dep, $deps) < 0) {
			$deplist .= "$dep ";
		}
	}

	return Utils::vxif_desp($deplist);
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%) {
  my ($self,$ctx,$nic) = @_;
  my ($nm);
  
  $nm = Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic | $self->{CMD}{GREP} netmask | $self->{CMD}{AWK} '{print \$4}'", "", $ctx);
  
  return "$nm";
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my (@pnics,$do);
  
  $do = Utils::vxif_dor("$self->{CMD}{NETSTAT} -i | $self->{CMD}{GREP} '^lan' | $self->{CMD}{SED} 's/:.*\$//' | $self->{CMD}{AWK} '{print \$1}' | $self->{CMD}{UNIQ}", "", $ctx);
  @pnics=split(/\s+/,$do);
  
  return \@pnics;
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) UPI;
#        2) the patch name;
#        3) the reference to the installation context;
#        4) the reference to the installed patch;
#        5) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 if older version is installed; 
#         2 if version or higher is installed.       
#
sub check_patch ($$$$$) {
  my ($self,$upi,$patch,$ctx,$installedpatch,$refresh) = @_;
  my ($iv);
  my ($rc) = 0;
  
  $iv = Utils::vxif_dor("$self->{CMD}{SWLIST} -l product -a revision $patch 2> /dev/null | $self->{CMD}{GREP} $patch | $self->{CMD}{AWK} '{print \$2}'", "", $ctx);

  return 0 if ($iv eq "");
  $$installedpatch = $iv;
  return 2;
}

#
# Check for and run post pkgadd operations.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
	my $self = shift;
	my ($upi,$pkg,$ctx) = (@_);
	my (@patchlist,$patch,$patchid);
	my ($rc,$installedpatchid,$msgl,$msgr);
	my $upi_inst = $self->{PRODUCT}{$upi}; # uh-oh!!!
	my $result = 1;

	if ($upi_inst->can("get_required_os_patches")) {
		@patchlist = $upi_inst->get_required_os_patches($pkg, $ctx);
	}

	if ($#patchlist >= 0 ) {
		Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 23, 1007, "$pkg"));
		foreach $patch (@patchlist) {
			$patchid = $patch;
			($upi_inst->can("get_patch_id")) && ($patchid = $upi_inst->get_patch_id($patch, $ctx));
			$msgl = Utils::_tr("Checking OS patch ${patchid}", 23, 1008, "${patchid}");
			Utils::vxif_pbl($msgl);

			$rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
			if ($rc == -1) {
				$msgr = Utils::_tr("invalid patch ID, ${patchid}", 23, 1009, "${patchid}");
				Utils::vxif_pbr($msgl, $msgr);
				$result = 0;
			} elsif ($rc == 0) {
				$msgr = Utils::_tr("not installed", 23, 1010);
				Utils::vxif_pbr($msgl, $msgr);
				$result = 0;
			} elsif ($rc == 1) {
				$msgr = Utils::_tr("${installedpatchid} installed", 23, 1011, "${installedpatchid}");
				Utils::vxif_pbr($msgl, $msgr);
				$result = 0;
			} else {
				$msgr = Utils::_tr("${installedpatchid} installed", 23, 1011, "${installedpatchid}");
				Utils::vxif_pbr($msgl, $msgr);
			}
		}
	}

	return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $task_type = $ctx->get_task_type();
	my $logfile = "${logdir}/${task_type}.${pkg}.${trg_host}";
	my $local_cmds = $ctx->get_local_cmds();
	my $local_grep_cmd = $local_cmds->{GREP};
	my $error = Utils::vxif_dol("${local_grep_cmd} ERROR: ${logfile}");

	if ($error eq "") {
		return 0;
	}

	$logger->info($error);

	return 1;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
  
	return $self->pkg_installerror($ctx, $pkg);
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = (@_);
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $trg_upi = $ctx->get_target_upi();
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $task_type = $ctx->get_task_type();
	my $swinstall_logfile = "${logdir}/${task_type}.${pkg}.${trg_host}";
	my $pkginfo = $pkgs->{$pkg};
	my $depots = $pkginfo->get_depot();
	my ($iver,$pkgver,$prefix);
	my $path = $self->{$trg_upi}{$pkg}{TLOC};
	
	# if depot is not specified, assuming the package name is the same as the depot name
	$depots = $pkg if (!$depots);
	
	# to support the B backward compatibility stuff, use {IOPTS} if it is available
	$depots = $$pkgs{$pkg}{IOPT} if (defined($$pkgs{$pkg}{IOPT}));
	
	# make sure the path to the package is a complete path
	if ($path =~ /^\.(.*)/) {
		# this is a local installation
		$path = $1;
		$prefix = Utils::vxif_dol("$self->{CMD}{PWD}");
		$path = $prefix . $path;
	}
	
	# check to make sure the given package is not already exist on the target host
	my $result = $self->check_for_existing_package($ctx, $pkg, $pkgs);
	if (0 == $result) {
		return 0;
	} elsif (2 == $result) {
		return 1;
	}

	my $xloc;
	if (Utils::vxif_localsys($ctx) && !$self->{${trg_upi}}{$pkg}{COMPRESSED}) {
		# if local install, and package not compressed (or archived),
		# install directly from source location.
		$xloc = "SLOC";
	} else {
		$xloc = "TLOC";
	}

	# The package itself may not be at the top of the "pkgadd" directory.
	# It may, rather, be one or more levels down.  Use the EXTRACTION_PATH
	# configuration variable (if present) to see where to dig.
	#
	# maybe put this in common code
	my $install_dir = $self->{$trg_upi}{$pkg}{$xloc};
	my $installer_info = $ctx->get_installer_info();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $package_info = $installer_info->get_platform_package_info($trg_os, $pkg);
	my $extra_path = $package_info->get_extraction_path();
	if ($extra_path) {
		$install_dir .= "/${extra_path}";
	} else {
		$install_dir .= "/${pkg}";
	}

	# now install the package
	#my $cmd = "$self->{CMD}{SWINSTALL} -s ${path}/$pkg ${depots}";
	my $cmd = "$self->{CMD}{SWINSTALL} -s ${install_dir} ${depots}";
	Utils::vxif_dor($cmd, "", $ctx, undef, $swinstall_logfile);
	
	return $self->is_package_install_successful($ctx, $pkg, $pkgs);
}

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = (@_);
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $swremove_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
    
	# now remove the package
	my $cmd = "$self->{CMD}{SWREMOVE} $pkg";
	if (Utils::vxif_localsys($ctx)) {
		Utils::vxif_dor($cmd, "", $ctx, undef, $swremove_logfile);
	  } else {
		  Utils::vxif_dor("\"${cmd}\"", "", $ctx, undef, $swremove_logfile);
		}
	
	# if the package is successfully uninstalled, the version number null.
	if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg) eq "")) {
		$$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from ${trg_host}", 23, 1012, "${pkg}", "${trg_host}"));
		return 1;
	}
	
	$$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from ${trg_host}", 23, 1013, "${pkg}", "${trg_host}"));
	return 0;
}

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%) {
	my $self = shift;
	my ($ctx) = @_;

	# (optimistically) stub this out for now
	return 1;
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  # for SunOS, the package name is the native package name
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
  my ($self,$ctx,$pkg,$fod,$pkgs) = @_;

  $$fod = "d";
  return $pkg;
}

sub get_subpackage_list {
	my $this = shift;
	my ($ctx, $pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $product_info = $upi_inst->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $subpkgs = $package_info->get_depot();

	return $subpkgs;
}

1;
