package CPIFreeBSD;

our @ISA = ("CPICommon");

require VxIF::CPICommon;

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#         SPACEWARNING	=> warning the users of low on space if the number of bytes on the 
#                          volume after the installation is less than this value; This is optional;
#                          The default is 5120 kilobytes;
#
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = { 	
                USESSH		=> 0,
  	  	        TMPPATH		=> '/var/tmp/SunOS',
  	 	          TIMEFILE	=> '/tmp/vxif_time',
  	 	          PKGSDIR		=> 'pkgs',
  	 	          PATCHSDIR	=> 'patches',
  	 	          OSINSTALLER	=> 'FreeBSD',
  	 	          SPACEWARNING	=> '5120',
                SUPPORTEDOS => [qw(SunOS Linux HPUX AIX OSF1 IRIX64 FreeBSD)],
                CMDSEARCHPATHS  => [qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin)],
          	    @_,
  	          };
    
  return bless $self, $class;
}

#
# Set the common system commands.
#
sub set_commands {
  my ($self) = @_;
  
  $self->SUPER::set_commands();
  $self->{CMD}{CAT}="/bin/cat";
  $self->{CMD}{CHMOD}="/bin/chmod";
  $self->{CMD}{CP}="/bin/cp -L";
  $self->{CMD}{DFK}="/bin/df -kl";
  $self->{CMD}{ECHO}="/usr/bin/echo";
  $self->{CMD}{GUNZIP}="/usr/sbin/gunzip";
  $self->{CMD}{IFCONFIG}="/sbin/ifconfig";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  $self->{CMD}{INST}="/usr/sbin/inst";
  $self->{CMD}{KILL}="/bin/kill";
  $self->{CMD}{LS}="/bin/ls";
  $self->{CMD}{MKDIR}="/bin/mkdir";
  $self->{CMD}{MOUNT}="/sbin/mount";
  $self->{CMD}{MV}="/bin/mv";
  $self->{CMD}{PING}="/bin/ping";
  $self->{CMD}{PS}="/bin/ps";
  $self->{CMD}{RCP}="/bin/rcp";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{RMR}="/bin/rm -rf";
  $self->{CMD}{SED}="/sbin/sed";
  $self->{CMD}{SHOWPRODS}="/usr/sbin/showprods";
  
  $self->{CMD}{PKG_ADD}=$CMD{PKGOBJ}="/usr/sbin/pkg_add";
  $self->{CMD}{PKG_CHECK}=$CMD{PKGOBJ}="/usr/sbin/pkg_check";
  $self->{CMD}{PKG_INFO}=$CMD{PKGOBJ}="/usr/sbin/pkg_info";
  $self->{CMD}{PKG_DELETE}=$CMD{PKGOBJ}="/usr/sbin/pkg_delete";
  $self->{CMD}{PKG_DEP}=$CMD{PKGOBJ}="/usr/sbin/pkg_info";
  $self->{CMD}{PKG_VERSION}=$CMD{PKGOBJ}="/usr/sbin/pkg_version";
  $self->{CMD}{MODINFO}="/sbin/lsmod";
  $self->{CMD}{MODUNLOAD}="/sbin/rmmod";
}

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  
  $self->{MSG}{PDFREN} 	= "package";
  $self->{MSG}{PDFR} 	= Utils::_tr("package", 28, 1000);
  $self->{MSG}{PDFRS}	= Utils::_tr("packages", 28, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}

#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
  
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the fileset name;
#        3) extension?;
#        4) the refeence to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers (\%$;$\%) {
  my ($self,$ctx,$pkg,$ext,$pkgs) = @_;
  my ($v,$output);
  
  $$ctx{LOGGER}->entering("CPIFreeBSD::get_instpkgvers");
  
  $output = Utils::vxif_dor("$self->{CMD}{PKG_INFO} -x $pkg", "", $ctx);
  $$ctx{LOGGER}->finer("output=$output");
  return "" if ($output =~ /no packages match/);
  
  if ($output =~ /(\d+\.\d+\.\d+\.\d+)/) {
    $v =  $1;
  }
  
  $$ctx{LOGGER}->finer("v=$v");
  
  return "$v";
}

#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
  my ($self,$ctx,$pkg,$pkgs,$ext) = @_;
  my ($v);
  
  $$ctx{LOGGER}->entering("CPIFreeBSD::get_mediapkgvers");
  
  $$ctx{LOGGER}->finer("$pkg TGZ=$$pkgs{$pkg}{TGZ}");
  
  if ($$pkgs{$pkg}{TGZ} =~ /(\d+(\.\d+)*)/) {
    $v = $1;
  }
  
  $$ctx{LOGGER}->finer("v=$v");
  
  return $v;
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
  my ($self,$ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
  my ($size) = 0;
  
  $$ctx{LOGGER}->entering("CPIFreeBSD::get_mediapkgsize");
  
  if ($compressed) {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 28, 1002, "${pkg}"));
    if (-f "${loc}/${filename}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      # assume compression of 3x. multiple size by 4 to account for the compressed package itself
      $size *= 4;
    } else {
      $$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on $$ctx{LOCAL}{HOST}.", 28, 1003, "${pkg}", "${loc}", "${filename}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  } else {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is uncompressed", 28, 1004, "${pkg}"));
    $$ctx{LOGGER}->finer("loc=${loc}\nTGZ=$$pkgs{$pkg}{TGZ}");
    
    if (-f "${loc}/$$pkgs{$pkg}{TGZ}") {
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/$$pkgs{$pkg}{TGZ} | $self->{CMD}{AWK} '{print \$1}'");
    } else {
      $$ctx{LOGGER}->error(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/$$pkgs{$pkg}{TGZ} does not exist on $$ctx{LOCAL}{HOST}.", 28, 1003, "${pkg}", "${loc}", "$$pkgs{$pkg}{TGZ}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  }
  
  $$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} is ${size} kbs", 28, 1005, "${pkg}", "${size}"));
  
  return $size;
}

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
  my ($self,$ctx,$pkg) = @_;
  
  return "";
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%) {
  my ($self,$ctx,$nic) = @_;
  my ($nm);
  
  $nm = Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic | $self->{CMD}{GREP} netmask | $self->{CMD}{AWK} '{print \$4}'", "", $ctx);
  # convert it to decimal format
  $nm =~ /^(..)(..)(..)(..)$/;
  $nm = sprintf("%d.%d.%d.%d", oct "0x$1", "0x$2", "0x$3", "0x$4");
  
  return "$nm";
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my (@pnics,$do);
  
  $do = Utils::vxif_dor("$self->{CMD}{IFCONFIG} -a | $self->{CMD}{GREP} 'flags=' | $self->{CMD}{SED} 's/:.*\$//' | $self->{CMD}{AWK} '{print \$1}' | $self->{CMD}{UNIQ}", "", $ctx);
  @pnics=split(/\s+/,$do);
  
  return \@pnics;
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) UPI;
#        2) the patch name;
#        3) the reference to the installation context;
#        4) the reference to the installed patch;
#        5) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 if older version is installed; 
#         2 if version or higher is installed.       
#
sub check_patch ($$$$$) {
  my ($self,$upi,$patch,$ctx,$installedpatch,$refresh) = @_;
  
  # stub this out for now
  return 0;
}

#
# Check for and run post pkgadd operations.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
  my ($self,$upi,$pkg,$ctx) = (@_);
  my (@patchlist,$patch,$patchid);
  my ($rc,$installedpatchid,$msgl,$msgr);
  my ($result) = 1;
  
  if ($self->{PRODUCT}{$upi}->can("get_required_os_patches")) {
    @patchlist = $self->{PRODUCT}{$upi}->get_required_os_patches($pkg, $ctx);
  }
  
  if ($#patchlist >= 0 ) {
    Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 28, 1006, "$pkg"));
    foreach $patch (@patchlist) {
      $patchid = $patch;
      ($self->{PRODUCT}{$upi}->can("get_patch_id")) && ($patchid = $self->{PRODUCT}{$upi}->get_patch_id($patch, $ctx));
      $msgl = Utils::_tr("Checking OS patch ${patchid}", 28, 1007, "${patchid}");
      Utils::vxif_pbl($msgl);
      
      $rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
      if ($rc == -1) {
        $msgr = Utils::_tr("invalid patch ID, ${patchid}", 28, 1008, "${patchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 0) {
        $msgr = Utils::_tr("not installed", 28, 1009);
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 1) {
        $msgr = Utils::_tr("${installedpatchid} installed", 28, 1010, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } else {
        $msgr = Utils::_tr("${installedpatchid} installed", 28, 1010, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
      }
    }
  }
  
  return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%) {
  my ($self,$ctx,$pkg) = @_;
  my ($logfile) = "$$ctx{TLOG}/$$ctx{TASK}.${pkg}.$$ctx{TARGET}{HOST}";
  my ($error);
  
  return 0;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%) {
  my ($self,$ctx,$pkg) = @_;
  
  return $self->pkg_installerror($ctx, $pkg);
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
  my ($self,$ctx,$pkgs,$pkg) = (@_);
  my ($logfile) = "$$ctx{TLOG}/$$ctx{TASK}.${pkg}.$$ctx{TARGET}{HOST}";
  
  $$ctx{LOGGER}->entering("CPIFreeBSD::install_native_package");
  
  # check to make sure the given package is not already exist on the target host
  return 0 if (!$self->check_for_existing_package($ctx, $pkg, $pkgs));
  $$ctx{LOGGER}->finer("TGZ=$self->{$$ctx{TARGET}{UPI}}{$pkg}{TLOC}/$$pkgs{$pkg}{TGZ}");
  
  Utils::vxif_dor("$self->{CMD}{PKG_ADD} $self->{$$ctx{TARGET}{UPI}}{$pkg}{TLOC}/$$pkgs{$pkg}{TGZ} 2>>${logfile} 1>>${logfile}", "", $ctx);
  
  return $self->is_package_install_successful($ctx, $pkg, $pkgs);
}

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
  my ($self,$ctx,$pkgs,$pkg) = (@_);
  my ($logfile) = "$$ctx{TLOG}/$$ctx{TASK}.${pkg}.$$ctx{TARGET}{HOST}";
  
  $$ctx{LOGGER}->entering("CPIFreeBSD::uninstall_native_package");
    
  # now remove the package
  Utils::vxif_dor("$self->{CMD}{PKG_DELETE} -f \"${pkg}*\" 2>${logfile} 1>${logfile}", "", $ctx);
 
  # if the package is successfully uninstalled, the version number null.
  if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg, "", $pkgs) eq "")) {
    $$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from $$ctx{TARGET}{HOST}", 28, 1011, "${pkg}", "$$ctx{TARGET}{HOST}"));
    return 1;
  }
   
  $$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from $$ctx{TARGET}{HOST}", 28, 1012, "${pkg}", "$$ctx{TARGET}{HOST}"));
  return 0;
}

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%) {
  my ($self,$ctx) = @_;
  my ($msgl,$msgr);

  # this has become moot.  It is only left for compatability - jpk 10 Feb 06
  #$msgl = Utils::_tr("Checking for external dependencies");
  #Utils::vxif_pbl($msgl);
  #$msgr = Utils::_tr("all external dependencies satisfied");
  #Utils::vxif_pbr($msgl, $msgr);
  return 1; # stub this out for now
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
  my ($self,$ctx,$pkg,$fod,$pkgs) = @_;
  
  $$fod = "f";
  
  return $$pkgs{$pkg}{TGZ};
}

1;
