package CPIDarwin;

use VxIF::CPICommon;
@CPIDarwin::ISA = qw(CPICommon);

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Darwin, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my @os = qw(SunOS Darwin Linux HPUX AIX OSF1);
  my @searchpaths = qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin);
  my $self = { 	
                USESSH		=> 0,
		IPv6            => 0,
  	  	TMPPATH		=> '/var/tmp/vxif',
  	 	TIMEFILE	=> '/tmp/vxif_time',
  	 	PKGSDIR		=> 'rpms',
  	 	PATCHSDIR	=> 'patches',
  	 	OSINSTALLER	=> 'Darwin',
  	 	SUPPORTEDOS	=> \@os,
  	 	CMDSEARCHPATHS	=> \@searchpaths,
          	@_,
  	     };

  return bless $self, $class;
}

#
# Set the common system commands.
#
# Input: 1) the reference of the installation context;
#
sub set_commands ($) {
  my ($self) = @_;

  $self->SUPER::set_commands();
  $self->{CMD}{AWK}="/usr/bin/awk";
  $self->{CMD}{CAT} = "/bin/cat";
  $self->{CMD}{CHMOD}="/bin/chmod";
  $self->{CMD}{CLEAR}="/usr/bin/clear";
  $self->{CMD}{CP}="/bin/cp -L";
  $self->{CMD}{DFK}="/bin/df -kl";
  $self->{CMD}{DOMAINNAME} = "/bin/domainname";
  $self->{CMD}{DU}="/usr/bin/du";
  $self->{CMD}{ECHO}="/bin/echo";
  $self->{CMD}{GREP}="/usr/bin/grep";
  $self->{CMD}{GUNZIP}="/usr/bin/gunzip";
  $self->{CMD}{ID}="/usr/bin/id";
  $self->{CMD}{IFCONFIG}="/sbin/ifconfig";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  $self->{CMD}{KILL}="/bin/kill";
  $self->{CMD}{LS}="/bin/ls";
  $self->{CMD}{MKDIR} = "/bin/mkdir";
  $self->{CMD}{MOUNT}="/sbin/mount";
  $self->{CMD}{MV}="/bin/mv";
  $self->{CMD}{PING}="/sbin/ping";
  $self->{CMD}{PS}="/bin/ps -w";
  $self->{CMD}{RCP}="/bin/rcp";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{RMDIR}="/bin/rmdir";
  $self->{CMD}{RMR} = "/bin/rm -rf";
  $self->{CMD}{RM} = "/bin/rm";
  $self->{CMD}{RSH}="/usr/bin/rsh";
  $self->{CMD}{SED}="/usr/bin/sed";
  #$self->{CMD}{SHUTDOWN}="/sbin/shutdown -r now";
  $self->{CMD}{SORT}="/usr/bin/sort";
  $self->{CMD}{TAR}="/usr/bin/tar";
  $self->{CMD}{TOUCH}="/usr/bin/touch";
  $self->{CMD}{TPUT}="/usr/bin/tput";
  $self->{CMD}{UNAME}="/usr/bin/uname";
  $self->{CMD}{UNIQ}="/usr/bin/uniq";
  $self->{CMD}{WC}="/usr/bin/wc";
  $self->{CMD}{WHICH}="/usr/bin/which";

}


#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  $self->{MSG}{PDFREN}  = "package";
  $self->{MSG}{PDFR}  = Utils::_tr("package", 77, 1000);
  $self->{MSG}{PDFRS} = Utils::_tr("packages", 77, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}

#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
 
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) extension;
#        4) the reference to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers ($\%;$) {
   my ($self,$ctx,$pkg,$ext,$pkgs) = @_;
   my ($iv,@pkgnames, $trg_upi, $trg_host, $upi_inst);
   $trg_host = $ctx->get_target_hostname();
   $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
   $trg_upi = $ctx->get_target_upi();

   $$ctx{LOGGER}->entering("CPIDarwin::get_instpkgvers");

   @pkgnames = keys(%{$pkgs});

   $$ctx{LOGGER}->finer("pkg=${pkg}");

   my $destination = "/var/SYMCif/".${trg_upi};
   my $build_file = $destination."/".$pkg."_build_version";

   if ((Utils::vxif_list($pkg, @pkgnames) >= 0) && $upi_inst->can("get_installed_package_version")) {
      $iv = $upi_inst->get_installed_package_version($ctx, $pkg);
   } else {
      if (Utils::vxif_fc("${build_file}", $ctx)) {
         $test = Utils::vxif_dor("$self->{CMD}{CAT} ${build_file}", "", $ctx);
         ($blank,$iv) = split(/-/,$test);
      } else {
         $iv = "not installed";
      }
      $$ctx{LOGGER}->info("Found version $iv");
      return "" if $iv =~ /not installed/;
   }
   return "$iv";
} # end of get_instpkgvers


#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs,$ext) = @_;
	my $rpmfile;
	my $v = "";

	$$ctx{LOGGER}->entering("CPIDarwin::get_mediapkgvers");

	my $trg_upi = $ctx->get_target_upi();
	my $loc = $self->{$trg_upi}{$pkg}{TLOC};

	# get the product packages to install
	$pkgs = $self->{PRODUCT}{$trg_upi}->get_product_packages($ctx, $prodver, $osver, $osarch);
	$tarfile = $$pkgs{$pkg}{COMPRESSEDFILE};

	if ($tarfile && Utils::vxif_fc("${loc}/${tarfile}", $ctx)) {
           ($blank,$v) = split(/-/,$tarfile);
           $v =~ s/.tar.gz//;
	} else {
           $$ctx{LOGGER}->error(Utils::_tr("Unable to locate the TAR file, ${tarfile}, associated with ${pkg} in ${loc} on $$ctx{TARGET}{HOST}.", 77, 1002, "${tarfile}", "${pkg}", "${loc}", "$$ctx{TARGET}{HOST}"));
	}

	return "$v";
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
  my ($self,$ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
  my($size) = 0;
  
  $$ctx{LOGGER}->entering("CPIDarwin::get_mediapkgsize");
  
  if ($compressed) {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 77, 1003, "${pkg}"));
    if (-f "${loc}/${filename}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${filename} | $self->{CMD}{AWK} '{print \$1}'");
      # assume compression of 3x. multiple size by 4 to account for the compressed package itself
      $size *= 4;
    } else {
      $$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on $$ctx{LOCAL}{HOST}.", 77, 1004, "${pkg}", "${loc}", "${filename}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  } else {
    $$ctx{LOGGER}->fine(Utils::_tr("${pkg} is uncompressed", 77, 1005, "${pkg}"));
    if (-f "${loc}/${pkg}") {
      $$ctx{LOGGER}->info("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
      $size = Utils::vxif_dol("$self->{CMD}{DU} -sk ${loc}/${pkg} | $self->{CMD}{AWK} '{print \$1}'");
    } else {
      $$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${pkg}/pkginfo does not exist on $$ctx{LOCAL}{HOST}.", 77, 1006, "${pkg}", "${loc}", "${pkg}", "$$ctx{LOCAL}{HOST}"));
      return 0;
    }
  }
  
  $$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} is ${size} kbs", 77, 1007, "${pkg}", "${size}"));
  
  return $size;
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%) {
  my ($self,$ctx,$nic) = @_;
  
  # special case for localhost. death to lunix.
  return "255.0.0.0" if $nic eq 'lo';

  #
  # run ifconfig and grab the lines with 'Mask:' in them
  # implicitly ifconfig for all interfaces if $nic is undefined                   
  #
  my @all_netmasks = split ( "\n", Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic", "", $ctx) );

  # in the case where we have multiple interfaces, we only need                   
  # the top line, and we assume it's the default
  my @def_netmask_line = split( " ", $all_netmasks[1] );

  # grab the last end of that line
  my @def_netmask = split( ":", $def_netmask_line[-1] );

  # split it at the : and take the end
  my $def_netmask = $def_netmask[-1];

  # return that as the netmask string
  return $def_netmask;
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my @all_interfaces;
  my @all_interface_lines = split( "\n",  Utils::vxif_dor("$self->{CMD}{IFCONFIG}", "", $ctx)) ;

  foreach (@all_interface_lines) {
    if ( grep /Link/i, $_ ) {
      my @temp = split( " ", $_);
      if ( ($temp[0] ne "lo")  && ($temp[0] !~ /:\d+$/)) {
        push( @all_interfaces, $temp[0] );
      }
    }
  }   

  return \@all_interfaces;
}   

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
   # this is commented out currently as there is not a way to check for dependents with MacOS. - jpk - 06 Dec 05
#  my ($self,$ctx,$pkg) = @_;
#  my ($dep,$deplist,$deprpms,$deppkgs,@deps);

#  if ($self->{PRODUCT}{$$ctx{TARGET}{UPI}}->can("check_for_package_dependents")) {
#    $deppkgs = $self->{PRODUCT}{$$ctx{TARGET}{UPI}}->check_for_package_dependents($ctx, $pkg);
#    @deps = keys(%{$deppkgs});
#    $deprpms = join(" ", @deps);
#    $deprpms .= " ";
#  }
  
#  @deplist = split("\n", Utils::vxif_dor("$self->{CMD}{RPM} -q --whatrequires $pkg", "", $ctx) );
#  foreach $dep(@deplist) {
#    if (!($dep =~ 'no package')) {
#      $dep =~ (s/-.*$//);
#      if (Utils::vxif_list($dep, @deps) < 0) {
#        $deprpms .= "$dep ";
#      }
#    }
#  }    
  
#  return Utils::vxif_desp($deprpms);
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) UPI;
#        2) the patch name;
#        3) the reference to the installation context;
#        4) the reference to the installed patch;
#        5) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 if older version is installed; 
#         2 if version or higher is installed.       
#
sub check_patch ($$$$$) {
  my ($self,$upi,$patch,$ctx,$installedpatch,$refresh) = @_;
  my ($pn,$pv,$iv);
  my ($rc) = 0;
  
  ($self->{PRODUCT}{$upi}->can("get_patch_id")) && ($patch = $self->{PRODUCT}{$upi}->get_patch_id($patch, $ctx));
  if ($patch =~ /-/) {
    ($pn,$pv) = split(/-/,$patch,2);
  } else {
    $$ctx{LOGGER}->severe(Utils::_tr("Invalid patch ID, $patch. The patch ID must be in ######-## format.", 77, 1008, "$patch"));
    return -1;
  }
  
  # get the OS patches for the installation host
  if ($refresh || (!$$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES})) {
    $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES} = $self->get_patches($ctx);
  }
  
  $iv = $$ctx{HOST}{$$ctx{INSTALL}{HOST}}{PATCHES}{$pn};
  if ($iv) {
    $$installedpatch = "${pn}-${iv}";
    ($iv < $pv) && ($rc = 1);
    ($iv >= $pv) && ($rc = 2);
  }
  
  return $rc;
}

#
# Get all the patches for the given host.
#
# Input: 1) the reference of the installation context;
#
# Return: the reference to the patches hash.
#
sub get_patches ($) {
   # Have not found a method of patching or finding out what OS patchs are on Darwin. jpk - 15 Feb 06
#  my ($self,$ctx) = @_;
#  my(%ph,@a,@op,$c,$j,$o,$p,$pkgs,$s,$v);
	

#  $s=Utils::vxif_dor("$self->{CMD}{SHOWREV} -p", "", $ctx);
#  @a=split(/^/,$s);
#  foreach $c (@a) {
#    ($j,$s,$j)=split(/\s/,$c,3);
#    ($p,$v)=split(/-/,$s,2);
#    # all patch keys are $p{######}=##
#    $ph{$p}=$v unless ($v<$ph{$p});
#    ($j,$o)=split(/Obsoletes: /,$c,2);
#    ($o,$j)=split(/ Requires/,$o,2);
#    @op=split(/, /,$o);
#    for $o (@op) {
#      # all obsoleted patch keys are $p{######-##}=######-##
#      $ph{$o}="$p-$v";
#    }
#    # all package references are $p{$PKG} = [ ######, ###### ];
#    chomp($c);
#    ($j,$pkgs)=split(/ Packages: /,$c,2);
#    if ($pkgs=~/VRTS/) {
#      @op=split(/, /,$pkgs);
#      for $o (@op) {
#        $ph{$o}||=[];
#	push(@{$ph{$o}},$p) if (_list($p,@{$ph{$o}})<0);
#      }
#    }
#  }

#  return \%ph;
   return 1;
}

#
# Check for patch dependencies.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
  my ($self,$upi,$pkg,$ctx) = (@_);
  my (@patchlist,$patch,$patchid);
  my ($rc,$installedpatchid,$msgl,$msgr);
  my ($result) = 1;
  
  if ($self->{PRODUCT}{$upi}->can("get_required_os_patches")) {
    @patchlist = $self->{PRODUCT}{$upi}->get_required_os_patches($pkg, $ctx);
  }
  
  if ($#patchlist >= 0 ) {
    Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 77, 1009, "$pkg"));
    foreach $patch (@patchlist) {
      $patchid = $patch;
      ($self->{PRODUCT}{$upi}->can("get_patch_id")) && ($patchid = $self->{PRODUCT}{$upi}->get_patch_id($patch, $ctx));
      $msgl = Utils::_tr("Checking OS patch ${patchid}", 77, 1010, "${patchid}");
      Utils::vxif_pbl($msgl);
      
      $rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
      if ($rc == -1) {
        $msgr = Utils::_tr("invalid patch ID, ${patchid}", 77, 1011, "${patchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 0) {
        $msgr = Utils::_tr("not installed", 77, 1012);
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } elsif ($rc == 1) {
        $msgr = Utils::_tr("${installedpatchid} installed", 77, 1013, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
        $result = 0;
      } else {
        $msgr = Utils::_tr("${installedpatchid} installed", 77, 1013, "${installedpatchid}");
        Utils::vxif_pbr($msgl, $msgr);
      }
    }
  }
  
  return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;

	return 0;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the package installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;

	return 0;
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
   my $self = shift;
   my ($ctx,$pkgs,$pkg) = @_;
   my $logger = $ctx->get_logger();
   my $logdir = $logger->get_logdir();
   my $trg_host = $ctx->get_target_hostname();
   my $tar_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
   my $trg_upi = $ctx->get_target_upi();
   my $pkginfo = $self->{$trg_upi}{$pkg};
   my $tloc = $pkginfo->{TLOC};
   my ($fe, $test, $result, $buildver_loc);

   $logger->entering("CPIDarwin::install_native_package");
   
   my $TOC_Path = "/var/SYMCif/".$trg_upi;
   my $TOC_File = $TOC_Path."/".$pkg."_TOC.txt";
   my $Build_Ver_File = $TOC_Path."/".$pkg."_build_version";

   # Create a Table Of Contents (TOC) to be used for package removal.
   # 1 - create the depository for it;
   $test = Utils::vxif_dc("${TOC_Path}", $ctx);
   if (!$test) {
      $$ctx{LOGGER}->fine("Creating TOC Path $self->{CMD}{MKDIR} -p ${TOC_Path}");
      # if there was a return from the command then $fe would be equal to something like "No file or directory by that name"
      $fe = Utils::vxif_dor("$self->{CMD}{MKDIR} -p ${TOC_Path}", "", $ctx);
      if ($fe) {
         Utils::vxif_lpl("Unable to create ${TOC_Path} on $${ctx{TARGET}{HOST}} with error ${fe}", "error", $ctx);
         return $test;
      }
   }

   # get the directory listing using the find command. - jpk - 08 Dec 05
   $result = Utils::vxif_dor("cd ${tloc}; $self->{CMD}{FIND} . -print  | sed -e '1d' -e 's#^.##' > $TOC_File", "", $ctx);
   if (!result) {
      Utils::vxif_lpl("Unable to create the TOC into ${TOC_File} on $${ctx{TARGET}{HOST}}", "error", $ctx);
      return $result;
   } else {
      $$ctx{LOGGER}->info("Getting build_version file");
      $buildver_loc = Utils::vxif_dor("cd ${tloc}; $self->{CMD}{FIND} . -name build_version -print | sed -e s/^\.//", "", $ctx);
      if (!buildver_loc) {
         $$ctx{LOGGER}->error("Could not find the build version file");
      }
   }

   # check to make sure the given package is not already exist on the target host
   $result = $self->check_for_existing_package($ctx, $pkg, $pkgs);
   if (0 == $result) {
      return 0;
   } elsif (2 == $result) {
      return 1;
   }

   # copy the build version to something AFTER we check for the installed version.  jpk 10 Feb 06
   $buildver_loc = $tloc."/".$buildver_loc;
   $buildver_file = $pkg."_build_version";
   # copy the build_version from the TOC_Path so we can later look up what version is installed. - jpk 08 Dec 05
   # create a name for the build version that shows the name of the package it comes from - jpk 10 Feb 06
   $$ctx{LOGGER}->info("$self->{CMD}{CP} -Rp ${buildver_loc} ${TOC_Path}/${buildver_file}");
   Utils::vxif_dor("$self->{CMD}{CP} -Rp ${buildver_loc} ${TOC_Path}/${buildver_file}", "", $ctx, undef, $tar_logfile);
   if (!Utils::vxif_fc("${TOC_Path}/${buildver_file}",$ctx)) {
      Utils::vxif_lpl("CPIDarwin::install_native_package:Unable to copy the build_version file from ${TOC_Path} to ${tloc}", "error", $ctx);
   }


   # now add the package
   $$ctx{LOGGER}->info("Adding the package $self->{CMD}{CP} -Rp ${tloc}/* /");
   $result = Utils::vxif_dor("$self->{CMD}{CP} -Rp ${tloc}/* /","", $ctx, undef, $tar_logfile);

   # add the TOC and build version file to the TOC File
   $$ctx{LOGGER}->info("$self->{CMD}{ECHO} ${TOC_File} >> ${TOC_File}");
   Utils::vxif_dor("$self->{CMD}{ECHO} ${TOC_File} >> ${TOC_File}", "", $ctx);
   $$ctx{LOGGER}->info("Add the build version information to the TOC $self->{CMD}{ECHO} ${TOC_Path}/${buildver_file} >> ${TOC_File}");
   Utils::vxif_dor("$self->{CMD}{ECHO} ${buildver_loc} >> ${TOC_File}", "", $ctx);

   if ($result) {
      Utils::vxif_lpl("CPIDarwin::install_native_package:Unable to place the software package ${pkg} on $${ctx{TARGET}{HOST}}", "error", $ctx);
      $logger->exiting("CPIDarwin::install_native_package - error");
      return $result;
   } else {
      $logger->exiting("CPIDarwin::install_native_package - success");
      return $self->is_package_install_successful($ctx, $pkg, $pkgs);
   }

} # end of install_native_package

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the package to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
        my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $tar_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
        my ($result, $test, $file ,$dir_check, @dir_list, @file_list);

	$logger->entering("CPIDarwin::uninstall_native_package");

        my $TOC_Path = "/var/SYMCif/".$trg_upi;
        my $TOC_File = $TOC_Path."/".$pkg."_TOC.txt";
        my $Build_Ver_File = $TOC_Path."/".$pkg."_build_version";

        # read the TOC formed during installation
        $test = Utils::vxif_fc("${TOC_File}", $ctx);
        if (!$test) {
           Utils::vxif_lpl("Unable to find ${TOC_File}", "error", $ctx);
           return 0;
        }

        $$ctx{LOGGER}->info("$self->{CMD}{CAT} ${TOC_File}");
        $file = Utils::vxif_dor("$self->{CMD}{CAT} ${TOC_File}", "", $ctx);
        @file_list = split(/\n/, $file);

        # Now remove the files that were installed. Find out which are the directories.  If the entry is a directory
        # do not remove it using rm -rf.  We want to use rmdir so we leave behind files that we did not create.
        foreach  $file (@file_list) {
           if (Utils::vxif_dc("${file}", $ctx)) {

              # Okay we don't want to remove the directories under root.  So count the number
              # of / in the list.  If it is greater than two, try and remove it.  If it is two
              # do not remove it as it too near the root. 08 Mar 06 - jpk
              my (@count) = split(/\//,$file);
              if ($#count > 2) {
                 push(@dir_list,$file);
              }
           } else {
              Utils::vxif_dor("$self->{CMD}{RM} ${file}", "", $ctx, undef, $tar_logfile);
              $test = Utils::vxif_fc("${file}", $ctx);
              if ($test) {
                 #  if we cannot remove the file, just list it and drive on.
                 $$ctx{LOGGER}->info(Utils::_tr("Unable to remove ${file}", 77, 1014, "${file}"));
              }  
           }
        } # end of foreach

        # Wack the directories no longer needed - 08 Mar 06 - jpk
        foreach my $dir (@dir_list) {
           if (Utils::vxif_dc("${dir}", $ctx)) {
              $result = Utils::vxif_dor("$self->{CMD}{RMR} ${dir}", "", $ctx, undef,);
              if ($result) {
                 $$ctx{LOGGER}->info(Utils::_tr("CPIDarwin::uninstall_native_package - Unable to remove directory ${dir} from ${trg_host}", 77, 1015, "${dir}", "${trg_host}"));
              }
           }
        }

        # Now to remove the build version file
        $result = Utils::vxif_dor("$self->{CMD}{RMR} ${Build_Ver_File}", "", $ctx);
        if ($result) {
           $$ctx{LOGGER}->info(Utils::_tr("CPIDarwin::uninstall_native_package - Unable to remove ${Build_Ver_File}", 77, 1016, "${Build_Ver_File}"));
           return 0;
        }
        # Now remove the entries under /var/SYMCif including the TOC file.
        $result = Utils::vxif_dor("$self->{CMD}{RMR} ${TOC_File}","", $ctx);
        if ($result) {
           $$ctx{LOGGER}->info(Utils::_tr("CPIDarwin::uninstall_native_package - Unable to remove ${TOC_File}", 77, 1016, "${TOC_File}"));
           return 0;
        }

	# if the package is successfully uninstalled, the version number null.
	if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg) eq "")) {
		$$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from ${trg_host}", 77, 1017, "${pkg}", "${trg_host}"));
		return 1;
	} else {
           $$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from ${trg_host}", 77, 1018, "${pkg}", "${trg_host}"));
           return 0;
        }
} # end of uninstall_native_package

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%) {
  my ($self,$ctx) = @_;
  my ($msgl,$msgr);

  # The messages are currently moot as they are done by the Interface::Classic::Events subroutines test_external_dependencies or 
  # external_dependencies_test_pass or external_dependencies_test_fail.

 # $$ctx{LOGGER}->entering("CPIDarwin::check_for_external_dependencies");
  #$msgl = Utils::_tr("Checking for external dependencies CPIDarwin");
  #Utils::vxif_pbl($msgl);
  #$msgr = Utils::_tr("all external dependencies satisfied");
  #Utils::vxif_pbr($msgl, $msgr);
  #$$ctx{LOGGER}->exiting("CPIDarwin::check_for_external_dependencies");
  return 1; # stub this out for now
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
	my $self = shift;
	my ($ctx,$pkg,$fod,$pkgs) = @_;
	my $package_info = $pkgs->{$pkg};
	my $rpm = $package_info->search_rpm();
	
	$$ctx{LOGGER}->entering("CPIDarwin::get_native_package_file");
	$$ctx{LOGGER}->fine("RPM file for ${pkg} = ${rpm}");
	
	$$fod = "f";
	return $rpm;
}

1;
