package Dictionary;
use strict;

sub new {
	my $pkg = shift;
	my $self = {};

	bless($self, $pkg);

	$self->init(@_);

	return $self;
}

sub init {
	my $self = shift;

	$self->set_defs_aref([]);
	$self->set_defs_href({});
}

sub get_defs_aref {
	my $self = shift;
	my $defs_aref = $self->{defs_aref};

	return $defs_aref;
}

sub set_defs_aref {
	my $self = shift;
	my ($defs_aref) = @_;

	$self->{defs_aref} = $defs_aref;
}

sub get_defs_href {
	my $self = shift;
	my $defs_href = $self->{defs_href};

	return $defs_href;
}

sub set_defs_href {
	my $self = shift;
	my ($defs_href) = @_;

	$self->{defs_href} = $defs_href;
}

sub add_definitions {
	my $self = shift;
	my ($definitions) = @_;
	my $defs_aref = $self->get_defs_aref();
	my $defs_href = $self->get_defs_href();

	for my $definition (@$definitions) {
		my $name = $definition->{name};
		$defs_href->{$name} = $definition;
		push(@$defs_aref, $name);
	}
}

sub lookup {
	my $self = shift;
	my ($name) = @_;
	my $defs_href = $self->get_defs_href();
	my $definition = $defs_href->{$name};

	return $definition;
}

# expand while type == definition (i.e. type == package ==> stop)
sub expand {
	my $self = shift;
	my ($name) = @_;
	my $expanded = [];
	my $seen = {}; # no recursive definitions
	my $def = $self->lookup($name);
	my $seq = $def->{sequence};
	my $stack = [];

	unshift(@$stack, {ix => 0, seq => $seq});

	while (scalar(@$stack)) {
		my $seq_exp = $stack->[0];
		my $seq = $seq_exp->{seq};
		my $ref = $seq->[$seq_exp->{ix}];
		my $name = $ref->{name};

		if ($seen->{$name}) {
			die "Recursive definitions not allowed (${name})\n";
		}

		$seen->{$name} = 1;

		if ($ref->{type} eq "definition") {
			# another sequence to expand
			my $new_def = $self->lookup($name);
			my $new_seq = $new_def->{sequence};
			unshift(@$stack, {ix => 0, seq => $new_seq});
		} else {
			# type == package
			push(@$expanded, $ref->{name});
			$seq_exp->{ix}++;
			while ($seq_exp->{ix} >= scalar(@{$seq_exp->{seq}})) {
				# done with this one, so pop stack
				shift(@$stack);
				if (!scalar(@$stack)) {
					last;
				}
				$seq_exp = $stack->[0];
				$stack->[0]{ix}++;
			}
		}
	}

	return $expanded;
}

sub list {
	my $self = shift;
	my ($indent) = @_;
	my $defs_aref = $self->get_defs_aref();
	my $defs_href = $self->get_defs_href();
	my $ind = " " x $indent;

	print("\n");

	my $title = "Definitions";
	my $underline = "-" x length($title);
	print("${ind}${title}\n");
	print("${ind}${underline}\n");

	print("\n");

	for my $name (@$defs_aref) {
		my $definition = $defs_href->{$name};
		print("${ind}  ${name} =");
		my $sequence = $definition->{sequence};
		for my $reference (@$sequence) {
			my $ref_name = $reference->{name};
			print(" ${ref_name}");
		}
		print("\n");

		my $expanded = $self->expand($name);
		my $exp_str = join(" ", @$expanded);
		print("${ind}    ${exp_str}\n");
	}

	print("\n");
}

1;

