package CPICommon;
#use strict;

use VxIF::License;
@CPICommon::ISA = qw(License);

use VxIF::SpaceMgr;

###############################
# Private Members             #
###############################


###############################
# Private Methods             #
###############################


###############################
# Public Methods       	      #
###############################

#
# Set the common system commands.
#
# Input: 1) the reference of the installation context;
#
sub set_commands ($) {
  my ($self) = @_;
  
  $self->{CMD}{AWK}="/usr/bin/awk";
  $self->{CMD}{CAT}="/usr/bin/cat";
  $self->{CMD}{CD}="/usr/bin/cd";
  $self->{CMD}{CHMOD}="/usr/bin/chmod";
  $self->{CMD}{CLEAR}="/usr/bin/clear";
  $self->{CMD}{CP}="/usr/bin/cp";
  $self->{CMD}{DFK}="/usr/bin/df -kl";
  $self->{CMD}{DU}="/usr/bin/du";
  $self->{CMD}{ECHO}="/usr/bin/echo";
  $self->{CMD}{FIND}="/usr/bin/find";
  $self->{CMD}{GREP}="/usr/bin/grep";
  $self->{CMD}{ID}="/usr/bin/id";
  $self->{CMD}{IFCONFIG}="/usr/sbin/ifconfig";
  $self->{CMD}{KILL}="/usr/bin/kill";
  $self->{CMD}{LN}="/usr/bin/ln";
  $self->{CMD}{LS}="/usr/bin/ls";
  $self->{CMD}{MKDIR}="/usr/bin/mkdir";
  $self->{CMD}{MOUNT}="/usr/sbin/mount";
  $self->{CMD}{MV}="/usr/bin/mv";
  $self->{CMD}{PING}="/usr/sbin/ping";
  $self->{CMD}{PS}="/usr/bin/ps";
  $self->{CMD}{RCP}="/usr/bin/rcp";
  $self->{CMD}{RMDIR}="/usr/bin/rmdir";
  $self->{CMD}{RMR}="/usr/bin/rm -rf";
  $self->{CMD}{RSH}="/usr/bin/rsh";
  $self->{CMD}{SED}="/usr/bin/sed";
  $self->{CMD}{SHUTDOWN}="/usr/sbin/shutdown -y -i6 -g0";
  $self->{CMD}{SORT}="/usr/bin/sort";
  $self->{CMD}{TAR}="/usr/bin/tar";
  $self->{CMD}{TOUCH}="/usr/bin/touch";
  $self->{CMD}{TPUT}="/usr/bin/tput";
  $self->{CMD}{UNAME}="/usr/bin/uname";
  $self->{CMD}{UNIQ}="/usr/bin/uniq";
  $self->{CMD}{WC}="/usr/bin/wc";
  $self->{CMD}{WHICH}="/usr/bin/which";

  # Define unique platform specific commands 
  $self->{CMD}{GUNZIP}="/opt/VRTScpi/bin/gunzip";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  $self->{CMD}{KSTAT}="/usr/bin/kstat";
  $self->{CMD}{MODINFO}="/usr/sbin/modinfo";
  $self->{CMD}{MODUNLOAD}="/usr/sbin/modunload";
  $self->{CMD}{PATCHADD}="/usr/sbin/patchadd";
  $self->{CMD}{PKGADD}="/usr/sbin/pkgadd";
  $self->{CMD}{PKGINFO}="/usr/bin/pkginfo";
  $self->{CMD}{PKGRM}=  $self->{CMD}{RMPOBJ}="/usr/sbin/pkgrm";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{SHOWREV}="/usr/bin/showrev";
  
  # VERITAS licensing commands
  $self->{CMD}{VXLICINST}="/sbin/vxlicinst";
  $self->{CMD}{VXLICREP}="/sbin/vxlicrep";
  $self->{CMD}{VXLICTEST}="/sbin/vxlictest";
}

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->License::set_msgs();
  
  $self->{MSG}{ERROR}           = Utils::_tr("ERROR", 16, 1000);
  $self->{MSG}{LICENSED}        = Utils::_tr("Licensed", 16, 1001);
  $self->{MSG}{NA}              = Utils::_tr("N/A", 16, 1002);
  $self->{MSG}{NO}              = Utils::_tr("no", 16, 1003);
  $self->{MSG}{VENDORPR}        = Utils::_tr("Veritas Product", 16, 1112);
  $self->{MSG}{VERSINST}        = Utils::_tr("Version Installed", 16, 1005);
  $self->{MSG}{YES}             = Utils::_tr("yes", 16, 1006);
  $self->{MSG}{YESKEY}{L}       = Utils::_tr("y", 16, 1113);
  $self->{MSG}{YESKEY}{U}       = Utils::_tr("Y", 16, 1114);
  $self->{MSG}{NOKEY}{L}        = Utils::_tr("n", 16, 1115);
  $self->{MSG}{NOKEY}{U}        = Utils::_tr("N", 16, 1116);
}

#
# Return the $CMD hash.
#
sub get_commands {
	my $self = shift;
	my $cmds = $self->{CMD};
	
	return $cmds;
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my ($self) = @_;
  
  $self->{LICENSEUPI} = "LIC";
  $self->{PERLPKG} = "PERL";
  
  return 1;
}

sub get_unlimit_filesize_cmd {
	my $self = shift;
	my ($ctx) = @_;
	my $cmd = undef;

	# override this in OS-specific common module.

	return $cmd;
}

#
# Get the local host name.
#
# Input: 1) the reference of the installation context;
#
sub get_local_host ($) {
	my $self = shift;
	my ($ctx) = @_;
	my $local_cmds = $ctx->get_local_cmds();
	my $uname_cmd = $local_cmds->{UNAME};
	my $fqdn = Utils::vxif_dol("${uname_cmd} -n");

	$fqdn =~ s/\..*$//;
	$$ctx{LOCAL}{HOST} = $fqdn;
	$$ctx{LOCAL}{OS}{NAME} = Utils::vxif_dol("${uname_cmd}");
	$$ctx{LOCAL}{OS}{NAME} =~ s/-//; # get rid of any - character. Mainly for HP-UX. 
	$$ctx{LOCAL}{OS}{VERSION} = Utils::vxif_dol("${uname_cmd} -r");
	$$ctx{LOCAL}{OS}{ARCHITECTURE} = Utils::vxif_dol("${uname_cmd} -m");
}

#
# Get the local domain name.
#
# Input: 1) the reference of the installation context;
#
sub get_local_domain ($) {
	my $self = shift;
	my ($ctx) = @_;
	my $domain = undef;
	my $resolve_fnam = "/etc/resolv.conf";
	my $local_cmds = $ctx->get_local_cmds();

	# first see if there is a local domain mapping
	if (-f $resolve_fnam) {
		my $grep_cmd = $local_cmds->{GREP};
		my $resolve_domain_cmd = "${grep_cmd} domain ${resolve_fnam}";
		my $domain_rec = Utils::vxif_dol($resolve_domain_cmd);
		my @domain_fields = split(/\s+/, $domain_rec);

		# it's in the 2nd field (starting at 0, of course :-)
		$domain = $domain_fields[1];
	}

	# If we can't get the domain name resolv.conf, use the 'domainname' command.
	if (!$domain) {
		my $domain_cmd = $local_cmds->{DOMAINNAME};
		my $local_domainname_cmd = $local_cmds->{DOMAINNAME};
		$domain = Utils::vxif_dol($local_domainname_cmd);
	}
    
	$$ctx{LOCAL}{DOMAIN}=$domain;
}

#

# Get the pid of a process string from ps -ef output.
#
# Input: 1) the reference to the installation context;
#        2) the process name;
#
# Return: the process ID of the given process if it is running or null.
#
sub get_pid (\%$) {
	my $self = shift;
	my ($ctx, $proc) = @_;
	my $quote = '"';
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	my $cmds = $ctx->get_host_os_cmds($trg_host);
	my $ps_cmd = $cmds->{PS};
	my $grep_cmd = $cmds->{GREP};
        my ($cmd, $cmd_outp);

	$quote = "" if (Utils::vxif_localsys($ctx));

	# filter the process records via grep   jpk
        if ($trg_os_name eq "Darwin") {
           $cmd = "${quote}${ps_cmd} -aj | ${grep_cmd} '$proc' | ${grep_cmd} -v grep${quote}";
        } else {
           $cmd = "${quote}${ps_cmd} -ef | ${grep_cmd} '$proc' | ${grep_cmd} -v grep${quote}";
        }
	$cmd_outp = Utils::vxif_dor($cmd, "", $ctx);

	# do the field extraction in perl
	#
	# strip off initial space
	$cmd_outp =~ s/^\s+//;

	# a simple chop into tokens
	my @fields = split(/\s+/, $cmd_outp);

	# we want the 2nd field (the process id)
	my $ret = $fields[1];
  
	return $ret;
}

#
# Kill a process with the given PID.
#
# Input: 1) the reference to the installation context;
#        2) the process name;
#        3) the process IDs associate with this process separated by a white space;
#
# Return: 1 if successful; 0 otherwise.
#
sub kill_proc {
	my $self = shift;
	my ($ctx,$name,$pid) = @_;
	my (@pids,$ps,$ukp,$count);
	my ($iter) = 5; # number of times to wait for the process to die.
	my $trg_host = $ctx->get_target_hostname();
	my $cmds = $ctx->get_host_os_cmds($trg_host);
	my $grep_cmd = $cmds->{GREP};
	my $kill_cmd = $cmds->{KILL};
	my $ps_cmd = $cmds->{PS};

	@pids = split(/\s+/,$pid);
	$$ctx{LOGGER}->info("Killing ${name}");

	foreach $pid (@pids) {
		$count = 0;
		while ($count < $iter) {
			my $kill_pid_cmd = "${kill_cmd} -9 ${pid}";
			Utils::vxif_dor($kill_pid_cmd, "", $ctx);
			$$ctx{LOGGER}->info("Killing ${pid} for ${name}");
			sleep 5; # wait for 5 seconds for the processes to die
			$ps = Utils::vxif_dor("${ps_cmd} -p ${pid} | ${grep_cmd} -v PID", "", $ctx);
			last if (!$ps);
			++$count;
		}
	}

	foreach $pid (@pids) {
		$ps = Utils::vxif_dor("${ps_cmd} -p ${pid} | ${grep_cmd} -v PID", "", $ctx);
		$ukp .= " ${pid}" if ($ps);
	}

	if ($ukp) {
		$$ctx{LOGGER}->info("Could not kill pids: $ukp");
		return 0;
	} else {
		$$ctx{LOGGER}->info("Done");
		return 1;
	}
}

#
# Get the netmask currently configured on the base IP of a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the base IP address;
#        3) the NIC;
#
# Return: the netmask.
#
sub get_defaultnetmask (\%;$$) {
  my ($self,$ctx,$ip,$nic) = @_;
  my (@f,$h,$hnm,$n,$nm);
  
  $hnm = $self->get_netmask($ctx, $nic) if ($nic);
  if ($hnm) {
    @f=split(/\./,$hnm);
    return "$hnm" if ($#f==3);
    $hnm=~s/^0x//;
    for $n(0..3) {
      $h = hex(substr($hnm,$n*2,2));
      $nm .= $h;
      $nm .= "." if ($n<3);
    }
  } else {
    $ip =~ s/\..*$//;
    $nm = ($ip<128) ? "255.0.0.0" : ($ip<192) ? "255.255.0.0" :
      "255.255.255.0";
  }
  
  return "$nm";
}

# 
# Ask the user if they are ready to stop a process.
#
# Input: 1) the reference of the installation context;
#        2) the process name;
#
# Return: "Y" or "N".
#
sub ask_stop ($$) {
  my ($self, $ctx, $process_name) = @_;
  my ($msg);
  
  my $quit_key_handler = sub {
    $self->cleanup($ctx);
  };
  Utils::vxif_pl(Utils::_tr("\n${process_name} processes must be stopped completely before it can be configured by $$ctx{PRODUCT}.", 16, 1025, "${process_name}", "$$ctx{PRODUCT}"));
  $msg = Utils::_tr("Do you want to stop ${process_name} processes?", 16, 1026, "${process_name}");
  return Utils::vxif_ayny($msg, "", "", $quit_key_handler);
}

#
# Check to make sure the package files exist and copy the package files to target host if
# necessary. If the package files are compressed, uncomress them.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name;
#        4) the reference to the packages hash;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_product_package_files (\%$$\%) {
  my ($self,$ctx,$pkg,$patch,$pkgs) = @_;
  
  # check to make sure the package files exist and copy the package files to target host if necessary
  return 0 if (!$self->copy_files_to_target_host($ctx, $pkg, $patch, $pkgs));
  
  # uncompress the package files if the are currently compressed
  return $self->uncompress_files_on_target_host($ctx, $pkg, $patch, $pkgs);
}

#
# Check to make sure the given package exist on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name if this is a patch;
#        4) the reference to the packages hash;
#
# Return: 1 if the require files exist on the installation media; 0 otherwise.
#
sub check_package_on_media (\%$$\%) {
	my $self = shift;
	my ($ctx,$pkg,$patch,$pkgs) = @_;
	my ($compressed,$sf,$filename);
	my ($fe) = 0; 	# file exist?
	my ($loc,@locs,$fod,$pkgloc);
	my ($iname) = "package";
	my ($dname) = $self->{PKGSDIR};
	my ($pkgpath) = $self->{PKGPATH};
	my ($func);
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_os_name = $ctx->get_host_os_name($trg_host);
	my $upi_inst = $ctx->get_os_upi_installer($trg_os_name);

	$$ctx{LOGGER}->entering("CPICommon::check_packages_on_media");

	# make sure we have a package name or patch name
	if ((!$pkg)&&(!$patch)) {
		$$ctx{LOGGER}->error("Unable to determine the file on the media. Must specify a package name or patch name.");
		return 0;
	}

	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
		$dname = $self->{PATCHESDIR};
	}

	$compressed = 0;
	$filename = $self->get_native_package_file($ctx, $pkg, \$fod, $pkgs);

	# Get the directory paths to search for the given package or patch.
	push(@locs, "./${trg_upi}/${dname}/${trg_os_name}");
	push(@locs,$pkgpath) if ($pkgpath);

	if ($upi_inst->can("get_product_train_dir")) {
		my $product_train_dir = $upi_inst->get_product_train_dir($ctx);
		push(@locs, $product_train_dir . "/${dname}");
	}
	$func = "get_${iname}_location";
	($upi_inst->can($func)) && (push(@locs, $upi_inst->$func($ctx, $pkg)));

	# Get the package file or directory name
	$func = "get_${iname}_compression_type";
	($upi_inst->can($func)) && ($compressed = $upi_inst->$func($ctx, $pkg));
	if ($compressed > 0) {
		$func = "get_${iname}_compressed_filename";
		if ($upi_inst->can($func)) {
			$filename = $upi_inst->$func($ctx, $pkg);
		} else {
			# default compressed file naming convention
			$filename = "${upi}${pkg}.tar.gz";
		}
	}

	# check to make sure the package exist on localhost
	foreach $loc (@locs) {
		$sf = "${loc}/${filename}";

		$pkgloc = $loc;
		$self->{$trg_upi}{$pkg}{TLOC} = $loc;
		$self->{$trg_upi}{$pkg}{SLOC} = $loc;
		$$ctx{LOGGER}->info("Checking ${sf} on $$ctx{LOCAL}{HOST}.");
		$fe = $compressed ? (-f $sf) : ($fod eq "d") ? (-d $sf) : (-f $sf);
		if ($fe) {
			last;
		}
	}

	# exist if package file not found
	if (!$fe) {
		Utils::vxif_lpl("Package file associated with ${pkg} not found on $$ctx{LOCAL}{HOST}.", "error", $ctx);
		$$ctx{FATALERROR}="One or more packages not found on $$ctx{LOCAL}{HOST}.";

		return 0;
	}

	$self->{$trg_upi}{$pkg}{COMPRESSED} = $compressed;
	$self->{$trg_upi}{$pkg}{FILENAME} = $filename;
	
	# determine the spaces needed for the package;
	$func = "get_${iname}_compressed_filesize";
	if (($compressed)&&($upi_inst->can($func))) {
		$self->{$trg_upi}{$pkg}{SPACE} = $upi_inst->$func($ctx, $pkg);
	} else {
		$self->{$trg_upi}{$pkg}{SPACE} = $self->get_mediapkgsize($ctx, $pkgs, $pkg, $pkgloc, $compressed, $filename);
	}
	
	if ((($compressed)||(!Utils::vxif_localsys($ctx)))&&($self->{$trg_upi}{$pkg}{SPACE} > $self->{$trg_upi}{MAXPKGSPACE})) {
		$self->{$trg_upi}{MAXPKGSPACE} = $self->{$trg_upi}{$pkg}{SPACE};
	}
    
	return 1;
}

#
# Copy the necessary files to the target host if it is a remote host.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#        3) patch name;
#        4) the reference to the packages hash;
#
# Return: 1 if everything is ok; 0 otherwise.
#
sub copy_files_to_target_host (\%$$\%) {
	my $self = shift;
	my ($ctx,$pkg,$patch,$pkgs) = @_;
	my ($fe) = 0;
	
	$$ctx{LOGGER}->entering("CPICommon::copy_files_to_target_host");
	
	if ($patch) {
		$pkg = $patch;
	}

	my $local_host = $ctx->get_local_hostname();
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();

	# I just *LLUUVV* having both of these (NOT!)
	my $pkginfo = $self->{$trg_upi}{$pkg};
	my $package_info = $pkgs->{$pkg};

	# for now, this will be just here
	my $tloc = $pkginfo->{TLOC};

	my $sloc =
		(
		 defined($pkginfo->{SLOC})
		 ? $pkginfo->{SLOC}
		 : $package_info->search_media_location()
		 );

	my $filename =
		(
		 defined($pkginfo->{FILENAME})
		 ? $pkginfo->{FILENAME}
		 : $package_info->search_compressed_file()
		 );

	my $compressed =
		(
		 defined($pkginfo->{COMPRESSED})
		 ? $pkginfo->{COMPRESSED}
		 : $package_info->search_compressed()
		 );

	if (!defined($sloc)) {
		$$ctx{LOGGER}->error("Unable to determine the location of ${pkg} on ${local_host}.");
		return 0;
	}
	
	# nothing to be done if install on localhost and the package is uncompressed.
	return 1 if (Utils::vxif_localsys($ctx)) && (!$compressed);

	# remove the old temporary directory on target host
	if (!Utils::vxif_rmdir($tloc, $ctx)) {
		Utils::vxif_lpl("Unable to remove ${tloc} on ${trg_host}.", "error", $ctx);
		return 0;
	}

	# create the temporary package directory on target host.
	if (!Utils::vxif_mkdir($tloc, $ctx)) {
		Utils::vxif_lpl("Unable to create directory, ${tloc}, on ${trg_host}", "error", $ctx);
		return 0;
	}

    # add '\' before space to ensure the path with spaces is valid during copying files later. Fixed defect 843525.
    if ($sloc =~ / /) {
        $sloc =~ s/ /\\ /;
    }
    
	# copy the stuff to the target host
	Utils::vxif_copy("${sloc}/${filename}", $tloc, $ctx, "", $trg_host, "");

	# check to see if the file/directory is copied over
	if ($compressed) {
		$fe = Utils::vxif_fc("${tloc}/${filename}", $ctx);
	} else {
		$fe = Utils::vxif_dc("${tloc}/${filename}", $ctx);
	}

	if (!$fe) {
		Utils::vxif_lpl("Unable to copy, ${sloc}/${filename}, on ${local_host} to, ${tloc}, on ${trg_host}.", "error", $ctx);
	}

	return $fe;
}

#
# Uncompress files on target host.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name;
#
# Return: 1 if everything is ok; 0 otherwise.
#
sub uncompress_files_on_target_host (\%$$$) {
	my $self = shift;
	my ($ctx,$pkg,$patch, $pkgs) = @_;
	my ($msg);
	my ($iname) = "package";
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_cmds = $ctx->get_host_os_cmds($trg_host);
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $rc;

	$$ctx{LOGGER}->entering("CPICommon::uncompress_files_on_target_host");

	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
	}

	my $pkginfo = $self->{$trg_upi}{$pkg};
	my $package_info = $pkgs->{$pkg};

	# for now, this will be just here
	my $tloc = $pkginfo->{TLOC};

	my $filename =
		(
		 defined($pkginfo->{FILENAME})
		 ? $pkginfo->{FILENAME}
		 : $package_info->search_compressed_file()
		 );

	my $compressed =
		(
		 defined($pkginfo->{COMPRESSED})
		 ? $pkginfo->{COMPRESSED}
		 : $package_info->search_compressed()
		 );

	# nothing to be done if the files are uncompressed
	return 1 if (!$compressed);
    
	# make sure the gunzip tools exist on target
	if (!Utils::vxif_fc($trg_cmds->{GUNZIP}, $ctx)) {
		# search in CMDSEARCHPATHS
		foreach my $path (@{$self->{CMDSEARCHPATHS}}) {
			if (Utils::vxif_fc("$path/gunzip", $ctx)) {
				$trg_cmds->{GUNZIP} = "$path/gunzip";
				last;
			}
		}
	}

	if (!Utils::vxif_fc($trg_cmds->{GUNZIP}, $ctx)) {
		$msg = Utils::_tr("Unable to uncompress files. $trg_cmds->{GUNZIP} not found on $trg_host.", 16, 1117, "$trg_cmds", "$trg_host");
		Utils::vxif_lpl($msg, "error", $ctx);
		return 0;
	}
	
	# uncompress the stuff
	if ($compressed == 1) {

		# get any file size unlimiting command, just in case
		my $unlimit_filesize_cmd = $self->get_unlimit_filesize_cmd($ctx);

		my $uncompress_cmd = "cd ${tloc}; $trg_cmds->{GUNZIP} ${filename}";
		if ($unlimit_filesize_cmd) {
			$uncompress_cmd = "${unlimit_filesize_cmd}; ${uncompress_cmd}";
		}

		$$ctx{LOGGER}->info("Uncompressing ${tloc}/${filename} on host ${trg_host}.");

		# the package is compressed in .tar.gz; Untar the stuff.
		$rc = Utils::vxif_dor($uncompress_cmd, 1, $ctx);
		if ($rc != 0) {
			$$ctx{LOGGER}->error("Unable to unzip ${tloc}/${filename} on ${trg_host}. Return code = ${rc}");
			return 0;
		}
		
		# filename will change after it is unzipped
		$filename=~s/\.gz//;

		# compose tar extraction command
		my $extract_cmd = "cd ${tloc}; $trg_cmds->{TAR} -xvf ${filename}";
		if ($unlimit_filesize_cmd) {
			$extract_cmd = "${unlimit_filesize_cmd}; ${extract_cmd}";
		}
		
		$$ctx{LOGGER}->info("Extracting ${tloc}/${filename} on host ${trg_host}.");

		# don't ask me why, but /usr/bin/cd doesn't work here
		$rc = Utils::vxif_dor($extract_cmd, 1, $ctx);
		if ($rc != 0) {
			$$ctx{LOGGER}->error("Unable to untar ${tloc}/${filename} on ${trg_host}. Return code = $rc");
			return 0;
		}
	} elsif ($compressed == 2) {
		# the package is compressed in some other format; Call custom decompressor here.
		my $func = "get_${iname}_decompressor";
		if ($upi_inst->can($func)) {
			my $decompressor = $upi_inst->$func($ctc, $pkg, $filename, $tloc, $tloc);
			if ($decompressor) {
				$$ctx{LOGGER}->info("Decompressing ${tloc}/${filename} on host ${trg_host}.");
				my $rc = $decompressor->($ctx, $pkg, $filename, $tloc, $tloc);
				return $rc;
			} else {
				$$ctx{LOGGER}->error("Unable to uncompress ${tloc}/${filename} on $trg_host. Decompressor is null.");
				return 0;
			}
		} else {
			$$ctx{LOGGER}->error("Unable to obtain the custom decompressor to uncompress ${tloc}/${filename} on $trg_host.");
			return 0;
		}
	} else {
		my $ct = $compressed;
		$$ctx{LOGGER}->error("Compression type, ${ct}, is an unsupported compression format.");
		return 0;
	}
	
	# now remove the compressed file since we no longer need it
	if (Utils::vxif_dor("$trg_cmds->{RMR} ${tloc}/${filename}", 1, $ctx)) {
		$$ctx{LOGGER}->error("Unable to remove ${tloc}/${filename} on $trg_host.");
		return 0;
	}
	
	return 1;
}

#
# Cleanup the temporary directories, package tar file, etc on target host.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name;
#
# Return: 1 if verything is happy; 0 otherwise.
#
sub cleanup_temporary_directory (\%$$) {
	my $self = shift;
	my ($ctx,$pkg,$patch) = (@_);
	my ($respfile,$loc);
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_cmds = $ctx->get_host_os_cmds($trg_host);
	
	$$ctx{LOGGER}->entering("CPICommon::cleanup_temporary_directory");
    
	if ($patch) {
		$pkg = $patch;
	}
	
	# remove the response file if it exists
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $trg_respfile = "${trg_tmppath}/pkgresp.${trg_upi}.${pkg}.${trg_host}";
	(Utils::vxif_fc($trg_respfile, $ctx)) && (Utils::vxif_dor("$trg_cmds->{RMR} ${trg_respfile}", "", $ctx));
	(Utils::vxif_fc($trg_respfile, $ctx)) && ($$ctx{LOGGER}->warning("Unable to remove ${trg_respfile} in ${trg_host}."));
	
	# remove the temporary package directory if it exist
	$loc = $self->{$trg_upi}{$pkg}{TLOC};
	if ($loc =~ /${trg_tmppath}/) {
		$$ctx{LOGGER}->fine("Removing ${loc} on ${trg_host}.");
		if (!Utils::vxif_rmdir($loc, $ctx)) {
			$$ctx{LOGGER}->warning("Unable to remove ${loc} on ${trg_host}.");
			return 0;
		}
		$$ctx{LOGGER}->fine("Removed ${loc} on ${trg_host}.");
	}
	
	return 1;
}

#
# Check whether a given package has already existing on the target host. It it does, determine
# whether the existing version is compatible.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the reference to the packages hash;
#
# Return: 4 if version installed on machine is newer than version being installed
#         3 if the version needs upgrading
#         2 if same version already installed
#         1 if pkg not installed; 
#	  0 failure.
#
sub check_for_existing_package (\%$\%) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs) = @_;
	my ($version,$pkgver,$decision);
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
        my $rtn = 1;

        $$ctx{LOGGER}->entering("CPICommon::check_for_existing_package");

	$version = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);  # this is returning blank jpk 4/26
	$pkgver = $pkgs->{$pkg}->search_version($pkg);
	$pkgver = $self->get_mediapkgvers($ctx, $pkg, $pkgs) if (!$pkgver);
	
    my $temp = $pkgver;
    my $temp1 = $version;
    my $local_host_os_name = Utils::vxif_dol("uname -s");
   chomp $local_host_os_name;
   if ($local_host_os_name eq "AIX" || $local_host_os_name eq "Linux" || $local_host_os_name eq "SunOS") 
   {
      $temp =~ s/-/./g;
      $temp1=~ s/-/./g;
   }

	if ($version) {
           if ($temp1 eq $temp) {
              $$ctx{LOGGER}->info("Version ${temp1} of ${pkg} found on ${trg_host}. This version is the same as the product to be installed.");
              $rtn = 2;
           } elsif ($temp1 gt $temp) {
              $$ctx{LOGGER}->info("Version ${temp1} of ${pkg} found on ${trg_host}. The installed version is newer than the product to be installed."); 
              $rtn = 4;
           } elsif ($upi_inst->can("is_package_version_supported")) {
              $decision = $upi_inst->is_package_version_supported($ctx, $pkg, $version);
              if ($decision == 0) {
                 $$ctx{LOGGER}->info("Version ${version} of ${pkg} found on ${trg_host}. This version requires upgrading.");
                 $rtn = 3;
              } elsif ($decision == 2){
                 $rtn = 0;
              }
           }
        }
        $$ctx{LOGGER}->exiting("CPICommon::check_for_existing_package");
	return $rtn;
}

#
# Check to make sure the given package has been successfully installed.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the reference to the packages hash;
#
# Return: 1 if the given package has been successfully installed; 0 otherwise.
#
sub is_package_install_successful (\%$\%) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs) = @_;
	my $trg_host = $ctx->get_target_hostname();

	# if the package is successfully installed, the version number should be the same.
	my $iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
	my $package_info = $pkgs->{$pkg};
	my $pkgver = $package_info->search_version($pkg);
	$pkgver = $self->get_mediapkgvers($ctx, $pkg, $pkgs) if (!$pkgver);
	
    $$ctx{LOGGER}->info("${pkg} successfully added to ${trg_host}");
	return 1;
}

#
# Check to make sure the given patch has been successfully installed.
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the reference to the patches hash;
#
# Return: 1 if the given patch has been successfully installed; 0 otherwise.
#
sub is_patch_install_successful (\%$\%) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs) = @_;
	my $trg_host = $ctx->get_target_hostname();

	$$ctx{LOGGER}->entering("CPICommon::is_patch_install_successful");

	if (!$self->patch_installerror($ctx, $pkg)) {
		$$ctx{LOGGER}->info("${patch} successfully added to ${trg_host}");
		return 1;
	}

	$$ctx{LOGGER}->info("Unable to add ${patch} to ${trg_host}");

	return 0;
}

#
# Sort the packages to make sure the dependencies are ordered correctly.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the new package list;
#        3) the reference to the packages hash;
#        4) the original package list;
#
sub sort_product_packages (\%\@\%@) {
  my ($self,$ctx,$npkgs,$pkgs,@opkgs) = @_;
  my ($pkg,$dependent,@dependents,$dc);
  
  return if ($#opkgs < 0);
  
  foreach $pkg (@opkgs) {
    @dependents = @{$$pkgs{$pkg}{DEPS}};
    $dc = $#dependents + 1;
    $$ctx{LOGGER}->fine("$pkg has ${dc} dependents.");
    if ($#dependents >= 0) {
      $self->sort_product_packages($ctx, $npkgs, $pkgs, @dependents);
    }
    
    if (Utils::vxif_list($pkg, @$npkgs) < 0) {
      $$ctx{LOGGER}->fine("Inserting $pkg into to be installed packages list.");
      push(@$npkgs, $pkg);
    }
  }
}

#
# Check for product licensing.
#
# Input: 1) the reference to the installation context;
#        2) indicate whether this is preinstall stage;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_product_licensing (\%\%$\@) {
	my $self = shift;
	my ($ctx,$preinstall) = @_;
	my ($host,$first);
	my @hosts = @{$ctx->get_target_systems()};
	my $trg_upi = $ctx->get_target_upi();
	my $rc = 1;

	$$ctx{LOGGER}->entering("CPICommon::check_product_licensing");
	return 1 if (!$trg_upi);

	$first = 1;
	foreach $host (@hosts) {
		$ctx->set_target_hostname($host);
		$$ctx{LOGGER}->info("Checking licensing on ${host}");
		$rc &= $self->check_product_licensing_on_target_host($ctx, $first);
		$first = 0;
	}
	
	return $rc;
}

#
# Stop the product processes.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#        3) the product version; this parameter is optional;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub stop_product_processes (\%\%;$) {
	my $self = shift;
	my ($ctx,$cfg,$version) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_installer = $ctx->get_host_os_upi_installer($trg_host);

	if ($upi_installer->can("stop_product_processes")) {
		if (!$upi_installer->stop_product_processes($ctx, $cfg, $version)) {
			return 0;
		}
	}

	return 1;
}

#
# Invoke product configuration.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_product_configuration (\%\%) {
	my $self = shift;
	my ($ctx, $cfg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	if ($upi_inst->can("install_product_configuration")) {
		($$ctx{RESPONSEFILE}) || ($$ctx{NOCONFIG} = (!$self->confirm_product_configuration($ctx)));

		if (!$$ctx{NOCONFIG}) {
			my $trg_mode = $ctx->get_host_inst_mode($trg_host);

			if (!$upi_inst->install_product_configuration($ctx, $cfg, $trg_mode)) {
				my $trg_upi = $ctx->get_target_upi();

				Utils::vxif_pl(Utils::_tr("No configuration changes are made to the systems until all configuration questions are completed and ${trg_upi} is installed successfully.", 16, 1126, "${trg_upi}"));
				return 0;
			}
		}
	}

	return 1;
}

# 
# Confirm that the user wants to complete configuration.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if users wants to perform configuration now; 0 otherwise.
#
sub confirm_product_configuration (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($answer,$quit_key_handler, $msg);
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();

	Utils::vxif_title($$ctx{INSTALLERTITLE});
	Utils::vxif_pl(Utils::_tr("It is optional to configure ${trg_upi} now. If you choose to configure ${trg_upi} later, you can either do so manually or run the $$ctx{PROGRAM} -configure command.\n", 16, 1127, "${trg_upi}", "${trg_upi}", "$$ctx{PROGRAM}"));
	$quit_key_handler = sub {
		$self->cleanup($ctx);
	};
        $msg = Utils::_tr("Are you ready to configure ${trg_upi} on ${trg_host}?", 16, 1128, "${trg_upi}", "${trg_host}");
	$answer = Utils::vxif_ayny($msg, "", "", $quit_key_handler);

	if ($answer eq $self->{MSG}{YESKEY}{U}) {
		Utils::vxif_title($$ctx{INSTALLERTITLE});
		Utils::vxif_pl(Utils::_tr("$$ctx{PROGRAM} will now ask sets of ${trg_upi} configuration-related questions.\n", 16, 1129, "$$ctx{PROGRAM}", "${trg_upi}"));
		Utils::vxif_pl(Utils::_tr("When a [b] is presented after a question, 'b' may be entered to go back to the first question of the configuration set.\n", 16, 1130));
		Utils::vxif_pl(Utils::_tr("When a [?] is presented after a question, '?' may be entered for help or additional information about the question.\n", 16, 1131));
		Utils::vxif_pl(Utils::_tr("Following each set of questions, the information you have entered will be presented for confirmation.  To repeat the set of questions and correct any previous errors, enter 'n' at the confirmation prompt.\n", 16, 1132));

		Utils::vxif_pl(Utils::_tr("No configuration changes are made to the systems until all configuration questions are completed and confirmed.\n\n", 16, 1133));
	}

	Utils::vxif_prtc($ctx);

	return ($answer eq $self->{MSG}{YESKEY}{U});
}

#
# Create product config file.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub create_product_config_file (\%\%) {
	my $self = shift;
	my ($ctx,$cfg) = @_;
	my ($rc) = 1;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_mode = $ctx->get_host_inst_mode($trg_host);
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $prodname = $upi_inst->get_product_title($ctx);
	
	if ($upi_inst->can("create_product_install_config")) {
		Utils::vxif_title($$ctx{INSTALLERTITLE});
		Utils::vxif_bpl(Utils::_tr("Configuring ${prodname}:\n", 16, 1059, "${prodname}"));
		if (!$upi_inst->create_product_install_config($ctx, $cfg, $trg_mode)) {
			Utils::vxif_bpl(Utils::_tr("\nFailed to configure ${prodname}.", 16, 1060, "${prodname}"));
			$rc = 0;
		} else {
			Utils::vxif_bpl(Utils::_tr("\n${prodname} configured successfully.", 16, 1061, "${prodname}"));
		}

		Utils::vxif_prtc($ctx);
	}

	return $rc;
}

#
# Start the product processes.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#        3) the product version; this parameter is optional;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub start_product_processes (\%\%;$) {
	my $self = shift;
	my ($ctx,$cfg,$version) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_installer = $ctx->get_host_os_upi_installer($trg_host);
	my $pname = $upi_installer->get_product_title($ctx);
	my $rc = 1;

	if ($upi_installer->can("start_product_processes")) {
		Utils::vxif_title($$ctx{INSTALLERTITLE});
		$rc = $upi_installer->start_product_processes($ctx, $cfg, $version);
		if (!$rc) {
			Utils::vxif_bpl(Utils::_tr("\n${pname} FAILED to start.", 16, 1134, "${pname}"));
		} else {
			Utils::vxif_bpl(Utils::_tr("\n${pname} was started successfully.", 16, 1135, "${pname}"));
		}
		Utils::vxif_prtc($ctx);

		return $rc;
	}

	return $rc;
}

#
# Start the product processes.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#        3) the product version; this parameter is optional;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_product_completion_messages (\%\%;$) {
	my $self = shift;
	my ($ctx,$cfg,$version) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $msg;

	if ($upi_inst->can("get_product_completion_message")) {
		# for the A and B stuff, the completion messages will be printed by the product itself
		if ($upi_inst->can("get_thunk_info")) {
			$upi_inst->get_product_completion_message($ctx, $cfg);
		} else {
			$msg = $upi_inst->get_product_completion_message($ctx, $cfg);

			if ($msg ne "") {
				Utils::vxif_title($$ctx{INSTALLERTITLE});
				Utils::vxif_pl($msg);
				Utils::vxif_pl();
				Utils::vxif_prtc($ctx);
			}
		}
	}

	return 1;
}

#
# Check for and run pre package/patch-install operations.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name
#
# Return: 1 if successful; 0 otherwise.
#
sub pre_package_patch_install (\%$$) {
	my $self = shift;
	my ($ctx,$pkg,$patch) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $os_name = $ctx->get_host_os_name($trg_host);
	my $os_version = $ctx->get_host_os_version($trg_host);
	my $logger = $ctx->get_logger();
	my $this_package = __PACKAGE__;

	$logger->entering("${this_package}::pre_package_patch_install");

	# make sure we have a package name or patch name
	if ((!$pkg)&&(!$patch)) {
		$logger->error(Utils::_tr("Unable to run pre-install operations. Must specify a package name or patch name.", 16, 1062));
		return 0;
	}

	my $iname = "package";
	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
	}

	my $func = "pre_${iname}_install";

	my $base_upi_inst = $upi_inst->package_base_installer($ctx, $pkg);
	if ($base_upi_inst) {
		if ($base_upi_inst->can($func)) {
			$rc = $base_upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
			if (!$rc) {
				$logger->error(Utils::_tr("Base pre ${iname} installation failed!", 16, 1136, "${iname}"));
				return 0;
			}
		}
	}

	if ($upi_inst->can($func)) {
		$rc = $upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
		if (!$rc) {
			$logger->error(Utils::_tr("Pre ${iname} installation failed!", 16, 1137, "${iname}"));
			return 0;
		}
	}

	return 1;
}

#
# Check for and run post package/patch-install operations.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name
#
# Return: 1 if successful; 0 otherwise.
#
sub post_package_patch_install (\%$) {
	my $self = shift;
	my ($ctx,$pkg,$patch) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $os_name = $ctx->get_host_os_name($trg_host);
	my $os_version = $ctx->get_host_os_version($trg_host);
	my $logger = $ctx->get_logger();
	my $this_package = __PACKAGE__;

	$logger->entering("${this_package}::post_package_patch_install");

	# make sure we have a package name or patch name
	if ((!$pkg)&&(!$patch)) {
		$logger->error(Utils::_tr("Unable to run post-install operations. Must specify a package name or patch name.", 16, 1064));
		return 0;
	}

	my $iname = "package";
	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
	}

	my $func = "post_${iname}_install";

	my $base_upi_inst = $upi_inst->package_base_installer($ctx, $pkg);
	if ($base_upi_inst) {
		if ($base_upi_inst->can($func)) {
			$rc = $base_upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
			if (!$rc) {
				$logger->error(Utils::_tr("Base post ${iname} installation failed!", 16, 1138, "${iname}"));
				return 0;
			}
		}
	}

	if ($upi_inst->can($func)) {
		$rc = $upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
		if (!$rc) {
			$logger->error(Utils::_tr("Post ${iname} installation failed!", 16, 1063, "${iname}"));
			return 0;
		}
	}

	return 1;
}

#
# Install all the product packages on the target host.
#
# Input: 1) the reference to the installation context;
#		 2) reference to @ of optional packages
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub display_product_packages_to_install (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($pkgs, @opkgs);
	my $upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $upi = $trg_upi_long;
    }
	my ($pdfrs) = $self->{MSG}{PDFRS};
	my ($os) = $self->{OSINSTALLER};
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	if ($upi_inst) {
		# get the product packages to install
		$pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch);
		if ($upi_inst->can("get_product_packages_ordering")) {
			@opkgs = $upi_inst->get_product_packages_ordering($ctx, $prodver, $osver, $osarch);
		} else {
			@opkgs = keys(%$pkgs);
		}

		my $trg_host = $ctx->get_target_hostname();
		my $title = $$ctx{INSTALLERTITLE};
		my $subtitle = Utils::_tr("$$ctx{PROGRAM} will install the following ${upi} ${pdfrs} on ${os} target system: ${trg_host}\n", 16, 1139, "$$ctx{PROGRAM}", "${upi}", "${pdfrs}", "${os}", "${trg_host}");
		my $page_msg = Utils::_tr("...continued packages list for ${os}:", 16, 1066, "${os}");

		my $msgs = {
			title => $title,
			subtitle => $subtitle,
			page_msg => $page_msg,
		};

		my $pkg_display_names = {};
		foreach my  $pkg (@opkgs) {
			my $package_info = $pkgs->{$pkg};
			my $package_name = $package_info->search_name($pkg);
			my $pkg_display_name = sprintf("%-16s $package_name", $self->get_package_name($ctx, $pkg, $pkgs));
			$pkg_display_names->{$pkg} = $pkg_display_name;
		}

		$events->display_os_packages($msgs, \@opkgs, $pkg_display_names);
	}

	return 1;
}

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%;\@) {
	my $self = shift;
	my ($ctx, $missing_pkgs) = @_;
	my $nipkgs = [];
	my $dpkgs = [];
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_mode = $ctx->get_host_inst_mode($trg_host);
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $logger = $ctx->get_logger();
	my $rc = 1;

	# supress dependency checking while upgrading
	my $task_type = $ctx->get_task_type();
	return $rc if ($task_type ne "upgrade");

	if ($upi_inst->can("get_external_dependent_packages")) {
		$dpkgs = $upi_inst->get_external_dependent_packages($ctx, $trg_mode);
	}

	foreach my $pkg (keys(%{$dpkgs})) {
		$logger->finer(Utils::_tr("checking dependent package $pkg", 16, 1140, "$pkg"));
		my $iver = $self->get_instpkgvers($ctx, $pkg, "", $dpkgs);
		if ($iver) {
			$logger->finer(Utils::_tr("$pkg $iver installed", 16, 1141, "$pkg", "$iver"));
			my $dver = $$dpkgs{$pkg}{VERSION};
			if ($iver ne $dver) {
				if ($upi_inst->can("is_external_dependent_version_supported")) {
					my $dvsrc = $upi_inst->is_external_dependent_version_supported($ctx, $pkg, $iver, $dver);
					if (!$dvsrc) {
						push(@$nipkgs, $pkg);
					}
				}
			}
		} else {
			$logger->finer(Utils::_tr("$pkg not installed", 16, 1142, "$pkg"));
			push(@$nipkgs, $pkg);
		}
	}

	my $num_nipkgs = scalar(@$nipkgs);

	if ($num_nipkgs > 0) {
		if ($missing_pkgs) {
			push(@$missing_pkgs, @$nipkgs);
		}
		$rc = 0;
	} else {
		$rc = 1;
	}

	return $rc;
}

#
# Check to make target host have the required spaces.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_required_spaces (\%) {
   my $self = shift;
   my ($ctx) = @_;
   my $ifmgr = $ctx->get_interface_manager();
   my $events = $ifmgr->get_events();
   my ($answer);
   my $trg_host = $ctx->get_target_hostname();
   my $trg_upi = $ctx->get_target_upi();
   my $rtn = 1;

   $$ctx{LOGGER}->entering("CPICommon::check_for_required_spaces");
   $events->test_required_space();

   my $space_mgr = new SpaceMgr();
   my $warnings = [];
   my $error = $space_mgr->check_for_required_spaces($ctx, $warnings);

	if ($error) {
           my $err_rc = $error->get_rc();
           my $err_msg = $error->get_msg();
           if ($err_rc == SpaceMgr::ERR_PRODUCT_SPACE_CALCULATION) {
              $rtn = $events->error_in_space_calculation($err_msg);
           } elsif ($err_rc == SpaceMgr::ERR_REQUIRED_SPACE_UNKNOWN) {
              $rtn = $events->required_space_unknown($err_msg);
           } elsif ($err_rc == SpaceMgr::ERR_INSUFFICIENT_SPACE) {
              $rtn = $events->required_space_test_fail();
              # this is added for BE so the user will be prompted to go ahead or to stop the install.
              if ($$ctx{RESPONSEFILE} && ($$ctx{DISKOVERRIDE} == 1)) {
                 # this is from the response file and will over ride the disk check results if they fail.
                 $rtn = 1;
              } 
              if ($$ctx{RESPONSEFILE} && ($$ctx{DISKOVERRIDE} == 0)){
                 $rtn = 0;
              }
              Utils::vxif_pl(Utils::_tr("There is not enough disk space on the required partitions", 16, 1143));
              $rtn = 0;
           }
	}
	for my $winfo (@$warnings) {
           my ($vol,$kr) = split(/,/, $winfo);
           $events->space_remaining_warning($trg_upi, $trg_host, $vol, $kr);
           my $ayn = $events->input_should_continue();
           if ($ayn == 0) {
              $self->cleanup($ctx);
              $rtn = 0;
           } elsif ($ayn ==1){
              $rtn = 1;
           }
	}
        if (($rtn == 1) && (!$answer) && (!$ayn)) {
           $events->required_space_test_pass();
        }

        $$ctx{LOGGER}->exiting("CPICommon::check_for_required_spaces");
        return $rtn;
}

#
# Check installation requirements on a host.
#
# Input: 1) the reference to the installation context;
#        2) the name of the host
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_installation_requirements_on_host {
	my $self = shift;
	my ($ctx, $host) = @_;
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $trg_upi = $trg_upi_long;
    }
	
	my $task_type = $ctx->get_task_type();
	my $trg_host = $ctx->get_target_hostname();
	my $rc = 1;

	$events->check_hostinst_reqs($trg_upi, $host);

        #jpk
	if ($task_type eq "install" || $task_type eq "precheck"){
           # check for external dependencies
           # for some reason we are testing teh external dependencies in events.  
           # Why since it only prints a message is unclear to me.  So I am NOT going to do the events check.
           # Let the O/S specific check happen.  - jpk Feb 8, 2006
           #$events->test_external_dependencies();
           $rc &= $self->check_for_external_dependencies($ctx);
           #if ($rc) {
           #$events->external_dependencies_test_pass();
           #} else {
           if (!$rc) {
              my $pkg = undef;
              my $missing = undef;
              $events->external_dependencies_test_fail($pkg, $missing);
           }
           # check for require spaces
           $rc &&= $self->check_for_required_spaces($ctx);
	}

	# stop product processes
	# for some reason the function still get called if $rc==0; that's why the if
	#
	# for configure, we'll do it there
	if (($task_type ne "configure") && ($rc)) {
           # this should be changed, so that it happens in, say,
           # an "unconfigure" step, or pre-uninstall (TBD). (DE 4-21-05)
           $rc &= $self->stop_product_processes($ctx, $cfg) if ($rc);
	}

	return $rc;
}

#
# Install a package on target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        3) the reference to the packages hash;
#        4) the package to be installed;
#
# Return: 2 if pkg already installed; 1 if the package is successfully added; 0 otherwise.
#
sub install_package (\%\%\%$) {
	my ($self,$ctx,$cfg,$pkgs,$pkg) = @_;
	my $rc = 1; # give benefit of doubt
	my $package_already_installed = 0;
	
	# copy and uncompress the package files to the target host if necessary
	($rc) && ($rc = $self->check_product_package_files($ctx, $pkg, "", $pkgs));
	
	# copy adminfile to target host
	if (($rc)&&($self->can("copy_admin_file_to_target_host"))) {
		$rc = $self->copy_admin_file_to_target_host($ctx);
	}
	
	# create and copy the response file to the install host
	if (($rc)&&($self->can("create_and_copy_response_file"))) {
		$rc = $self->create_and_copy_response_file($ctx, $cfg, $pkgs, $pkg, "");
	}
	
	# pre package installation operations
	($rc) && ($rc = $self->pre_package_patch_install($ctx, $pkg, ""));
	
	# add the package to host
	($rc) && ($rc = $self->install_native_package($ctx, $pkgs, $pkg));
	if ($rc == 2) {
		$package_already_installed = 1;
		$rc = 1;
	}

	# should we do the next steps if already installed ???
	# maybe we should pass a flag indicating if it was already installed.

	# post package installation operations
	($rc) && ($rc = $self->check_post_package_install($ctx, $cfg, $pkg));

	($rc) && ($rc = $self->post_package_patch_install($ctx, $pkg, ""));
	
	# cleanup the temporary directory
	$self->cleanup_temporary_directory($ctx, $pkg, "");

	if ($rc && $package_already_installed) {
		$rc = 2;
	}

	return $rc;
}

#
# Post package install callbacks.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#        3) the name of the package;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_post_package_install (\%\%$) {
	my $self = shift;
	my ($ctx, $cfg, $pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $installer = $ctx->get_host_os_upi_installer($trg_host);

	my $base_installer = $installer->package_base_installer($ctx, $pkg);
	if ($base_installer) {
		if ($base_installer->can("post_package_install")) {
			my $rc = $base_installer->post_package_install($ctx, $cfg, $pkg);
			if (!$rc) {
				return 0;
			}
		}
	}

	if ($installer->can("post_package_install")) {
		my $rc = $installer->post_package_install($ctx, $cfg, $pkg);
		return $rc;
	}

	return 1;
}

#
# Install a patch on target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        3) the reference to the patches hash;
#        4) the patch to be installed;
#
# Return: 2 if patch already installed; 1 if the patch is successfully added; 0 otherwise.
#
sub install_patch (\%\%\%$) {
  my ($self,$ctx,$cfg,$pchs,$pch) = @_;
  my ($rc) = 1;
  my ($rci) =1;
  
  # copy and uncompress the patch files to the target host if necessary
  ($rc) && ($rc = $self->check_product_package_files($ctx, "", $pch, $pchs));
  
  # copy adminfile to target host
  if (($rc)&&($self->can("copy_admin_file_to_target_host"))) {
    $rc = $self->copy_admin_file_to_target_host($ctx);
  }
  
  # create and copy the response file to the install host
  if (($rc)&&($self->can("create_and_copy_response_file"))) {
    $rc = $self->create_and_copy_response_file($ctx, $cfg, $pchs, "", $pch);
  }
  
  # pre patch installation operations
  ($rc) && ($rc = $self->pre_package_patch_install($ctx, "", $pch));
  
  # add the patch to host
  ($rc) && ($rci = $self->install_native_patch($ctx, $pchs, $pch));
  
  # post patch installation operations
  ($rci) && ($rc = $self->post_package_patch_install($ctx, "", $pch));
  
  # cleanup the temporary directory
  $self->cleanup_temporary_directory($ctx, "", $pch);

# need to do this nasty if here because I want to return a 2 incase a pch
# is already installed.
  if ($rc) {
	return $rci;
  } else {
	return $rc;
  }
}


sub get_effective_package_order (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $trg_host = $ctx->get_target_hostname();
	$osver = $ctx->get_host_os_version($trg_host);
	$osarch = $ctx->get_host_os_architecture($trg_host);
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $prodver = $upi_inst->get_product_version($ctx);
	my $eff_order = [];

	# sort the install packages to make sure the dependencies are ordered correctly
	if ($upi_inst->can("get_product_packages_ordering")) {
		my @opkgs = $upi_inst->get_product_packages_ordering($ctx, $prodver, $osver, $osarch);
		@$eff_order = @opkgs;
	} else {
		my @opkgs = keys(%{$pkgs});
		@opkgs = split(/ /, Utils::vxif_desp(join(" ", @opkgs)));
		$self->sort_product_packages($ctx, $eff_order, $pkgs, @opkgs);
		my $task_type = $ctx->get_task_type();
		if ($task_type eq "uninstall") {
			@$eff_order = reverse(@$eff_order);
		}
	}

	return $eff_order;
}

sub compute_package_status (\%$\%\@) {
	my $self = shift;
	my ($ctx, $pkg, $pkgs) = @_;
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $trg_tmppath = $ctx->get_host_tmppath($trg_host);
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $pkg_loc;
	my $rc = 0;
	my $logger = $ctx->get_logger();

        $logger->entering("CPICommon::compute_package_status");

	if ($upi_inst->can("rel_inst_pkg_loc")) {
		$pkg_loc = $upi_inst->rel_inst_pkg_loc($ctx, $pkg);
	} else {
		$pkg_loc = $pkg;
	}
	# do this
	$self->{$trg_upi}{$pkg}{TLOC} = "${trg_tmppath}/${trg_upi}/${pkg_loc}"; # target package location
	# before this
	$rc = $self->check_for_existing_package($ctx,$pkg,$pkgs);

        $logger->exiting("CPICommon::compute_package_status");
	return $rc;
}

#
# Install all the product packages on the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %CFG;
#		 3) reference to @ of optional pkgs
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub install_product_packages (\%\%) {
	my $self = shift;
	my ($ctx,$cfg) = @_;
	my ($osver,$osarch,$pkgcount,$totalpkgs,$msgl,$msgr);
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);  # this is where you should create the upgrade client
	my $title = $upi_inst->get_product_title($ctx);
	my $prodver = $upi_inst->get_product_version($ctx);

	# get the product packages to install
	my $pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch);
	my $npkgs = $self->get_effective_package_order($ctx);
	# the ones that need to be installed
	my $ipkgs = [];
        my ($rc, $pkg, $rtn, $ret, $iver, @upgrade_pkgs);

        $$ctx{LOGGER}->entering("CPICommon::install_product_packages");

        Utils::vxif_bpl(Utils::_tr("Checking ${title} on ${trg_host}:\n", 16, 1144, "${title}", "${trg_host}"));

	# display the status of all packages before beginning installation
	# like in IFRTA
	foreach $pkg (@$npkgs) {
		$msgl = Utils::_tr("Checking ${pkg} package", 16, 1084, "${pkg}");
		Utils::vxif_pbl($msgl);

		# clear package status
		$rc = 0;

                # do computation ... no "UI"
		if (!$rc) {
                   # status not already known (i.e. from previous partial upgrade).
                   # compute based on current state.
                   $rc = $self->compute_package_status($ctx, $pkg, $pkgs);   
		}

		if (2 == $rc) {
			# installed
			$iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
			$msgr = Utils::_tr("$iver installed", 16, 1085, "$iver");
			Utils::vxif_pbr($msgl, $msgr);		
		} elsif (1 == $rc) {
			# not installed
			push(@$ipkgs, $pkg);
			$msgr = Utils::_tr("not installed", 16, 1086);
			Utils::vxif_pbr($msgl, $msgr);		
		} elsif (3 == $rc) {
			# installed, but will be upgraded
			push(@$ipkgs, $pkg);
                        push(@upgrade_pkgs, $pkg);
                        $iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
                        $msgr = Utils::_tr("will upgrade from ${iver}", 16, 1145, "${iver}");
                        Utils::vxif_pbr($msgl, $msgr);		
                        $iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
      } elsif (4 == $rc) {
                        # installed version is newer; Will not downgrade
                        $iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
                        $msgr = Utils::_tr("${iver} installed", 16, 1085, "${iver}");
                        Utils::vxif_pbr($msgl, $msgr);
                        Utils::vxif_pl("");
                        $msgr = Utils::vxif_tr("Downgrading ${pkg} is not supported on ${trg_host}", 16, 1157, "${pkg}", "${trg_host}");
                        Utils::vxif_bpl($msgr);
		} else {
			#error#let's say not an error, but don't know status
			#return 0;
			$msgr = Utils::_tr("status unknown", 16, 1146);
			Utils::vxif_pbr($msgl, $msgr);		
		}
	}

	#list and get the patches to be installed here
	my @ipchs = $self->check_for_patches($ctx);

	Utils::vxif_prtc($ctx);

	# install the packages now
	$pkgcount= 1;
	my $num_ipkgs = scalar(@$ipkgs);
	$totalpkgs = $num_ipkgs + $#ipchs + 1; # $# gives the last index of the array.
	Utils::vxif_pl("");

	Utils::vxif_title($$ctx{INSTALLERTITLE});
	#Utils::vxif_bpl(Utils::_tr("Installing ${title} ${prodver} on ${trg_host}:\n"));
        Utils::vxif_bpl(Utils::_tr("Installing ${title} on ${trg_host}:\n", 16, 1147, "${title}", "${trg_host}"));

	if ($upgrade_client) {
		$rc = $upgrade_client->check_pre_product_upgrade();  # This is in HAPUpgradeClient
	}

        foreach my $pkg (@$ipkgs) {
           my $package_info = $pkgs->{$pkg};
           my $pkgver = $package_info->search_version($pkg);
           if (!$pkgver) {
              $pkgver = $self->get_mediapkgvers($ctx, $pkg, $pkgs);
           }
           #$$ctx{LOGGER}->info(Utils::_tr("Installing package ${pkg} ${pkgver} on ${trg_host}.")); # Changed ET 517162 - 10 Mar 06 jpk

           $ret = 0;

           if (grep(/$pkg/,@upgrade_pkgs)) {
              if($upi_inst->can("pre_upgrade")){
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages starting pre-upgrade work for package ${pkg}");
                 $rtn = $upi_inst->pre_upgrade($ctx, $pkg, $version);
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages finished pre-upgrade work for package ${pkg}");
                 # jpk May 10 - now uninstall the package
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages is now removing ${pkg} for upgrading");
                 $rtn = $self->uninstall_native_package($ctx, $pkgs, $pkg);
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages finished upgrade removal of ${pkg}");
              }
           }
           $$ctx{LOGGER}->info("Installing package ${pkg} ${pkgver} on ${trg_host}.");
           $msgl = Utils::_tr("Installing ${pkg} ${pkgver} on ${trg_host}", 16, 1073, "${pkg}", "${pkgver}", "${trg_host}");
           Utils::vxif_pbl($msgl);
           $ret = $self->install_package($ctx, $cfg, $pkgs, $pkg);

              
           if ($ret) {
              $msgr = Utils::_tr("done ${pkgcount} of ${totalpkgs} steps", 16, 1074, "${pkgcount}", "${totalpkgs}");
              Utils::vxif_pbr($msgl, $msgr);

              # set the reboot flag if the package requires a reboot
              my $reboot = $package_info->search_requires_reboot($pkg);
              if ($reboot) {
                 $$ctx{REBOOT} = 1;
              }
           } else {
              $msgr = Utils::_tr("failed to install", 16, 1075);
              Utils::vxif_pbr($msgl, $msgr);
              return 0;
           }

           if (grep(/$pkg/,@upgrade_pkgs)) {
              if($upi_inst->can("post_upgrade")){
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages starting post-upgrade work for package ${pkg}");
                 $rtn = $upi_inst->post_upgrade($ctx, $pkg, $version);
                 $$ctx{LOGGER}->info("CPICommon::install_product_packages finished post-upgrade work for package ${pkg}");
              }
           }

           ++$pkgcount;
        }

	#install the patches here
	#foreach my $pkg (@ipchs) {
	#	# may need to alter this if ever get actual patches
	#	my $package_info = $pkgs->{$pkg};
	#	$$ctx{LOGGER}->info(Utils::_tr("Installing patch ${pkg} on ${trg_host}."));
	#	$msgl = Utils::_tr("Installing ${pkg} on ${trg_host}");
	#	Utils::vxif_pbl($msgl);
#
	#	my $ret = $self->install_patch($ctx, $cfg, $pkgs, $pkg);
	#	if ($ret) {
	#		$msgr = Utils::_tr("done ${pkgcount} of ${totalpkgs} steps");
	#		Utils::vxif_pbr($msgl, $msgr);
#
	#		# set the reboot flag if the patch requires a reboot
	#		my $reboot = $package_info->search_requires_reboot($pkg);
	#		if ($reboot) {
	#			$$ctx{REBOOT} = 1;
	#		}
	#	} else {
	#		$msgr = Utils::_tr("failed to install");
	#		Utils::vxif_pbr($msgl, $msgr);
	#		return 0;
	#	}
	#	++$pkgcount;
	#}

	if ($upgrade_client) {
		$rc = $upgrade_client->check_post_product_upgrade();
	}

	$$ctx{LOGGER}->info("All packages for ${trg_upi} successfully installed on ${trg_host}.");

        $$ctx{LOGGER}->exiting("CPICommon::install_product_packages");

	return 1;
}

##############################################################
# Product uninstall operations.                              #
##############################################################

#
# Check whether a given package is installed on target host.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#
# Return: 1 if the package is installed on target host; 0 otherwise.
#
sub check_package_installed_on_target_host (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my ($iver,$msgl,$msgr);
	my ($pkgs,$osver,$osarch,$prodver);
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	# get the product packages to install
	$prodver = $upi_inst->get_product_version($ctx);
	$osver = $ctx->get_host_os_version($trg_host);
	$osarch = $ctx->get_host_os_architecture($trg_host);
	$pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch);

	$msgl = Utils::_tr("Checking ${pkg} package", 16, 1084, "${pkg}");
	Utils::vxif_pbl($msgl);
	$iver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);

	if ($iver ne "") {
		$msgr = Utils::_tr("${iver} installed", 16, 1085, "${iver}");
		Utils::vxif_pbr($msgl, $msgr);

		return 1;
	} else {
		$msgr = Utils::_tr("not installed", 16, 1086);
		Utils::vxif_pbr($msgl, $msgr);

		return 0;
	}
}

#
# Check prerequisite packages on target host for the given pacakge.
#
# Input: 1) the reference to the installation context;
#        2) the prerequisite package or subpackage name;
#        3) a reference to the uninstall order array
#
# Return: 1 if there is a problem; 0 otherwise.
#
sub have_uninstall_prereq_issue {
	my $self = shift;
	my ($ctx, $dep, $opkgs) = @_;
	my $verdict = 0;
	my $handled = 0;

#	if (Utils::vxif_list($dep, @{$opkgs}) < 0) {
#		# we are not uninstalling the dependent package
#		$verdict = 1;
#	}

	for my $opkg (@$opkgs) {
		if ($dep eq $opkg) {
			# we are handling it
			$handled = 1;
			last;
		}
		# not out of the woods yet
		my $method_name = "get_subpackage_list";
		my $method = $self->can($method_name);
		if ($method) {
			# filesets on hpux, or depots on aix, for example
			my $subpkgs_str = $self->$method($ctx, $opkg);
			my $subpkg_aref = [ split(/\s+/, $subpkgs_str) ];
			for my $subpkg (@$subpkg_aref) {
				if ($dep eq $subpkg) {
					$handled = 1;
					last;
				}
			}
			if ($handled) {
				last;
			}
		}
	}

	if (!$handled) {
		# we are not handling this dependency via uninstall order
		$verdict = 1;
	}

	return $verdict;
}

#
# Check prerequisite packages on target host for the given pacakge.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the reference to the original package list;
#        4) the reference to the installed package list;
#        5) the reference to the not installed package list;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_for_prerequisite_packages (\%$\@\@\@) {
	my $self = shift;
	my ($ctx,$pkg,$opkgs,$ipkgs,$nipkgs) = @_;
	my ($dep,$deps,@da,$pdl,$quit_key_handler,$ans,$msgl,$msgr);
	my $task_type = $ctx->get_task_type();
	my $trg_host = $ctx->get_target_hostname();

	# skip dependency checking incase of upgrade.
	return 1 if ($task_type eq "upgrade");

	$msgl = Utils::_tr("Checking ${pkg} dependencies", 16, 1087, "${pkg}");
	Utils::vxif_pbl($msgl);
	$deps = $self->get_pkgdep($ctx, $pkg);

	if (!$deps) { 
		$msgr = Utils::_tr("no dependencies", 16, 1088);
		Utils::vxif_pbr($msgl, $msgr);
		return 1;
	}

	@da=split(/\s+/,$deps);
	$pdl=join(" ",@da);
	Utils::vxif_pbr($msgl, $pdl);

	$quit_handler = sub {
		$self->cleanup($ctx);
	};

	foreach $dep (@da) {
		my $have_prereq_issue = $self->have_uninstall_prereq_issue($ctx, $dep, $opkgs);
		#if (Utils::vxif_list($dep, @{$opkgs}) < 0) {
		if ($have_prereq_issue) {
			# TRY THIS (417514).
			$$ctx{DEPCHK}{$dep} = $self->{MSG}{NOKEY}{U};
			#
			$msgl = Utils::_tr("Do you want to uninstall package ${dep} from ${trg_host} which is dependent on package ${pkg}", 16, 1089, "${dep}", "${trg_host}", "${pkg}");
			if (defined($$ctx{DEPCHK}{${dep}})) {
				$ans = $$ctx{DEPCHK}{${dep}};
			} else {
				$$ctx{DEPCHK}{${dep}} = $ans = Utils::vxif_ayny($msgl, "", "", $quit_handler);
			}

			if ($ans eq $self->{MSG}{NOKEY}{U}) {
				Utils::vxif_bpl(Utils::_tr("Leaving ${pkg} installed on ${trg_host} due to its package dependencies: ${dep}", 16, 1090, "${pkg}", "${trg_host}", "${dep}"));
				return 0;
			} else {
				if ($self->check_package_installed_on_target_host($ctx, $dep)) {
					return 0 if (!$self->check_for_prerequisite_packages($ctx, $dep, $opkgs, $ipkgs, $nipkgs));
					push(@$ipkgs, $dep);
				} else {
					push(@$nipkgs, $dep);
				}
			}
		}
	}
	
	return 1;
}

#
# Check for additional package status.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_packages_status (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my ($pkg,@nipkgs,@ipkgs,$pls,$ans);
	my ($prodver,$osver,$osarch,$quit_handler,$msg);
	my ($optpkgs);
	my $trg_upi = $ctx->get_target_upi();
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	# get the product packages to install
	$osver = $ctx->get_host_os_version($trg_host);
	$osarch = $ctx->get_host_os_architecture($trg_host);
	$prodver = $upi_inst->get_product_version($ctx);
	my $pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch);
	my $npkgs = $self->get_effective_package_order($ctx);

	# also need to check for the optional pkgs.
	if ($upi_inst->can("get_optional_packages_to_uninstall")) {
		$optpkgs = $upi_inst->get_optional_packages_to_uninstall($ctx);
		unshift(@$npkgs, @$optpkgs);
	}

	foreach $pkg (@$npkgs) {
		#check if $pkg is null.
		Utils::vxif_desp($pkg);
		next if (!$pkg);

		# check to see if the given package installed on target host
		my $rc = $self->check_package_installed_on_target_host($ctx, $pkg);
		if ($rc) {
			# package is installed
			$rc = $self->check_for_prerequisite_packages($ctx, $pkg, $npkgs, \@ipkgs, \@nipkgs);
			if ($rc) {
				# no dependent packages
				push(@ipkgs, $pkg);
			} else {
				push(@nipkgs, $pkg);
			}
		} else {
                        # if we are uninstalling then we need to call our product installer
                        # to handle any type of specific handling for uninstalling if product
                        # isn't installed
                        my $task_type = $ctx->get_task_type();
                        if (($task_type eq "uninstall") && ($upi_inst->can("pre_check_product_uninstall")))
                        {
                                # If pre-check product uninstall returns error we need to cleanup and exit
                                $error = $upi_inst->pre_check_product_uninstall($ctx);
                                if ($error) {
                                        $self->cleanup($ctx);
                                }
                        }

			# package is not installed
			push(@nipkgs, $pkg);
		}
	}

#removing this to be consistent with IFRT A  
#  if ($#nipkgs >= 0) {
#    $pls = join(',', @nipkgs);
#    Utils::vxif_bpl(Utils::_tr("$$ctx{PROGRAM} will not attempt to remove the following packages since they do not exist on ${trg_host}: ${pls}"));
#  
#    $quit_handler = sub {
#        $self->cleanup($ctx);
#    };
#    
#    $pls = join(',', @ipkgs);
#    $msg = Utils::_tr("Do you want to uninstall only the following packages from ${trg_host}: ${pls}");
#    $ans = Utils::vxif_ayny($msg, "", "", $quit_handler);
#    return 0 if ($ans eq $self->{MSG}{NOKEY}{U});
#  }

	$self->{$trg_host}{$trg_upi}{UPKGS} = \@ipkgs;

	return 1;
}

#
# Check system uninstallation requirements.
#
# Input: 1) the reference to the installation context;
#        2) the name of the host
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_uninstall_requirements_on_host {
	my $self = shift;
	my ($ctx, $host) = @_;
	my $ifmgr = $ctx->get_interface_manager();
	my $events = $ifmgr->get_events();
	my $trg_upi = $ctx->get_target_upi();
	#if user specify the UPI_LONG in beinst.conf we will use it instead of UPI for display
    #for Backup Exec for Linux Server, the UPI is BE and the UPI_LONG is "Backup Exec" which is more meaningful
    my $trg_upi_long = $ctx->get_target_upi_long();
    if ($trg_upi_long){
      $trg_upi = $trg_upi_long;
    }

	$events->check_hostinst_reqs($trg_upi, $host);

	# check for addition package dependencies
	my $rc = $self->check_for_packages_status($ctx);

	return $rc;
}

#
# Pre package uninstall callbacks.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg;
#        3) the name of the package;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_pre_package_uninstall (\%\%$) {
	my $self = shift;
	my ($ctx, $cfg, $pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $installer = $ctx->get_host_os_upi_installer($trg_host);

	my $base_installer = $installer->package_base_installer($ctx, $pkg);
	if ($base_installer) {
		if ($base_installer->can("pre_package_uninstall")) {
			my $rc = $base_installer->pre_package_uninstall($ctx, $cfg, $pkg);
			if (!$rc) {
				return 0;
			}
		}
	}

	if ($installer->can("pre_package_uninstall")) {
		my $rc = $installer->pre_package_uninstall($ctx, $cfg, $pkg);
		return $rc;
	}

	return 1;
}

#
# Check for and run pre package/patch-uninstall operations.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name
#
# Return: 1 if successful; 0 otherwise.
#
sub pre_package_patch_uninstall (\%$$) {
	my $self = shift;
	my ($ctx,$pkg,$patch) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $os_name = $ctx->get_host_os_name($trg_host);
	my $os_version = $ctx->get_host_os_version($trg_host);
	my $logger = $ctx->get_logger();
	my $this_package = __PACKAGE__;

	$logger->entering("${this_package}::pre_package_patch_uninstall");

	# make sure we have a package name or patch name
	if ((!$pkg)&&(!$patch)) {
		$logger->error(Utils::_tr("Unable to run pre-uninstall operations. Must specify a package name or patch name.", 16, 1098));
		return 0;
	}

	my $iname = "package";
	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
	}

	my $func = "pre_${iname}_uninstall";

	my $base_upi_inst = $upi_inst->package_base_installer($ctx, $pkg);
	if ($base_upi_inst) {
		if ($base_upi_inst->can($func)) {
			$rc = $base_upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
			if (!$rc) {
				$logger->error(Utils::_tr("Base pre ${iname} uninstall failed!", 16, 1148, "${iname}"));
				return 0;
			}
		}
	}

	if ($upi_inst->can($func)) {
		$rc = $upi_inst->$func($ctx, $trg_host, $os_name, $os_version);
		if (!$rc) {
			$logger->error(Utils::_tr("Pre ${iname} uninstall failed!", 16, 1099, "${iname}"));
			return 0;
		}
	}

	return 1;
}

#
# Check for and run post package/patch-uninstall operations.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name
#
# Return: 1 if successful; 0 otherwise.
#
sub post_package_patch_uninstall (\%$$) {
	my $self = shift;
	my ($ctx,$pkg,$patch) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $os_name = $ctx->get_host_os_name($trg_host);
	my $os_version = $ctx->get_host_os_version($trg_host);
	my $logger = $ctx->get_logger();
	my $this_package = __PACKAGE__;

	$logger->entering("${this_package}::post_package_patch_uninstall");

	# make sure we have a package name or patch name
	if ((!$pkg)&&(!$patch)) {
		$logger->error(Utils::_tr("Unable to run post-uninstall operations. Must specify a package name or patch name.", 16, 1100));
		return 0;
	}

	my $iname = "package";
	if ($patch) {
		$pkg = $patch;
		$iname = "patch";
	}

	my $func = "post_${iname}_uninstall";

	my $base_upi_inst = $upi_inst->package_base_installer($ctx, $pkg);
	if ($base_upi_inst) {
		if ($base_upi_inst->can($func)) {
			$rc = $base_upi_inst->$func($ctx, $pkg, $trg_host, $os_name, $os_version);
			if (!$rc) {
				$logger->error(Utils::_tr("Base post ${iname} uninstall failed!", 16, 1149, "${iname}"));
				return 0;
			}
		}
	}

	if ($upi_inst->can($func)) {
		$rc = $upi_inst->$func($ctx, $trg_host, $os_name, $os_version);
		if (!$rc) {
			$logger->error(Utils::_tr("Post ${iname} uninstall failed!", 16, 1101, "${iname}"));
			return 0;
		}
	}

	return 1;
}

#
# Uninstall a package on target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        3) the reference to the packages hash;
#        4) the package to be installed;
#
# Return: 1 if the package is successfully removed; 0 otherwise.
#
sub uninstall_package (\%\%\%$) {
	my $self = shift;
	my ($ctx,$cfg,$pkgs,$pkg) = @_;
	my ($rc) = 1;

	# copy adminfile to target host
	if ($self->can("copy_admin_file_to_target_host")) {
		($rc) && ($rc = $self->copy_admin_file_to_target_host($ctx));
	}

	# create and copy the response file to the install host
	if (($rc)&&($self->can("create_and_copy_response_file"))) {
		$rc = $self->create_and_copy_response_file($ctx, $cfg, $pkgs, $pkg, "");
	}

	# pre package uninstall operations
	($rc) && ($rc = $self->check_pre_package_uninstall($ctx, $cfg, $pkg));

	($rc) && ($rc = $self->pre_package_patch_uninstall($ctx, $pkg, ""));

	# remove the package from the host
	($rc) && ($rc = $self->uninstall_native_package($ctx, $pkgs, $pkg));

	# post package uninstall operations
	($rc) && ($rc = $self->post_package_patch_uninstall($ctx, $pkg, ""));

	return $rc;
}

#
# Unnstall all the product packages on the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %CFG;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub uninstall_product_packages (\%\%) {
	my $self = shift;
	my ($ctx,$cfg) = @_;
	my ($pkg,$pkgs,$osver,$osarch,$prodver,$pkgver,$pkgcount,$totalpkgs,$msgl,$msgr);
	my $trg_host = $ctx->get_target_hostname();
	my $trg_upi = $ctx->get_target_upi();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my (@pls) = @{$self->{$trg_host}{$trg_upi}{UPKGS}};
	my $version = $upi_inst->get_product_version($ctx);
	my $task_type = $ctx->get_task_type();
	
	# get the product packages to install
	$prodver = $upi_inst->get_product_version($ctx);
	$osver = $ctx->get_host_os_version($trg_host);
	$osarch = $ctx->get_host_os_architecture($trg_host);
	$pkgs = $upi_inst->get_product_packages($ctx, $prodver, $osver, $osarch);

	# uninstall the packages
	$pkgcount= 1;
	$totalpkgs = $#pls + 1;
	Utils::vxif_pl("");
	foreach $pkg (@pls) {
		$pkgver = $self->get_instpkgvers($ctx, $pkg, "", $pkgs);
		$$ctx{LOGGER}->info("Uninstalling package ${pkg} ${pkgver} on ${trg_host}.");
		$msgl = Utils::_tr("Uninstalling ${pkg} ${pkgver} on ${trg_host}", 16, 1103, "${pkg}", "${pkgver}", "${trg_host}");
		Utils::vxif_pbl($msgl) if ($task_type ne "upgrade");
		if ($self->uninstall_package($ctx, $cfg, $pkgs, $pkg)) {
			$msgr = Utils::_tr("done ${pkgcount} of ${totalpkgs} steps", 16, 1074, "${pkgcount}", "${totalpkgs}");
			Utils::vxif_pbr($msgl, $msgr) if ($task_type ne "upgrade");
		} else {
			$msgr = Utils::_tr("failed to uninstall", 16, 1150);
			Utils::vxif_pbr($msgl, $msgr);
			#my $logmsg = Utils::_tr("Package ${pkg} not uninstalled on ${trg_host}.");
			$$ctx{LOGGER}->warning("Package ${pkg} not uninstalled on ${trg_host}.");
			# keep going
			#return 0;
		}
		++$pkgcount;
	}
	$$ctx{LOGGER}->info("All packages for ${trg_upi} successfully uninstalled on ${trg_host}.");
	
	return 1;
}

#
# Configure the product on target host
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub configure_product_on_target_host (\%\%) {
	my $self = shift;
	my ($ctx,$cfg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $trg_upi = $ctx->get_target_upi();
	my $task_type = $ctx->get_task_type();
	my $rc = 1;

	if (!$$ctx{NOCONFIG}) {
		# product configuration questions
		if (!$self->check_product_configuration($ctx, $cfg)) {
			return 0;
		}

		if ($$ctx{NOCONFIG}) {
			Utils::vxif_title($$ctx{INSTALLERTITLE});
			Utils::vxif_pl(Utils::_tr("\nYou have elected not to configure ${trg_upi} on ${trg_host} at this time. ${trg_upi} may not function properly without configuration. If you decide to configure ${trg_upi} later, you can either do so manually or run the $$ctx{PROGRAM} -configure command.\n", 16, 1152, "${trg_upi}", "${trg_host}", "${trg_upi}", "${trg_upi}", "$$ctx{PROGRAM}"));
			Utils::vxif_prtc($ctx);
			return 1;
		}

		if ($task_type eq "configure") {
			# should happen regardless, but for tasks other than "configure",
			# this now happens in "check_system_installation_requirements()".
			# that is a misnomer and should be changed soon (DE 4-21-05).
			$rc &= $self->stop_product_processes($ctx, $cfg) if ($rc);
		}

		# create configure files
		if (!$self->create_product_config_file($ctx, $cfg)) {
			return 0;
		}

		# start product processes
		if (!$self->start_product_processes($ctx, $cfg)) {
			return 0;
		}

		# display product completion messages
		$self->check_product_completion_messages($ctx, $cfg);
	}

	$$ctx{LOGGER}->info("${trg_upi} successfully configured on ${trg_host}\n");

	return 1;
}

#
# Check for patches. This includes checking for required and optional OS patches 
# and VRTS patches.
#
# Input: 1) the reference to the installation context;
#
# Return: list of patches to install.
#
sub check_for_patches (\%) {
  my ($self,$ctx) = @_;
  my (@patches);

  $$ctx{LOGGER}->entering("CPICommon::check_for_patches");
  
  unshift(@patches, $self->check_for_os_patches($ctx));
  unshift(@patches, $self->check_for_vrts_patches($ctx));

  return @patches;
}

#
# Check for VRTS patches.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_for_vrts_patches (\%) {
}

#
# Check for OS patches.
#
# Input: 1) the reference to the installation context;
#
# Return: list of OS patches to install
#
sub check_for_os_patches (\%) {
  my ($self,$ctx) = @_;
  my (@patches);
  
  $$ctx{LOGGER}->entering("CPICommon::check_for_os_patches");
  
  unshift(@patches, $self->check_for_required_os_patches($ctx));
  unshift(@patches, $self->check_for_optional_os_patches($ctx));

  return @patches;
}

#
# Check for required OS patches.
#
# Input: 1) the reference to the installation context;
#
# Return: list of required OS patches to be installed
#
sub check_for_required_os_patches (\%) {
	my $self = shift;
	my ($ctx) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my ($patch,$patches,$msgl,$msgr, @pchs);
	my ($os) = $self->{OSINSTALLER};
	my ($prodver);
	my ($ret);

	$$ctx{LOGGER}->entering("CPICommon::check_for_required_os_patches");

	my $osver = $ctx->get_host_os_version($trg_host);
	my $osarch = $ctx->get_host_os_architecture($trg_host);
	if ($upi_inst->can("get_required_os_patches")) {
		my $trg_mode = $ctx->get_host_inst_mode($trg_host);
		$patches = $upi_inst->get_required_os_patches($ctx, $osver, $osarch, $trg_mode);
	}

	my $num_patches = scalar(@$patches);
	if ($num_patches > 0) {
		foreach $patch (@{$patches}) {
			$msgl = Utils::_tr("Checking required $os patch $patch", 16, 1154, "$os", "$patch");
			Utils::vxif_pbl($msgl);

			$ret = $self->check_patch($ctx, $patch);
			if (0 == $ret) {
				$msgr = Utils::_tr("not installed", 16, 1086);
				push(@pchs, $patch);
			} elsif (1 == $ret) {
				$msgr = Utils::_tr("obsoleted", 16, 1155);
			} elsif (2 == $ret) {
				$msgr = Utils::_tr("installed", 16, 1156);
			}

			Utils::vxif_pbr($msgl, $msgr);
		}
	}

	return @pchs;
}

#
# Check for optional OS patches.
#
# Input: 1) the reference to the installation context;
#
# Return: list of optional OS patches to be installed
#
sub check_for_optional_os_patches (\%) {
  my ($self,$ctx) = @_;
  
  return;
}

#
# Finalization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub finalize ($) {
}

#
# Cleanup.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub cleanup ($) {
	my $self = shift;
	my ($ctx) = @_;

	my $root_installer = $ctx->get_root_installer();
	$root_installer->cleanup($ctx);
}

sub print_products {
  my($self) = @_;
  my($product,$upi);
  
  if ($self->{PRODUCT}) {
    foreach $upi (keys(%{$self->{PRODUCT}})) {
      $product = $self->{PRODUCT}{$upi};
      
      Utils::vxif_pl($product->get_product_name());
    }
  }
}

1;
