package ThunkAIX;

our @ISA = ("ThunkCommon");

require VxIF::ThunkCommon;
# load the IFRTA module
require VxIF::IFRTA::CPI40AIX4C;
require VxIF::Utils;

###############################
# Public Members              #
###############################


###############################
# Private Members             #
###############################

#
# The reference to the global installation context.
#
my $gctx;

###############################
# Private Methods             #
###############################


###############################
# Public Methods       	      #
###############################

#
# Create an instance of ThunkAIX.
#
sub new {
  my $invocant = shift;
  my $class = ref($invocant) || $invocant;
  my $self = {
        @_,
  };
  
  return bless $self, $class;
}

#
# Set all the necessary globals from the installation context.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %CFG hash;
#
sub set_context_globals (\%;\%) {
  my ($self,$ctx,$cfg) = @_;
  my (%TASKMAP);
  
  $TASKMAP{install} = "INSTALL";
  $TASKMAP{uninstall} = "UNINSTALL";
  $TASKMAP{precheck} = "PRECHECK";
  $TASKMAP{configure} = "CONFIG";
  $TASKMAP{license} = "LICENSE";
  $TASKMAP{describe} = "DESC";
  $TASKMAP{rti} = "RTI";
  
  # copy the installation context and store it locally
  $gctx = $ctx;
  
  # $SYS is the target host
  $SYS = $$ctx{TARGET}{HOST};
  
  # @SYSI is the list of target hosts
  @SYSI = @{$$ctx{SYSTEMS}};
  
  # set local host and domain
  $COMM{LOCALSYS} = $$ctx{LOCAL}{HOST};
  $COMM{LOCALDOMAIN} = $$ctx{LOCAL}{DOMAIN};
  
  # $COMM{LOG} points to the current log filename and path
  $COMM{LOG} = $$ctx{LOGGER}->{LOGFILE};
  
  # update the SSH/RSH SCP/RCP commands
  $CMD{RSH} = $$ctx{LOCAL}{CMD}{RSH};
  $CMD{RCP} = $$ctx{LOCAL}{CMD}{RCP};
  
  # set the response file
  $OPT{RESPONSEFILE} = $$ctx{RESPONSEFILE};
  
  # set the title
  $MSG{TITLE} = $$ctx{INSTALLERTITLE};
  
  # set the task
  $COMM{TASK} = $$ctx{TASK};
  # For now, lets map the options to task. We'll deal with things like INSTALLONLY later.
  # vikas
  #TBD:
  $OPT{$TASKMAP{$COMM{TASK}}} = 1;
  
  # set mySQL status
  #$COMM{MYSQLPRESENT}
  
  #set up the licensing globals
  $COMM{KEYS}{$$ctx{TARGET}{HOST}} = $$ctx{KEYS}{$$ctx{TARGET}{HOST}};
  $COMM{PKEY}{$$ctx{TARGET}{HOST}} = $$ctx{PKEY}{$$ctx{TARGET}{HOST}};
  $COMM{ALLSYSKEYS} = $$ctx{ALLSYSKEYS};
}

#
# Set the MSG globals;
#
sub set_msg_globals {
  my ($self) = @_;
  
  $self->SUPER::set_msg_globals(\%MSG, \%COMM);
  
  # set the product descript messages in rti.
  _rti_set_msgs();
}

#
# Set the COMM globals;
#
sub set_comm_globals {
  my ($self) = @_;
  my ($v);
  
  $self->SUPER::set_comm_globals(\%COMM, \%CMD, \%FLAG);

  # Define train information
  $COMM{OS}="AIX";
  $COMM{SUPPOSVERS}=[ qw(4.3.3 5.1.0 5.2.0) ];
  $COMM{TRAINVERS}="4.0";
  $COMM{LICPKG}="VRTSvlic";
  $COMM{INFRASTRUCTURE}=[ qw(VRTScpi VRTSvlic) ]; 
  $COMM{INFRASTRUCTURE2}=[ qw(VRTSperl VRTSob VRTSmuob VRTSobgui) ]; 
  $COMM{SHAREDPKGS}=[ qw(VRTSllt VRTSgab VRTSjre VRTSweb VRTStep VRTSap) ];
  $COMM{UGPKGS}=[ qw(VRTSweb VRTSvcsor) ];
  $COMM{CKVOLS}=[$PATH{TMP},$PATH{OPT},$PATH{USR},$PATH{VAR},$PATH{ROOT}];
  $COMM{PATCHREL}=1;
  
  # setup the backup file suffix
  if (!$COMM{BUCF}) {
    $v=$COMM{TRAINVERS};
    $v=~s/\.//g;
    $COMM{BUCF}=".init$v";
  }
}

#
# Set the PATH globals;
#
sub set_path_globals {
  my ($self) = @_;
  
  $self->SUPER::set_path_globals(\%PATH);
  
  $PATH{PKGDIR}="pkgs";
  $PATH{CPILIB}="CPI40AIX4C.pm";
  $PATH{PKGINFO}="info";
}

#
# Set the FLAG globals;
#
sub set_flag_globals {
  my ($self) = @_;
  
  $self->SUPER::set_flag_globals(\%FLAG);
}

#
# Set the CMD globals;
#
sub set_cmd_globals {
  my ($self) = @_;
  
  $self->SUPER::set_cmd_globals(\%CMD);

	$CMD{AWK}="/usr/bin/awk";
	$CMD{CAT}="/usr/bin/cat";
	$CMD{CP}="/usr/bin/cp";
	$CMD{CHMOD}="/usr/bin/chmod";
	$CMD{DU}="/usr/bin/du";
	$CMD{DFK}="/usr/bin/df -kt";
	$CMD{ECHO}="/usr/bin/echo";
	$CMD{GREP}="/usr/bin/grep";
	$CMD{ID}="/usr/bin/id";
	$CMD{IFCONFIG}="/usr/sbin/ifconfig";
	$CMD{KILL}="/usr/bin/kill";
	$CMD{LS}="/usr/bin/ls";
	$CMD{MKDIR}="/usr/bin/mkdir";
	$CMD{MV}="/usr/bin/mv";
	$CMD{NETSTAT}="/usr/bin/netstat";
	$CMD{PING}="/usr/sbin/ping";
	$CMD{PS}="/usr/bin/ps";
	$CMD{RCP}="/usr/bin/rcp";
	$CMD{REX}="/opt/VRTScpi/bin/scripts/rex";
	$CMD{RMR}="/usr/bin/rm -rf";
	$CMD{RSH}="/usr/bin/rsh";
	$CMD{SED}="/usr/bin/sed";
	$CMD{SHUTDOWN}="/usr/sbin/shutdown -r";
	$CMD{SORT}="/usr/bin/sort";
	$CMD{TOUCH}="/usr/bin/touch";
	$CMD{TPUT}="/usr/bin/tput";
	$CMD{UNAME}="/usr/bin/uname";
	$CMD{UNIQ}="/usr/bin/uniq";
	$CMD{WC}="/usr/bin/wc";
	$CMD{WHICH}="/usr/bin/which";
	$CMD{GUNZIP}="/opt/VRTScpi/bin/gunzip";
	$CMD{FIND}="/usr/bin/find";  

  # Define unique platform specific commands 
	$CMD{INSTALLP}=$CMD{RMPOBJ}="/usr/sbin/installp";
	$CMD{LSDEV}="/usr/sbin/lsdev";
	$CMD{LSLPP}="/usr/bin/lslpp";
	$CMD{STRLOAD}="/usr/sbin/strload";
	$CMD{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  
  $CMD{STOP}{vxsvc}="/etc/rc2.d/S50isisd stop";
  $CMD{START}{vxsvc}="/etc/rc2.d/S50isisd start > /dev/null 2>&1 &";
  $CMD{STOPTIME}{vxsvc}=60;

  # vikas
  # Define unique product specific commands
	$CMD{STOP}{lmx}="$CMD{LMXCONFIG} -U";
	$CMD{STOP}{vcsmm}="$CMD{VCSMMCONFIG} -U";
	$CMD{STOP}{vxfen}="$CMD{VXFENCONFIG} -U";

  $CMD{STOP}{sfrac_odm}   = "umount /dev/odm";
  $CMD{STOP}{sfrac_had}   = "/opt/VRTSvcs/bin/hastop -all";
}

#
# Initialize the thunk layer.
#
sub initialize {
  my ($self) = @_;
  
  # set the CMD globals
  $self->set_cmd_globals();
  
  # set the COMM globals
  $self->set_comm_globals();
  
  # set the MSG globals
  $self->set_msg_globals();
  
  # set the PATH globals
  $self->set_path_globals();
  
  # set the FLAG globals
  $self->set_flag_globals();
}

#
# Find out whether a given product exist.
#
# Input: 1) the reference to the installation context;
#        2) product UPI;
#
# Return: 1 if the product exist; Otherwise, return 0.
#
sub is_product_exist ($) {
  my ($self,$ctx,$upi) = @_;
  
  $$ctx{LOGGER}->entering("ThunkAIX::is_product_exist");
  
  return ((Utils::vxif_list($upi, $COMM{LUPIS}) >= 0) && (Utils::vxif_list($upi, $COMM{UPIS}) >= 0) && (defined($PROD{${upi}}{NAME})));
}

################################################
# Product Info Interfaces                      #
################################################

#
# Get the localized product title for this product. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#
# Return: the complete localized title for this product.
#
sub get_product_title (\%) {
  my ($self,$ctx) = @_;
  
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_title");
  
  return $PROD{$$ctx{TARGET}{UPI}}{NAME};
}

#
# Determine if the given product name, could be different for different license type, 
# is a valid product name for this product.
#
# This interface is mandatory for licensable proudct.
#
# Input: 1) the reference to the installation context;
#        2) the product name;
#
# Return: 1 if the given product name is a valid product name for this product;
#         0 otherwise.
#
sub is_product_name (\%$) {
  my ($self,$ctx,$pn) = @_;
  my ($mode);
  my ($upi) = $$ctx{TARGET}{UPI};
  
  $$ctx{LOGGER}->entering("ThunkAIX::is_proudct_name");
  
  $$ctx{LOGGER}->finer("upi=${upi}\npn=${pn}");
  return 1 if (($PROD{$upi}{NAME} eq $pn) || ($PROD{$upi}{LNAME} eq $pn));
  foreach $mode (@{$PROD{$upi}{MODES}}) {
    return 1 if (($pn eq $PROD{$mode}{NAME}) || ($pn eq $PROD{$mode}{LNAME}));
  }
  
  return 0;
}

#
# Get the product version number (i.e. 4.1.3.10). For some products, the product version
# number may be different from the main package version number. The main package version 
# number is optional and will be absent for the current product version. If present, 
# the main package version will be different the current main package version. 
# The reason is that for product upgrade and product downgrade, VxIF needs a list of 
# packages associated with a particular product version so they can properly removed.
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the main package version; this parameter is optional;
#
# Return: product version number
#
sub get_product_version (\%;$) {
  my ($self,$ctx) = @_;
    
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_version");
  
  # For the A stuff, should the product version be the train version of the 
  # product main package version? For now, just use the train version.
  return $COMM{TRAINVERS};
}

#
# Get the localized product description for this product. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: localized product description.
#
sub get_product_description (\%) {
  my ($self,$ctx) = @_;
  my ($n) = 1;
  my ($desc);
      
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_description");
  
  $desc = "$MSG{$$ctx{PRODUCT}{UPI}}{DESC}";
  while ($MSG{$$ctx{PRODUCT}{UPI}}{"DESC${n}"}) {
    $desc .= "\n$MSG{$$ctx{PRODUCT}{UPI}}{\"DESC${n}\"}";
    $n++;
  }
  
  return $desc;
}

#
# Get the localized product copyright information. 
#
# This interface is optional. If product copyright information is not provided, 
# VxIF will use the generic VERITAS copyright information. If product copyright 
# information is given, VxIF will use it instead of the generic VERITAS 
# copyright information.
#
# Input: 1) the reference to the installation context;
#
# Return: the localized product copyright information.
#
#sub get_product_copyright (\%) {
#}

#
# Indicate whether this is a licenable product, which requires a legitimate
# license from VERITAS in order to install. 
#
# This interface is optional. By default VxIF assumes all products are licensable.
#
# Input: 1) the reference to the installation context;
#        2) the selected installation mode if applicable;
#
# Return: 1 if this product is a licenable product; 0 otherwise.
#
sub is_product_licensable (\%$) {
  my ($self,$ctx,$mode) = @_;
        
  $$ctx{LOGGER}->entering("ThunkAIX::is_product_licensable");

  return (Utils::vxif_list($$ctx{TARGET}{UPI}, $COMM{LUPIS}) >= 0);
}

#
# Get the product license feature.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license feature.
#
sub get_product_license_feature (\%) {
  my ($self,$ctx) = @_;
          
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_license_feature");
  
  return $PROD{$$ctx{TARGET}{UPI}}{LICFEATURE};
}

#
# Get the product license ID.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license ID.
#
sub get_product_license_id (\%) {
  my ($self,$ctx) = @_;
            
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_license_id");
    
  return $PROD{$$ctx{TARGET}{UPI}}{LICID};
}

#
# Get the product license UPI.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: the product license UPI.
#
sub get_product_license_upi (\%) {
  my ($self,$ctx) = @_;
              
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_license_upi");
      
  return $PROD{$$ctx{TARGET}{UPI}}{LICUPI};
}

#
# Get the product key mode associated with the given registration key.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the reference to registration keys hash;
#
# Return: the product key mode.
#
sub get_product_key_mode (\%\%) {
  my ($self,$ctx,$rk) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_get_mode';
  my ($mode);
          
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_key_mode");

  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    $mode = &$fc($rk);
  }
  $$ctx{LOGGER}->finer("mode=${mode}");
    
  return $mode;
}

#
# Get the product license modes.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: the reference to the list of product license modes.
#
sub get_product_license_modes (\%) {
  my ($self,$ctx) = @_;
    
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_license_modes");
    
  return $PROD{$$ctx{TARGET}{UPI}}{MODES};
}

#
# Return a list of product modes of which previous version keys existed, but are not supported 
# for the current version.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#
# Return: A list of product modes of which previous version keys existed, but are not supported
#         for the current version.
#
sub get_unsupported_product_key_modes (\%) {
  my ($self,$ctx) = @_;
  
  $$ctx{LOGGER}->entering("ThunkAIX::get_unsupported_product_key_modes");
  
  return $COMM{UNSUPPORTEDMODES};
}

#
# Get the product key mode name associated with the given product mode key.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product key mode;
#
# Return: the product key mode name.
#
sub get_product_key_mode_name (\%$) {
  my ($self,$ctx,$mode) = @_;
  my ($mname) = $$ctx{TARGET}{UPI};
    
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_key_mode");
  $$ctx{LOGGER}->finer("mode=${mode}");
  
  if (defined($PROD{$mode}{NAME})) {
    $mname = $PROD{$mode}{NAME};
  }
  $$ctx{LOGGER}->finer("mode name =${mname}");
      
  return $mname;
}

#
# Determine if the given registration key is a valid product registration key.
#
# This interface is optional. By default, VxIF the given registration key is
# not a valid product registration key.
#
# Input: 1) the reference to the installation context;
#        2) the product registration key;
#
# Return: 1 if the given key is a valid product registration key; 
#         -1 if _<prod>_comm_is_prodkey() is not defined;
#         0 otherwise.
#
sub is_product_key (\%$) {
  my ($self,$ctx,$rk) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_is_prodkey';
  my ($ipk) = 0;
            
  $$ctx{LOGGER}->entering("ThunkAIX::is_product_key");
  $$ctx{LOGGER}->finer("rk=${rk}");
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    $ipk = &$fc($rk);
  } else {
    $ipk = -1;
  }
  
  $$ctx{LOGGER}->finer("ipk=${ipk}");
      
  return $ipk;
}

#
# Determine if the given license mode and license type are valide.
#
# This interface is optional. By default, VxIF assumes the given license mode and 
# license type are valid.
#
# Input: 1) the reference to the installation context;
#        2) license mode;
#        3) license type;
#
# Return: 1 if the given license mode and license type are valid; 0 otherwise.
#
sub check_proudct_license (\%$$) {
  my ($self,$ctx,$mode,$type) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_check_license';
              
  $$ctx{LOGGER}->entering("ThunkAIX::validate_proudct_license");
  
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, "");
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc($mode);
  }
  
  return ($COMM{PKEY}{$SYS} ne "");
}

#
# Determine whether a older version of the product can be upgraded to the current version. 
# This interface will be invoked if an older version of this product exist on the
# destination host. If this product is upgradable, VxIF will try to perform a product
# upgrade rather than new product install. VxIF will invoke product upgrade interfaces,
# such as pre_product_upgrade(), post_product_upgrade(), etc instead of the new installation
# interfaces.
#
# This interface is option. If not implemented, VxIF assumes any existing version of this
# product is not upgradable.
#
# Product upgrade may not be supported on all VxIF platforms. If product upgrade is not
# supported by a VxIF platform, VxIF will first attempt remove the existing product and
# perform a new install with the current version. Product group should implement 
# pre_product_upgrade() to properly backup the data, and implement post_product_upgrade() 
# to properly migrate the data over after the packages are installed.
#
# Input: 1) the reference to the installation context;
#        2) the version of the product currently installed on the destination host;
#
# Return: 1 if this version can be upgraded to the current version. Otherwise, return
#         0;
#
sub is_product_upgradable (\%$) {
  my ($self,$ctx,$iver) = @_;
          
  $$ctx{LOGGER}->entering("ThunkAIX::is_product_upgradable");

  return 0 if (!$PROD{$$ctx{TARGET}{UPI}}{UGVERS});
  
  # the existing product is upgradable if the existing version exist in {UGVERS}
  return (Utils::vxif_list($iver, $PROD{$$ctx{TARGET}{UPI}}{UGVERS}) >= 0);
}

#
# Determine whether a newer version of the product can be downgraded to the current version. 
# This interface will be invoked if an newer version of this product exist on the
# destination host. If this product is downgradable, VxIF will try to perform a product
# downgrade rather than new product install. VxIF will invoke product downgrade interfaces,
# such as pre_product_downgrade(), post_product_downgrade(), etc instead of the new installation
# interfaces.
#
# This interface is option. If not implemented, VxIF assumes any existing version of this
# product is not downgradable.
#
# Product downgrade may not be supported on all VxIF platforms. If product downgrade is not
# supported by a VxIF platform, VxIF will first attempt remove the existing product and
# perform a new install with the current version. Product group should implement 
# pre_product_downgrade() to properly backup the data, and implement post_product_downgrade() 
# to properly migrate the data over after the packages are installed.
#
# Input: 1) the reference to the installation context;
#        2) the version of the product currently installed on the destination host;
#
# Return: 1 if this version can be downgraded to the current version. Otherwise, return
#         0;
#
sub is_product_downgradable (\%$) {
  my ($self,$ctx,$iver) = @_;
            
  $$ctx{LOGGER}->entering("ThunkAIX::is_product_downgradable");
  
  # none of the A stuff is downgradable
  return 0;
}

#
# Get the product train directory. This is similar to getting the {TRAINDIR} in IFRTA and IFRTB.
# This interface, if implemented, will be used to determine the location of the product packages.
#
# This interface is optional. If not implement, VxIF to determine the packages location from
# the get_package_location() interface. This interface must be implemented in the absence of 
# get_package_location() interface.
#
# Input: 1) the reference to the installation context;
#        
# Return: the product directory.
#
sub get_product_train_dir (\%) {
  my ($self,$ctx) = @_;
              
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_train_dir");
  
  #vikas.
  #::return $PROD{$$ctx{TARGET}{UPI}}{TRAINDIR};
  #return "/space/SxRT/storage_solutions_4.0_cd1/file_system/pkgs"; # hardcode the path for testing purposes
  #return "/space/SxRT/storage_solutions_4.0_cd1/storage_foundation_for_oracle/pkgs"; # hardcode the path for testing purposes
  return "/space/SxRT/cluster_file_solutions_cd1/storage_foundation_for_oracle_rac"; # hardcode the path for testing purposes
}

#
# Return the different installation modes for this product. For example, "Complete", 
# "Typical", etc. 
#
# This interface is optional and must be omitted if there is no installation modes.
#
# This interface shall return the reference of a hash with the modes as the keys
# and the localized mode description as the value.
#
# $modes{<MODE>} = "<LOCALIZED MODE DESCRIPTION>"
#
# For example,
#
# $modes{Typical} = "Install the Foo service only.";
# $modes{Complete} = "Install Foo service and client.";
# $modes{Optional} = "Install optional debug files for Foo.";
#
# Input: 1) the reference to the installation context;
#
# Return: the reference of a hash with the installation modes as the keys and the localized
#         description as the values.
#
#sub get_product_installation_modes (\%) {
#}


################################################
# System Validation Interfaces                 #
################################################

#
# Determine whether the destination host OS is supported.
#
# Input: 1) the reference to the installation context;
#        2) host name;
#        3) host OS;
#        4) host OS version number;
#        5) host OS architecture;
#
# Return: 1 if the given OS is supported. Otherwise, return 0;
#
sub is_os_supported (\%$$$$) {
  my ($self,$ctx,$host,$os,$osver,$osarch) = @_;
  my ($supported,$pkgsup,$pkg);
                
  $$ctx{LOGGER}->entering("ThunkAIX::is_os_supported");
  
  #
  # For the A stuff, we need to check three places: $COMM{SUPPOSVERS},
  # $PROD{<UPI>}{SUPPOSVERS}, and $PKGI{<PKG}}{SUPPOSVERS}. We will first
  # check $COMM{SUPPOSVERS}, then $PROD{<UPI>}{SUPPOSVERS}, and then
  # $PKGI{<PKG}}{SUPPOSVERS}. For the decision tree, $PKGI{<PKG}}{SUPPOSVERS}
  # overwrites $PROD{<UPI>}{SUPPOSVERS} and $PROD{<UPI>}{SUPPOSVERS} overwrites
  # $COMM{SUPPOSVERS}.
  #
  $supported = (Utils::vxif_wclist($osver, @{$COMM{SUPPOSVERS}}) >= 0);
  
  if ($PROD{$$ctx{TARGET}{UPI}}{SUPPOSVERS}) {
    $supported = (Utils::vxif_wclist($osver, @{$PROD{$$ctx{TARGET}{UPI}}{SUPPOSVERS}}) >= 0);
  }
  
  $pkgsup = 1;
  foreach $pkg ($PROD{$$ctx{TARGET}{UPI}}{PKGS}) {
    last if (!$pkgsup);
    if ($PKGI{$pkg}{SUPPOSVERS}) {
      $pkgsup = (Utils::vxif_wclist($osver, @{$PKGI{$pkg}{SUPPOSVERS}}) >= 0);
      if (!$pkgsub) {
        $$ctx{LOGGER}->error(Utils::_tr("${pkg} of $$ctx{TARGET}{UPI} does not support ${os} ${osver}.", 26, 1000, "${pkg}", "$$ctx{TARGET}{UPI}", "${os}", "${osver}"));
      }
    }
  }
  
  $supported &= $pkgsup;
  
  return $supported;
  #return 1;
}

#
# Check to make sure the system names are valid.
#
# This interface is optional. By default, VxIF do some generic checking, such as making
# sure the name contains the valid characters. Apart from that it is up the each individual
# product to impose additional contraints.
#
# Input: 1) the reference to the installation context;
#        2) the array of systems that will be installing this product;
#
# Return: 1 if the system names are valid; 0 otherwise.
#
sub check_system_names (\%@) {
  my ($self,$ctx,@systems) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_singlesystem_areyousure';
                  
  $$ctx{LOGGER}->entering("ThunkAIX::check_system_names");

  # vikas - need to add more conditions to it.
  # && ($OPT{INSTALL}) && (!$OPT{LICENSE}) && (!$OPT{INSTALLONLY}) && (!$OPT{RESPONSEFILE}) && (!$OPT{PRECHECK})
  if (!$#systems && defined(&$fc)) {
	  $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
  
  return 1;
}

################################################
# Common Package and Patch Interfaces          #
################################################
#
# These interfaces will likely be implemented by the train coordination team. For example, 
# the train coordination team can determine whether a product package can be compressed 
# to reduce space on a train CD.
#

#
# Get the name of the main package. 
#
# This interface is mandatory.
#
# Input: 1) the reference to the installation context;
#        2) the selected installation mode if one is available;
#
# Return: the name of the main package.
# 
sub get_product_main_package (\%$) {
  my ($self,$ctx,$mode) = @_;
                    
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_main_package");

  return $PROD{$$ctx{TARGET}{UPI}}{MAINPKG};
}

#
# For a given package, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: Return the directory path where the given package is located.
#
#sub get_package_location (\%$) {
# For the A stuff, always use the train dir.
#}

#
# For a given patch name, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>/patches.
#
# Input: 1) the reference to the installation context;
#        2) patch name;
#
# Return: Return the directory path where the given patch is located.
#
#sub get_patch_location (\%$) {
# For the A stuff, always use the train dir.
#}

#
# Indicate whether a given package is in a compressed file. 
#
# This interface is optional. By default, VxIF assumes a given package is not compressed.
#
# Input: 1) the reference to the installation context;
#        2) package name; 
#
# Return: 0 if the given package is not compressed;
#         1 if the given package is compressed using gunzip (.tar.gz);
#         2 if the given package is compressed in other format; If the package
#           is compressed in other format, product must implement get_package_decompressor() 
#           method to provide a custom decompressor;
#
sub get_package_compression_type (\%$) {
  my ($self,$ctx,$pkg) = @_;
                      
  $$ctx{LOGGER}->entering("ThunkAIX::get_package_compression_type");

  # The A stuff is always compressed in $pkg.tar.gz?
  return 1;
}

#
# Indicate whether a given patch is in a compressed file. 
#
# This interface is optional. By default, VxIF assumes a given patch is not compressed.
#
# Input: 1) the reference to the installation context;
#        2) patch name; 
#
# Return: 0 if the given patch is not compressed;
#         1 if the given patch is compressed using gunzip (.tar.gz);
#         2 if the given patch is compressed in other format; If the patch
#           is compressed in other format, product must implement get_patch_decompressor() 
#           method to provide a custom decompressor;
#
sub get_patch_compression_type (\%$) {
  my ($self,$ctx,$patch) = @_;
                        
  $$ctx{LOGGER}->entering("ThunkAIX::get_patch_compression_type");
  
  # The A stuff is always compressed in $patch.tar.gz?
  return 1;
}

#
# Return the compressed filename, without the path, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed filename, without the path, for the given package.
#
sub get_package_compressed_filename (\%$) {
  my ($self,$ctx,$pkg) = @_;
                        
  $$ctx{LOGGER}->entering("ThunkAIX::get_package_compressed_filename");
  
  # The A stuff is always compressed in $pkg.tar.gz?
  return "${pkg}.tar.gz";
}

#
# Return the compressed filename, without the path, for the given patch.
#
# This interface is mandatory if any product patches are compressed. This interface 
# must be absent all product patches are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) patch name;
#
# Return: the compressed filename, without the path, for the given patch.
#
sub get_patch_compressed_filename (\%$) {
  my ($self,$ctx,$patch) = @_;
                          
  $$ctx{LOGGER}->entering("ThunkAIX::get_patch_compressed_filename");
    
  # The A stuff is always compressed in $patch.tar.gz?
  return "${patch}.tar.gz";
}

#
# Return the compressed file size, in kilobytes, for the given package.
#
# This interface is mandatory if any product packages are compressed. This interface 
# must be absent all product packages are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed file size, in kilobytes, for the given package.
#
sub get_package_compressed_filesize (\%$) {
  my ($self,$ctx,$pkg) = @_;
                          
  $$ctx{LOGGER}->entering("ThunkAIX::get_package_compressed_filesize");
  
  #TBD Hardcode a dummy value for now. Revisited this later since time is kicking my ass!
  return "5000"; 
}

#
# Return the compressed file size, in kilobytes, for the given patch.
#
# This interface is mandatory if any product patches are compressed. This interface 
# must be absent all product patches are uncompressed.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: the compressed file size, in kilobytes, for the given patches.
#
sub get_patch_compressed_filesize (\%$) {
  my ($self,$ctx,$patch) = @_;
                            
  $$ctx{LOGGER}->entering("ThunkAIX::get_patch_compressed_filesize");
    
  #TBD Hardcode a dummy value for now. Revisited this later since time is kicking my ass!
  return "5000"; 
}

#
# This is the package decompression callback. Given a package name, return the 
# reference of the interface to handle the decompression of the compressed package file.
#
# This interface is mandatory if the package compression type is 2.
#
# VxIF will pass the following parameters into the callback in the order as specified.
# 
# 1) the reference to the installation context;
# 2) the package name;
# 3) the compressed file name associated with this package;
# 4) the directory where the compressed file is located;
# 5) the directory to uncompress the files into;
#
# This callback shall return 1 if the given package has been successfully uncomressed;
# otherwise, 0 shall be returned.
#
# Here's an example on how to customize the decompression operation.
#
# sub get_package_decompressor (\%$$$$$) {
#   my ($self,$ctx,$pkg,$cfile,$cdir, $ucdir) = @_;
#   my $my_decompressor;
#   if ($pkg eq "package1") {
#      $my_decompressor = sub (\%$$$$) {
#        my ($ctx, $pkg, $cfile, $cdir, $ucdir) = @_;
#        ... customize the decompression for package1 here
#      };
#   } elsif ($pkg eq "package2") {
#      $my_decompressor = sub (\%$$$) {
#        my ($ctx, $pkg, $cfile, $cdur, $ucdir) = @_;
#        ... customize the decompression for package2 here
#      };
#   }
#
#   return \&$my_decompressor;
# }
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the compressed file name associated with this pacakge;
#        4) the directory where the compressed file is localted;
#        5) the directory to uncompress the files into;
#
# Return: the reference of the customized package decompressor.
#
#sub get_package_decompressor (\%$$$$) {
#}

#
# This is the patch decompression callback. Given a patch name, return the 
# reference of the interface to handle the decompression of the compressed patch file.
#
# This interface is mandatory if the patch compression type is 2.
#
# VxIF will pass the following parameters into the callback in the order as specified.
# 
# 1) the reference to the installation context;
# 2) the patch name;
# 3) the compressed file name associated with this patch;
# 4) the directory where the compressed file is located;
# 5) the directory to uncompress the files into;
#
# This callback shall return 1 if the given patch has been successfully uncomressed;
# otherwise, 0 shall be returned.
#
# Here's an example on how to customize the decompression operation.
#
# sub get_patch_decompressor (\%$$$$) {
#   my ($self,$patch,$cfile,$cdir, $ucdir) = @_;
#   my $my_decompressor;
#   if ($patch eq "patch1") {
#      $my_decompressor = sub (%$$$$) {
#        my (%ctx, $pkg, $cfile, $ucdir) = @_;
#        ... customize the decompression for package1 here
#      };
#   } elsif ($patch eq "patch2") {
#      $my_decompressor = sub (%$$$$) {
#        my (%ctx, $pkg, $cfile, $ucdir) = @_;
#        ... customize the decompression for patch2 here
#      };
#   }
#
#   return \&$my_decompressor;
# }
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#        3) the compressed file name associated with this patch;
#        4) the directory where the compressed file is localted;
#        5) the directory to uncompress the files into;
#
# Return: the reference of the customized package decompressor.
#
#sub get_patch_decompressor (\%$$$$) {
#}


################################################
# Common Product Installation Interfaces       #
################################################

#
# Product pre-installation script. This interface will be called before the product
# is being installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-installation is successful. Otherwise, return 0;
#
sub pre_product_install (\%\%;$) {
  my ($self,$ctx,$cfg,$mode) = @_;
                              
  $$ctx{LOGGER}->entering("ThunkAIX::pre_product_install");
  
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  #TBD: 
  #vikas
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_preinstall_questions';

 if (defined(&$fc)) {
   $$ctx{LOGGER}->entering("$fc");
   &$fc();
 }

  $fc = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_preinstall';
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
  
  return 1;
}

#
# Product post-installation script. This interface will be called after all the packages
# has been installed.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
sub post_product_install (\%\%;$) {
  my ($self,$ctx,$cfg,$mode) = @_;
                                
  $$ctx{LOGGER}->entering("ThunkAIX::post_product_install");
    
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  #TBD: 
  #vikas
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_postinstall';
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
    
  return 1;
}

#
# Product configuration. This interface will be called after all the packages
# has been installed and post_product_install().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-installation is successful. Otherwise, return 0;
#
sub install_product_configuration (\%\%;$) {
  my ($self,$ctx,$cfg,$mode) = @_;
                                  
  $$ctx{LOGGER}->entering("ThunkAIX::install_product_configuration");
      
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  #TBD: 
#vikas
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_postinstall_questions';
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
  
  
  return 1;
}

#
# Product configuration. This interface will be called after post-installation.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
sub create_product_install_config (\%\%;$) {
  my ($self,$ctx,$cfg,$mode) = @_;
                                    
  $$ctx{LOGGER}->entering("ThunkAIX::install_product_configuration");
        
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  #TBD: 
    
  return 1;
}

#
# Product pre-patch script. This interface will be called before the patches
# are being applied. This interface will only be invoked for patch-only install.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-patch is successful. Otherwise, return 0;
#
#sub pre_product_patch (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product post-patch script. This interface will be called after all the product patches
# have been installed. This interface will only be invoked for patch-only install.
#
# This interface is optional.
#
# Input: 1) license key if the product is a licensable product;
#        2) %cfg array reference;
#        3) the reference to the installation context;
#
# Return: 1 if the post-patch is successful. Otherwise, return 0;
#
#sub post_product_patch (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product configuration. This interface will be called after all the packages
# has been patched and post_product_patch().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the patch configuration is successful. Otherwise, return 0;
#
#sub patch_product_configuration (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Patch configuration. This interface will be called after post-patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_patch_install_config (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product pre-upgrade script. This interface will be called before the product
# are being upgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-upgrade is successful. Otherwise, return 0;
#
#sub pre_product_upgrade (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product post-upgrade script. This interface will be called after the product
# have been upgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the post-upgrade is successful. Otherwise, return 0;
#
#sub post_product_upgrade (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product configuration. This interface will be called after all the packages
# has been upgraded and post_product_upgrade().
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the upgrade configuration is successful. Otherwise, return 0;
#
#sub upgrade_product_configuration (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product upgrade configuration. This interface will be called after post-upgrade.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_product_upgrade_config (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product pre-downgrade script. This interface will be called before the product
# are being downgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: 1 if the pre-downgrade is successful. Otherwise, return 0;
#
#sub pre_product_downgrade (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product post-downgrade script. This interface will be called after the product
# have been downgraded.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the installation mode if applicable;
#        3) %cfg array reference;
#
# Return: 1 if the post-downgrade is successful. Otherwise, return 0.
#
#sub post_product_downgrade (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product downgrade configuration. This interface will be called after post-downgrade.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the reference to the %cfg array;
#
#sub create_product_downgrade_config (\%\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Start the product processees after product installation operations.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version; this parameter is optional;
#        3) the installation mode if applicable;
#
# Return: 1 if successful; 0 otherwise.
#
sub start_product_processees (\%;$$) {
  my ($self,$ctx,$prodver,$mode) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_start_product';
                                      
  $$ctx{LOGGER}->entering("ThunkAIX::start_product_processees");
          
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
    
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
      
  return 1;
}

#
# Stop the product processees, for a given version of the product, before product
# installation operations.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version; this parameter is optional;
#        3) the installation mode if applicable; this parameter is optional;
#
# Return: 1 if successful; 0 otherwise.
#
sub stop_product_processees (\%;$$) {
  my ($self,$ctx,$prodver,$mode) = @_;
  my ($psfc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_prestop';
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_stop_processes';
  
  $$ctx{LOGGER}->entering("ThunkAIX::stop_product_processees");
  
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  # call prestop if it exist
  if (defined(&$psfc)) {
    $$ctx{LOGGER}->entering("$psfc");
    &$psfc();
  }
  
  # stop the processes
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
  
  return 1;
}

#
# Serialize %cfg array. The serialized %cfg will be written directly to the
# response file. The handler must return a buffer which contains the serialzied %cfg array.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the installation type;
#           0 - product install;
#           1 - product patch;
#           2 - product upgrade;
#           3 - product downgrade;
#        3) %cfg array reference;
#        4) the installation mode if applicable;
#
# Return: the reference to the %cfg serialization hander or null ("").
#
#sub serialize_cfg (\%$\%;$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# Product completion messages. This interface will be called after the product installation
# operations completed successfully.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) %cfg array reference;
#        3) the installation mode if applicable;
#
# Return: the localized product completion message.
#
sub get_product_completion_message (\%\%;$) {
  my ($self,$ctx,$prodver,$mode) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_completion_messages';
                                        
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_completion_message");
            
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
      
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }
        
  return 1;
}


################################################
# Common Package Installation Interfaces       #
################################################

#
# Stop the processes associated with a given package. This interface, if implemented, will be
# invoked before a package, or a patch for this package, is installed/upgraded/downgraded.
#
# This interface is optional. This interface need not be implemented if the stop_product_processes()
# has already shutdown all the processes.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name of the patch to be applied;
#
# Return: 1 if successful; 0 otherwise.
#
#sub stop_package_processes (\%$$) {
# TBD: Stub this out for now since time is kicking my ass!
#{

#
# Start the processes associated with a given package. This interface, if implemented, will be
# invoked after a package, or a patch for this package, is installed/upgraded/downgraded.
#
# This interface is optional. This interface need not be implemented if the start_product_processes()
# will start all the processes.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the patch name of the patch applied;
#
# Return: 1 if successful; 0 otherwise.
#
#sub start_package_processes (\%$$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

# 
# Pre system reboot script. This interface will be invoked before system reboot take place.
# This interface will be invoked if {REQREBOOT} is set to 1 for a given package.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#
# Return: 1 if pre-system reboot is successful. Otherwise, return 0.
#
#sub pre_package_system_reboot (\%$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

# 
# Pre system reboot script. This interface will be invoked before system reboot take place.
# This interface will be invoked if {REQREBOOT} is set to 1 for a given patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#
# Return: 1 if pre-system reboot is successful. Otherwise, return 0.
#
#sub pre_patch_system_reboot (\%$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# This is the system reboot callback. Given a package name, return the 
# reference of the interface to handle the system reboot. 
# Instead of using default reboot mechanism (i.e. reboot command),
# VxIF will using this handler to handle system reboot.
#
# This interface is optional.
#
# Here's an example on how to customize the reboot operation.
#
# sub get_package_system_reboot_handler (%$) {
#   my($self,%context,$package_name) = @_;
#   my $my_reboot = sub {
#     # handle the system reboot here.
#   }
#
#   return \&$my_reboot;
# }
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#
# Return: the reference of the system reboot handler.
#
#sub get_package_system_reboot_handler (\%$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# This is the system reboot callback. Given a patch name, return the 
# reference of the interface to handle the system reboot. 
# Instead of using default reboot mechanism (i.e. reboot command),
# VxIF will using this handler to handle system reboot.
#
# This interface is optional.
#
# Here's an example on how to customize the reboot operation.
#
# sub get_patch_system_reboot_handler (%$) {
#   my($self,%context,$patch_name) = @_;
#   my $my_reboot = sub {
#     # handle the system reboot here.
#   }
#
#   return \&$my_reboot;
# }
#
# Input: 1) the reference to the installation context;
#        2) the patch name;
#
# Return: the reference of the system reboot handler.
#
#sub get_patch_system_reboot_handler (\%$) {
# TBD: Stub this out for now since time is kicking my ass!
#}

#
# What to do if a given package already exist on the target host and its version is different 
# from the one this product is trying to install? 
#
# This interface is option. By default, VxIF assumes the existing package is not compatible
# with this product and stop the installation.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the actual existing package version on the target host;
#        4) the package version to be installed;
#
# Return: 1 if the existing package on the target host is compatible with this product
#         and use it instead of installing the new version; 2 if the existing package on
#         target host is not compatible with this product and abort the installation;
#         0 to remove the existing package on the target host and install the current
#         version.
# 
sub is_package_version_supported (\%$$$) {
  my ($self,$ctx,$pkg,$iver,$pkgver) = @_;

  $$ctx{LOGGER}->entering("ThunkAIX::is_package_version_supported");
  
  # For the A stuff, assuming any existing package is not supported.
  return 2;
}

#
# Pre package installation operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
sub pre_package_install (\%$$$$) {
  my ($self,$ctx,$pkg,$host,$os,$osver) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_aix_prepkgadd';
                                      
  $$ctx{LOGGER}->entering("ThunkAIX::pre_package_install");
          
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
    
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  } 
      
  return 1;
}

#
# Post package installation operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
sub post_package_install (\%$$$$) {
  my ($self,$ctx,$pkg,$host,$os,$osver) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_aix_postpkgadd';
                                        
  $$ctx{LOGGER}->entering("ThunkAIX::post_package_install");
            
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
      
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }  
        
  return 1;
}

#
# Pre package removal operations. 
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
sub pre_package_remove (\%$$$$) {
  my ($self,$ctx,$pkg,$host,$os,$osver) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_aix_prepkgrm';
                                          
  $$ctx{LOGGER}->entering("ThunkAIX::pre_package_remove");
              
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
        
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }  
          
  return 1;
}

#
# Post package removal operations. 
#
# This interface is optional
#
# Input: 1) the reference to the installation context;
#        2) the package name;
#        3) the name of the host for which this package will be removed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: 1 if successful; 0 otherwise.
#
sub post_package_remove (\%$$$$) {
  my ($self,$ctx,$pkg,$host,$os,$osver) = @_;
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_aix_postpkgrm';
                                            
  $$ctx{LOGGER}->entering("ThunkAIX::post_package_remove");
                
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
          
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    &$fc();
  }  
            
  return 1;
}

################################################
# Product Dependency Interfaces                #
################################################

#
# Get the external packages for which this product depended on.
# Any package that is not part of this product are consider external
# packages. VxIF will first check for the external dependent packages before 
# installing the product.
#
# This interface is optional.
#
# This interface shall return the reference of a hash with the dependent 
# package name as the keys with the following properties.
#
# {NAME}	- dependent package name; This is mandatory;
# {VERSION}	- dependent package version; This is optional; If the dependent
#                 version is not specified; VxIF will assume this product depended
#                 on any version of this package;
# {DESC}	- localized dependent package description; This is optional;
#
# Input: 1) the reference to the installation context;
#        2) host OS version;
#        3) host OS architecture;
#        4) the selected installation mode if applicable;
#
# Return: the reference of a hash with the dependent package name as the key and the
#         dependent package version and the localized description as the values.
#
#sub get_external_dependent_packages (\%$$;$) {
# TBD:
#}

#
# What to do if a given external dependent package exist on the install host
# and its version is different from the one this product is expecting? 
#
# This interface is option. By default, VxIF assumes the existing external dependent
# package is not supported and stop the installation.
#
# Input: 1) the reference to the installation context;
#        2) the external dependent package name;
#        3) the actual dependent package version;
#        4) the expected dependent package version;
#        5) host OS version;
#        6) host OS architecture;
#
# Return: 1 if the existing external dependent package is compatible with this product
#         and continue with the installation; 0 if the existing external dependent 
#         package is not supported and stop the installation.
#
sub is_external_dependent_version_supported (\%$$$$$) {
  my ($self,$ctx,$dpkg,$idver,$edver,$ver,$osver,$osarch) = @_;
  
  $$ctx{LOGGER}->entering("ThunkAIX::is_external_dependent_version_supported");
  
  # For the A stuff, assuming any existing dependent package is not supported.
  return 0;
}

#
# Get the list of requred patches, both external application and OS patches, for this product. 
# VxIF will first check for the required patches before installing the product.
#
# This interface is optional.
#
# This interface shall return the a list which contains the patch ID of the required
# patches for this product.
#
# Input: 1) the reference to the installation context;
#        2) host OS version;
#        3) host OS architecture;
#        4) the selected installation mode if applicable;
#
# Return: a list which contains the patch IDs of the required patches.
#
#sub get_required_external_patches (\%$$;$) {
# TBD:
#}


################################################
# Packages and Patches Definition Interfaces   #
################################################

#
# Get the order in which the product packages will be installed. This interface
# is used for backward compatibility support only. Some old product scripts
# may not set the {DEPS} correctly. Basically, $PROD{<UPI>}{PKGS} instructs the
# installer the order in which to install the product packages.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the list of product packages in the order in which to be installed.
#
sub get_product_packages_ordering (\%$$$;$) {
  my ($self,$ctx,$pdver,$osver,$osarch,$mode) = @_;  
  my ($arrayref) = $PROD{$$ctx{TARGET}{UPI}}{PKGS};
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_define_pkgs';
  
  $$ctx{LOGGER}->entering("ThunkAIX::get_product_packages_ordering");
  
  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
      
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    $arrayref = &$fc();
  }
    
  return @{$arrayref};
}

#
# Return a list of required packages for a given version of this product. The given 
# version may not be the current version. The reason is that for product upgrade 
# and product downgrade, VxIF needs a list of packages associated with a particular 
# version so they can properly removed.
#
# This interface is mandatory for package install. This interface is optional 
# for patch-only install.
#
# This interface shall return the reference of a hash with package names as the keys
# with the following properties.
#
# {NAME}    	- the package name; This property is mandatory;
# {VERSION} 	- the package version; This property is mandatory;
# {SPACE}	- a list which specified the spaces, in kilobytes, needed for this
#                 package in /opt, /usr, /var, and /; this property is mandatory;
# {REQREBOOT}	- require system reboot after adding this package? This property is
#                 optional;
# {RELOC}	- is this package relocatable; This property is optional;
# {DEPS}	- a list packages, within this product, that this package depended on;
#                 If a dependent package is not part of this product, it must be
#                 included in get_external_dependents(); This property is optional;
# {VRTSPATCHES}	- a list of patches to be installed with this package; This property is
#                 optional;
# {OPTIONAL}	- localized package description; This property is optional;
# {RESPONSEFILE}- response file content for this package; This property is optional;
#
# For example,
#
# $pkgs{VRTSfoo}{NAME} = "VERITAS Foo Package";
# $pkgs{VRTSfoo}{VERSION} = "1.2.3.4";
# $pkgs{VRTSfoo}{SPACE} = [100, 200, 100, 10];
# $pkgs{VRTSfoo}{DEPS} = ["VRTSbar"];
# $pkgs{VRTSfoo}{OPTIONAL} = "This is the foo package of product foobar.";
# $pkgs{VRTSfoo}{VRTSPATCHES} = ["123456", "434200"];
#
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the reference of the hash which contains the packages to be installed.
#
sub get_product_packages (\%$$$;$) {
  my ($self,$ctx,$pdver,$osver,$osarch,$mode) = @_;
  my (%PKGS, $val);
  my ($arrayref) = $PROD{$$ctx{TARGET}{UPI}}{PKGS};
  my ($fc) = '_' . lc($$ctx{TARGET}{UPI}) . '_comm_define_pkgs';

  $$ctx{LOGGER}->entering("ThunkAIX::get_product_packages");

  # Alreays adjust the globals using infos from the installation context before
  # we dive into the A stuff.
  $self->set_context_globals($ctx, $cfg);
  
  if (defined(&$fc)) {
    $$ctx{LOGGER}->entering("$fc");
    $arrayref = &$fc();
  }

  foreach $val (@{$arrayref}) {
    $PKGS{$val} = $PKGI{$val};
  }

  return \%PKGS;
}

#
# Return a list of patches for a given version of this product. The given 
# version may not be the current version. The reason is that for product upgrade 
# and product downgrade, VxIF needs a list of patches associated with a particular 
# version so they can properly removed.
#
# This interface is mandatory for patch-only install.
#
# This interface shall return the reference of a hash with patch IDs as the keys
# with the following properties.
#
# {NAME}    	- the patch ID; This property is mandatory;
# {PACKAGE}	- the package this patch is intended for; This property is mandatory;
# {SPACE}	- a list which specified the spaces, in kilobytes, needed for this
#                 patch in /opt, /usr, /var, and /; this property is mandatory;
# {REQREBOOT}	- require system reboot after applying this patch? This property is
#                 optional;
# {OBSOLETES} 	- the obsoletes patches; This property is optional;
# {INCOMPAT}	- incompatible patches; This property is optional;
# {REQUIRES}	- a list patches, within this product, that this patch depended on;
#                 If a dependent patch is not part of this product, it must be
#                 included in get_required_external_patches(); This property is optional;
# {DESC}	- localized patch description; This property is optional;
#
# Input: 1) the reference to the installation context;
#        2) the product version;
#        3) host OS version;
#        4) host OS architecture;
#        5) the selected installation mode if one is available;
#
# Return: the reference of the hash which contains the packages to be installed.
#
#sub get_product_patches (\%$$$;$) {
# TBD:
#}


################################################
# Package and Patch Installation Interfaces    #
################################################

#
# Get the content of the admin file for a given package or patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the package name or patch ID;
#        3) the name of the host for which this package will be installed;
#        4) host OS version;
#        5) host OS architecture;
#
# Return: the content of the admin file for the given package or patch ID.
#
#sub get_adminfile_content (\%$$$$) {
# TBD: this interface probably not needed.
#}

#
# Get the content of the response file for a given package or patch.
#
# This interface is optional.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the %cfg hash;
#        3) the package to be installed;
#        4) the name of the host for which this package will be installed;
#        5) host OS version;
#        6) host OS architecture;
#
# Return: the content of the response file for the given package or patch ID.
#
sub get_responsefile_content (\%\%$$$$) {
  my ($self,$ctx,$cfg,$pkg,$host,$osver,$osarch) = @_;
  
  $$ctx{LOGGER}->entering("ThunkAIX::get_responsefile_content");

  return Utils::vxif_desp($PKGI{$pkg}{RESPONSEFILE});
}

#
# For a given package, return the directory path, relative to the installation program,
# where this it is located.
#
# This interface is mandatory if the get_product_train_dir() interface has not been implemented.
# This interface is optional if the get_product_train_dir() interace has been implemented.
# By default, VxIF to look for packages in <TRAINDIR>/<pkgs | rpms>.
#
# Input: 1) the reference to the installation context;
#        2) package name;
#
# Return: Return the directory path where the given package is located.
#
#sub get_package_location (\%$) {
# For the A stuff, always use the train dir.
#}

###############################
# A subroutines need to be    #
# translated into C calls     #
###############################

#
# found in common/cpi.pl
#
sub _comm_cleanup {
  my ($exit,$sum)=@_;
  $$gctx{INSTALLERS}{AIX}->cleanup($gctx, 0, $exit, $sum);
}

#
# translate a message accordingly
#
sub _tr {
  Utils::_tr(@_);
}

# call a subroutine from the correct platform/product library
sub _prod_plat_sub {
  return if ($FLAG{NOPL});
  my($func,$lcpl,$lcu,$pf,$rtn);
  $func=shift;
  if (_list($func,@{$COMM{UPIS}})>=0) {
    $lcu=lc($func);
    $func=shift;
  } else {
    $lcu=$COMM{LCUPI};
  }
  $lcpl=$COMM{OSLIB}{$COMM{OS}};
  if (!$lcpl) {
    _die("Unsupported Platform: $COMM{OS}") 
  }
  $pf="_$lcu"."_$lcpl"."_$func";
  if (defined(&$pf)) {
    $rtn=&$pf(@_) 
  } elsif ($FLAG{DEBUG}) {
    _log("Skipping undefined subroutine: $pf");	
  }
  return $rtn;
}

#
# Get the package version on the target host.
#
sub _aix_get_instpkgvers { 
  $$gctx{INSTALLERS}{AIX}->get_instpkgvers($gctx, $PKG);
}

#
# Get the package version on the installation media.
#
sub _aix_get_mediapkgvers {
  $$gctx{INSTALLERS}{AIX}->get_mediapkgvers($gctx, $PKG, \%PKGI);
}

#vikas
# Get the package version on the target host.
#
sub _aix_get_mediapkginfovalue { 
  $$gctx{INSTALLERS}{AIX}->get_mediapkginfovalue($gctx, $PKG);
}

sub _comm_unload_drivers {
	my($cmd,$dr,$mn,$opt,$rd);
	for $dr(@_) {
		($cmd,$opt)=split(/\s/,$CMD{STOP}{$dr},2);
		_pb1("Checking $dr driver");
		$mn=_plat_sub("check_driver", $dr);
		if (!$mn) { 
			_pb2("not running");
			next;
		} elsif ($mn eq "?") {
			_pb2("$dr check command not installed");
			next;
		}
		_pb2("$dr module loaded");
		if ($OPT{PRECHECK}) {
			_ckmsg("The $dr driver is running on $SYS.  This driver will be stopped and unloaded during the installation process of $PROD{$UPI}{ABBR}");
			next;
		}
		_comm_ask_stop if (($OPT{CONFIGURE}) && (!$FLAG{STOPOK}));
		if (!$CMD{STOP}{$dr}) {
			$cmd = "$SUB{STOP}{$dr}";
			if (defined( &$cmd )) {
				&$cmd();
			} else {
				_log("$dr stop command not defined");
			}
		} elsif (!_fc($cmd)) {
			_log("$dr stop command $cmd not installed");
		} else {
			my ($txt, $res);
			_pb1("Stopping $dr driver");
			$rd = _dor($CMD{STOP}{$dr}, 2);
			($res, $txt) = (@$rd);
			if ($res) {
				_pb2("failed");
				_pl_notr($txt);
				_pl_notr($MSG{$dr}{STOPFAIL}) if ($MSG{$dr}{STOPFAIL});
				$FLAG{FUD}{$SYS} .= "$dr ";
				next;
			}
			sleep 3;
			_pb2("Done");
		}
		if ($FLAG{NOUNLOAD}{$dr}) {
			_pb1("Rechecking $dr driver");
		} else {
			_pb1("Unloading $dr module on $SYS");
			_plat_sub("unload_driver", $dr, $mn);
			sleep 3;
		}
		$mn=_plat_sub("check_driver", $dr);
		if (!$mn) { 
			_pb2("Done"); 
		} else {
			_pb2("$dr unload failed");
			$FLAG{FUD}{$SYS} .= "$dr ";
		}
	}
}

#vikas
sub _aix_create_responsefile {
	#so nothing. responsefile is already being generated.
}

#vikas
sub _comm_get_defaultnetmask {
        ($vip,$nic)=(@_);

        $$gctx{INSTALLERS}{AIX}->get_defaultnetmask($gctx,$vip,$nic);
}

#vikas
sub _plat_sub {
        my($func,$lcpl,$pf,$rtn);
        $func=shift;
        $lcpl=$COMM{OSLIB}{$COMM{OS}};
        if (!$lcpl) {
                _die("Unsupported Platform: $COMM{OS}")
        }
        $pf="_$lcpl"."_$func";
        if (defined(&$pf)) {
                $rtn=&$pf(@_)
        } elsif ($FLAG{DEBUG}) {
                _log("Skipping undefined subroutine: $pf");
        }
        return $rtn;
}

#vikas
# Check to see whether a driver is loaded on $SYS
# return module number if so
sub _aix_check_driver {
        my ($dr)=(@_);
        my $mn=_dor("$CMD{MODINFO} | $CMD{GREP} -w $dr | $CMD{AWK} '{print \$1}'");
        return "$mn";
}

#vikas
# unload a driver on $SYS
sub _aix_unload_driver {
        my ($dr,$mn)=(@_);
        _dor("$CMD{MODUNLOAD} -i $mn");
}

1;
