package CPIAIX;

use VxIF::CPICommon;
@CPIAIX::ISA = qw(CPICommon);

###############################
# Private Methods             #
###############################

###############################
# Public Methods       	      #
###############################

#
# Create an Installer object.
#
# Inputs: PRODUCT	=> the name of the product to install;
#         LOGGER 	=> reference to the Logger object; This is optional;
#         TITLE		=> product title; This is option; The default title is
#                          VERITAS Installer;
#         TMPPATH 	=> the complete to the tmp directory; This is optional;
#                          The default is /var/tmp;
#         INSTALLONLY 	=> install without configure; This is optional;
#                          The default is install and configure;
#         RESPONSEFILE	=> response file; This is optional; If a responsible is provided,
#                          the pre install and post install steps will be skipped;
#         USESSH	=> indicate whether to use SSH for remote installation; This is optional;
#                          The default is to use rsh;
#         SUPPORTEDOS	=> a list of supported OSs; This is optional; By default, the supported
#                          OSs are SunOS, Linux, HPUX, AIX, and OSF1;
#         CMDSEARCHPATHS=> a list of system paths to search for a shell command; This is optional;
#                          By default, the search paths are '/usr/bin', '/bin', '/usr/local/bin', 
#                          '/sbin', '/usr/sbin', and '/usr/local/sbin';
#         SPACEWARNING	=> warning the users of low on space if the number of bytes on the 
#                          volume after the installation is less than this value; This is optional;
#                          The default is 5120 kilobytes;
#
sub new {
	my $invocant = shift;
	my $class = ref($invocant) || $invocant;
	my $os = [qw(SunOS Linux HPUX AIX OSF1)];
	my $searchpaths = [qw(/usr/bin /bin /usr/local/bin /sbin /usr/sbin /usr/local/sbin)];
	my $self = { 	
		USESSH		=> 0,
		IPv6            => 0,
		TMPPATH		=> '/var/tmp/vxif',
		TIMEFILE	=> '/tmp/vxif_time',
		PKGSDIR		=> 'pkgs',
		PATCHSDIR	=> 'patches',
		OSINSTALLER	=> 'Aix',
		SPACEWARNING	=> '5120',
		SUPPORTEDOS	=> $os,
		CMDSEARCHPATHS	=> $searchpaths,
		@_,
	};
    
	return bless $self, $class;
}

#
# Set the common system commands.
#
sub set_commands {
  my ($self) = @_;
  
  $self->SUPER::set_commands();
  $self->{CMD}{DFK}="/usr/bin/df -kt";
  $self->{CMD}{GUNZIP}="/usr/bin/gunzip";
  $self->{CMD}{NETSTAT}="/usr/bin/netstat";
  $self->{CMD}{REX}="/opt/VRTScpi/bin/scripts/rex";
  $self->{CMD}{SHUTDOWN}="/usr/sbin/shutdown -r";
  
  # Define unique platform specific commands
  $self->{CMD}{INSTALLP}=$self->{CMD}{RMPOBJ}="/usr/sbin/installp";
  $self->{CMD}{LSDEV}="/usr/sbin/lsdev";
  $self->{CMD}{LSLPP}="/usr/bin/lslpp";
  $self->{CMD}{STRLOAD}="/usr/sbin/strload";
  $self->{CMD}{INSTALL}="/opt/VRTScpi/bin/scripts/install/install";
  
  # Begin Hack for beta of vcs
  $self->{CMD}{VEKIEXTADM}="/etc/methods/vekikextadm";
  $self->{CMD}{VXKEXTADM}="/etc/methods/vxkextadm";
  
  $self->{CMD}{UNLOAD}{vxfs}="$self->{CMD}{VXKEXTADM} unload";
  $self->{CMD}{UNLOAD}{veki}="$self->{CMD}{VEKIEXTADM} unload";
}

#
# Set message strings for package, packages
#
sub set_msgs {
  my ($self) = @_;
  
  $self->SUPER::set_msgs();
  $self->{MSG}{PDFREN}  = "fileset";
  $self->{MSG}{PDFR}  = Utils::_tr("fileset", 24, 1000);
  $self->{MSG}{PDFRS} = Utils::_tr("filesets", 24, 1001);
}

#
# Initialization.
#
# Input: 1) the reference of the installation context;
#
# Return: 1 if successfull; 0 otherwise.
#
sub initialize ($) {
  my($self,$context) = @_;
  
  $self->SUPER::initialize($context);
  
  $self->set_msgs();
  
  return 1;
}

#
# Add a product to this platform.
#
# Input: 1) the UPI;
#        2) the reference of the product object;
# 
sub add_product ($&) {
  my($self,$upi,$product) = @_;
  
  $self->{PRODUCT}{$upi} = $product;
}

#
# Add a target host to this platform.
#
# Input: 1) the target host name;
#        2) the reference of the installation context;
#
sub add_host (\%$) {
  my ($self,$ctx,$host) = @_;
  
  push(@{$self->{HOSTS}}, $host);
}

#
# For use during unzip, uncompress, or tar extraction.
# If someone has set ulimit, this is for that case.
#
# Input: 1) the reference of the installation context;
#
# Return: the command to (presumably) prepend
#
sub get_unlimit_filesize_cmd {
	my $self = shift;
	my ($ctx) = @_;
	my $cmd = "ulimit -f unlimited";

	return $cmd;
}

#
# Determine if a package is installed on a given host and return the version.
#
# Input: 1) the reference to the installation context;
#        2) the fileset;
#        3) extension;
#        4) the reference to the packages hash;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_instpkgvers (\%$;$) {
	my $self = shift;
	my ($ctx,$pkg,$ext,$pkgs) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my ($iv,@tmp);

	$$ctx{LOGGER}->entering("CPIAIX::get_instpkgvers");

	@pkgsnames = keys(%{$pkgs});

	$$ctx{LOGGER}->finer("pkg=${pkg}");

	return "" if !$pkg;

	if ((Utils::vxif_list($pkg, @pkgnames) >= 0) && $upi_inst->can("get_installed_package_version")) {
		return $upi_inst->get_installed_package_version($ctx, $pkg);
	}

	if (Utils::vxif_localsys($ctx)) {
		#
		# $PKGI{$PKGI{$pkg}{FILESET}}{EXT} assumes a global which may not be set
		# conditional hacks around the case of calling with an arg when global isn't
		# initiated
		# bleh.
		# 
		if ($ext eq "") {
			$iv = Utils::vxif_dol("$self->{CMD}{LSLPP} -L | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \$2}' 2>/dev/null");
		} else {
			$iv = Utils::vxif_dol("$self->{CMD}{LSLPP} -L | $self->{CMD}{GREP} $ext | $self->{CMD}{AWK} '{print \$2}' 2>/dev/null");
		}
	} else { 
		if ($ext eq "") {
			$iv = Utils::vxif_dor("\"$self->{CMD}{LSLPP} -L | $self->{CMD}{GREP} $pkg | $self->{CMD}{AWK} '{print \\\$2}'\" 2>/dev/null", "", $ctx);
		} else {
			$iv = Utils::vxif_dor("\"$self->{CMD}{LSLPP} -L | $self->{CMD}{GREP} $ext | $self->{CMD}{AWK} '{print \\\$2}'\" 2>/dev/null", "", $ctx);
		}
	}

	$$ctx{LOGGER}->finer("iv=$iv");

	@tmp = split "\n", $iv;
	return $tmp[0];
}

#
# Determine the version number of the package on the installation media.
#
# Input: 1) the reference to the installation context;
#        2) the native package name;
#        3) the reference to the packages hash;
#        4) extension;
#
# Return: the version number, or null ("") if unable to determine the version number.
#
sub get_mediapkgvers (\%$\%;$) {
	my $self = shift;
	my ($ctx,$pkg,$pkgs,$ext) = @_;
	my ($line, $name, $garbage, $pp, $remainder, $vers, $vtmp, $tocl, @toc);
	my $package_info = $pkgs->{$pkg};
	my $fileset = $package_info->search_fileset($pkg);
	
	$$ctx{LOGGER}->entering("CPIAIX::get_mediapkgvers");
	
	$pp = $self->{$$ctx{TARGET}{UPI}}{$pkg}{TLOC};
	
	#
	# use .toc file on CD for gzipped packages
	#
	if ( (-e "$pp/.toc") && (-e "$pp/${pkg}.tar.gz" && $ext )) {
		$tocl = _dol("$self->{CMD}{GREP} $ext $pp/.toc | $self->{CMD}{GREP} -v usr | $self->{CMD}{GREP} -v opt | $self->{CMD}{GREP} -v prereq | $self->{CMD}{GREP} -v coreq 2>/dev/null" );
		@toc = split "\n", $tocl;
		($garbage, $vtmp, $remainder) = split " ", $toc[1];
		$vers = sprintf("%d.%d.%d.%d", split (/\./, $vtmp));
	} else {
		$$ctx{LOGGER}->finer("pp=$pp\nfileset=$fileset");
		$line = Utils::vxif_dol("$self->{CMD}{INSTALLP} -L -d $pp/$fileset 2> /dev/null");
		$$ctx{LOGGER}->finer("line=$line");
		($name, $garbage, $vers, $remainder) = split /:/, $line, 4;
		$$ctx{LOGGER}->finer("name=$name\n$garbage=$garbage\nvers=$vers\nremainder=$remainder");
	}
    
	return $vers;
}

#
# Determine size of the given package on the media. If the package is compressed, we are assuming
# the compression of 3x.
#
# Input: 1) the reference of the installation context;
#        2) the reference of the packages hash;
#        3) the fileset name;
#        4) the location of the given package;
#        5) indicating whether the given package is in a compressed file;
#        6) the compressed filename;
#
# Return: the size of the package in kilobytes.
#
sub get_mediapkgsize {
	my $self = shift;
	my ($ctx,$pkgs,$pkg,$loc,$compressed,$filename) = @_;
	my $local_host = $ctx->get_local_hostname();
	my $size = 0;

	$$ctx{LOGGER}->entering("CPIAIX::get_mediapkgsize");

	if ($compressed) {
		$$ctx{LOGGER}->fine(Utils::_tr("${pkg} is compressed", 24, 1002, "${pkg}"));
		if (-f "${loc}/${filename}") {
			$size = Utils::vxif_filesize_in_k("${loc}/${filename}");
			# assume compression of 3x. multiple size by 4 to account for the compressed package itself
			$size *= 4;
		} else {
			$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${pkg} because ${loc}/${filename} does not exist on ${local_host}.", 24, 1003, "${pkg}", "${loc}", "${filename}", "${local_host}"));
			return 0;
		}
	} else {
		my $imagefile = $$pkgs{$pkg}{IMAGE};
		$$ctx{LOGGER}->finer("$pkg image = $imagefile\nlocation = $loc");
		$$ctx{LOGGER}->fine(Utils::_tr("${imagefile} is uncompressed", 24, 1004, "${imagefile}"));
		if (-f "${loc}/${imagefile}") {
			$size = Utils::vxif_filesize_in_k("${loc}/${imagefile}");
		} else {
			$$ctx{LOGGER}->warning(Utils::_tr("Unable to determine the size of ${imagefile} because ${loc}/${imagefile} does not exist on ${local_host}.", 24, 1003, "${imagefile}", "${loc}", "${imagefile}", "${local_host}"));
			return 0;
		}
	}

	$$ctx{LOGGER}->info(Utils::_tr("Size of ${pkg} image is ${size} kbs", 24, 1005, "${pkg}", "${size}"));

	return $size;
}

#
# Determine if a package has any dependencies on target host.
#
# Input: 1) the reference to the installation context;
#        2) the fileset;
#
# Return: a string of dependent packages or null if there are no dependents.
#
sub get_pkgdep {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my ($dep,$deplist,$deppkgs,@deps,@ideps);
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);

	$$ctx{LOGGER}->entering("CPIAIX::get_pkgdep");

	if ($upi_inst->can("check_for_package_dependents")) {
		$deppkgs = $upi_inst->check_for_package_dependents($ctx, $pkg);
		@deps = keys(%{$deppkgs});
		$deplist = join(" ", @deps);
		$deplist .= " ";
	}


	# Just do the lslpp command remotely.  Do the rest in perl.
	my $dep_href = {};
	my $dep_cmd = "$self->{CMD}{LSLPP} -d -c '${pkg}*'";
	my $dep_outp = Utils::vxif_dor($dep_cmd, "", $ctx);
	my $dep_lines = [ split(/\n/, $dep_outp) ];
	for my $dep_line (@$dep_lines) {
		$dep_line = Utils::vxif_desp($dep_line);
		$dep_line =~ s/[\r\n]+$//;
		if ($dep_line =~ /${pkg}/) {
			my @fields = split(/\s*[:\s]\s*/, $dep_line);
			my $dep_pkg = $fields[3];
			if ($dep_pkg ne "NONE") {
				$dep_href->{$dep_pkg} = 1;
			}
		}
	}
	$dep = join("\n", sort(keys(%{$dep_href})));

	$$ctx{LOGGER}->finer("dep=$dep");
	@ideps = split(" ", $dep);
	foreach $dep (@ideps) {
		if (Utils::vxif_list($dep, $deps) < 0) {
			$deplist .= "$dep ";
		}
	}

	my $desp_deplist = Utils::vxif_desp($deplist);

	return $desp_deplist;
}

#
# Determine the netmask of the base IP configured on a NIC.
#
# Input: 1) the reference to the installation context;
#        2) the NIC;
#
# Return: the netmask.
#
sub get_netmask (\%$) {
  my ($self,$ctx,$nic) = @_;
  my ($nm);
  
  $$ctx{LOGGER}->entering("CPIAIX::get_netmask");
  
  $nm = Utils::vxif_dor("$self->{CMD}{IFCONFIG} $nic | $self->{CMD}{GREP} netmask | $self->{CMD}{AWK} '{print \$4}' | $self->{CMD}{AWK} -Fx '{print \$2}' 2>/dev/null", "", $ctx);
  
  $$ctx{LOGGER}->finer("nm=$nm");
  return "$nm";
}

#
# Determine the NICs on target host which have IP addresses configured on them
# return a reference to an array holding the NICs
#
# Input: 1) the reference to the installation context;
#
# Return: a reference to an array holding the NICs.
#
sub get_publicnics (\%) {
  my ($self,$ctx) = @_;
  my (@pnics,$do);
  
  $$ctx{LOGGER}->entering("CPIAIX::get_publicnics");
  
  $do = Utils::vxif_dor("$self->{CMD}{NETSTAT} -i | $self->{CMD}{GREP} '^en' | $self->{CMD}{SED} 's/:.*\$//' | $self->{CMD}{AWK} '{print \$1}' | $self->{CMD}{UNIQ} 2>/dev/null", "", $ctx);
  $$ctx{LOGGER}->finer("do=$do");
  @pnics=split(/\s+/,$do);
  
  return \@pnics;
}

#
# Check whether a patch is installed on the installation host.
#
# Input: 1) UPI;
#        2) the patch name;
#        3) the reference to the installation context;
#        4) the reference to the installed patch;
#        5) indicate whether to refresh the current patches list;
#
# Return: -1 if the patch ID is invalid;
#         0 if the patch does not exist;
#         1 if older version is installed; 
#         2 if version or higher is installed.       
#
sub check_patch ($$$$$) {
  my ($self,$upi,$patch,$ctx,$installedpatch,$refresh) = @_;
  my ($pn,$pv,$iv);
  my ($rc) = 0;
  
  # stub this out for now until we implement the patch support.
  
  return $rc;
}

#
# Get all the patches for the given host.
#
# Input: 1) the reference of the installation context;
#
# Return: the reference to the patches hash.
#
sub get_patches ($) {
  my ($self,$ctx) = @_;
  my (%ph,@a,$j,$p,$s,$v);
  
  $$ctx{LOGGER}->entering("CPIAIX::get_patches");
    
  $s = Utils::vxif_dor("$self->{CMD}{LSLPP} -qLc", "", $ctx);
  @a = split(/^/,$s);
  foreach $s (@a) {
    ($j,$p,$v,$j) = split(/:/,$s,4);
    $ph{$p} = $v;
  }
  
  return \%ph;
}

#
# Check for and run post pkgadd operations.
#
# Input: 1) UPI;
#        2) package name;
#        3) the reference to the installation context;
#
# Return: 1 if successful; 0 otherwise.
#
sub check_os_patch_dependencies ($$$) {
	my $self = shift;
	my ($upi,$pkg,$ctx) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my (@patchlist,$patch,$patchid);
	my ($rc,$installedpatchid,$msgl,$msgr);
	my ($result) = 1;

	if ($upi_inst->can("get_required_os_patches")) {
		@patchlist = $upi_inst->get_required_os_patches($pkg, $ctx);
	}
	
	if ($#patchlist >= 0 ) {
		Utils::vxif_pl(Utils::_tr("Package, $pkg, require one or more OS patches.", 24, 1006, "$pkg"));
		  foreach $patch (@patchlist) {
			  $patchid = $patch;
			  ($upi_inst->can("get_patch_id")) && ($patchid = $upi_inst->get_patch_id($patch, $ctx));
			  $msgl = Utils::_tr("Checking OS patch ${patchid}", 24, 1007, "${patchid}");
			  Utils::vxif_pbl($msgl);
			  
			  $rc = $self->check_patch($upi, $pkg, $ctx, \$installedpatchid, "");
			  if ($rc == -1) {
				  $msgr = Utils::_tr("invalid patch ID, ${patchid}", 24, 1008, "${patchid}");
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } elsif ($rc == 0) {
				  $msgr = Utils::_tr("not installed", 24, 1009);
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } elsif ($rc == 1) {
				  $msgr = Utils::_tr("${installedpatchid} installed", 24, 1010, "${installedpatchid}");
				  Utils::vxif_pbr($msgl, $msgr);
				  $result = 0;
			  } else {
				  $msgr = Utils::_tr("${installedpatchid} installed", 24, 1010, "${installedpatchid}");
				  Utils::vxif_pbr($msgl, $msgr);
			  }
		  }
	  }
	
	return 1;
}

#
# Check for pkgadd errors.
#
# Input: 1) the reference to the installation context;
#        2) the fileset installed;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_installerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
	my ($error);
	my (@s,$if);
	my ($name,$level,$part,$event,$result);

	$$ctx{LOGGER}->entering("CPIAIX::pkg_installerror");
	my $local_cmds = $ctx->get_local_cmds();
	my $local_grep_cmd = $local_cmds->{GREP};
	# maybe should redirect stderr to /dev/null
	$if = Utils::vxif_dol("${local_grep_cmd} '^$pkg' $logfile");
	$$ctx{LOGGER}->finer("if=${if}");
	@s=split(/\n/,$if);
	foreach $if(@s) { 
		($name,$level,$part,$event,$result) = split(/\s+/, $if, 5);
		$name = Utils::vxif_desp($name);
		$result = Utils::vxif_desp($result);
		$$ctx{LOGGER}->finer("name=${name}\nresult=${result}");

		# the internal name of the installed package(s) may be different from $pkg which might be a single fileset.
		# e.g. VRTSaz.image has VRTSaz.server and VRTSaz.client. so removing a strict match.
		#return 0 if (($name eq $pkg) && ($result eq "SUCCESS"));
		return 0 if (($name =~ /$pkg/) && ($result eq "SUCCESS"));
	}

	return 1;
}

#
# Check for pkgrm errors.
#
# Input: 1) the reference to the installation context;
#        2) the fileset uninstalled;
#
# Return: 0 if no error; 1 otherwise.
#
sub pkg_uninstallerror (\%$) {
	my $self = shift;
	my ($ctx,$pkg) = @_;
  
	return $self->pkg_installerror($ctx, $pkg);
}

#
# Add the native package to the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the fileset to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub install_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $installp_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
	my $trg_upi = $ctx->get_target_upi();
	my $package_info = $pkgs->{$pkg};
	my $imagefile = $package_info->search_image($pkg);
	my $filesets = $package_info->search_fileset();

	# if fileset is not specified, assume the package name is the same as the fileset name
	$filesets = $pkg if (!$filesets);
	
	# check to make sure the given package is not already exist on the target host
	my $result = $self->check_for_existing_package($ctx, $pkg, $pkgs);
	if (0 == $result) {
		return 0;
	} elsif (2 == $result) {
		return 1;
	}

	#
	# run cleanup in case of failed prior install
	#
	my $cleanup_cmd = "$self->{CMD}{INSTALLP} -C";
	Utils::vxif_dor($cleanup_cmd, "", $ctx, undef, $installp_logfile);

	$logger->finer("$pkg image = $imagefile");

	my $xloc;
	if (Utils::vxif_localsys($ctx) && !$self->{${trg_upi}}{$pkg}{COMPRESSED}) {
		# if local install, and package not compressed (or archived),
		# install directly from source location.
		$xloc = "SLOC";
	} else {
		$xloc = "TLOC";
	}

	# The package itself may not be at the top of the "pkgadd" directory.
	# It may, rather, be one or more levels down.  Use the EXTRACTION_PATH
	# configuration variable (if present) to see where to dig.
	#
	# maybe put this in common code
	my $install_dir = $self->{$trg_upi}{$pkg}{$xloc};
	my $installer_info = $ctx->get_installer_info();
	my $trg_os = $ctx->get_host_os_name($trg_host);
	my $pkginfo = $installer_info->get_platform_package_info($trg_os, $pkg);
	my $extra_path = $pkginfo->get_extraction_path();
	if ($extra_path) {
		$install_dir .= "/${extra_path}";
	}

	# now install the package
	#my $install_cmd = "$self->{CMD}{INSTALLP} -aXd ${install_dir}/${imagefile} ${pkg}";
	my $install_cmd = "$self->{CMD}{INSTALLP} -aXd ${install_dir}/${imagefile} ${filesets}";
	Utils::vxif_dor($install_cmd, "", $ctx, undef, $installp_logfile);

	return $self->is_package_install_successful($ctx, $pkg, $pkgs);
}

#
# Remove the native package from the target host.
#
# Input: 1) the reference to the installation context;
#        2) the reference to the packages hash;
#        3) the fileset to install;
#
# Return: 1 if successful; 0 otherwise.
#
sub uninstall_native_package (\%\%$) {
	my $self = shift;
	my ($ctx,$pkgs,$pkg) = @_;
	my $logger = $ctx->get_logger();
	my $logdir = $logger->get_logdir();
	my $trg_host = $ctx->get_target_hostname();
	my $installp_logfile = "${logdir}/$$ctx{TASK}.${pkg}.${trg_host}";
	my ($iver);
	
	#
	# run cleanup in case of failed prior install
	# 
	Utils::vxif_dor("$self->{CMD}{INSTALLP} -C", "", $ctx, undef, $installp_logfile);
	
	# now remove the package
	Utils::vxif_dor("$self->{CMD}{INSTALLP} -u $pkg", "", $ctx, undef, $installp_logfile);
    
	# if the package is successfully uninstalled, the version number null.
	if ((!$self->pkg_uninstallerror($ctx, $pkg))&&($self->get_instpkgvers($ctx, $pkg) eq "")) {
		$$ctx{LOGGER}->info(Utils::_tr("${pkg} successfully uninstalled from ${trg_host}", 24, 1011, "${pkg}", "${trg_host}"));
		return 1;
	}
	
	$$ctx{LOGGER}->info(Utils::_tr("Unable to uninstall ${pkg} from ${trg_host}", 24, 1012, "${pkg}", "${trg_host}"));
	return 0;
}

#
# Check to make sure all the external dependencies are satisfied.
#
# Input: 1) the reference to the installation context;
#
# Return: 1 if everything is happy; 0 otherwise.
#
sub check_for_external_dependencies (\%) {
	my $self = shift;
	my ($ctx) = @_;

	# (optimistically) stub this out for now
	return 1;
}

#
# Get the native package name for the given package.
#
# Input: 1) reference to the installation context;
#        2) the package name;
#        3) reference the the packages hash;
#
# Return: the native package name.
#
sub get_package_name (\%$\%) {
  my ($self,$ctx,$pkg,$pkgs) = @_;
  
  return $pkg;
}

sub get_native_package_file (\%$\$\%) {
  my ($self,$ctx,$pkg,$fod,$pkgs) = @_;
  
  $$fod = "f";
  my $package_info = $pkgs->{$pkg};
  my $filename = $package_info->search_image($pkg);

  if (!$filename) {
	  $filename = $package_info->search_fileset($pkg);
  }

  return $filename;
}


sub get_subpackage_list {
	my $this = shift;
	my ($ctx, $pkg) = @_;
	my $trg_host = $ctx->get_target_hostname();
	my $upi_inst = $ctx->get_host_os_upi_installer($trg_host);
	my $product_info = $upi_inst->obtain_product_info($ctx);
	my $package_info = $product_info->get_package_info($pkg);
	my $subpkgs = $package_info->get_fileset();

	return $subpkgs;
}

1;

